<?php
defined('BASEPATH') or exit('No direct script access allowed');

class Reporting_dashboard_permissions_model extends App_Model
{
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Get staff permission for reporting dashboard
     * @param int $staff_id
     * @return array ['type' => string, 'allowed_staff_ids' => array]
     */
    public function get_staff_permission($staff_id)
    {
        // Admin always has view_all_staff permission
        if (is_admin($staff_id)) {
            return [
                'type' => 'view_all_staff',
                'allowed_staff_ids' => []
            ];
        }

        $this->db->where('staff_id', $staff_id);
        $permission = $this->db->get(db_prefix() . 'reporting_dashboard_permissions')->row();
        
        if ($permission) {
            $allowed_staff_ids = [];
            if ($permission->permission_type === 'view_specific_staff' && !empty($permission->allowed_staff_ids)) {
                $allowed_staff_ids = json_decode($permission->allowed_staff_ids, true) ?: [];
            }
            
            return [
                'type' => $permission->permission_type,
                'allowed_staff_ids' => $allowed_staff_ids
            ];
        }
        
        return [
            'type' => 'view_own_only',
            'allowed_staff_ids' => []
        ];
    }

    /**
     * Set staff permission for reporting dashboard
     * @param int $staff_id
     * @param string $permission_type 'view_all_staff', 'view_own_only', or 'view_specific_staff'
     * @param int $granted_by
     * @param array $allowed_staff_ids Required when permission_type is 'view_specific_staff'
     * @param bool $can_generate_reports Permission to generate and schedule staff reports
     * @return bool
     */
    public function set_staff_permission($staff_id, $permission_type, $granted_by, $allowed_staff_ids = [], $can_generate_reports = false)
    {
        // Validate permission type
        if (!in_array($permission_type, ['view_all_staff', 'view_own_only', 'view_specific_staff'])) {
            return false;
        }

        // Validate allowed_staff_ids for specific staff permission
        if ($permission_type === 'view_specific_staff') {
            if (empty($allowed_staff_ids) || !is_array($allowed_staff_ids)) {
                return false;
            }
            // Ensure all IDs are numeric
            $allowed_staff_ids = array_filter(array_map('intval', $allowed_staff_ids));
            if (empty($allowed_staff_ids)) {
                return false;
            }
        }

        // Check if permission already exists
        $this->db->where('staff_id', $staff_id);
        $existing = $this->db->get(db_prefix() . 'reporting_dashboard_permissions')->row();

        $data = [
            'staff_id' => $staff_id,
            'permission_type' => $permission_type,
            'allowed_staff_ids' => $permission_type === 'view_specific_staff' ? json_encode($allowed_staff_ids) : null,
            'can_generate_reports' => $can_generate_reports ? 1 : 0,
            'granted_by' => $granted_by,
            'updated_at' => date('Y-m-d H:i:s')
        ];

        if ($existing) {
            // Update existing permission
            $this->db->where('staff_id', $staff_id);
            return $this->db->update(db_prefix() . 'reporting_dashboard_permissions', $data);
        } else {
            // Insert new permission
            $data['created_at'] = date('Y-m-d H:i:s');
            return $this->db->insert(db_prefix() . 'reporting_dashboard_permissions', $data);
        }
    }

    /**
     * Get all staff with their permissions
     * @return array
     */
    public function get_all_staff_permissions()
    {
        $this->db->select('s.staffid, s.firstname, s.lastname, s.email, p.permission_type, p.allowed_staff_ids, p.created_at, p.updated_at, 
                          gb.firstname as granted_by_firstname, gb.lastname as granted_by_lastname');
        $this->db->from(db_prefix() . 'staff s');
        $this->db->join(db_prefix() . 'reporting_dashboard_permissions p', 's.staffid = p.staff_id', 'left');
        $this->db->join(db_prefix() . 'staff gb', 'p.granted_by = gb.staffid', 'left');
        $this->db->where('s.active', 1);
        $this->db->order_by('s.firstname', 'ASC');
        
        $result = $this->db->get()->result_array();
        
        // Set default permissions for staff without explicit permissions and decode allowed_staff_ids
        foreach ($result as &$staff) {
            if (empty($staff['permission_type'])) {
                $staff['permission_type'] = is_admin($staff['staffid']) ? 'view_all_staff' : 'view_own_only';
                $staff['is_default'] = true;
                $staff['allowed_staff_ids'] = null;
            } else {
                $staff['is_default'] = false;
                if ($staff['permission_type'] === 'view_specific_staff' && !empty($staff['allowed_staff_ids'])) {
                    $staff['allowed_staff_ids'] = json_decode($staff['allowed_staff_ids'], true);
                } else {
                    $staff['allowed_staff_ids'] = null;
                }
            }
        }
        
        return $result;
    }

    /**
     * Remove staff permission (revert to default)
     * @param int $staff_id
     * @return bool
     */
    public function remove_staff_permission($staff_id)
    {
        $this->db->where('staff_id', $staff_id);
        return $this->db->delete(db_prefix() . 'reporting_dashboard_permissions');
    }

    /**
     * Get filtered staff IDs based on current user's permission
     * @param int $current_staff_id
     * @return array Empty array means all staff, populated array means specific staff only
     */
    public function get_allowed_staff_ids($current_staff_id)
    {
        $permission = $this->get_staff_permission($current_staff_id);
        
        switch ($permission['type']) {
            case 'view_all_staff':
                return []; // Empty array means all staff
            
            case 'view_specific_staff':
                $allowed_ids = $permission['allowed_staff_ids'];
                // Always include the current staff member's own ID
                if (!in_array($current_staff_id, $allowed_ids)) {
                    $allowed_ids[] = $current_staff_id;
                }
                return $allowed_ids;
            
            case 'view_own_only':
            default:
                return [$current_staff_id]; // Only current staff
        }
    }

    /**
     * Check if current staff can view all staff data
     * @param int $staff_id
     * @return bool
     */
    public function can_view_all_staff($staff_id)
    {
        $permission = $this->get_staff_permission($staff_id);
        return $permission['type'] === 'view_all_staff';
    }

    /**
     * Check if current staff can generate and schedule reports
     * @param int $staff_id
     * @return bool
     */
    public function can_generate_reports($staff_id)
    {
        // Admins always can generate reports
        if (is_admin($staff_id)) {
            return true;
        }

        $this->db->where('staff_id', $staff_id);
        $permission = $this->db->get(db_prefix() . 'reporting_dashboard_permissions')->row();
        
        if ($permission) {
            return (bool)$permission->can_generate_reports;
        }
        
        return false; // Default: no report generation permission
    }

    /**
     * Get available staff options for specific staff permission
     * @param int $exclude_staff_id Staff ID to exclude from the list (usually the staff being configured)
     * @return array
     */
    public function get_available_staff_for_permission($exclude_staff_id = null)
    {
        $this->db->select('staffid, firstname, lastname, email');
        $this->db->from(db_prefix() . 'staff');
        $this->db->where('active', 1);
        if ($exclude_staff_id) {
            $this->db->where('staffid !=', $exclude_staff_id);
        }
        $this->db->order_by('firstname', 'ASC');
        
        return $this->db->get()->result_array();
    }

    /**
     * Get staff names by IDs
     * @param array $staff_ids
     * @return array
     */
    public function get_staff_names_by_ids($staff_ids)
    {
        if (empty($staff_ids)) {
            return [];
        }

        $this->db->select('staffid, firstname, lastname');
        $this->db->from(db_prefix() . 'staff');
        $this->db->where_in('staffid', $staff_ids);
        $this->db->where('active', 1);
        
        $result = $this->db->get()->result_array();
        
        $names = [];
        foreach ($result as $staff) {
            $names[$staff['staffid']] = $staff['firstname'] . ' ' . $staff['lastname'];
        }
        
        return $names;
    }
}
?>
