<?php defined('BASEPATH') or exit('No direct script access allowed');

class Reporting_dashboard extends AdminController
{
    public function __construct()
    {
        parent::__construct();
        
        // Load models using CodeIgniter approach
        $this->load->model('reporting_dashboard/reporting_dashboard_model');
        $this->load->model('reporting_dashboard/reporting_dashboard_permissions_model');
        $this->load->model('staff_model');
        
        // Check if user is logged in
        if (!is_staff_logged_in()) {
            redirect(admin_url('authentication'));
        }
    }
    
    /**
     * Check if user has access to reporting dashboard and get allowed staff IDs
     */
    private function check_access_and_get_filters()
    {
        // Everyone can access the dashboard, but data is filtered based on permissions
        $current_staff_id = get_staff_user_id();
        
        // Direct database query for permissions
        $this->db->select('*');
        $this->db->from(db_prefix() . 'reporting_dashboard_permissions');
        $this->db->where('staff_id', $current_staff_id);
        $permission_query = $this->db->get();
        
        $allowed_staff_ids = [$current_staff_id]; // default: own only
        $can_view_all = false;
        
        if ($permission_query->num_rows() > 0) {
            $permission = $permission_query->row();
            if ($permission->permission_type === 'view_all_staff') {
                $allowed_staff_ids = [];
                $can_view_all = true;
            } elseif ($permission->permission_type === 'view_specific_staff') {
                $allowed_staff_ids = json_decode($permission->allowed_staff_ids, true) ?: [];
                if (!in_array($current_staff_id, $allowed_staff_ids)) {
                    $allowed_staff_ids[] = $current_staff_id;
                }
            }
        }
        
        // Admins always can view all
        if (is_admin($current_staff_id)) {
            $allowed_staff_ids = [];
            $can_view_all = true;
        }
        
        return [
            'has_access' => true,
            'allowed_staff_ids' => $allowed_staff_ids,
            'can_view_all' => $can_view_all
        ];
    }

    public function index()
    {
        if (!is_staff_logged_in()) {
            redirect(admin_url('authentication'));
        }
        
        $data['title'] = 'Reporting Dashboard';
        
        // Get current user info
        $current_staff_id = get_staff_user_id();
        $data['user_staff_id'] = $current_staff_id;
        
        // Check permissions using direct database queries (bypassing model issues)
        $permission_type = 'view_own_only'; // default
        $allowed_staff_ids = [$current_staff_id]; // default
        
        // Check if user has custom permissions
        $this->db->select('*');
        $this->db->from(db_prefix() . 'reporting_dashboard_permissions');
        $this->db->where('staff_id', $current_staff_id);
        $permission_query = $this->db->get();
        
        if ($permission_query->num_rows() > 0) {
            $permission = $permission_query->row();
            $permission_type = $permission->permission_type;
            
            if ($permission_type === 'view_specific_staff') {
                $allowed_staff_ids = json_decode($permission->allowed_staff_ids, true) ?: [];
                // Always include current user
                if (!in_array($current_staff_id, $allowed_staff_ids)) {
                    $allowed_staff_ids[] = $current_staff_id;
                }
            } elseif ($permission_type === 'view_all_staff') {
                $allowed_staff_ids = []; // empty means all
            }
        }
        
        // Admins always have view_all_staff permission
        if (is_admin($current_staff_id)) {
            $permission_type = 'view_all_staff';
            $allowed_staff_ids = [];
        }
        
        // Get staff list based on permissions
        $this->db->select('staffid, firstname, lastname, email, admin, active');
        $this->db->from(db_prefix() . 'staff');
        $this->db->where('active', 1);
        
        if (!empty($allowed_staff_ids)) {
            // Filter to specific staff only
            $this->db->where_in('staffid', $allowed_staff_ids);
        }
        
        $staff_query = $this->db->get();
        $data['staff_list'] = $staff_query->result_array();
        
        $data['can_view_all_staff'] = ($permission_type === 'view_all_staff');
        
        // Debug logging
        log_message('debug', 'Reporting Dashboard - Current staff ID: ' . $current_staff_id);
        log_message('debug', 'Reporting Dashboard - Permission type: ' . $permission_type);
        log_message('debug', 'Reporting Dashboard - Allowed staff IDs: ' . json_encode($allowed_staff_ids));
        log_message('debug', 'Reporting Dashboard - Staff list count: ' . count($data['staff_list']));
        
        // Add permission management link for admins
        if (is_admin()) {
            $data['show_permission_link'] = true;
        }
        
        $this->load->view('dashboard', $data);
    }

    public function get_kpis()
    {
        // Check access permissions
        $access_info = $this->check_access_and_get_filters();
        if (!$access_info['has_access']) {
            header('HTTP/1.1 403 Forbidden');
            echo json_encode(['error' => 'Access denied']);
            return;
        }
        
        try {
            $filters = $this->get_filters();
            $kpis = $this->reporting_dashboard_model->get_kpis($filters);
            
            header('Content-Type: application/json');
            echo json_encode($kpis);
        } catch (Exception $e) {
            error_log('Reporting Dashboard KPI Error: ' . $e->getMessage());
            error_log('Reporting Dashboard KPI Error Trace: ' . $e->getTraceAsString());
            
            header('Content-Type: application/json');
            http_response_code(500);
            echo json_encode([
                'error' => true,
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
        } catch (Error $e) {
            error_log('Reporting Dashboard KPI Fatal Error: ' . $e->getMessage());
            error_log('Reporting Dashboard KPI Fatal Error Trace: ' . $e->getTraceAsString());
            
            header('Content-Type: application/json');
            http_response_code(500);
            echo json_encode([
                'error' => true,
                'message' => 'Fatal error: ' . $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
        }
    }

    public function get_staff_activity_chart()
    {
        // Check access permissions
        $access_info = $this->check_access_and_get_filters();
        if (!$access_info['has_access']) {
            header('HTTP/1.1 403 Forbidden');
            echo json_encode(['error' => 'Access denied']);
            return;
        }
        
        try {
            $filters = $this->get_filters();
            $data = $this->reporting_dashboard_model->get_staff_activity_chart($filters);
            header('Content-Type: application/json');
            echo json_encode($data);
        } catch (Exception $e) {
            error_log('Reporting Dashboard Staff Chart Error: ' . $e->getMessage());
            header('Content-Type: application/json');
            http_response_code(500);
            echo json_encode([
                'error' => true,
                'message' => $e->getMessage()
            ]);
        }
    }

    public function get_activity_over_time_chart()
    {
        // Check access permissions
        $access_info = $this->check_access_and_get_filters();
        if (!$access_info['has_access']) {
            header('HTTP/1.1 403 Forbidden');
            echo json_encode(['error' => 'Access denied']);
            return;
        }
        
        try {
            $filters = $this->get_filters();
            $data = $this->reporting_dashboard_model->get_activity_over_time_chart($filters);
            header('Content-Type: application/json');
            echo json_encode($data);
        } catch (Exception $e) {
            error_log('Reporting Dashboard Activity Over Time Error: ' . $e->getMessage());
            header('Content-Type: application/json');
            http_response_code(500);
            echo json_encode([
                'error' => true,
                'message' => $e->getMessage()
            ]);
        }
    }

    public function get_activity_type_pie()
    {
        // Check access permissions
        $access_info = $this->check_access_and_get_filters();
        if (!$access_info['has_access']) {
            header('HTTP/1.1 403 Forbidden');
            echo json_encode(['error' => 'Access denied']);
            return;
        }
        
        try {
            $filters = $this->get_filters();
            $data = $this->reporting_dashboard_model->get_activity_type_pie($filters);
            header('Content-Type: application/json');
            echo json_encode($data);
        } catch (Exception $e) {
            error_log('Reporting Dashboard Activity Type Pie Error: ' . $e->getMessage());
            header('Content-Type: application/json');
            http_response_code(500);
            echo json_encode([
                'error' => true,
                'message' => $e->getMessage()
            ]);
        }
    }

    public function get_activity_table()
    {
        // Check access permissions
        $access_info = $this->check_access_and_get_filters();
        if (!$access_info['has_access']) {
            header('HTTP/1.1 403 Forbidden');
            echo json_encode(['error' => 'Access denied']);
            return;
        }
        
        try {
            $filters = $this->get_filters();
            $data = $this->reporting_dashboard_model->get_activity_table($filters);
            header('Content-Type: application/json');
            echo json_encode($data);
        } catch (Exception $e) {
            error_log('Reporting Dashboard Activity Table Error: ' . $e->getMessage());
            header('Content-Type: application/json');
            http_response_code(500);
            echo json_encode([
                'error' => true,
                'message' => $e->getMessage()
            ]);
        }
    }

    public function get_followup_types_chart()
    {
        // Check access permissions
        $access_info = $this->check_access_and_get_filters();
        if (!$access_info['has_access']) {
            header('HTTP/1.1 403 Forbidden');
            echo json_encode(['error' => 'Access denied']);
            return;
        }
        
        try {
            $filters = $this->get_filters();
            $data = $this->reporting_dashboard_model->get_followup_types_chart($filters);
            header('Content-Type: application/json');
            echo json_encode($data);
        } catch (Exception $e) {
            error_log('Reporting Dashboard Follow-up Types Error: ' . $e->getMessage());
            header('Content-Type: application/json');
            http_response_code(500);
            echo json_encode([
                'error' => true,
                'message' => $e->getMessage()
            ]);
        }
    }

    public function get_reminders()
    {
        // Check access permissions
        $access_info = $this->check_access_and_get_filters();
        if (!$access_info['has_access']) {
            header('HTTP/1.1 403 Forbidden');
            echo json_encode(['error' => 'Access denied']);
            return;
        }
        
        try {
            $filters = $this->get_filters();
            
            // Build reminder data with proper names
            $data = $this->get_reminders_with_names($filters, $access_info['allowed_staff_ids']);
            
            header('Content-Type: application/json');
            echo json_encode($data);
        } catch (Exception $e) {
            error_log('Reporting Dashboard Reminders Error: ' . $e->getMessage());
            header('Content-Type: application/json');
            http_response_code(500);
            echo json_encode([
                'error' => true,
                'message' => $e->getMessage()
            ]);
        }
    }
    
    private function get_reminders_with_names($filters, $allowed_staff_ids)
    {
        $today = date('Y-m-d');
        
        // Base query for reminders
        $this->db->select('r.*, s.firstname, s.lastname, s.email as staff_email');
        $this->db->from(db_prefix() . 'reminders r');
        $this->db->join(db_prefix() . 'staff s', 'r.staff = s.staffid', 'left');
        $this->db->where('r.isnotified', 0); // Only unread reminders
        
        // Apply staff filtering based on permissions
        if (!empty($allowed_staff_ids)) {
            $this->db->where_in('r.staff', $allowed_staff_ids);
        }
        
        // Apply date filters if provided
        if (!empty($filters['from'])) {
            $this->db->where('r.date >=', $filters['from']);
        }
        if (!empty($filters['to'])) {
            $this->db->where('r.date <=', $filters['to']);
        }
        
        $this->db->order_by('r.date', 'ASC');
        $reminders_query = $this->db->get();
        $reminders = $reminders_query->result_array();
        
        // Process reminders and add related information
        foreach ($reminders as &$reminder) {
            // Add staff name
            $reminder['staff_name'] = trim($reminder['firstname'] . ' ' . $reminder['lastname']);
            
            // Add related information based on rel_type
            if ($reminder['rel_type'] && $reminder['rel_id']) {
                switch ($reminder['rel_type']) {
                    case 'lead':
                        $this->db->select('name');
                        $this->db->from(db_prefix() . 'leads');
                        $this->db->where('id', $reminder['rel_id']);
                        $lead_query = $this->db->get();
                        if ($lead_query->num_rows() > 0) {
                            $lead = $lead_query->row();
                            $reminder['lead_name'] = $lead->name;
                        }
                        break;
                        
                    case 'customer':
                        $this->db->select('company');
                        $this->db->from(db_prefix() . 'clients');
                        $this->db->where('userid', $reminder['rel_id']);
                        $client_query = $this->db->get();
                        if ($client_query->num_rows() > 0) {
                            $client = $client_query->row();
                            $reminder['customer_name'] = $client->company;
                        }
                        break;
                        
                    case 'project':
                        $this->db->select('name');
                        $this->db->from(db_prefix() . 'projects');
                        $this->db->where('id', $reminder['rel_id']);
                        $project_query = $this->db->get();
                        if ($project_query->num_rows() > 0) {
                            $project = $project_query->row();
                            $reminder['project_name'] = $project->name;
                        }
                        break;
                }
            }
        }
        
        // Categorize reminders
        $todays = [];
        $due = [];
        $upcoming = [];
        
        foreach ($reminders as $reminder) {
            $reminder_date = $reminder['date'];
            
            if ($reminder_date == $today) {
                $todays[] = $reminder;
            } elseif ($reminder_date < $today) {
                $due[] = $reminder;
            } else {
                $upcoming[] = $reminder;
            }
        }
        
        return [
            'todays' => $todays,
            'due' => $due,
            'upcoming' => $upcoming
        ];
    }

    private function get_filters()
    {
        // Accept both POST and GET for flexibility
        $input = array_merge($_GET, $_POST);
        
        $filters = [
            'from' => $input['from'] ?? '',
            'to' => $input['to'] ?? '',
            'staff' => $input['staff'] ?? [],
            'activity_type' => $input['activity_type'] ?? [],
        ];
        
        // Ensure staff is always an array
        if (!is_array($filters['staff'])) {
            if (empty($filters['staff'])) {
                $filters['staff'] = [];
            } else {
                // Convert string to array (in case it's a single ID)
                $filters['staff'] = [$filters['staff']];
            }
        }
        
        // Ensure activity_type is always an array
        if (!is_array($filters['activity_type'])) {
            if (empty($filters['activity_type'])) {
                $filters['activity_type'] = [];
            } else {
                $filters['activity_type'] = [$filters['activity_type']];
            }
        }
        
        // Apply permission-based filtering
        $current_staff_id = get_staff_user_id();
        $allowed_staff_ids = $this->reporting_dashboard_permissions_model->get_allowed_staff_ids($current_staff_id);
        
        // If user has limited permissions, override the staff filter
        if (!empty($allowed_staff_ids)) {
            // If user requested specific staff, filter to only those they can see
            if (!empty($filters['staff'])) {
                $filters['staff'] = array_intersect($filters['staff'], $allowed_staff_ids);
            } else {
                // If no specific staff requested, limit to allowed staff
                $filters['staff'] = $allowed_staff_ids;
            }
        }
        
        return $filters;
    }

    public function download_pdf()
    {
        // Check access permissions
        $access_info = $this->check_access_and_get_filters();
        if (!$access_info['has_access']) {
            header('HTTP/1.1 403 Forbidden');
            show_error('Access denied');
            return;
        }
        
        try {
            // Clear any output buffer and suppress warnings that could break PDF generation
            ob_clean();
            $old_error_reporting = error_reporting(E_ALL & ~E_WARNING);
            
            $filters = $this->get_filters();
            
            // Load required dependencies first
            if (!class_exists('App_pdf')) {
                require_once(APPPATH . 'libraries/pdf/App_pdf.php');
            }
            
            // Now load our custom PDF class
            $pdfClassPath = FCPATH . 'modules/reporting_dashboard/libraries/Reporting_dashboard_pdf.php';
            if (!file_exists($pdfClassPath)) {
                error_reporting($old_error_reporting);
                show_error('PDF class file not found at: ' . $pdfClassPath);
                return;
            }
            
            require_once($pdfClassPath);
            
            // Check if class exists after include
            if (!class_exists('Reporting_dashboard_pdf')) {
                error_reporting($old_error_reporting);
                show_error('Reporting_dashboard_pdf class not found after include');
                return;
            }
            
            // Create instance manually
            $pdf = new Reporting_dashboard_pdf($filters);
            
            // Generate the PDF
            $result = $pdf->prepare();
            
            // Set filename
            $filename = 'Staff_Activity_Report_' . date('Y-m-d_H-i-s') . '.pdf';
            
            // Restore error reporting
            error_reporting($old_error_reporting);
            
            // Output the PDF for download
            $pdf->Output($filename, 'D');
            
        } catch (Exception $e) {
            error_reporting($old_error_reporting);
            show_error('Error generating PDF report: ' . $e->getMessage());
        }
    }

    /**
     * Permissions management page (Admin only)
     */
    public function permissions()
    {
        // Only admin can access permission management
        if (!is_admin()) {
            access_denied('Reporting Dashboard Permissions');
        }

        $data['title'] = 'Reporting Dashboard - Permissions Management';
        $data['staff_permissions'] = $this->reporting_dashboard_permissions_model->get_all_staff_permissions();
        $data['reporting_dashboard_permissions_model'] = $this->reporting_dashboard_permissions_model;
        $this->load->view('permissions_management', $data);
    }

    /**
     * Update staff permission via AJAX
     */
    public function update_permission()
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        // Only admin can update permissions
        if (!is_admin()) {
            echo json_encode([
                'success' => false,
                'message' => 'Access denied'
            ]);
            return;
        }

        $staff_id = $this->input->post('staff_id');
        $permission_type = $this->input->post('permission_type');
        $allowed_staff_ids = $this->input->post('allowed_staff_ids');
        $current_user_id = get_staff_user_id();

        // Validate inputs
        if (empty($staff_id) || empty($permission_type)) {
            echo json_encode([
                'success' => false,
                'message' => 'Invalid parameters'
            ]);
            return;
        }

        // Validate permission type
        if (!in_array($permission_type, ['view_all_staff', 'view_own_only', 'view_specific_staff'])) {
            echo json_encode([
                'success' => false,
                'message' => 'Invalid permission type'
            ]);
            return;
        }

        // For view_specific_staff, validate allowed_staff_ids
        $allowed_staff_array = [];
        if ($permission_type === 'view_specific_staff') {
            if (empty($allowed_staff_ids)) {
                echo json_encode([
                    'success' => false,
                    'message' => 'Please select at least one staff member for specific staff permission'
                ]);
                return;
            }
            
            $allowed_staff_array = is_array($allowed_staff_ids) ? $allowed_staff_ids : explode(',', $allowed_staff_ids);
            $allowed_staff_array = array_filter(array_map('intval', $allowed_staff_array));
            
            if (empty($allowed_staff_array)) {
                echo json_encode([
                    'success' => false,
                    'message' => 'Invalid staff selection'
                ]);
                return;
            }
        }

        // Update permission
        $success = $this->reporting_dashboard_permissions_model->set_staff_permission(
            $staff_id,
            $permission_type,
            $current_user_id,
            $allowed_staff_array
        );

        if ($success) {
            echo json_encode([
                'success' => true,
                'message' => 'Permission updated successfully'
            ]);
        } else {
            echo json_encode([
                'success' => false,
                'message' => 'Failed to update permission'
            ]);
        }
    }

    /**
     * Remove staff permission (revert to default)
     */
    public function remove_permission()
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        if (!is_admin()) {
            echo json_encode([
                'success' => false,
                'message' => 'Access denied'
            ]);
            return;
        }

        $staff_id = $this->input->post('staff_id');

        if (empty($staff_id)) {
            echo json_encode([
                'success' => false,
                'message' => 'Invalid staff ID'
            ]);
            return;
        }

        $success = $this->reporting_dashboard_permissions_model->remove_staff_permission($staff_id);

        if ($success) {
            echo json_encode([
                'success' => true,
                'message' => 'Permission reset to default successfully'
            ]);
        } else {
            echo json_encode([
                'success' => false,
                'message' => 'Failed to reset permission'
            ]);
        }
    }

    /**
     * Get available staff for selection (AJAX)
     */
    public function get_available_staff()
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        if (!is_admin()) {
            echo json_encode([
                'success' => false,
                'message' => 'Access denied'
            ]);
            return;
        }

        $exclude_staff_id = $this->input->post('exclude_staff_id');
        $staff_list = $this->reporting_dashboard_permissions_model->get_available_staff_for_permission($exclude_staff_id);

        echo json_encode([
            'success' => true,
            'staff_list' => $staff_list
        ]);
    }

    /**
     * Bulk update permissions
     */
    public function bulk_update()
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        if (!is_admin()) {
            echo json_encode([
                'success' => false,
                'message' => 'Access denied'
            ]);
            return;
        }

        $permissions = $this->input->post('permissions');
        $current_user_id = get_staff_user_id();

        if (empty($permissions) || !is_array($permissions)) {
            echo json_encode([
                'success' => false,
                'message' => 'No permissions provided'
            ]);
            return;
        }

        $updated_count = 0;
        $errors = [];

        foreach ($permissions as $staff_id => $permission_data) {
            $permission_type = $permission_data['type'] ?? '';
            $allowed_staff_ids = $permission_data['allowed_staff_ids'] ?? [];
            
            if (in_array($permission_type, ['view_all_staff', 'view_own_only', 'view_specific_staff'])) {
                $success = $this->reporting_dashboard_permissions_model->set_staff_permission(
                    $staff_id,
                    $permission_type,
                    $current_user_id,
                    $allowed_staff_ids
                );
                
                if ($success) {
                    $updated_count++;
                } else {
                    $errors[] = "Failed to update permission for staff ID: $staff_id";
                }
            }
        }

        echo json_encode([
            'success' => empty($errors),
            'message' => empty($errors) ? 
                "Successfully updated $updated_count permissions" :
                "Updated $updated_count permissions with " . count($errors) . " errors",
            'errors' => $errors
        ]);
    }
}