<?php

defined('BASEPATH') or exit('No direct script access allowed');

// Include App_pdf from the correct path
require_once(APPPATH . 'libraries/pdf/App_pdf.php');

class Reporting_dashboard_pdf extends App_pdf
{
    protected $filters;
    protected $data;

    public function __construct($filters = [])
    {
        parent::__construct();
        $this->filters = $filters;
    }

    public function prepare()
    {
        $this->SetTitle('Staff Activity Report - ' . date('Y-m-d'));
        
        // Get CodeIgniter instance
        $CI = &get_instance();
        
        // Load the model if not already loaded
        if (!isset($CI->reporting_dashboard_model)) {
            $CI->load->model('reporting_dashboard/reporting_dashboard_model');
        }
        
        // Get all report data using standalone scripts for now (to avoid model dependency issues)
        $this->data = $this->get_report_data($this->filters);

        $this->set_view_vars($this->data);

        return $this->build();
    }
    
    private function get_report_data($filters)
    {
        // For now, use direct database queries to avoid model loading issues
        $host = 'localhost';
        $username = 'root';
        $password = '';
        $database = 'crm';
        $prefix = 'tbl';
        
        $mysqli = new mysqli($host, $username, $password, $database);
        
        if ($mysqli->connect_error) {
            return [
                'kpis' => ['today' => 0, 'week' => 0, 'month' => 0, 'logins_today' => 0, 'top_staff' => []],
                'activity_types' => ['chart_data' => ['labels' => [], 'datasets' => [['data' => []]]]],
                'activity_log' => ['data' => []],
                'filters' => $filters
            ];
        }
        
        // Get date filters with proper defaults
        $dateFrom = (!empty($filters['from'])) ? $filters['from'] : date('Y-m-01');
        $dateTo = (!empty($filters['to'])) ? $filters['to'] : date('Y-m-d');
        
        // Get KPIs
        $today = date('Y-m-d');
        $weekStart = date('Y-m-d', strtotime('monday this week'));
        $monthStart = date('Y-m-01');
        
        // Today's activities
        $todayQuery = "SELECT COUNT(*) as count FROM {$prefix}lead_activity_log WHERE DATE(date) = '$today'";
        $todayResult = $mysqli->query($todayQuery);
        $todayCount = $todayResult->fetch_assoc()['count'];
        
        // Week's activities
        $weekQuery = "SELECT COUNT(*) as count FROM {$prefix}lead_activity_log WHERE DATE(date) >= '$weekStart'";
        $weekResult = $mysqli->query($weekQuery);
        $weekCount = $weekResult->fetch_assoc()['count'];
        
        // Month's activities
        $monthQuery = "SELECT COUNT(*) as count FROM {$prefix}lead_activity_log WHERE DATE(date) >= '$monthStart'";
        $monthResult = $mysqli->query($monthQuery);
        $monthCount = $monthResult->fetch_assoc()['count'];
        
        // Top staff
        $staffQuery = "SELECT s.firstname, s.lastname, COUNT(*) as total 
                       FROM {$prefix}lead_activity_log a 
                       LEFT JOIN {$prefix}staff s ON s.staffid = a.staffid 
                       WHERE DATE(a.date) >= '$monthStart'
                       GROUP BY a.staffid 
                       ORDER BY total DESC LIMIT 5";
        $staffResult = $mysqli->query($staffQuery);
        $topStaff = [];
        while ($row = $staffResult->fetch_assoc()) {
            $topStaff[] = [
                'staff_name' => trim($row['firstname'] . ' ' . $row['lastname']),
                'total' => $row['total']
            ];
        }
        
        // Activity types for the selected period
        $typesQuery = "SELECT 
                        CASE 
                            WHEN description LIKE '%not_activity_new_reminder_created%' THEN 'Reminder Created'
                            WHEN description LIKE '%not_lead_activity_added_followup%' THEN 'Followup Added'
                            WHEN description LIKE '%not_lead_activity_lead_created%' THEN 'Lead Created'
                            WHEN description LIKE '%not_lead_activity_assigned%' THEN 'Lead Assigned'
                            ELSE 'Other'
                        END as activity_type,
                        COUNT(*) as count
                       FROM {$prefix}lead_activity_log 
                       WHERE DATE(date) >= '$dateFrom' AND DATE(date) <= '$dateTo'
                       GROUP BY activity_type
                       ORDER BY count DESC";
        $typesResult = $mysqli->query($typesQuery);
        $activityLabels = [];
        $activityData = [];
        while ($row = $typesResult->fetch_assoc()) {
            $activityLabels[] = $row['activity_type'];
            $activityData[] = (int)$row['count'];
        }
        
        // Activity log (limited)
        $logQuery = "SELECT 
                        a.date, a.description, a.staffid,
                        s.firstname, s.lastname,
                        l.name as lead_name, l.company as lead_company
                     FROM {$prefix}lead_activity_log a
                     LEFT JOIN {$prefix}staff s ON s.staffid = a.staffid
                     LEFT JOIN {$prefix}leads l ON l.id = a.leadid
                     WHERE DATE(a.date) >= '$dateFrom' AND DATE(a.date) <= '$dateTo'
                     ORDER BY a.date DESC
                     LIMIT 50";
        $logResult = $mysqli->query($logQuery);
        $activityLog = [];
        while ($row = $logResult->fetch_assoc()) {
            $staffName = trim($row['firstname'] . ' ' . $row['lastname']);
            $related = !empty($row['lead_name']) ? $row['lead_name'] : (!empty($row['lead_company']) ? $row['lead_company'] : 'Unknown');
            
            // Parse activity type from description
            $activityType = 'Activity';
            if (strpos($row['description'], 'not_activity_new_reminder_created') !== false) {
                $activityType = 'Reminder Created';
            } elseif (strpos($row['description'], 'not_lead_activity_added_followup') !== false) {
                $activityType = 'Followup Added';
            } elseif (strpos($row['description'], 'not_lead_activity_lead_created') !== false) {
                $activityType = 'Lead Created';
            } elseif (strpos($row['description'], 'not_lead_activity_assigned') !== false) {
                $activityType = 'Lead Assigned';
            }
            
            $activityLog[] = [
                'date' => $row['date'],
                'staff_name' => $staffName,
                'activity_type' => $activityType,
                'related' => $related,
                'description' => 'Activity performed on lead'
            ];
        }
        
        // Get follow-up types by staff (from both followup_type column and description parsing)
        $followupQuery = "SELECT 
                            s.firstname, s.lastname,
                            ln.followup_type,
                            ln.description,
                            ln.id,
                            ln.dateadded
                          FROM {$prefix}lead_notes ln
                          LEFT JOIN {$prefix}staff s ON s.staffid = ln.addedfrom
                          WHERE DATE(ln.dateadded) >= '$dateFrom' AND DATE(ln.dateadded) <= '$dateTo'
                          ORDER BY ln.dateadded DESC";
        
        $followupResult = $mysqli->query($followupQuery);
        $followupByStaff = [];
        $allFollowupTypes = [];
        

        
        while ($row = $followupResult->fetch_assoc()) {
            $staffName = trim($row['firstname'] . ' ' . $row['lastname']);
            $followupType = $row['followup_type'];
            $extractedType = '';
            
            // If followup_type column is empty, try to extract from description
            if (empty($followupType) && !empty($row['description'])) {
                if (preg_match('/<strong>Type:\s*([^<]+)<\/strong>/i', $row['description'], $matches)) {
                    $extractedType = trim($matches[1]);
                } elseif (preg_match('/Type:\s*([^\r\n<]+)/i', $row['description'], $matches)) {
                    $extractedType = trim($matches[1]);
                }
            }
            
            $finalType = !empty($followupType) ? $followupType : $extractedType;
            
            // Only process if we have a valid followup type
            if (!empty($finalType)) {
                if (!isset($followupByStaff[$staffName])) {
                    $followupByStaff[$staffName] = [];
                }
                
                if (!isset($followupByStaff[$staffName][$finalType])) {
                    $followupByStaff[$staffName][$finalType] = 0;
                }
                
                $followupByStaff[$staffName][$finalType]++;
                
                if (!in_array($finalType, $allFollowupTypes)) {
                    $allFollowupTypes[] = $finalType;
                }
            }
        }
        
        // Also get follow-up data from activity log for those that don't have followup_type in notes
        $activityFollowupQuery = "SELECT 
                                    s.firstname, s.lastname,
                                    la.description,
                                    COUNT(*) as count
                                  FROM {$prefix}lead_activity_log la
                                  LEFT JOIN {$prefix}staff s ON s.staffid = la.staffid
                                  WHERE DATE(la.date) >= '$dateFrom' AND DATE(la.date) <= '$dateTo'
                                  AND la.description LIKE '%not_lead_activity_added_followup%'
                                  GROUP BY la.staffid
                                  ORDER BY s.firstname, s.lastname";
        
        $activityFollowupResult = $mysqli->query($activityFollowupQuery);
        while ($row = $activityFollowupResult->fetch_assoc()) {
            $staffName = trim($row['firstname'] . ' ' . $row['lastname']);
            
            if (!isset($followupByStaff[$staffName])) {
                $followupByStaff[$staffName] = [];
            }
            
            // Add general followup count if no specific types found
            if (empty($followupByStaff[$staffName])) {
                $followupByStaff[$staffName]['General Followup'] = (int)$row['count'];
                if (!in_array('General Followup', $allFollowupTypes)) {
                    $allFollowupTypes[] = 'General Followup';
                }
            }
        }
        
        $mysqli->close();
        
        return [
            'kpis' => [
                'today' => $todayCount,
                'week' => $weekCount,
                'month' => $monthCount,
                'logins_today' => 0, // Not tracking logins in this simplified version
                'top_staff' => $topStaff
            ],
            'activity_types' => [
                'chart_data' => [
                    'labels' => $activityLabels,
                    'datasets' => [['data' => $activityData]]
                ]
            ],
            'activity_log' => ['data' => $activityLog],
            'followup_types' => [
                'by_staff' => $followupByStaff,
                'all_types' => $allFollowupTypes
            ],
            'filters' => $filters
        ];
    }

    protected function type()
    {
        return 'reporting-dashboard';
    }

    protected function file_path()
    {
        $customPath = FCPATH . 'modules/reporting_dashboard/views/pdf_template.php';
        $actualPath = FCPATH . 'modules/reporting_dashboard/views/pdf_template.php';

        if (file_exists($customPath)) {
            return $customPath;
        }

        return $actualPath;
    }

    public function get_format_array()
    {
        return [
            'orientation' => 'P', // Portrait
            'format'      => 'A4',
        ];
    }
}
