<?php

defined('BASEPATH') or exit('No direct script access allowed');

/**
 * Facebook Leads Integration Model
 * Handles database operations and Facebook API interactions
 */
class Facebook_leads_integration_model extends App_Model
{
    private $settings_table;
    private $logs_table;

    public function __construct()
    {
        parent::__construct();
        $this->settings_table = db_prefix() . 'facebook_leads_settings';
        $this->logs_table = db_prefix() . 'facebook_leads_logs';
        $this->load->helper('facebook_helper');
    }

    /**
     * Get a single setting value
     * @param string $name Setting name
     * @return string|null
     */
    public function get_setting($name)
    {
        $result = $this->db->select('setting_value')
            ->where('setting_name', $name)
            ->get($this->settings_table)
            ->row();

        return $result ? $result->setting_value : null;
    }

    /**
     * Get all settings as associative array
     * @return array
     */
    public function get_all_settings()
    {
        $results = $this->db->get($this->settings_table)->result();
        $settings = [];

        foreach ($results as $row) {
            $settings[$row->setting_name] = $row->setting_value;
        }

        return $settings;
    }

    /**
     * Update a setting value
     * @param string $name Setting name
     * @param string $value Setting value
     * @return bool
     */
    public function update_setting($name, $value)
    {
        $data = [
            'setting_value' => $value,
            'updated_at' => date('Y-m-d H:i:s')
        ];

        $this->db->where('setting_name', $name);
        $exists = $this->db->get($this->settings_table)->row();

        if ($exists) {
            $this->db->where('setting_name', $name);
            return $this->db->update($this->settings_table, $data);
        } else {
            $data['setting_name'] = $name;
            return $this->db->insert($this->settings_table, $data);
        }
    }

    /**
     * Fetch leads from Facebook and import them into Perfex CRM
     * @return array Statistics about the import
     */
    public function fetch_and_import_leads()
    {
        $settings = $this->get_all_settings();
        $page_access_token = $settings['page_access_token'];
        $form_ids = explode(',', $settings['form_ids']);
        
        $stats = [
            'imported' => 0,
            'duplicates' => 0,
            'errors' => 0
        ];

        foreach ($form_ids as $form_id) {
            $form_id = trim($form_id);
            if (empty($form_id)) {
                continue;
            }

            try {
                $leads = $this->fetch_leads_from_facebook($form_id, $page_access_token);
                
                foreach ($leads as $lead) {
                    $result = $this->import_lead($lead, $form_id, $settings);
                    
                    if ($result === 'imported') {
                        $stats['imported']++;
                    } elseif ($result === 'duplicate') {
                        $stats['duplicates']++;
                    } else {
                        $stats['errors']++;
                    }
                }
            } catch (Exception $e) {
                log_facebook_error('fetch_form_' . $form_id, $e->getMessage());
                $stats['errors']++;
            }
        }

        return $stats;
    }

    /**
     * Fetch leads from Facebook API for a specific form
     * @param string $form_id Facebook form ID
     * @param string $access_token Page access token
     * @return array Array of leads
     */
    private function fetch_leads_from_facebook($form_id, $access_token)
    {
        $params = [
            'fields' => 'id,created_time,field_data',
            'limit' => 100
        ];

        $response = facebook_api_request("/{$form_id}/leads", $params, $access_token);

        if (isset($response['error'])) {
            throw new Exception($response['error']['message']);
        }

        return $response['data'] ?? [];
    }

    /**
     * Fetch and import a single lead (used by webhook)
     * @param string $leadgen_id Facebook lead ID
     * @param string $form_id Form ID
     * @return bool
     */
    public function fetch_and_import_single_lead($leadgen_id, $form_id)
    {
        $settings = $this->get_all_settings();
        $page_access_token = $settings['page_access_token'];

        // Check if already imported
        $this->db->where('facebook_lead_id', $leadgen_id);
        $exists = $this->db->get($this->logs_table)->row();
        
        if ($exists) {
            return false; // Duplicate
        }

        // Fetch lead details from Facebook
        $params = ['fields' => 'id,created_time,field_data'];
        $lead = facebook_api_request("/{$leadgen_id}", $params, $page_access_token);

        if (isset($lead['error'])) {
            throw new Exception($lead['error']['message']);
        }

        $this->import_lead($lead, $form_id, $settings);
        return true;
    }

    /**
     * Import a single lead into Perfex CRM
     * @param array $facebook_lead Lead data from Facebook
     * @param string $form_id Form ID
     * @param array $settings Module settings
     * @return string Result status: 'imported', 'duplicate', or 'error'
     */
    private function import_lead($facebook_lead, $form_id, $settings)
    {
        $facebook_lead_id = $facebook_lead['id'];

        // Check for duplicates
        $this->db->where('facebook_lead_id', $facebook_lead_id);
        $existing_log = $this->db->get($this->logs_table)->row();

        if ($existing_log) {
            return 'duplicate';
        }

        try {
            // Parse lead data from Facebook
            $lead_data = $this->parse_facebook_lead_data($facebook_lead);

            // Check if email already exists in Perfex
            if (!empty($lead_data['email'])) {
                $this->db->where('email', $lead_data['email']);
                $existing_lead = $this->db->get(db_prefix() . 'leads')->row();
                
                if ($existing_lead) {
                    // Log as duplicate
                    $this->log_lead($facebook_lead_id, $form_id, null, $facebook_lead, 'duplicate', 'Email already exists in CRM');
                    return 'duplicate';
                }
            }

            // Prepare lead data for Perfex CRM
            $perfex_lead_data = [
                'name' => $lead_data['name'] ?? 'Facebook Lead',
                'email' => $lead_data['email'] ?? '',
                'phonenumber' => $lead_data['phone'] ?? '',
                'source' => $settings['lead_source'] ?: null,
                'status' => $settings['lead_status'] ?: 1,
                'assigned' => $settings['default_assigned_staff'] ?: null,
                'dateadded' => date('Y-m-d H:i:s'),
                'description' => $this->build_lead_description($lead_data),
                'addedfrom' => 0 // System added
            ];

            // Insert lead using Perfex's Leads model
            $this->load->model('leads_model');
            $lead_id = $this->leads_model->add($perfex_lead_data);

            if ($lead_id) {
                // Log successful import
                $this->log_lead($facebook_lead_id, $form_id, $lead_id, $facebook_lead, 'success');
                
                // Send notification to assigned staff
                if (!empty($settings['default_assigned_staff'])) {
                    $this->send_lead_notification($lead_id, $settings['default_assigned_staff']);
                }

                return 'imported';
            } else {
                throw new Exception('Failed to insert lead into Perfex CRM');
            }
        } catch (Exception $e) {
            // Log error
            $this->log_lead($facebook_lead_id, $form_id, null, $facebook_lead, 'failed', $e->getMessage());
            log_facebook_error('import_lead_' . $facebook_lead_id, $e->getMessage());
            return 'error';
        }
    }

    /**
     * Parse Facebook lead field_data into usable format
     * @param array $facebook_lead Raw lead data from Facebook
     * @return array Parsed data
     */
    private function parse_facebook_lead_data($facebook_lead)
    {
        $parsed = [];
        $field_data = $facebook_lead['field_data'] ?? [];

        foreach ($field_data as $field) {
            $name = strtolower($field['name']);
            $value = $field['values'][0] ?? '';

            // Map common field names
            if (in_array($name, ['full_name', 'name', 'full name'])) {
                $parsed['name'] = $value;
            } elseif (in_array($name, ['email', 'e-mail'])) {
                $parsed['email'] = $value;
            } elseif (in_array($name, ['phone', 'phone_number', 'phone number', 'mobile'])) {
                $parsed['phone'] = $value;
            } elseif (in_array($name, ['first_name', 'first name'])) {
                $parsed['first_name'] = $value;
            } elseif (in_array($name, ['last_name', 'last name'])) {
                $parsed['last_name'] = $value;
            } else {
                // Store custom fields
                $parsed['custom_' . $name] = $value;
            }
        }

        // Combine first and last name if full name not provided
        if (empty($parsed['name']) && (isset($parsed['first_name']) || isset($parsed['last_name']))) {
            $parsed['name'] = trim(($parsed['first_name'] ?? '') . ' ' . ($parsed['last_name'] ?? ''));
        }

        return $parsed;
    }

    /**
     * Build description field with all lead data
     * @param array $lead_data Parsed lead data
     * @return string
     */
    private function build_lead_description($lead_data)
    {
        $description = "Lead imported from Facebook\n\n";

        foreach ($lead_data as $key => $value) {
            if (!in_array($key, ['name', 'email', 'phone']) && !empty($value)) {
                $label = ucwords(str_replace(['_', 'custom '], ' ', $key));
                $description .= "{$label}: {$value}\n";
            }
        }

        return trim($description);
    }

    /**
     * Log lead import to database
     * @param string $facebook_lead_id Facebook lead ID
     * @param string $form_id Form ID
     * @param int|null $lead_id Perfex lead ID
     * @param array $data Full Facebook lead data
     * @param string $status Status: success, failed, duplicate
     * @param string|null $error_message Error message if failed
     */
    private function log_lead($facebook_lead_id, $form_id, $lead_id, $data, $status, $error_message = null)
    {
        $log_data = [
            'facebook_lead_id' => $facebook_lead_id,
            'form_id' => $form_id,
            'lead_id' => $lead_id,
            'data' => json_encode($data),
            'status' => $status,
            'error_message' => $error_message,
            'created_at' => date('Y-m-d H:i:s')
        ];

        $this->db->insert($this->logs_table, $log_data);
    }

    /**
     * Get logs with pagination
     * @param int $limit Number of logs to retrieve
     * @param int $offset Offset for pagination
     * @return array
     */
    public function get_logs($limit = 100, $offset = 0)
    {
        return $this->db->order_by('created_at', 'DESC')
            ->limit($limit, $offset)
            ->get($this->logs_table)
            ->result();
    }

    /**
     * Get a single log by ID
     * @param int $id Log ID
     * @return object|null
     */
    public function get_log($id)
    {
        return $this->db->where('id', $id)
            ->get($this->logs_table)
            ->row();
    }

    /**
     * Get statistics
     * @return array
     */
    public function get_statistics()
    {
        $total_imported = $this->db->where('status', 'success')
            ->count_all_results($this->logs_table);

        $this_month = $this->db->where('status', 'success')
            ->where('MONTH(created_at)', date('m'))
            ->where('YEAR(created_at)', date('Y'))
            ->count_all_results($this->logs_table);

        $last_fetch = $this->get_setting('last_fetch_time');

        return [
            'total_imported' => $total_imported,
            'this_month' => $this_month,
            'last_fetch' => $last_fetch ? date('M d, Y H:i', strtotime($last_fetch)) : null
        ];
    }

    /**
     * Send notification to assigned staff member
     * @param int $lead_id Lead ID
     * @param int $staff_id Staff ID
     */
    private function send_lead_notification($lead_id, $staff_id)
    {
        $notification_data = [
            'description' => 'New lead from Facebook',
            'touserid' => $staff_id,
            'fromuserid' => 0,
            'link' => 'leads/index/' . $lead_id,
            'date' => date('Y-m-d H:i:s')
        ];

        $this->db->insert(db_prefix() . 'notifications', $notification_data);
    }
}

