<?php

/**
 * Facebook Leads Integration - Cron Job Script
 * 
 * This script fetches leads from Facebook and imports them into Perfex CRM
 * 
 * Usage:
 * Add this to your crontab to run every 15 minutes:
 * */15 * * * * php /path/to/perfex/modules/facebook_leads_integration/cron/fetch_facebook_leads.php
 * 
 * Or run manually:
 * php fetch_facebook_leads.php
 */

// Prevent direct browser access
if (php_sapi_name() !== 'cli') {
    exit('This script can only be run from the command line.');
}

// Define the base path to Perfex CRM
define('FCPATH', realpath(dirname(__FILE__) . '/../../../') . '/');

// Check if index.php exists
if (!file_exists(FCPATH . 'index.php')) {
    die("Error: Could not find Perfex CRM installation. Please check the path.\n");
}

// Bootstrap Perfex CRM
$_SERVER['HTTP_HOST'] = 'localhost'; // Required for CLI
$_SERVER['REQUEST_URI'] = '/';

require_once(FCPATH . 'application/config/app-config.php');

// Start output buffering to prevent CodeIgniter output
ob_start();

// Bootstrap CodeIgniter
require_once(FCPATH . 'application/libraries/App_Controller.php');

// Get CodeIgniter instance
$CI = &get_instance();

// Load the Facebook Leads Integration model
$CI->load->model('facebook_leads_integration/facebook_leads_integration_model');
$CI->load->helper('facebook_helper');

// Clear output buffer
ob_end_clean();

// Start execution
echo "[" . date('Y-m-d H:i:s') . "] Facebook Leads Cron Job Started\n";

try {
    // Check if integration is enabled
    $enabled = $CI->facebook_leads_integration_model->get_setting('enabled');
    
    if ($enabled != '1') {
        echo "[" . date('Y-m-d H:i:s') . "] Integration is disabled. Exiting.\n";
        exit(0);
    }
    
    // Get settings
    $settings = $CI->facebook_leads_integration_model->get_all_settings();
    
    // Validate required settings
    if (empty($settings['page_access_token']) || empty($settings['form_ids'])) {
        echo "[" . date('Y-m-d H:i:s') . "] Error: Missing required settings (page_access_token or form_ids)\n";
        log_facebook_error('cron_job', 'Missing required settings');
        exit(1);
    }
    
    // Check cron interval
    $cron_interval = intval($settings['cron_interval'] ?: 15); // Default 15 minutes
    $last_fetch = $settings['last_fetch_time'];
    
    if ($last_fetch) {
        $last_fetch_timestamp = strtotime($last_fetch);
        $next_fetch_timestamp = $last_fetch_timestamp + ($cron_interval * 60);
        
        if (time() < $next_fetch_timestamp) {
            $minutes_remaining = ceil(($next_fetch_timestamp - time()) / 60);
            echo "[" . date('Y-m-d H:i:s') . "] Too soon to fetch. Next fetch in {$minutes_remaining} minutes.\n";
            exit(0);
        }
    }
    
    echo "[" . date('Y-m-d H:i:s') . "] Fetching leads from Facebook...\n";
    
    // Fetch and import leads
    $result = $CI->facebook_leads_integration_model->fetch_and_import_leads();
    
    // Update last fetch time
    $CI->facebook_leads_integration_model->update_setting('last_fetch_time', date('Y-m-d H:i:s'));
    
    // Output results
    echo "[" . date('Y-m-d H:i:s') . "] Fetch completed:\n";
    echo "  - Imported: {$result['imported']}\n";
    echo "  - Duplicates: {$result['duplicates']}\n";
    echo "  - Errors: {$result['errors']}\n";
    
    // Log to file
    log_facebook_info('cron_job', "Fetch completed - Imported: {$result['imported']}, Duplicates: {$result['duplicates']}, Errors: {$result['errors']}");
    
    // Clean old log files
    facebook_clean_old_logs();
    
    echo "[" . date('Y-m-d H:i:s') . "] Cron job finished successfully\n";
    exit(0);
    
} catch (Exception $e) {
    echo "[" . date('Y-m-d H:i:s') . "] Error: " . $e->getMessage() . "\n";
    log_facebook_error('cron_job', $e->getMessage());
    exit(1);
}

