<?php
// Backup and create enhanced activity table functionality
defined('BASEPATH') or exit('No direct script access allowed');

/**
 * Enhanced Activity Table Functions for Reporting Dashboard
 */
class Enhanced_Activity_Model extends App_Model
{
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Get modern, enhanced activity table with advanced features
     */
    public function get_enhanced_activity_table(array $filters)
    {
        // Detect available activity tables
        $table_variations = ['lead_activity_log', 'activity_log', 'activities'];
        $actual_table = null;
        
        foreach ($table_variations as $variation) {
            if ($this->db->table_exists($variation)) {
                $actual_table = db_prefix() . $variation;
                break;
            }
        }
        
        if (!$actual_table) {
            error_log('No activity table found in database');
            return $this->empty_datatable_response();
        }

        error_log('Using enhanced activity table: ' . $actual_table);

        // DataTables parameters
        $draw = (int)$this->input->post('draw');
        $start = (int)$this->input->post('start');
        $length = (int)$this->input->post('length');
        if ($length <= 0 || $length > 100) { $length = 25; }

        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');

        // Enhanced followup type extraction
        $followup_type_case = $this->get_followup_type_sql();
        
        // Build comprehensive select with modern features
        $select_fields = [
            'a.*',
            'CONCAT(s.firstname, " ", s.lastname) as staff_name',
            's.firstname', 
            's.lastname',
            'l.name as lead_name', 
            'l.company as lead_company',
            'l.phonenumber as lead_phone',
            'l.email as lead_email',
            $followup_type_case . ' as followup_type',
            'DATE_FORMAT(a.date, "%Y-%m-%d %H:%i") as formatted_date',
            'DATE_FORMAT(a.date, "%W, %M %d, %Y at %H:%i") as readable_date',
            'TIMESTAMPDIFF(HOUR, a.date, NOW()) as hours_ago',
            'CASE 
                WHEN TIMESTAMPDIFF(HOUR, a.date, NOW()) <= 2 THEN "recent"
                WHEN TIMESTAMPDIFF(HOUR, a.date, NOW()) <= 24 THEN "today"
                WHEN TIMESTAMPDIFF(HOUR, a.date, NOW()) <= 168 THEN "week"
                ELSE "older"
            END as time_category'
        ];
        
        $this->db->select(implode(', ', $select_fields));
        $this->db->from($actual_table . ' a');
        $this->db->join(db_prefix() . 'staff s', 's.staffid = a.staffid', 'left');
        $this->db->join(db_prefix() . 'leads l', 'l.id = a.leadid', 'left');
        
        // Apply filters
        if ($from) $this->db->where('DATE(a.date) >=', $from);
        if ($to) $this->db->where('DATE(a.date) <=', $to);
        
        // Staff filtering
        if (!empty($filters['staff']) && is_array($filters['staff'])) {
            $this->db->where_in('a.staffid', $filters['staff']);
        }

        // Activity type filtering
        if (!empty($filters['activity_type']) && is_array($filters['activity_type'])) {
            $this->apply_activity_type_filter($filters['activity_type']);
        }

        // Search functionality
        $search = $this->input->post('search');
        if (!empty($search['value'])) {
            $search_term = $search['value'];
            $this->db->group_start();
            $this->db->like('a.description', $search_term);
            $this->db->or_like('CONCAT(s.firstname, " ", s.lastname)', $search_term);
            $this->db->or_like('l.name', $search_term);
            $this->db->or_like('l.company', $search_term);
            $this->db->group_end();
        }

        // Get total count
        $total_query = clone $this->db;
        $total = $total_query->count_all_results();
        
        // Apply ordering
        $this->apply_ordering();
        
        // Apply pagination
        $this->db->limit($length, $start);
        
        try {
            $query_result = $this->db->get();
            $rows = $query_result->result_array();
            error_log('Enhanced activity query successful: ' . count($rows) . ' rows');
        } catch (Exception $e) {
            error_log('Enhanced activity query error: ' . $e->getMessage());
            return $this->empty_datatable_response($draw);
        }

        // Process and enhance the data
        $data = [];
        foreach ($rows as $row) {
            $data[] = $this->process_activity_row($row);
        }

        return [
            'draw' => $draw,
            'recordsTotal' => $total,
            'recordsFiltered' => $total,
            'data' => $data
        ];
    }

    /**
     * Get SQL for enhanced followup type detection
     */
    private function get_followup_type_sql()
    {
        return "(
            CASE 
                WHEN LOWER(a.description) REGEXP '(whatsapp|whats.app|wa )' THEN 'WhatsApp'
                WHEN LOWER(a.description) REGEXP '(call|phone|dial|rang|spoke)' THEN 'Phone Call'
                WHEN LOWER(a.description) REGEXP '(email|mail|sent.message)' THEN 'Email'
                WHEN LOWER(a.description) REGEXP '(sms|text.message|texted)' THEN 'SMS'
                WHEN LOWER(a.description) REGEXP '(meeting|appointment|scheduled|met)' THEN 'Meeting'
                WHEN LOWER(a.description) REGEXP '(reminder|follow.?up|callback)' THEN 'Reminder'
                WHEN LOWER(a.description) REGEXP '(status|updated|changed|moved)' THEN 'Status Update'
                WHEN LOWER(a.description) REGEXP '(proposal|quote|estimate)' THEN 'Proposal'
                WHEN LOWER(a.description) REGEXP '(contract|agreement|signed)' THEN 'Contract'
                WHEN LOWER(a.description) REGEXP '(invoice|bill|payment|paid)' THEN 'Invoice'
                WHEN LOWER(a.description) REGEXP '(note|comment|remark)' THEN 'Note'
                WHEN LOWER(a.description) REGEXP '(assigned|delegated|transferred)' THEN 'Assignment'
                WHEN LOWER(a.description) REGEXP '(qualified|qualification)' THEN 'Qualification'
                WHEN LOWER(a.description) REGEXP '(demo|demonstration|presentation)' THEN 'Demo'
                WHEN LOWER(a.description) REGEXP '(visit|visited|on.site)' THEN 'Site Visit'
                ELSE 'General'
            END
        )";
    }

    /**
     * Apply activity type filtering based on user selection
     */
    private function apply_activity_type_filter($activity_types)
    {
        if (empty($activity_types)) return;
        
        $this->db->group_start();
        foreach ($activity_types as $type) {
            switch (strtolower(trim($type))) {
                case 'whatsapp':
                    $this->db->or_where("LOWER(a.description) REGEXP", '(whatsapp|whats.app|wa )');
                    break;
                case 'call':
                case 'phone':
                    $this->db->or_where("LOWER(a.description) REGEXP", '(call|phone|dial|rang)');
                    break;
                case 'email':
                    $this->db->or_where("LOWER(a.description) REGEXP", '(email|mail)');
                    break;
                case 'sms':
                    $this->db->or_where("LOWER(a.description) REGEXP", '(sms|text)');
                    break;
                case 'meeting':
                    $this->db->or_where("LOWER(a.description) REGEXP", '(meeting|appointment)');
                    break;
                default:
                    $this->db->or_like('a.description', $type);
            }
        }
        $this->db->group_end();
    }

    /**
     * Apply ordering based on DataTables request
     */
    private function apply_ordering()
    {
        $order = $this->input->post('order');
        if (!empty($order)) {
            $order_column = (int)$order[0]['column'];
            $order_dir = $order[0]['dir'] === 'asc' ? 'ASC' : 'DESC';
            
            $order_columns = [
                0 => 'a.date',
                1 => 'staff_name',
                2 => 'lead_name',
                3 => 'followup_type',
                4 => 'a.description'
            ];
            
            if (isset($order_columns[$order_column])) {
                $this->db->order_by($order_columns[$order_column], $order_dir);
            } else {
                $this->db->order_by('a.date', 'DESC');
            }
        } else {
            $this->db->order_by('a.date', 'DESC');
        }
    }

    /**
     * Process and enhance individual activity row
     */
    private function process_activity_row($row)
    {
        // Staff information
        $staff_name = trim(($row['firstname'] ?? '') . ' ' . ($row['lastname'] ?? '')) ?: 'System';
        
        // Lead information
        $lead_id = (int)($row['leadid'] ?? 0);
        $lead_name = trim($row['lead_name'] ?? '');
        $lead_company = trim($row['lead_company'] ?? '');
        $lead_phone = trim($row['lead_phone'] ?? '');
        $lead_email = trim($row['lead_email'] ?? '');
        
        // Enhanced lead display
        $lead_display = $this->build_lead_display($lead_name, $lead_company, $lead_phone, $lead_email);
        
        // Enhanced followup type
        $followup_type = $row['followup_type'] ?? 'General';
        
        // Time formatting
        $hours_ago = (int)($row['hours_ago'] ?? 0);
        $time_ago = $this->format_time_ago($hours_ago);
        
        // Priority and status
        $priority = $this->determine_priority($row);
        $status = $this->determine_status($row);
        
        // Enhanced description with emoji
        $description = $this->enhance_description($row['description'] ?? '');

        return [
            'DT_RowId' => 'activity_' . $row['id'],
            'DT_RowClass' => $this->get_row_class($row),
            'id' => $row['id'],
            'date' => $row['formatted_date'] ?? '',
            'readable_date' => $row['readable_date'] ?? '',
            'time_ago' => $time_ago,
            'time_category' => $row['time_category'] ?? 'older',
            'staff' => $staff_name,
            'staff_id' => $row['staffid'] ?? 0,
            'lead' => $lead_display,
            'lead_id' => $lead_id,
            'lead_name' => $lead_name,
            'lead_company' => $lead_company,
            'lead_phone' => $lead_phone,
            'lead_email' => $lead_email,
            'followup_type' => $followup_type,
            'description' => $description,
            'raw_description' => $row['description'] ?? '',
            'priority' => $priority,
            'status' => $status,
            'additional_data' => $row['additional_data'] ?? ''
        ];
    }

    /**
     * Build enhanced lead display
     */
    private function build_lead_display($name, $company, $phone, $email)
    {
        $parts = [];
        
        if ($name) $parts[] = '<strong>' . htmlspecialchars($name) . '</strong>';
        if ($company) $parts[] = '<em>' . htmlspecialchars($company) . '</em>';
        
        $main = !empty($parts) ? implode(' - ', $parts) : 'Unknown Lead';
        
        $contacts = [];
        if ($phone) $contacts[] = '<small>📞 ' . htmlspecialchars($phone) . '</small>';
        if ($email) $contacts[] = '<small>📧 ' . htmlspecialchars($email) . '</small>';
        
        if (!empty($contacts)) {
            $main .= '<br>' . implode(' | ', $contacts);
        }
        
        return $main;
    }

    /**
     * Format time ago in human-readable format
     */
    private function format_time_ago($hours)
    {
        if ($hours < 1) {
            return '<span class="badge badge-success">Just now</span>';
        } elseif ($hours < 24) {
            $h = $hours == 1 ? '1 hour' : $hours . ' hours';
            return '<span class="badge badge-info">' . $h . ' ago</span>';
        } elseif ($hours < 48) {
            return '<span class="badge badge-warning">Yesterday</span>';
        } else {
            $days = floor($hours / 24);
            $d = $days == 1 ? '1 day' : $days . ' days';
            return '<span class="badge badge-secondary">' . $d . ' ago</span>';
        }
    }

    /**
     * Enhance description with emoji and formatting
     */
    private function enhance_description($description)
    {
        $description = htmlspecialchars(trim($description));
        
        // Add emoji based on content
        $emoji_map = [
            'whatsapp' => '💬',
            'call' => '📞',
            'email' => '📧',
            'sms' => '💬',
            'meeting' => '🤝',
            'reminder' => '⏰',
            'proposal' => '📄',
            'contract' => '📋',
            'invoice' => '💰',
            'status' => '🔄',
            'note' => '📝'
        ];
        
        $lower_desc = strtolower($description);
        foreach ($emoji_map as $keyword => $emoji) {
            if (strpos($lower_desc, $keyword) !== false) {
                $description = $emoji . ' ' . $description;
                break;
            }
        }
        
        return $description;
    }

    /**
     * Determine activity priority
     */
    private function determine_priority($row)
    {
        $hours_ago = (int)($row['hours_ago'] ?? 999);
        $description = strtolower($row['description'] ?? '');
        
        if ($hours_ago <= 2 && (
            strpos($description, 'urgent') !== false ||
            strpos($description, 'important') !== false ||
            strpos($description, 'asap') !== false
        )) {
            return 'high';
        }
        
        if ($hours_ago <= 24) {
            return 'medium';
        }
        
        return 'normal';
    }

    /**
     * Determine activity status
     */
    private function determine_status($row)
    {
        $description = strtolower($row['description'] ?? '');
        
        if (strpos($description, 'completed') !== false || strpos($description, 'done') !== false) {
            return 'completed';
        }
        
        if (strpos($description, 'pending') !== false || strpos($description, 'waiting') !== false) {
            return 'pending';
        }
        
        return 'active';
    }

    /**
     * Get CSS row class based on activity properties
     */
    private function get_row_class($row)
    {
        $classes = [];
        
        $hours_ago = (int)($row['hours_ago'] ?? 999);
        if ($hours_ago <= 2) {
            $classes[] = 'activity-recent';
        } elseif ($hours_ago <= 24) {
            $classes[] = 'activity-today';
        }
        
        $followup_type = strtolower($row['followup_type'] ?? '');
        $classes[] = 'activity-type-' . str_replace(' ', '-', $followup_type);
        
        return implode(' ', $classes);
    }

    /**
     * Return empty DataTables response
     */
    private function empty_datatable_response($draw = 0)
    {
        return [
            'draw' => $draw,
            'recordsTotal' => 0,
            'recordsFiltered' => 0,
            'data' => []
        ];
    }
}
?>
