<?php

defined('BASEPATH') or exit('No direct script access allowed');

class Reporting_dashboard_model extends App_Model
{
    /**
     * Flag to prevent repeated schema initialization checks.
     *
     * @var bool
     */
    private $staffReportTablesInitialized = false;
    /**
     * Lead notes follow-up type column availability.
     *
     * @var bool
     */
    private $leadNotesHasFollowupType = false;

    /**
     * Lead notes stats (call connected) column availability.
     *
     * @var bool
     */
    private $leadNotesHasStats = false;
    
    /**
     * Lead notes is_new_call column availability.
     *
     * @var bool
     */
    private $leadNotesHasIsNewCall = false;

    public function __construct()
    {
        parent::__construct();
        $this->leadNotesHasFollowupType = $this->db->field_exists('followup_type', db_prefix() . 'lead_notes');
        $this->leadNotesHasStats = $this->db->field_exists('stats', db_prefix() . 'lead_notes');
        $this->leadNotesHasIsNewCall = $this->db->field_exists('is_new_call', db_prefix() . 'lead_notes');
    }

    /**
     * Ensure supporting tables for staff reports and schedules exist.
     *
     * The table creation queries are idempotent via CREATE TABLE IF NOT EXISTS.
     *
     * @return void
     */
    private function ensure_staff_report_tables()
    {
        if ($this->staffReportTablesInitialized) {
            return;
        }

        $reportsTable = db_prefix() . 'reporting_dashboard_staff_reports';
        $schedulesTable = db_prefix() . 'reporting_dashboard_staff_report_schedules';
        $logsTable = db_prefix() . 'reporting_dashboard_schedule_logs';

        // Staff reports history table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS `{$reportsTable}` (
                `id` INT(11) NOT NULL AUTO_INCREMENT,
                `generated_at` DATETIME NOT NULL,
                `generated_by` INT(11) DEFAULT NULL,
                `date_from` DATETIME NOT NULL,
                `date_to` DATETIME NOT NULL,
                `staff_ids` TEXT NULL,
                `summary_json` LONGTEXT NULL,
                `analysis_json` LONGTEXT NULL,
                `ai_model` VARCHAR(100) NULL,
                `message_preview` TEXT NULL,
                `whatsapp_numbers` TEXT NULL,
                `whatsapp_status_json` TEXT NULL,
                `notes` TEXT NULL,
                PRIMARY KEY (`id`),
                KEY `idx_generated_at` (`generated_at`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
        ");

        // Staff report schedules table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS `{$schedulesTable}` (
                `id` INT(11) NOT NULL AUTO_INCREMENT,
                `name` VARCHAR(191) NOT NULL,
                `created_by` INT(11) DEFAULT NULL,
                `frequency` ENUM('daily','weekly','monthly') NOT NULL DEFAULT 'daily',
                `range_scope` ENUM('auto','today','yesterday','last_7_days','last_30_days','this_month','last_month') NOT NULL DEFAULT 'auto',
                `schedule_time` TIME NOT NULL DEFAULT '09:00:00',
                `day_of_week` TINYINT(1) DEFAULT NULL COMMENT '0=Sunday ... 6=Saturday',
                `day_of_month` TINYINT(2) DEFAULT NULL,
                `staff_ids` TEXT NULL,
                `whatsapp_numbers` TEXT NULL,
                `active` TINYINT(1) NOT NULL DEFAULT 1,
                `last_run` DATETIME DEFAULT NULL,
                `next_run` DATETIME DEFAULT NULL,
                `last_result_json` TEXT NULL,
                `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
                `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                KEY `idx_active_next_run` (`active`,`next_run`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
        ");

        // Schedule execution logs table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS `{$logsTable}` (
                `id` INT(11) NOT NULL AUTO_INCREMENT,
                `schedule_id` INT(11) NOT NULL,
                `schedule_name` VARCHAR(255) NULL,
                `execution_time` DATETIME NOT NULL,
                `status` ENUM('success','failed','partial') NOT NULL DEFAULT 'success',
                `report_generated` TINYINT(1) NOT NULL DEFAULT 0,
                `whatsapp_sent` TINYINT(1) NOT NULL DEFAULT 0,
                `whatsapp_success_count` INT(11) DEFAULT 0,
                `whatsapp_failed_count` INT(11) DEFAULT 0,
                `whatsapp_numbers` TEXT NULL,
                `error_message` TEXT NULL,
                `execution_duration_ms` INT(11) NULL,
                `report_id` INT(11) NULL,
                `staff_count` INT(11) DEFAULT 0,
                `activity_count` INT(11) DEFAULT 0,
                `details` TEXT NULL COMMENT 'JSON data with additional details',
                PRIMARY KEY (`id`),
                KEY `idx_schedule_id` (`schedule_id`),
                KEY `idx_execution_time` (`execution_time`),
                KEY `idx_status` (`status`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
        ");

        $this->staffReportTablesInitialized = true;
    }

    /**
     * Log schedule execution
     *
     * @param int $scheduleId
     * @param string $scheduleName
     * @param array $logData
     * @return int Log ID
     */
    public function log_schedule_execution($scheduleId, $scheduleName, array $logData)
    {
        $this->ensure_staff_report_tables();

        $table = db_prefix() . 'reporting_dashboard_schedule_logs';
        
        $data = [
            'schedule_id' => (int)$scheduleId,
            'schedule_name' => $scheduleName,
            'execution_time' => date('Y-m-d H:i:s'),
            'status' => $logData['status'] ?? 'success',
            'report_generated' => isset($logData['report_generated']) ? (int)$logData['report_generated'] : 0,
            'whatsapp_sent' => isset($logData['whatsapp_sent']) ? (int)$logData['whatsapp_sent'] : 0,
            'whatsapp_success_count' => (int)($logData['whatsapp_success_count'] ?? 0),
            'whatsapp_failed_count' => (int)($logData['whatsapp_failed_count'] ?? 0),
            'whatsapp_numbers' => isset($logData['whatsapp_numbers']) ? json_encode($logData['whatsapp_numbers']) : null,
            'error_message' => $logData['error_message'] ?? null,
            'execution_duration_ms' => isset($logData['execution_duration_ms']) ? (int)$logData['execution_duration_ms'] : null,
            'report_id' => isset($logData['report_id']) ? (int)$logData['report_id'] : null,
            'staff_count' => (int)($logData['staff_count'] ?? 0),
            'activity_count' => (int)($logData['activity_count'] ?? 0),
            'details' => isset($logData['details']) ? json_encode($logData['details']) : null,
        ];

        $this->db->insert($table, $data);
        return $this->db->insert_id();
    }

    /**
     * Get schedule execution logs
     *
     * @param int|null $scheduleId Filter by schedule ID
     * @param int $limit
     * @return array
     */
    public function get_schedule_logs($scheduleId = null, $limit = 100)
    {
        $this->ensure_staff_report_tables();

        $table = db_prefix() . 'reporting_dashboard_schedule_logs';
        
        $this->db->select('*');
        $this->db->from($table);
        
        if ($scheduleId !== null) {
            $this->db->where('schedule_id', (int)$scheduleId);
        }
        
        $this->db->order_by('execution_time', 'DESC');
        $this->db->limit($limit);
        
        $results = $this->db->get()->result_array();
        
        // Decode JSON fields
        foreach ($results as &$log) {
            if (!empty($log['whatsapp_numbers'])) {
                $log['whatsapp_numbers'] = json_decode($log['whatsapp_numbers'], true);
            }
            if (!empty($log['details'])) {
                $log['details'] = json_decode($log['details'], true);
            }
        }
        
        return $results;
    }

    /**
     * Get schedule execution statistics
     *
     * @param int $scheduleId
     * @return array
     */
    public function get_schedule_stats($scheduleId)
    {
        $this->ensure_staff_report_tables();

        $table = db_prefix() . 'reporting_dashboard_schedule_logs';
        
        $this->db->select('
            COUNT(*) as total_executions,
            SUM(CASE WHEN status = "success" THEN 1 ELSE 0 END) as successful,
            SUM(CASE WHEN status = "failed" THEN 1 ELSE 0 END) as failed,
            SUM(CASE WHEN status = "partial" THEN 1 ELSE 0 END) as partial,
            SUM(whatsapp_success_count) as total_whatsapp_sent,
            SUM(whatsapp_failed_count) as total_whatsapp_failed,
            MAX(execution_time) as last_execution,
            AVG(execution_duration_ms) as avg_duration_ms
        ', false);
        $this->db->from($table);
        $this->db->where('schedule_id', (int)$scheduleId);
        
        return $this->db->get()->row_array() ?: [];
    }

    public function get_kpis(array $filters)
    {
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        
        $today = date('Y-m-d');
        $weekStart = date('Y-m-d', strtotime('monday this week'));
        $monthStart = date('Y-m-01');
        
        return [
            'today' => $this->count_lead_activities(['from' => $today, 'to' => $today, 'staff' => $filters['staff'] ?? []]),
            'week' => $this->count_lead_activities(['from' => $weekStart, 'to' => $today, 'staff' => $filters['staff'] ?? []]),
            'month' => $this->count_lead_activities(['from' => $monthStart, 'to' => $today, 'staff' => $filters['staff'] ?? []]),
            'logins_today' => $this->count_logins_today($filters['staff'] ?? []),
            'new_calls_today' => $this->count_new_calls(['from' => $today, 'to' => $today, 'staff' => $filters['staff'] ?? []]),
            'followup_calls_today' => $this->count_followup_calls(['from' => $today, 'to' => $today, 'staff' => $filters['staff'] ?? []]),
            'new_calls_week' => $this->count_new_calls(['from' => $weekStart, 'to' => $today, 'staff' => $filters['staff'] ?? []]),
            'followup_calls_week' => $this->count_followup_calls(['from' => $weekStart, 'to' => $today, 'staff' => $filters['staff'] ?? []]),
            'top_staff' => $this->get_top_staff_simple($filters),
            'dealer_stats' => $this->get_dealer_statistics($filters)
        ];
    }

    public function get_staff_activity_chart(array $filters)
    {
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        
        // Try to find the correct activity table
        $table = $this->get_activity_table_name();
        if (!$table) {
            return ['labels' => [], 'data' => []];
        }
        
        $this->db->select('s.staffid, CONCAT(s.firstname, " ", s.lastname) as staff_name, COUNT(a.id) as total');
        $this->db->from($table . ' a');
        $this->db->join(db_prefix() . 'staff s', 's.staffid = a.staffid', 'left');
        
        if ($from) $this->db->where('DATE(a.date) >=', $from);
        if ($to) $this->db->where('DATE(a.date) <=', $to);
        if (!empty($filters['staff'])) {
            $this->db->where_in('a.staffid', $filters['staff']);
        }
        
        $this->db->group_by('s.staffid');
        $this->db->order_by('total', 'DESC');
        $this->db->limit(10);
        
        $result = $this->db->get()->result_array();
        
        return [
            'labels' => array_column($result, 'staff_name'),
            'data' => array_column($result, 'total')
        ];
    }

    public function get_activity_over_time_chart(array $filters)
    {
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        
        $table = $this->get_activity_table_name();
        if (!$table) {
            return ['labels' => [], 'data' => []];
        }
        
        $this->db->select('DATE(date) as activity_date, COUNT(id) as total');
        $this->db->from($table);
        $this->db->where('DATE(date) >=', $from);
        $this->db->where('DATE(date) <=', $to);
        
        if (!empty($filters['staff'])) {
            $this->db->where_in('staffid', $filters['staff']);
        }
        
        $this->db->group_by('DATE(date)');
        $this->db->order_by('activity_date', 'ASC');
        
        $result = $this->db->get()->result_array();
        
        return [
            'labels' => array_column($result, 'activity_date'),
            'data' => array_column($result, 'total')
        ];
    }

    /**
     * Build a staff-centric activity summary with call counts and productivity metrics.
     *
     * @param array $filters
     * @return array
     */
    public function get_staff_activity_summary(array $filters)
    {
        $table = $this->get_activity_table_name();
        if (!$table) {
            return [
                'period' => [
                    'from' => null,
                    'to' => null,
                    'days' => 0,
                ],
                'staff' => [],
                'totals' => [
                    'total_activities' => 0,
                    'total_calls' => 0,
                    'average_per_staff' => 0,
                    'call_ratio_overall' => 0,
                ],
                'filters' => $filters,
            ];
        }

        $from = !empty($filters['from']) ? date('Y-m-d', strtotime($filters['from'])) : date('Y-m-01');
        $to = !empty($filters['to']) ? date('Y-m-d', strtotime($filters['to'])) : date('Y-m-d');

        $fromDateTime = $from . ' 00:00:00';
        $toDateTime = $to . ' 23:59:59';

        $daysInPeriod = max(1, (int)ceil((strtotime($toDateTime) - strtotime($fromDateTime)) / 86400) + 1);

        $useNotes = $this->leadNotesHasFollowupType || $this->leadNotesHasStats;
        $followupCase = $useNotes
            ? $this->get_followup_type_sql('a', 'ln')
            : $this->get_followup_type_sql('a');

        // Calculate calls based on followup_type instead of description
        $callTypeCase = "CASE WHEN {$followupCase} IN ('Phone Call', 'WhatsApp Call') THEN 1 ELSE 0 END";

        $this->db->select("
            a.staffid,
            CONCAT(COALESCE(s.firstname, ''), ' ', COALESCE(s.lastname, '')) AS staff_name,
            COUNT(a.id) AS total_activities,
            SUM({$callTypeCase}) AS total_calls,
            COUNT(DISTINCT CASE WHEN a.leadid IS NULL OR a.leadid = 0 THEN NULL ELSE a.leadid END) AS unique_leads,
            MAX(a.date) AS last_activity_date
        ", false);
        $this->db->from($table . ' a');
        if ($useNotes) {
            $this->db->join(
                db_prefix() . 'lead_notes ln',
                'ln.leadid = a.leadid AND ABS(TIMESTAMPDIFF(MINUTE, ln.dateadded, a.date)) <= 120',
                'left'
            );
        }
        $this->db->join(db_prefix() . 'staff s', 's.staffid = a.staffid', 'left');

        $this->db->where('a.date >=', $fromDateTime);
        $this->db->where('a.date <=', $toDateTime);

        // Filter out system-only activities
        $this->db->where("(a.description = 'not_lead_activity_added_followup' OR a.description NOT LIKE 'not_%')");

        if (!empty($filters['staff']) && is_array($filters['staff'])) {
            $this->db->where_in('a.staffid', $filters['staff']);
        }

        $this->db->group_by('a.staffid');
        $this->db->having('total_activities >', 0);
        $this->db->order_by('total_activities', 'DESC');

        $rows = $this->db->get()->result_array();

        $staffSummaries = [];
        $totalActivities = 0;
        $totalCalls = 0;
        $totalNewCalls = 0;
        $totalFollowupCalls = 0;
        $staffIds = [];

        foreach ($rows as $row) {
            $staffId = isset($row['staffid']) ? (int)$row['staffid'] : 0;
            $activities = (int)($row['total_activities'] ?? 0);
            $calls = (int)($row['total_calls'] ?? 0);
            $uniqueLeads = (int)($row['unique_leads'] ?? 0);
            $lastActivity = !empty($row['last_activity_date']) ? date('Y-m-d H:i:s', strtotime($row['last_activity_date'])) : null;

            if ($staffId === 0 && $activities === 0) {
                continue;
            }

            $staffIds[] = $staffId;
            $totalActivities += $activities;
            $totalCalls += $calls;

            $staffSummaries[] = [
                'staff_id' => $staffId,
                'staff_name' => trim($row['staff_name']) !== '' ? trim($row['staff_name']) : 'System / Unassigned',
                'total_activities' => $activities,
                'total_calls' => $calls,
                'unique_leads' => $uniqueLeads,
                'activities_per_day' => $activities > 0 ? round($activities / $daysInPeriod, 2) : 0,
                'call_ratio' => $activities > 0 ? round(($calls / $activities) * 100, 1) : 0,
                'last_activity' => $lastActivity,
            ];
        }
        
        // Get new calls and followup calls per staff (if column exists)
        $newCallsByStaff = [];
        $followupCallsByStaff = [];
        
        if ($this->leadNotesHasIsNewCall) {
            // Query for new calls per staff
            $this->db->select('addedfrom as staffid, COUNT(*) as new_calls');
            $this->db->from(db_prefix() . 'lead_notes');
            $this->db->where('dateadded >=', $fromDateTime);
            $this->db->where('dateadded <=', $toDateTime);
            $this->db->where('is_new_call', 1);
            if (!empty($filters['staff']) && is_array($filters['staff'])) {
                $this->db->where_in('addedfrom', $filters['staff']);
            }
            $this->db->group_by('addedfrom');
            $newCallsRows = $this->db->get()->result_array();
            
            foreach ($newCallsRows as $row) {
                $sid = (int)$row['staffid'];
                $newCallsByStaff[$sid] = (int)$row['new_calls'];
                $totalNewCalls += (int)$row['new_calls'];
            }
            
            // Query for followup calls per staff
            $this->db->select('addedfrom as staffid, COUNT(*) as followup_calls');
            $this->db->from(db_prefix() . 'lead_notes');
            $this->db->where('dateadded >=', $fromDateTime);
            $this->db->where('dateadded <=', $toDateTime);
            $this->db->where('is_new_call', 0);
            if (!empty($filters['staff']) && is_array($filters['staff'])) {
                $this->db->where_in('addedfrom', $filters['staff']);
            }
            $this->db->group_by('addedfrom');
            $followupCallsRows = $this->db->get()->result_array();
            
            foreach ($followupCallsRows as $row) {
                $sid = (int)$row['staffid'];
                $followupCallsByStaff[$sid] = (int)$row['followup_calls'];
                $totalFollowupCalls += (int)$row['followup_calls'];
            }
        }
        
        // Add new_calls and followup_calls to each staff summary
        foreach ($staffSummaries as &$summary) {
            $sid = $summary['staff_id'];
            $summary['new_calls'] = $newCallsByStaff[$sid] ?? 0;
            $summary['followup_calls'] = $followupCallsByStaff[$sid] ?? 0;
        }
        unset($summary);

        usort($staffSummaries, function ($a, $b) {
            return $b['total_activities'] <=> $a['total_activities'];
        });

        // Follow-up type distribution per staff
        $this->db->select("
            a.staffid,
            {$followupCase} AS followup_type,
            COUNT(a.id) AS total
        ", false);
        $this->db->from($table . ' a');
        if ($useNotes) {
            $this->db->join(
                db_prefix() . 'lead_notes ln',
                'ln.leadid = a.leadid AND ABS(TIMESTAMPDIFF(MINUTE, ln.dateadded, a.date)) <= 120',
                'left'
            );
        }
        $this->db->where('a.date >=', $fromDateTime);
        $this->db->where('a.date <=', $toDateTime);
        $this->db->where("(a.description = 'not_lead_activity_added_followup' OR a.description NOT LIKE 'not_%')");
        if (!empty($filters['staff']) && is_array($filters['staff'])) {
            $this->db->where_in('a.staffid', $filters['staff']);
        }
        $this->db->group_by(['a.staffid', 'followup_type']);

        $followupRows = $this->db->get()->result_array();

        $followupCategories = [
            'Visit',
            'Phone Call',
            'WhatsApp Call',
            'WhatsApp Message',
            'Message',
            'Email',
            'Meeting Arranged',
            'Meeting Done',
            'General'
        ];

        $overallFollowups = [];
        foreach ($followupCategories as $cat) {
            $overallFollowups[$cat] = 0;
        }

        $staffFollowups = [];
        foreach ($followupRows as $row) {
            $staffId = (int)($row['staffid'] ?? 0);
            $type = $row['followup_type'] ?? 'General';
            $count = (int)($row['total'] ?? 0);

            if (!isset($overallFollowups[$type])) {
                $overallFollowups[$type] = 0;
            }
            $overallFollowups[$type] += $count;

            if (!isset($staffFollowups[$staffId])) {
                $staffFollowups[$staffId] = [];
            }
            $staffFollowups[$staffId][$type] = $count;
        }

        $callConnectTotals = [
            'logged' => 0,
            'connected' => 0,
            'not_connected' => 0,
            'unknown' => 0,
            'connect_rate' => 0,
        ];
        $callConnectByStaff = [];

        if ($this->leadNotesHasStats) {
            $normalizedStats = "LOWER(TRIM(REPLACE(REPLACE(REPLACE(COALESCE(ln.stats, ''), 'call connected', ''), ':', ''), '-', '')))";
            $callFlag = "({$callTypeCase})";
            $connectedCaseSql = "CASE WHEN {$callFlag} = 1 AND {$normalizedStats} IN ('yes','y','connected','1','success') THEN 1 ELSE 0 END";
            $notConnectedCaseSql = "CASE WHEN {$callFlag} = 1 AND {$normalizedStats} IN ('no','n','0','not connected','missed','busy') THEN 1 ELSE 0 END";
            $loggedCaseSql = "CASE WHEN {$callFlag} = 1 AND {$normalizedStats} <> '' THEN 1 ELSE 0 END";

            $this->db->select("
                a.staffid,
                SUM({$connectedCaseSql}) AS connected_calls,
                SUM({$notConnectedCaseSql}) AS not_connected_calls,
                SUM({$loggedCaseSql}) AS logged_calls
            ", false);
            $this->db->from($table . ' a');
            $this->db->join(
                db_prefix() . 'lead_notes ln',
                'ln.leadid = a.leadid AND ABS(TIMESTAMPDIFF(MINUTE, ln.dateadded, a.date)) <= 120',
                'left'
            );
            $this->db->where('a.date >=', $fromDateTime);
            $this->db->where('a.date <=', $toDateTime);
            $this->db->where("(a.description = 'not_lead_activity_added_followup' OR a.description NOT LIKE 'not_%')");
            if (!empty($filters['staff']) && is_array($filters['staff'])) {
                $this->db->where_in('a.staffid', $filters['staff']);
            }
            $this->db->group_by('a.staffid');

            $callStatsRows = $this->db->get()->result_array();
            foreach ($callStatsRows as $row) {
                $staffId = (int)($row['staffid'] ?? 0);
                $logged = (int)($row['logged_calls'] ?? 0);
                $connected = (int)($row['connected_calls'] ?? 0);
                $notConnected = (int)($row['not_connected_calls'] ?? 0);
                $unknown = max(0, $logged - ($connected + $notConnected));
                $rate = $logged > 0 ? round(($connected / $logged) * 100, 1) : 0;

                $callConnectByStaff[$staffId] = [
                    'logged' => $logged,
                    'connected' => $connected,
                    'not_connected' => $notConnected,
                    'unknown' => $unknown,
                    'connect_rate' => $rate,
                ];

                $callConnectTotals['logged'] += $logged;
                $callConnectTotals['connected'] += $connected;
                $callConnectTotals['not_connected'] += $notConnected;
                $callConnectTotals['unknown'] += $unknown;
            }
        }

        foreach ($staffSummaries as &$summaryRow) {
            $sid = $summaryRow['staff_id'];
            $counts = $staffFollowups[$sid] ?? [];
            foreach ($followupCategories as $cat) {
                if (!isset($counts[$cat])) {
                    $counts[$cat] = 0;
                }
            }
            // sort counts desc preserving category order
            arsort($counts);
            $summaryRow['followup_counts'] = $counts;
            $summaryRow['top_followup_type'] = !empty($counts) ? array_key_first($counts) : null;

            $callStats = $callConnectByStaff[$sid] ?? [
                'logged' => 0,
                'connected' => 0,
                'not_connected' => 0,
                'unknown' => 0,
                'connect_rate' => 0,
            ];
            if ($callStats['logged'] > 0 && (!isset($callStats['connect_rate']) || $callStats['connect_rate'] === null)) {
                $callStats['connect_rate'] = round(($callStats['connected'] / max(1, $callStats['logged'])) * 100, 1);
            } else {
                $callStats['connect_rate'] = $callStats['connect_rate'] ?? 0;
            }
            $summaryRow['call_connected'] = $callStats;
        }
        unset($summaryRow);

        if ($callConnectTotals['logged'] > 0) {
            $callConnectTotals['connect_rate'] = round(($callConnectTotals['connected'] / max(1, $callConnectTotals['logged'])) * 100, 1);
        }

        // Ensure all categories exist in totals
        foreach ($followupCategories as $cat) {
            if (!isset($overallFollowups[$cat])) {
                $overallFollowups[$cat] = 0;
            }
        }

        return [
            'period' => [
                'from' => $fromDateTime,
                'to' => $toDateTime,
                'days' => $daysInPeriod,
                'label' => date('M j, Y', strtotime($from)) . ' - ' . date('M j, Y', strtotime($to)),
            ],
            'staff' => $staffSummaries,
            'staff_ids' => array_values(array_unique($staffIds)),
            'totals' => [
                'total_activities' => $totalActivities,
                'total_calls' => $totalCalls,
                'total_new_calls' => $totalNewCalls,
                'total_followup_calls' => $totalFollowupCalls,
                'average_per_staff' => !empty($staffSummaries) ? round($totalActivities / count($staffSummaries), 2) : 0,
                'call_ratio_overall' => $totalActivities > 0 ? round(($totalCalls / $totalActivities) * 100, 1) : 0,
                'staff_count' => count($staffSummaries),
                'call_connected' => $callConnectTotals,
            ],
            'followup_totals' => $overallFollowups,
            'followup_categories' => array_keys($overallFollowups),
            'generated_at' => date('Y-m-d H:i:s'),
            'filters' => [
                'from' => $filters['from'] ?? $from,
                'to' => $filters['to'] ?? $to,
                'staff' => $filters['staff'] ?? [],
            ],
        ];
    }

    /**
     * Get due/overdue reminders and missed follow-ups by staff
     *
     * @param array $filters
     * @return array
     */
    public function get_staff_reminders_followups(array $filters = [])
    {
        $currentDateTime = date('Y-m-d H:i:s');
        $staffFilter = !empty($filters['staff']) && is_array($filters['staff']) ? $filters['staff'] : [];

        // 1. Get overdue reminders (not dismissed, date passed, no completed activity)
        $this->db->select('
            r.id as reminder_id,
            r.description as reminder_description,
            r.date as due_date,
            r.isnotified,
            r.staff as staff_id,
            CONCAT(COALESCE(s.firstname, ""), " ", COALESCE(s.lastname, "")) as staff_name,
            r.rel_id as lead_id,
            r.rel_type,
            l.name as lead_name,
            l.company as lead_company,
            l.phonenumber as lead_phone,
            TIMESTAMPDIFF(DAY, r.date, NOW()) as days_overdue
        ', false);
        $this->db->from(db_prefix() . 'reminders r');
        $this->db->join(db_prefix() . 'staff s', 's.staffid = r.staff', 'left');
        $this->db->join(db_prefix() . 'leads l', 'l.id = r.rel_id AND r.rel_type = "lead"', 'left');
        $this->db->where('r.isnotified', 0);
        $this->db->where('r.date <', $currentDateTime);
        if (!empty($staffFilter)) {
            $this->db->where_in('r.staff', $staffFilter);
        }
        $this->db->order_by('r.date', 'ASC');
        $this->db->limit(500); // reasonable limit

        $overdueReminders = $this->db->get()->result_array();

        // 2. Get upcoming due reminders (within next 7 days)
        $futureDate = date('Y-m-d H:i:s', strtotime('+7 days'));
        $this->db->select('
            r.id as reminder_id,
            r.description as reminder_description,
            r.date as due_date,
            r.isnotified,
            r.staff as staff_id,
            CONCAT(COALESCE(s.firstname, ""), " ", COALESCE(s.lastname, "")) as staff_name,
            r.rel_id as lead_id,
            r.rel_type,
            l.name as lead_name,
            l.company as lead_company,
            l.phonenumber as lead_phone,
            TIMESTAMPDIFF(DAY, NOW(), r.date) as days_until_due
        ', false);
        $this->db->from(db_prefix() . 'reminders r');
        $this->db->join(db_prefix() . 'staff s', 's.staffid = r.staff', 'left');
        $this->db->join(db_prefix() . 'leads l', 'l.id = r.rel_id AND r.rel_type = "lead"', 'left');
        $this->db->where('r.isnotified', 0);
        $this->db->where('r.date >=', $currentDateTime);
        $this->db->where('r.date <=', $futureDate);
        if (!empty($staffFilter)) {
            $this->db->where_in('r.staff', $staffFilter);
        }
        $this->db->order_by('r.date', 'ASC');
        $this->db->limit(500);

        $upcomingReminders = $this->db->get()->result_array();

        // 3. Aggregate by staff
        $staffSummary = [];
        
        // Process overdue reminders
        foreach ($overdueReminders as $reminder) {
            $staffId = (int)$reminder['staff_id'];
            if (!isset($staffSummary[$staffId])) {
                $staffSummary[$staffId] = [
                    'staff_id' => $staffId,
                    'staff_name' => $reminder['staff_name'] ?: 'Unassigned',
                    'overdue_count' => 0,
                    'upcoming_count' => 0,
                    'overdue_reminders' => [],
                    'upcoming_reminders' => [],
                    'oldest_overdue_days' => 0,
                ];
            }
            $staffSummary[$staffId]['overdue_count']++;
            $staffSummary[$staffId]['overdue_reminders'][] = [
                'reminder_id' => $reminder['reminder_id'],
                'description' => $reminder['reminder_description'],
                'due_date' => $reminder['due_date'],
                'days_overdue' => (int)$reminder['days_overdue'],
                'lead_name' => $reminder['lead_name'] ?: 'N/A',
                'lead_company' => $reminder['lead_company'] ?: '',
                'lead_phone' => $reminder['lead_phone'] ?: '',
                'lead_id' => $reminder['lead_id'],
            ];

            if ((int)$reminder['days_overdue'] > $staffSummary[$staffId]['oldest_overdue_days']) {
                $staffSummary[$staffId]['oldest_overdue_days'] = (int)$reminder['days_overdue'];
            }
        }

        // Process upcoming reminders
        foreach ($upcomingReminders as $reminder) {
            $staffId = (int)$reminder['staff_id'];
            if (!isset($staffSummary[$staffId])) {
                $staffSummary[$staffId] = [
                    'staff_id' => $staffId,
                    'staff_name' => $reminder['staff_name'] ?: 'Unassigned',
                    'overdue_count' => 0,
                    'upcoming_count' => 0,
                    'overdue_reminders' => [],
                    'upcoming_reminders' => [],
                    'oldest_overdue_days' => 0,
                ];
            }
            $staffSummary[$staffId]['upcoming_count']++;
            $staffSummary[$staffId]['upcoming_reminders'][] = [
                'reminder_id' => $reminder['reminder_id'],
                'description' => $reminder['reminder_description'],
                'due_date' => $reminder['due_date'],
                'days_until_due' => (int)$reminder['days_until_due'],
                'lead_name' => $reminder['lead_name'] ?: 'N/A',
                'lead_company' => $reminder['lead_company'] ?: '',
                'lead_phone' => $reminder['lead_phone'] ?: '',
                'lead_id' => $reminder['lead_id'],
            ];
        }

        // Sort staff by most overdue count first
        usort($staffSummary, function($a, $b) {
            if ($b['overdue_count'] !== $a['overdue_count']) {
                return $b['overdue_count'] <=> $a['overdue_count'];
            }
            return $b['upcoming_count'] <=> $a['upcoming_count'];
        });

        // Calculate totals
        $totalOverdue = array_sum(array_column($staffSummary, 'overdue_count'));
        $totalUpcoming = array_sum(array_column($staffSummary, 'upcoming_count'));

        return [
            'staff_summary' => array_values($staffSummary),
            'totals' => [
                'total_overdue' => $totalOverdue,
                'total_upcoming' => $totalUpcoming,
                'staff_with_overdue' => count(array_filter($staffSummary, function($s) {
                    return $s['overdue_count'] > 0;
                })),
                'staff_with_upcoming' => count(array_filter($staffSummary, function($s) {
                    return $s['upcoming_count'] > 0;
                })),
            ],
            'all_overdue_reminders' => $overdueReminders,
            'all_upcoming_reminders' => $upcomingReminders,
            'generated_at' => $currentDateTime,
        ];
    }

    /**
     * Enhanced activity table with modern features
     */
    public function get_activity_table(array $filters)
    {
        $table = $this->get_activity_table_name();
        if (!$table) {
            error_log('No activity table found');
            return ['data' => [], 'recordsTotal' => 0, 'recordsFiltered' => 0];
        }

        error_log('Using activity table: ' . $table);

        // DataTables parameters - check POST first, then GET as fallback
        $draw = (int)($this->input->post('draw') ?: $this->input->get('draw'));
        $start = (int)($this->input->post('start') ?: $this->input->get('start'));
        $length = (int)($this->input->post('length') ?: $this->input->get('length'));
        
        // For client-side DataTables, return all data (up to reasonable limit)
        // Frontend will handle pagination
        if ($length <= 0) {
            $length = 1000; // Default to getting all data for client-side pagination
        } elseif ($length == -1) {
            $length = 1000; // Set a reasonable maximum
        }
        
        error_log("Activity table pagination - Start: $start, Length: $length, Draw: $draw");

        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');

        error_log('Activity table query - From: ' . $from . ', To: ' . $to);

        // Enhanced followup type extraction from actual notes table
        $joinLeadNotes = $this->leadNotesHasFollowupType || $this->leadNotesHasStats;
        if ($this->leadNotesHasFollowupType) {
            $followupCaseActivity = $this->get_followup_type_sql('a', 'ln');
            $this->db->select("a.*, s.firstname, s.lastname, l.name as lead_name, l.company as lead_company, l.phonenumber as lead_phone, l.email as lead_email, {$followupCaseActivity} as followup_type, COALESCE(ln.description, a.description) as original_description", false);
        } else {
            $followupCaseActivity = $this->get_followup_type_sql('a');
            $this->db->select("a.*, s.firstname, s.lastname, l.name as lead_name, l.company as lead_company, l.phonenumber as lead_phone, l.email as lead_email, {$followupCaseActivity} as followup_type, a.description as original_description", false);
        }
        if ($this->leadNotesHasStats) {
            $callTypeCase = "CASE WHEN {$followupCaseActivity} IN ('Phone Call','WhatsApp Call') THEN 1 ELSE 0 END";
            $normalizedStats = "LOWER(TRIM(REPLACE(REPLACE(REPLACE(COALESCE(ln.stats, ''), 'call connected', ''), ':', ''), '-', '')))";
            $callConnectCase = "
                CASE
                    WHEN {$callTypeCase} = 1 AND {$normalizedStats} IN ('yes','y','connected','1','success','answer','answered') THEN 'Connected'
                    WHEN {$callTypeCase} = 1 AND {$normalizedStats} IN ('no','n','0','not connected','missed','busy','failed') THEN 'Not Connected'
                    WHEN {$callTypeCase} = 1 AND {$normalizedStats} <> '' THEN 'Unknown'
                    ELSE NULL
                END
            ";
            $this->db->select("{$callConnectCase} AS call_connected_status", false);
        }
        $this->db->from($table . ' a');
        $this->db->join(db_prefix() . 'staff s', 's.staffid = a.staffid', 'left');
        $this->db->join(db_prefix() . 'leads l', 'l.id = a.leadid', 'left');
        if ($joinLeadNotes) {
            $this->db->join(db_prefix() . 'lead_notes ln', 'ln.leadid = a.leadid AND ABS(TIMESTAMPDIFF(MINUTE, ln.dateadded, a.date)) <= 120', 'left');
        }
        
        // Only show actual followup activities, not system notifications
        $this->db->where("(a.description = 'not_lead_activity_added_followup' OR a.description NOT LIKE 'not_%')");
        
        if ($from) $this->db->where('DATE(a.date) >=', $from);
        if ($to) $this->db->where('DATE(a.date) <=', $to);
        
        // Staff filtering
        if (!empty($filters['staff']) && is_array($filters['staff'])) {
            $this->db->where_in('a.staffid', $filters['staff']);
        }

        // Search functionality
        $search = $this->input->post('search');
        if (!empty($search['value'])) {
            $search_term = $search['value'];
            $this->db->group_start();
            $this->db->like('a.description', $search_term);
            $this->db->or_like('CONCAT(s.firstname, " ", s.lastname)', $search_term);
            $this->db->or_like('l.name', $search_term);
            $this->db->or_like('l.company', $search_term);
            $this->db->group_end();
        }

        // Get total count with a fresh query to avoid issues with JOINs
        $count_sql = "SELECT COUNT(DISTINCT a.id) as total_count FROM " . $table . " a ";
        $count_sql .= "LEFT JOIN " . db_prefix() . "staff s ON s.staffid = a.staffid ";
        $count_sql .= "LEFT JOIN " . db_prefix() . "leads l ON l.id = a.leadid ";
        if ($this->leadNotesHasFollowupType || $this->leadNotesHasStats) {
            $count_sql .= "LEFT JOIN " . db_prefix() . "lead_notes ln ON ln.leadid = a.leadid AND ABS(TIMESTAMPDIFF(MINUTE, ln.dateadded, a.date)) <= 120 ";
        }
        $count_sql .= "WHERE (a.description = 'not_lead_activity_added_followup' OR a.description NOT LIKE 'not_%') ";
        
        $count_params = [];
        if ($from) {
            $count_sql .= "AND DATE(a.date) >= ? ";
            $count_params[] = $from;
        }
        if ($to) {
            $count_sql .= "AND DATE(a.date) <= ? ";
            $count_params[] = $to;
        }
        
        if (!empty($filters['staff']) && is_array($filters['staff'])) {
            $placeholders = str_repeat('?,', count($filters['staff']) - 1) . '?';
            $count_sql .= "AND a.staffid IN ($placeholders) ";
            $count_params = array_merge($count_params, $filters['staff']);
        }
        
        // Search filter for count
        $search = $this->input->post('search');
        if (!empty($search['value'])) {
            $search_term = '%' . $search['value'] . '%';
            $count_sql .= "AND (a.description LIKE ? OR CONCAT(s.firstname, ' ', s.lastname) LIKE ? OR l.name LIKE ? OR l.company LIKE ?) ";
            $count_params[] = $search_term;
            $count_params[] = $search_term;
            $count_params[] = $search_term;
            $count_params[] = $search_term;
        }
        
        try {
            $count_result = $this->db->query($count_sql, $count_params)->row();
            $total = $count_result ? (int)$count_result->total_count : 0;
            error_log('Activity table total count: ' . $total);
        } catch (Exception $e) {
            error_log('Count query error: ' . $e->getMessage());
            $total = 0;
        }
        
        $this->db->order_by('a.date', 'DESC');
        
        // For client-side DataTables, don't apply LIMIT - get all data
        // Apply LIMIT only if specifically requested for server-side pagination
        if ($start > 0 || ($length < 1000 && $length > 0)) {
            $this->db->limit($length, $start);
            error_log("Applying LIMIT: $length OFFSET: $start");
        } else {
            error_log("Getting all records for client-side pagination (no LIMIT applied)");
        }
        
        try {
            $query_result = $this->db->get();
            $rows = $query_result->result_array();
            error_log('Activity table query successful: ' . count($rows) . ' rows');
        } catch (Exception $e) {
            error_log('Activity table query error: ' . $e->getMessage());
            return ['data' => [], 'recordsTotal' => 0, 'recordsFiltered' => 0];
        }

        $data = [];
        foreach ($rows as $r) {
            $staff_name = trim(($r['firstname'] ?? '') . ' ' . ($r['lastname'] ?? '')) ?: 'System';
            
            $lead_id = (int)($r['leadid'] ?? 0);
            $lead_name = trim($r['lead_name'] ?? '');
            $lead_company = trim($r['lead_company'] ?? '');
            $lead_phone = trim($r['lead_phone'] ?? '');
            $lead_email = trim($r['lead_email'] ?? '');
            
            $lead_display = $this->build_lead_display($lead_name, $lead_company, $lead_phone, $lead_email);
            $followup_type = $r['followup_type'] ?? 'General';
            
            // Use original description from notes if available, otherwise use activity description
            $description = !empty($r['original_description']) ? $r['original_description'] : ($r['description'] ?? '');
            
            $data[] = [
                'id' => $r['id'],
                'date' => date('Y-m-d H:i', strtotime($r['date'] ?? 'now')),
                'staff' => $staff_name,
                'staff_id' => $r['staffid'] ?? 0,
                'lead' => $lead_display,
                'lead_id' => $lead_id,
                'lead_name' => $lead_name,
                'lead_company' => $lead_company,
                'lead_phone' => $lead_phone,
                'lead_email' => $lead_email,
                'followup_type' => $followup_type,
                'description' => $description,
                'priority' => $this->determine_priority($r),
                'call_connected_status' => $r['call_connected_status'] ?? null,
            ];
        }

        return [
            'draw' => $draw,
            'recordsTotal' => $total,
            'recordsFiltered' => $total,
            'data' => $data
        ];
    }

    /**
     * Find the correct activity table name
     */
    private function get_activity_table_name()
    {
        $table_variations = ['lead_activity_log', 'activity_log', 'activities'];
        
        foreach ($table_variations as $variation) {
            if ($this->db->table_exists($variation)) {
                return db_prefix() . $variation;
            }
        }
        
        return null;
    }

    /**
     * Get SQL for followup type detection
     */
    private function get_followup_type_sql($activityAlias = 'a', $noteAlias = null)
    {
        $activityField = "LOWER({$activityAlias}.description)";
        $noteTypeField = ($noteAlias && $this->leadNotesHasFollowupType) ? "LOWER({$noteAlias}.followup_type)" : 'NULL';
        $noteDescField = $noteAlias ? "LOWER({$noteAlias}.description)" : 'NULL';

        $activityCase = $this->build_followup_case_sql($activityField);
        $noteTypeCase = $this->build_followup_case_sql($noteTypeField);
        $noteDescCase = $this->build_followup_case_sql($noteDescField);

        if ($noteAlias) {
            $cases = [];
            if ($this->leadNotesHasFollowupType) {
                $cases[] = "WHEN {$noteAlias}.followup_type IS NOT NULL AND {$noteAlias}.followup_type != '' THEN {$noteTypeCase}";
            }
            $cases[] = "WHEN {$noteAlias}.description IS NOT NULL AND {$noteAlias}.description != '' THEN {$noteDescCase}";
            $cases[] = "ELSE {$activityCase}";

            return "(CASE " . implode(' ', $cases) . " END)";
        }

        return "({$activityCase})";
    }

    /**
     * Build SQL CASE statement to classify follow-up type based on text expression.
     *
     * @param string $expression
     * @return string
     */
    private function build_followup_case_sql($expression)
    {
        return "CASE
            WHEN {$expression} IS NULL THEN 'General'
            WHEN {$expression} REGEXP '(meeting.*arranged|arranged.*meeting|meeting.*scheduled|scheduled.*meeting)' THEN 'Meeting Arranged'
            WHEN {$expression} REGEXP '(meeting.*done|done.*meeting|meeting.*completed|completed.*meeting|meeting.*finished|finished.*meeting)' THEN 'Meeting Done'
            WHEN {$expression} REGEXP '(visit|site visit|meeting|appointment)' THEN 'Visit'
            WHEN {$expression} REGEXP '(whatsapp|whats.app|wa )' AND {$expression} REGEXP '(call|phone|dial|rang|ring|spoke|connected)' THEN 'WhatsApp Call'
            WHEN {$expression} REGEXP '(whatsapp|whats.app|wa )' THEN 'WhatsApp Message'
            WHEN {$expression} REGEXP '(email|mail|sent.*email)' THEN 'Email'
            WHEN {$expression} REGEXP '(sms|text|message)' THEN 'Message'
            WHEN {$expression} REGEXP '(call|phone|dial|rang|ring|spoke|connected)' THEN 'Phone Call'
            ELSE 'General'
        END";
    }

    /**
     * SQL CASE statement that approximates whether an activity represents a call.
     *
     * @param string $alias Table alias for the activity table.
     * @return string
     */
    private function get_call_detection_case_sql($alias = 'a')
    {
        $field = $alias . '.description';

        return "
            CASE
                WHEN LOWER({$field}) REGEXP '(phone|call|dial|rang|spoke|discussion|connected)' THEN 1
                WHEN LOWER({$field}) LIKE '%call%' THEN 1
                WHEN LOWER({$field}) LIKE '%dial%' THEN 1
                WHEN LOWER({$field}) LIKE '%spoke%' THEN 1
                WHEN LOWER({$field}) LIKE '%connected%' THEN 1
                ELSE 0
            END
        ";
    }

    /**
     * Build lead display with contact info
     */
    private function build_lead_display($name, $company, $phone, $email)
    {
        $parts = [];
        if ($name) $parts[] = $name;
        if ($company) $parts[] = '(' . $company . ')';
        
        $main = !empty($parts) ? implode(' ', $parts) : 'Unknown Lead';
        
        $contacts = [];
        if ($phone) $contacts[] = $phone;
        // Show company in contact details instead of email
        if ($company && !in_array('(' . $company . ')', $parts)) {
            $contacts[] = $company;
        }
        
        if (!empty($contacts)) {
            $main .= ' • ' . implode(' • ', $contacts);
        }
        
        return $main;
    }

    /**
     * Determine activity priority
     */
    private function determine_priority($row)
    {
        $date = $row['date'] ?? '';
        $hours_ago = $date ? (time() - strtotime($date)) / 3600 : 999;
        
        if ($hours_ago <= 2) return 'high';
        if ($hours_ago <= 24) return 'medium';
        return 'normal';
    }

    /**
     * Helper functions for KPI calculation
     */
    private function count_lead_activities($filters) 
    {
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        $staff = $filters['staff'] ?? [];
        
        $table = $this->get_activity_table_name();
        if (!$table) return 0;
        
        $this->db->from($table);
        $this->db->where('DATE(date) >=', $from);
        $this->db->where('DATE(date) <=', $to);
        
        if (!empty($staff)) {
            $this->db->where_in('staffid', $staff);
        }
        
        return $this->db->count_all_results();
    }
    
    private function count_logins_today($staff) 
    {
        return 0; // Placeholder
    }
    
    /**
     * Count new calls (first-time calls to leads) within the date range
     */
    private function count_new_calls($filters) 
    {
        if (!$this->leadNotesHasIsNewCall) {
            return 0;
        }
        
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        $staff = $filters['staff'] ?? [];
        
        $this->db->from(db_prefix() . 'lead_notes');
        $this->db->where('DATE(dateadded) >=', $from);
        $this->db->where('DATE(dateadded) <=', $to);
        $this->db->where('is_new_call', 1); // Only count new calls
        
        if (!empty($staff)) {
            $this->db->where_in('addedfrom', $staff);
        }
        
        return $this->db->count_all_results();
    }
    
    /**
     * Count follow-up calls (subsequent calls to already-called leads) within the date range
     */
    private function count_followup_calls($filters) 
    {
        if (!$this->leadNotesHasIsNewCall) {
            return 0;
        }
        
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        $staff = $filters['staff'] ?? [];
        
        $this->db->from(db_prefix() . 'lead_notes');
        $this->db->where('DATE(dateadded) >=', $from);
        $this->db->where('DATE(dateadded) <=', $to);
        $this->db->where('is_new_call', 0); // Only count follow-up calls
        
        if (!empty($staff)) {
            $this->db->where_in('addedfrom', $staff);
        }
        
        return $this->db->count_all_results();
    }
    
    private function get_top_staff_simple($filters) 
    {
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        
        $table = $this->get_activity_table_name();
        if (!$table) return [];
        
        $this->db->select('s.staffid, CONCAT(s.firstname, " ", s.lastname) as staff_name, COUNT(a.id) as total');
        $this->db->from($table . ' a');
        $this->db->join(db_prefix() . 'staff s', 's.staffid = a.staffid', 'left');
        $this->db->where('DATE(a.date) >=', $from);
        $this->db->where('DATE(a.date) <=', $to);
        $this->db->group_by('s.staffid');
        $this->db->order_by('total', 'DESC');
        $this->db->limit(5);
        
        return $this->db->get()->result_array();
    }

    // Method for email summary
    public function get_email_summary_data()
    {
        return [
            'kpis' => $this->get_kpis([])
        ];
    }

    /**
     * Return reminders grouped for dashboard
     */
    public function get_reminders(array $filters)
    {
        return ['todays' => [], 'due' => [], 'upcoming' => []];
    }

    /**
     * Get activity type pie chart data
     */
    public function get_activity_type_pie(array $filters)
    {
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        
        $table = $this->get_activity_table_name();
        if (!$table) {
            return ['labels' => [], 'data' => []];
        }
        
        $followupCaseChart = $this->leadNotesHasFollowupType
            ? $this->get_followup_type_sql('a', 'ln')
            : $this->get_followup_type_sql('a');
        $this->db->select("{$followupCaseChart} as followup_type, COUNT(a.id) as count", false);
        $this->db->from($table . ' a');
        if ($this->leadNotesHasFollowupType) {
            $this->db->join(db_prefix() . 'lead_notes ln', 'ln.leadid = a.leadid AND ABS(TIMESTAMPDIFF(MINUTE, ln.dateadded, a.date)) <= 120', 'left');
        }
        
        // Only show actual followup activities, not system notifications
        $this->db->where("(a.description = 'not_lead_activity_added_followup' OR a.description NOT LIKE 'not_%')");
        
        if ($from) $this->db->where('DATE(a.date) >=', $from);
        if ($to) $this->db->where('DATE(a.date) <=', $to);
        
        if (!empty($filters['staff'])) {
            $this->db->where_in('a.staffid', $filters['staff']);
        }
        
        $this->db->group_by('followup_type');
        $this->db->order_by('count', 'DESC');
        
        $result = $this->db->get()->result_array();
        
        return [
            'labels' => array_column($result, 'followup_type'),
            'data' => array_column($result, 'count')
        ];
    }

    /**
     * Get followup types chart data
     */
    public function get_followup_types_chart(array $filters)
    {
        return $this->get_activity_type_pie($filters);
    }

    /**
     * Get dealer statistics
     */
public function get_dealer_statistics(array $filters)
    {
        try {
            // NOTE: Dealer statistics show ALL dealers regardless of date range
            // because dealer interest is tracked over time, not limited to when leads were created
            
            // Get total dealers (no date filter - we want all dealers)
            $this->db->select('COUNT(*) as total_dealers');
            $this->db->from(db_prefix() . 'leads l');
            $this->db->join(db_prefix() . 'leads_status ls', 'ls.id = l.status', 'left');
            $this->db->where('LOWER(ls.name)', 'dealer');
            
            // No date filtering for total dealers count
            
            $result = $this->db->get();
            if (!$result) {
                error_log('Dealer Statistics Error: Failed to get total dealers - ' . $this->db->last_query());
                return ['total_dealers' => 0, 'interested_dealers' => 0, 'not_interested_dealers' => 0, 'no_response_dealers' => 0];
            }
            
            $total_dealers = $result->row()->total_dealers;
            
            // Get interested dealers - count distinct leads with interested status
            $this->db->select('COUNT(DISTINCT l.id) as interested_dealers');
            $this->db->from(db_prefix() . 'leads l');
            $this->db->join(db_prefix() . 'leads_status ls', 'ls.id = l.status', 'left');
            $this->db->join(db_prefix() . 'lead_notes ln', 'ln.leadid = l.id', 'left');
            $this->db->where('LOWER(ls.name)', 'dealer');
            $this->db->where('ln.dealer_interest', 'interested');
            
            // No date filtering - we want all interested dealers
            
            $interested_result = $this->db->get();
            $interested_dealers = $interested_result ? $interested_result->row()->interested_dealers : 0;
            
            // Get not interested dealers - count distinct leads with not_interested status
            $this->db->select('COUNT(DISTINCT l.id) as not_interested_dealers');
            $this->db->from(db_prefix() . 'leads l');
            $this->db->join(db_prefix() . 'leads_status ls', 'ls.id = l.status', 'left');
            $this->db->join(db_prefix() . 'lead_notes ln', 'ln.leadid = l.id', 'left');
            $this->db->where('LOWER(ls.name)', 'dealer');
            $this->db->where('ln.dealer_interest', 'not_interested');
            
            // No date filtering - we want all not interested dealers
            
            $not_interested_result = $this->db->get();
            $not_interested_dealers = $not_interested_result ? $not_interested_result->row()->not_interested_dealers : 0;
            
            // Calculate no response dealers (total - interested - not_interested)
            $no_response_dealers = $total_dealers - $interested_dealers - $not_interested_dealers;
            
            // Ensure no negative values
            if ($no_response_dealers < 0) {
                $no_response_dealers = 0;
            }
            
            // Debug logging
            error_log('Dealer Statistics Debug: Total=' . $total_dealers . ', Interested=' . $interested_dealers . ', Not Interested=' . $not_interested_dealers . ', No Response=' . $no_response_dealers);
        
            return [
                'total_dealers' => (int)$total_dealers,
                'interested_dealers' => (int)$interested_dealers,
                'not_interested_dealers' => (int)$not_interested_dealers,
                'no_response_dealers' => (int)$no_response_dealers
            ];
        } catch (Exception $e) {
            error_log('Dealer Statistics Error: ' . $e->getMessage());
            error_log('Dealer Statistics Error Trace: ' . $e->getTraceAsString());
            return [
                'total_dealers' => 0,
                'interested_dealers' => 0,
                'not_interested_dealers' => 0,
                'no_response_dealers' => 0
            ];
        }
    }

    /**
     * Get dealer list by interest status
     */
    public function get_dealer_list_by_interest($interest_status, array $filters)
    {
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        
        // Debug logging
        error_log('Dealer List Query Debug: Interest Status=' . $interest_status . ', From=' . $from . ', To=' . $to);
        
        try {
            if ($interest_status === 'no_response') {
                // For no response dealers, use LEFT JOIN with IS NULL
                $this->db->select('l.id, l.name, l.email, l.phonenumber, l.company, l.dateadded, NULL as interest_date');
                $this->db->from(db_prefix() . 'leads l');
                $this->db->join(db_prefix() . 'leads_status ls', 'ls.id = l.status', 'left');
                $this->db->join(db_prefix() . 'lead_notes ln', 'ln.leadid = l.id AND ln.dealer_interest IS NOT NULL', 'left');
                $this->db->where('LOWER(ls.name)', 'dealer');
                $this->db->where('ln.leadid IS NULL');
                
                if ($from) $this->db->where('DATE(l.dateadded) >=', $from);
                if ($to) $this->db->where('DATE(l.dateadded) <=', $to);
                
                $this->db->order_by('l.dateadded', 'DESC');
            } else {
                // For interested/not_interested dealers, find those with specific dealer_interest values
                $this->db->select('l.id, l.name, l.email, l.phonenumber, l.company, l.dateadded, ln.dateadded as interest_date');
                $this->db->from(db_prefix() . 'leads l');
                $this->db->join(db_prefix() . 'leads_status ls', 'ls.id = l.status', 'left');
                $this->db->join(db_prefix() . 'lead_notes ln', 'ln.leadid = l.id', 'left');
                $this->db->where('LOWER(ls.name)', 'dealer');
                $this->db->where('ln.dealer_interest', $interest_status);
                
                if ($from) $this->db->where('DATE(l.dateadded) >=', $from);
                if ($to) $this->db->where('DATE(l.dateadded) <=', $to);
                
                $this->db->order_by('ln.dateadded', 'DESC');
            }
            
            $query = $this->db->get();
            error_log('Dealer List Query: ' . $this->db->last_query());
            
            if (!$query) {
                error_log('Dealer List Query Error: ' . $this->db->error()['message']);
                return [];
            }
            
            return $query->result_array();
            
        } catch (Exception $e) {
            error_log('Dealer List Query Exception: ' . $e->getMessage());
            error_log('Dealer List Query Exception Trace: ' . $e->getTraceAsString());
            return [];
        }
    }

    /**
     * Get activity log for PDF reports
     */
    public function get_activity_log(array $filters, $limit = 50)
    {
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        
        $table = $this->get_activity_table_name();
        if (!$table) {
            return [];
        }
        
        $joinLeadNotes = $this->leadNotesHasFollowupType || $this->leadNotesHasStats;
        if ($this->leadNotesHasFollowupType) {
            $followupCaseLog = $this->get_followup_type_sql('a', 'ln');
            $this->db->select("a.*, s.firstname, s.lastname, l.name as lead_name, l.company as lead_company, l.phonenumber as lead_phone, l.email as lead_email, {$followupCaseLog} as followup_type, COALESCE(ln.description, a.description) as original_description", false);
        } else {
            $followupCaseLog = $this->get_followup_type_sql('a');
            $this->db->select("a.*, s.firstname, s.lastname, l.name as lead_name, l.company as lead_company, l.phonenumber as lead_phone, l.email as lead_email, {$followupCaseLog} as followup_type, a.description as original_description", false);
        }
        if ($this->leadNotesHasStats) {
            $callTypeCase = "CASE WHEN {$followupCaseLog} IN ('Phone Call','WhatsApp Call') THEN 1 ELSE 0 END";
            $normalizedStats = "LOWER(TRIM(REPLACE(REPLACE(REPLACE(COALESCE(ln.stats, ''), 'call connected', ''), ':', ''), '-', '')))";
            $callConnectCase = "
                CASE
                    WHEN {$callTypeCase} = 1 AND {$normalizedStats} IN ('yes','y','connected','1','success','answer','answered') THEN 'Connected'
                    WHEN {$callTypeCase} = 1 AND {$normalizedStats} IN ('no','n','0','not connected','missed','busy','failed') THEN 'Not Connected'
                    WHEN {$callTypeCase} = 1 AND {$normalizedStats} <> '' THEN 'Unknown'
                    ELSE NULL
                END
            ";
            $this->db->select("{$callConnectCase} AS call_connected_status", false);
        }
        $this->db->from($table . ' a');
        $this->db->join(db_prefix() . 'staff s', 's.staffid = a.staffid', 'left');
        $this->db->join(db_prefix() . 'leads l', 'l.id = a.leadid', 'left');
        if ($joinLeadNotes) {
            $this->db->join(db_prefix() . 'lead_notes ln', 'ln.leadid = a.leadid AND ABS(TIMESTAMPDIFF(MINUTE, ln.dateadded, a.date)) <= 120', 'left');
        }
        
        // Only show actual followup activities, not system notifications
        $this->db->where("(a.description = 'not_lead_activity_added_followup' OR a.description NOT LIKE 'not_%')");
        
        if ($from) $this->db->where('DATE(a.date) >=', $from);
        if ($to) $this->db->where('DATE(a.date) <=', $to);
        
        if (!empty($filters['staff']) && is_array($filters['staff'])) {
            $this->db->where_in('a.staffid', $filters['staff']);
        }
        
        $this->db->order_by('a.date', 'DESC');
        
        // Only apply limit if it's a positive number
        if ($limit > 0) {
            $this->db->limit($limit);
        }
        
        $result = $this->db->get()->result_array();
        
        // Format the data for PDF display
        $formatted_data = [];
        foreach ($result as $row) {
            $staff_name = trim(($row['firstname'] ?? '') . ' ' . ($row['lastname'] ?? '')) ?: 'System';
            $lead_name = trim($row['lead_name'] ?? '');
            $lead_company = trim($row['lead_company'] ?? '');
            $lead_display = $this->build_lead_display($lead_name, $lead_company, $row['lead_phone'] ?? '', $row['lead_email'] ?? '');
            
            $formatted_data[] = [
                'id' => $row['id'],
                'date' => date('Y-m-d H:i', strtotime($row['date'] ?? 'now')),
                'staff' => $staff_name,
                'lead' => $lead_display,
                'followup_type' => $row['followup_type'] ?? 'General',
                'description' => !empty($row['original_description']) ? $row['original_description'] : ($row['description'] ?? ''),
                'priority' => $this->determine_priority($row),
                'call_connected_status' => $row['call_connected_status'] ?? null,
            ];
        }
        
        return $formatted_data;
    }

    /**
     * Get followup types by staff for PDF reports
     */
    public function get_followup_types_by_staff(array $filters)
    {
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        
        $table = $this->get_activity_table_name();
        if (!$table) {
            return [];
        }
        
        $followupCaseByStaff = $this->leadNotesHasFollowupType
            ? $this->get_followup_type_sql('a', 'ln')
            : $this->get_followup_type_sql('a');
        $this->db->select("s.staffid, CONCAT(s.firstname, ' ', s.lastname) as staff_name, {$followupCaseByStaff} as followup_type, COUNT(a.id) as count", false);
        $this->db->from($table . ' a');
        $this->db->join(db_prefix() . 'staff s', 's.staffid = a.staffid', 'left');
        if ($this->leadNotesHasFollowupType) {
            $this->db->join(db_prefix() . 'lead_notes ln', 'ln.leadid = a.leadid AND ABS(TIMESTAMPDIFF(MINUTE, ln.dateadded, a.date)) <= 120', 'left');
        }
        
        // Only show actual followup activities, not system notifications
        $this->db->where("(a.description = 'not_lead_activity_added_followup' OR a.description NOT LIKE 'not_%')");
        
        if ($from) $this->db->where('DATE(a.date) >=', $from);
        if ($to) $this->db->where('DATE(a.date) <=', $to);
        
        if (!empty($filters['staff']) && is_array($filters['staff'])) {
            $this->db->where_in('a.staffid', $filters['staff']);
        }
        
        $this->db->group_by(['s.staffid', 'followup_type']);
        $this->db->order_by('staff_name, count', 'DESC');
        
        $result = $this->db->get()->result_array();
        
        // Group by staff for better presentation
        $grouped_data = [];
        $allTypesSet = [];
        $staffIdMap = [];
        foreach ($result as $row) {
            $staffId = isset($row['staffid']) ? (int)$row['staffid'] : 0;
            $staff_name = $row['staff_name'] ?: 'System';
            if (!isset($grouped_data[$staff_name])) {
                $grouped_data[$staff_name] = [];
            }
            if (!isset($staffIdMap[$staff_name])) {
                $staffIdMap[$staff_name] = $staffId;
            }
            $grouped_data[$staff_name][] = [
                'followup_type' => $row['followup_type'],
                'count' => (int)$row['count']
            ];
            if (!empty($row['followup_type'])) {
                $allTypesSet[$row['followup_type']] = true;
            }
        }

        $callConnectedByStaff = [];
        $callConnectTotals = [
            'logged' => 0,
            'connected' => 0,
            'not_connected' => 0,
            'unknown' => 0,
            'connect_rate' => 0,
        ];

        if ($this->leadNotesHasStats && $table) {
            $callStatsRows = $this->fetch_call_connect_stats($filters, $table);
            foreach ($staffIdMap as $name => $id) {
                if (isset($callStatsRows['by_staff'][$id])) {
                    $callConnectedByStaff[$name] = $callStatsRows['by_staff'][$id];
                }
            }
            $callConnectTotals = $callStatsRows['totals'];
        }
        
        // Get new calls and followup calls per staff
        $newCallsByStaff = [];
        $followupCallsByStaff = [];
        $totalNewCalls = 0;
        $totalFollowupCalls = 0;
        
        if ($this->leadNotesHasIsNewCall) {
            $fromDateTime = $from . ' 00:00:00';
            $toDateTime = $to . ' 23:59:59';
            
            // Query for new calls per staff
            $this->db->select('ln.addedfrom as staffid, COUNT(*) as new_calls');
            $this->db->from(db_prefix() . 'lead_notes ln');
            $this->db->where('ln.dateadded >=', $fromDateTime);
            $this->db->where('ln.dateadded <=', $toDateTime);
            $this->db->where('ln.is_new_call', 1);
            if (!empty($filters['staff']) && is_array($filters['staff'])) {
                $this->db->where_in('ln.addedfrom', $filters['staff']);
            }
            $this->db->group_by('ln.addedfrom');
            $newCallsRows = $this->db->get()->result_array();
            
            foreach ($newCallsRows as $row) {
                $sid = (int)$row['staffid'];
                $count = (int)$row['new_calls'];
                $totalNewCalls += $count;
                // Find staff name
                foreach ($staffIdMap as $name => $id) {
                    if ($id === $sid) {
                        $newCallsByStaff[$name] = $count;
                        break;
                    }
                }
            }
            
            // Query for followup calls per staff
            $this->db->select('ln.addedfrom as staffid, COUNT(*) as followup_calls');
            $this->db->from(db_prefix() . 'lead_notes ln');
            $this->db->where('ln.dateadded >=', $fromDateTime);
            $this->db->where('ln.dateadded <=', $toDateTime);
            $this->db->where('ln.is_new_call', 0);
            if (!empty($filters['staff']) && is_array($filters['staff'])) {
                $this->db->where_in('ln.addedfrom', $filters['staff']);
            }
            $this->db->group_by('ln.addedfrom');
            $followupCallsRows = $this->db->get()->result_array();
            
            foreach ($followupCallsRows as $row) {
                $sid = (int)$row['staffid'];
                $count = (int)$row['followup_calls'];
                $totalFollowupCalls += $count;
                // Find staff name
                foreach ($staffIdMap as $name => $id) {
                    if ($id === $sid) {
                        $followupCallsByStaff[$name] = $count;
                        break;
                    }
                }
            }
        }

        return [
            'by_staff' => $grouped_data,
            'all_types' => array_keys($allTypesSet),
            'call_connected' => $callConnectedByStaff,
            'call_connected_totals' => $callConnectTotals,
            'new_calls_by_staff' => $newCallsByStaff,
            'followup_calls_by_staff' => $followupCallsByStaff,
            'new_calls_total' => $totalNewCalls,
            'followup_calls_total' => $totalFollowupCalls,
        ];
    }

    /**
     * Persist a generated staff report for audit and re-use.
     *
     * @param array $summary
     * @param array|null $analysis
     * @param array $filters
     * @param array $context
     * @return int Inserted report ID
     */
    public function save_staff_report(array $summary, $analysis = null, array $filters = [], array $context = [])
    {
        $this->ensure_staff_report_tables();

        $table = db_prefix() . 'reporting_dashboard_staff_reports';

        $generatedBy = isset($context['generated_by']) ? (int)$context['generated_by'] : (function_exists('get_staff_user_id') ? (int)get_staff_user_id() : null);
        $whatsappNumbers = isset($context['whatsapp_numbers']) ? $this->normalize_whatsapp_numbers($context['whatsapp_numbers']) : [];

        $data = [
            'generated_at' => isset($context['generated_at']) ? $context['generated_at'] : date('Y-m-d H:i:s'),
            'generated_by' => $generatedBy ?: null,
            'date_from' => $summary['period']['from'] ?? (($filters['from'] ?? date('Y-m-d')) . ' 00:00:00'),
            'date_to' => $summary['period']['to'] ?? (($filters['to'] ?? date('Y-m-d')) . ' 23:59:59'),
            'staff_ids' => !empty($summary['staff_ids']) ? json_encode($summary['staff_ids']) : null,
            'summary_json' => json_encode($summary),
            'analysis_json' => !empty($analysis) ? json_encode($analysis) : null,
            'ai_model' => $context['ai_model'] ?? null,
            'message_preview' => $context['message_preview'] ?? null,
            'whatsapp_numbers' => !empty($whatsappNumbers) ? json_encode($whatsappNumbers) : null,
            'whatsapp_status_json' => isset($context['whatsapp_status']) ? json_encode($context['whatsapp_status']) : null,
            'notes' => $context['notes'] ?? null,
        ];

        $this->db->insert($table, $data);

        return (int)$this->db->insert_id();
    }

    /**
     * Update stored report metadata (e.g., WhatsApp delivery status).
     *
     * @param int   $reportId
     * @param array $data
     * @return bool
     */
    public function update_staff_report($reportId, array $data)
    {
        $this->ensure_staff_report_tables();

        $table = db_prefix() . 'reporting_dashboard_staff_reports';
        $update = [];

        if (array_key_exists('whatsapp_numbers', $data)) {
            $numbers = $this->normalize_whatsapp_numbers($data['whatsapp_numbers']);
            $update['whatsapp_numbers'] = !empty($numbers) ? json_encode($numbers) : null;
        }

        if (array_key_exists('whatsapp_status', $data)) {
            $update['whatsapp_status_json'] = $data['whatsapp_status'] !== null ? json_encode($data['whatsapp_status']) : null;
        }

        if (array_key_exists('message_preview', $data)) {
            $update['message_preview'] = $data['message_preview'];
        }

        if (array_key_exists('analysis_json', $data)) {
            $update['analysis_json'] = $data['analysis_json'] !== null ? json_encode($data['analysis_json']) : null;
        }

        if (array_key_exists('notes', $data)) {
            $update['notes'] = $data['notes'];
        }

        if (empty($update)) {
            return false;
        }

        $this->db->where('id', (int)$reportId);
        return $this->db->update($table, $update);
    }

    /**
     * Retrieve recent staff reports with decoded JSON payloads.
     *
     * @param int $limit
     * @return array
     */
    public function get_staff_reports($limit = 20)
    {
        $this->ensure_staff_report_tables();

        $table = db_prefix() . 'reporting_dashboard_staff_reports';

        $this->db->order_by('generated_at', 'DESC');
        if ($limit > 0) {
            $this->db->limit($limit);
        }

        $rows = $this->db->get($table)->result_array();

        foreach ($rows as &$row) {
            $row['summary'] = !empty($row['summary_json']) ? json_decode($row['summary_json'], true) : [];
            $row['analysis'] = !empty($row['analysis_json']) ? json_decode($row['analysis_json'], true) : [];
            $row['whatsapp_numbers_list'] = !empty($row['whatsapp_numbers']) ? json_decode($row['whatsapp_numbers'], true) : [];
            $row['whatsapp_status'] = !empty($row['whatsapp_status_json']) ? json_decode($row['whatsapp_status_json'], true) : [];
            unset($row['summary_json'], $row['analysis_json'], $row['whatsapp_numbers'], $row['whatsapp_status_json']);
        }

        return $rows;
    }

    /**
     * Create a new automated staff report schedule.
     *
     * @param array $data
     * @return int
     */
    public function create_staff_report_schedule(array $data)
    {
        $this->ensure_staff_report_tables();

        $table = db_prefix() . 'reporting_dashboard_staff_report_schedules';
        $payload = $this->prepare_schedule_payload($data);

        if (empty($payload['next_run'])) {
            $payload['next_run'] = $this->calculate_schedule_next_run($payload);
        }

        $payload['created_at'] = date('Y-m-d H:i:s');
        $payload['updated_at'] = date('Y-m-d H:i:s');

        $this->db->insert($table, $payload);
        return (int)$this->db->insert_id();
    }

    /**
     * Update an existing schedule.
     *
     * @param int   $scheduleId
     * @param array $data
     * @return bool
     */
    public function update_staff_report_schedule($scheduleId, array $data)
    {
        $this->ensure_staff_report_tables();

        $table = db_prefix() . 'reporting_dashboard_staff_report_schedules';
        $payload = $this->prepare_schedule_payload($data);

        $payload['next_run'] = $this->calculate_schedule_next_run($payload, $data['reference_time'] ?? null);
        $payload['updated_at'] = date('Y-m-d H:i:s');

        $this->db->where('id', (int)$scheduleId);
        return $this->db->update($table, $payload);
    }

    /**
     * Toggle schedule active state.
     *
     * @param int  $scheduleId
     * @param bool $active
     * @return bool
     */
    public function toggle_staff_report_schedule($scheduleId, $active)
    {
        $this->ensure_staff_report_tables();

        $table = db_prefix() . 'reporting_dashboard_staff_report_schedules';
        $this->db->where('id', (int)$scheduleId);
        return $this->db->update($table, [
            'active' => (int)(bool)$active,
            'updated_at' => date('Y-m-d H:i:s')
        ]);
    }

    /**
     * Delete schedule.
     *
     * @param int $scheduleId
     * @return bool
     */
    public function delete_staff_report_schedule($scheduleId)
    {
        $this->ensure_staff_report_tables();

        $table = db_prefix() . 'reporting_dashboard_staff_report_schedules';
        $this->db->where('id', (int)$scheduleId);
        return $this->db->delete($table);
    }

    /**
     * Fetch a single schedule by ID.
     *
     * @param int $scheduleId
     * @return array|null
     */
    public function get_staff_report_schedule($scheduleId)
    {
        $this->ensure_staff_report_tables();

        $table = db_prefix() . 'reporting_dashboard_staff_report_schedules';
        $this->db->where('id', (int)$scheduleId);
        $row = $this->db->get($table)->row_array();

        if (!$row) {
            return null;
        }

        return $this->hydrate_schedule_row($row);
    }

    /**
     * Retrieve all schedules.
     *
     * @param bool $includeInactive
     * @return array
     */
    public function get_staff_report_schedules($includeInactive = true)
    {
        $this->ensure_staff_report_tables();

        $table = db_prefix() . 'reporting_dashboard_staff_report_schedules';
        if (!$includeInactive) {
            $this->db->where('active', 1);
        }

        $this->db->order_by('next_run IS NULL', 'ASC', false);
        $this->db->order_by('next_run', 'ASC');
        $this->db->order_by('name', 'ASC');

        $rows = $this->db->get($table)->result_array();
        return array_map([$this, 'hydrate_schedule_row'], $rows);
    }

    /**
     * Schedules that should run now.
     *
     * @return array
     */
    public function get_due_staff_report_schedules()
    {
        $this->ensure_staff_report_tables();

        $table = db_prefix() . 'reporting_dashboard_staff_report_schedules';
        
        // Get current server time for comparison
        $currentTime = date('Y-m-d H:i:s');
        
        // Log for debugging
        log_activity('Checking due schedules at server time: ' . $currentTime);

        $this->db->where('active', 1);
        $this->db->where('next_run IS NOT NULL', null, false);
        $this->db->where('next_run <=', $currentTime);
        $rows = $this->db->get($table)->result_array();
        
        // Log the SQL query and results for debugging
        log_activity('Due schedules query: ' . $this->db->last_query());
        log_activity('Found ' . count($rows) . ' due schedule(s)');

        return array_map([$this, 'hydrate_schedule_row'], $rows);
    }

    /**
     * Update schedule run metadata after execution.
     *
     * @param int   $scheduleId
     * @param array $result
     * @param string|null $runTime
     * @return bool
     */
    public function mark_staff_schedule_run($scheduleId, array $result, $runTime = null)
    {
        $this->ensure_staff_report_tables();

        $schedule = $this->get_staff_report_schedule($scheduleId);
        if (!$schedule) {
            return false;
        }

        $reference = $runTime ?: date('Y-m-d H:i:s');
        $nextRun = $this->calculate_schedule_next_run($schedule, $reference);

        $table = db_prefix() . 'reporting_dashboard_staff_report_schedules';
        $this->db->where('id', (int)$scheduleId);
        return $this->db->update($table, [
            'last_run' => $reference,
            'next_run' => $nextRun,
            'last_result_json' => json_encode($result),
            'updated_at' => date('Y-m-d H:i:s')
        ]);
    }

    /**
     * Build filters for schedule execution.
     *
     * @param array|object $schedule
     * @return array
     */
    public function build_filters_for_schedule($schedule)
    {
        $schedule = (array)$schedule;

        $staffIds = [];
        if (!empty($schedule['staff_ids'])) {
            $decoded = json_decode($schedule['staff_ids'], true);
            if (is_array($decoded)) {
                $staffIds = array_values(array_filter(array_map('intval', $decoded)));
            }
        }

        $range = $this->resolve_schedule_range($schedule);

        return [
            'from' => substr($range['from'], 0, 10),
            'to' => substr($range['to'], 0, 10),
            'staff' => $staffIds
        ];
    }

    /**
     * Normalize schedule payload prior to insert/update.
     *
     * @param array $data
     * @return array
     */
    private function prepare_schedule_payload(array $data)
    {
        $frequency = isset($data['frequency']) && in_array($data['frequency'], ['daily', 'weekly', 'monthly'], true)
            ? $data['frequency']
            : 'daily';

        $rangeScope = isset($data['range_scope']) && in_array($data['range_scope'], ['auto', 'today', 'yesterday', 'last_7_days', 'last_30_days', 'this_month', 'last_month'], true)
            ? $data['range_scope']
            : 'auto';

        $scheduleTime = $data['schedule_time'] ?? '09:00:00';
        if (strlen($scheduleTime) === 5) {
            $scheduleTime .= ':00';
        }

        $staffIds = isset($data['staff_ids']) ? array_values(array_filter(array_map('intval', (array)$data['staff_ids']))) : [];
        $whatsappNumbers = isset($data['whatsapp_numbers']) ? $this->normalize_whatsapp_numbers($data['whatsapp_numbers']) : [];

        return [
            'name' => trim($data['name'] ?? 'Staff Performance Report'),
            'created_by' => isset($data['created_by']) ? (int)$data['created_by'] : (function_exists('get_staff_user_id') ? (int)get_staff_user_id() : null),
            'frequency' => $frequency,
            'range_scope' => $rangeScope,
            'schedule_time' => $scheduleTime,
            'day_of_week' => $frequency === 'weekly' ? (isset($data['day_of_week']) ? (int)$data['day_of_week'] : null) : null,
            'day_of_month' => $frequency === 'monthly' ? (isset($data['day_of_month']) ? min(max((int)$data['day_of_month'], 1), 31) : null) : null,
            'staff_ids' => !empty($staffIds) ? json_encode(array_values(array_unique($staffIds))) : null,
            'whatsapp_numbers' => !empty($whatsappNumbers) ? json_encode($whatsappNumbers) : null,
            'active' => isset($data['active']) ? (int)(bool)$data['active'] : 1,
            'next_run' => $data['next_run'] ?? null,
        ];
    }

    /**
     * Hydrate schedule row with decoded JSON for consumers.
     *
     * @param array $row
     * @return array
     */
    private function hydrate_schedule_row(array $row)
    {
        $row['staff_ids_list'] = !empty($row['staff_ids']) ? json_decode($row['staff_ids'], true) : [];
        $row['whatsapp_numbers_list'] = !empty($row['whatsapp_numbers']) ? json_decode($row['whatsapp_numbers'], true) : [];
        $row['last_result'] = !empty($row['last_result_json']) ? json_decode($row['last_result_json'], true) : [];
        return $row;
    }

    /**
     * Determine the next run datetime for a schedule.
     *
     * @param array $schedule
     * @param string|null $referenceTime
     * @return string
     */
    private function calculate_schedule_next_run(array $schedule, $referenceTime = null)
    {
        $frequency = $schedule['frequency'] ?? 'daily';
        $time = $schedule['schedule_time'] ?? '09:00:00';
        $dayOfWeek = isset($schedule['day_of_week']) ? (int)$schedule['day_of_week'] : null;
        $dayOfMonth = isset($schedule['day_of_month']) ? (int)$schedule['day_of_month'] : null;

        $reference = $referenceTime ? new DateTime($referenceTime) : new DateTime();

        $timeParts = array_map('intval', explode(':', $time));
        $hour = $timeParts[0] ?? 9;
        $minute = $timeParts[1] ?? 0;
        $second = $timeParts[2] ?? 0;

        $next = clone $reference;
        $next->setTime($hour, $minute, $second);

        if ($frequency === 'daily') {
            if ($next <= $reference) {
                $next->modify('+1 day');
            }
        } elseif ($frequency === 'weekly') {
            $targetDow = $dayOfWeek !== null ? $dayOfWeek : (int)$reference->format('w');
            $currentDow = (int)$next->format('w');
            $diff = $targetDow - $currentDow;
            if ($diff < 0) {
                $diff += 7;
            }
            if ($diff === 0 && $next <= $reference) {
                $diff = 7;
            }
            $next->modify('+' . $diff . ' days');
        } else { // monthly
            $targetDay = $dayOfMonth ?: (int)$reference->format('j');

            $daysInCurrentMonth = (int)$next->format('t');
            if ($targetDay > $daysInCurrentMonth) {
                $targetDay = $daysInCurrentMonth;
            }
            $next->setDate((int)$next->format('Y'), (int)$next->format('n'), $targetDay);

            if ($next <= $reference) {
                $next->modify('first day of next month');
                $daysInNextMonth = (int)$next->format('t');
                $targetDay = min($targetDay, $daysInNextMonth);
                $next->setDate((int)$next->format('Y'), (int)$next->format('n'), $targetDay);
            }
        }

        return $next->format('Y-m-d H:i:s');
    }

    /**
     * Resolve the date range that a schedule should analyze.
     *
     * @param array $schedule
     * @return array
     */
    private function resolve_schedule_range(array $schedule)
    {
        $scope = $schedule['range_scope'] ?? 'auto';
        $frequency = $schedule['frequency'] ?? 'daily';

        $now = new DateTime();
        $start = clone $now;
        $end = clone $now;

        $setFullDay = function (DateTime $date, $endOfDay = false) {
            if ($endOfDay) {
                $date->setTime(23, 59, 59);
            } else {
                $date->setTime(0, 0, 0);
            }
            return $date;
        };

        switch ($scope) {
            case 'today':
                $start = $setFullDay(clone $now, false);
                $end = $setFullDay(clone $now, true);
                break;

            case 'yesterday':
                $start = $setFullDay((clone $now)->modify('-1 day'), false);
                $end = $setFullDay((clone $now)->modify('-1 day'), true);
                break;

            case 'last_7_days':
                $start = $setFullDay((clone $now)->modify('-7 days'), false);
                $end = $setFullDay(clone $now, true);
                break;

            case 'last_30_days':
                $start = $setFullDay((clone $now)->modify('-30 days'), false);
                $end = $setFullDay(clone $now, true);
                break;

            case 'this_month':
                $start = $setFullDay(new DateTime('first day of this month'), false);
                $end = $setFullDay(clone $now, true);
                break;

            case 'last_month':
                $start = $setFullDay(new DateTime('first day of last month'), false);
                $end = $setFullDay(new DateTime('last day of last month'), true);
                break;

            case 'auto':
            default:
                if ($frequency === 'daily') {
                    $start = $setFullDay((clone $now)->modify('-1 day'), false);
                    $end = $setFullDay((clone $now)->modify('-1 day'), true);
                } elseif ($frequency === 'weekly') {
                    $start = $setFullDay((clone $now)->modify('-7 days'), false);
                    $end = $setFullDay(clone $now, true);
                } else { // monthly
                    $start = $setFullDay(new DateTime('first day of last month'), false);
                    $end = $setFullDay(new DateTime('last day of last month'), true);
                }
                break;
        }

        return [
            'from' => $start->format('Y-m-d H:i:s'),
            'to' => $end->format('Y-m-d H:i:s')
        ];
    }

    /**
     * Normalize an array/string of WhatsApp numbers.
     *
     * @param array|string $numbers
     * @return array
     */
    private function normalize_whatsapp_numbers($numbers)
    {
        if (empty($numbers)) {
            return [];
        }

        if (is_string($numbers)) {
            $numbers = preg_split('/[\s,;]+/', $numbers);
        }

        if (!is_array($numbers)) {
            return [];
        }

        $normalized = [];

        foreach ($numbers as $number) {
            if ($number === null) {
                continue;
            }

            $clean = preg_replace('/[^0-9+]/', '', (string)$number);
            if ($clean === '') {
                continue;
            }

            // Prepend '+' if the number looks like an international number without it
            if ($clean[0] !== '+' && strlen($clean) >= 10) {
                $clean = '+' . $clean;
            }

            $normalized[] = $clean;
        }

        return array_values(array_unique($normalized));
    }

    /**
     * Aggregate call connect stats for staff within the requested filter window.
     *
     * @param array  $filters
     * @param string $table
     * @return array
     */
    private function fetch_call_connect_stats(array $filters, $table)
    {
        $result = [
            'by_staff' => [],
            'totals' => [
                'logged' => 0,
                'connected' => 0,
                'not_connected' => 0,
                'unknown' => 0,
                'connect_rate' => 0,
            ],
        ];

        if (!$this->leadNotesHasStats || !$table) {
            return $result;
        }

        $followupCase = $this->leadNotesHasFollowupType
            ? $this->get_followup_type_sql('a', 'ln')
            : $this->get_followup_type_sql('a');
        $callTypeCase = "CASE WHEN {$followupCase} IN ('Phone Call','WhatsApp Call') THEN 1 ELSE 0 END";
        $normalizedStats = "LOWER(TRIM(REPLACE(REPLACE(REPLACE(COALESCE(ln.stats, ''), 'call connected', ''), ':', ''), '-', '')))";
        $connectedCaseSql = "CASE WHEN {$callTypeCase} = 1 AND {$normalizedStats} IN ('yes','y','connected','1','success','answer','answered') THEN 1 ELSE 0 END";
        $notConnectedCaseSql = "CASE WHEN {$callTypeCase} = 1 AND {$normalizedStats} IN ('no','n','0','not connected','missed','busy','failed') THEN 1 ELSE 0 END";
        $loggedCaseSql = "CASE WHEN {$callTypeCase} = 1 AND {$normalizedStats} <> '' THEN 1 ELSE 0 END";

        $this->db->select("
            a.staffid,
            SUM({$connectedCaseSql}) AS connected_calls,
            SUM({$notConnectedCaseSql}) AS not_connected_calls,
            SUM({$loggedCaseSql}) AS logged_calls
        ", false);
        $this->db->from($table . ' a');
        $this->db->join(
            db_prefix() . 'lead_notes ln',
            'ln.leadid = a.leadid AND ABS(TIMESTAMPDIFF(MINUTE, ln.dateadded, a.date)) <= 120',
            'left'
        );
        $this->db->where("(a.description = 'not_lead_activity_added_followup' OR a.description NOT LIKE 'not_%')");
        if (!empty($filters['from'])) {
            $this->db->where('DATE(a.date) >=', $filters['from']);
        }
        if (!empty($filters['to'])) {
            $this->db->where('DATE(a.date) <=', $filters['to']);
        }
        if (!empty($filters['staff']) && is_array($filters['staff'])) {
            $this->db->where_in('a.staffid', $filters['staff']);
        }
        $this->db->group_by('a.staffid');

        $rows = $this->db->get()->result_array();
        foreach ($rows as $row) {
            $staffId = (int)($row['staffid'] ?? 0);
            $logged = (int)($row['logged_calls'] ?? 0);
            $connected = (int)($row['connected_calls'] ?? 0);
            $notConnected = (int)($row['not_connected_calls'] ?? 0);
            $unknown = max(0, $logged - ($connected + $notConnected));
            $rate = $logged > 0 ? round(($connected / $logged) * 100, 1) : 0;

            $result['by_staff'][$staffId] = [
                'logged' => $logged,
                'connected' => $connected,
                'not_connected' => $notConnected,
                'unknown' => $unknown,
                'connect_rate' => $rate,
            ];

            $result['totals']['logged'] += $logged;
            $result['totals']['connected'] += $connected;
            $result['totals']['not_connected'] += $notConnected;
            $result['totals']['unknown'] += $unknown;
        }

        if ($result['totals']['logged'] > 0) {
            $result['totals']['connect_rate'] = round(
                ($result['totals']['connected'] / max(1, $result['totals']['logged'])) * 100,
                1
            );
        }

        return $result;
    }
}
?>
