<?php
defined('BASEPATH') or exit('No direct script access allowed');

class Reporting_dashboard_permissions extends AdminController
{
    public function __construct()
    {
        parent::__construct();
        
        $this->load->model('reporting_dashboard/reporting_dashboard_permissions_model');
        $this->load->model('staff_model');
        
        // Only admin can access permission management
        if (!is_admin()) {
            access_denied('Reporting Dashboard Permissions');
        }
    }

    /**
     * Permission management page
     */
    public function index()
    {
        $data['title'] = 'Reporting Dashboard - Permissions Management';
        $data['staff_permissions'] = $this->reporting_dashboard_permissions_model->get_all_staff_permissions();
        $data['available_staff'] = $this->reporting_dashboard_permissions_model->get_available_staff_for_permission();
        $this->load->view('permissions_management', $data);
    }

    /**
     * Get available staff for specific permission assignment via AJAX
     */
    public function get_available_staff()
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $exclude_staff_id = $this->input->get('exclude_staff_id');
        $available_staff = $this->reporting_dashboard_permissions_model->get_available_staff_for_permission($exclude_staff_id);
        
        echo json_encode([
            'success' => true,
            'staff' => $available_staff
        ]);
    }

    /**
     * Update staff permission via AJAX
     */
    public function update_permission()
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $staff_id = $this->input->post('staff_id');
        $permission_type = $this->input->post('permission_type');
        $allowed_staff_ids = $this->input->post('allowed_staff_ids');
        $can_generate_reports = (bool)$this->input->post('can_generate_reports');
        $current_user_id = get_staff_user_id();

        // Validate inputs
        if (empty($staff_id) || empty($permission_type)) {
            echo json_encode([
                'success' => false,
                'message' => 'Invalid parameters'
            ]);
            return;
        }

        // Validate permission type
        if (!in_array($permission_type, ['view_all_staff', 'view_own_only', 'view_specific_staff'])) {
            echo json_encode([
                'success' => false,
                'message' => 'Invalid permission type'
            ]);
            return;
        }

        // For view_specific_staff, validate allowed_staff_ids
        $allowed_staff_array = [];
        if ($permission_type === 'view_specific_staff') {
            if (empty($allowed_staff_ids)) {
                echo json_encode([
                    'success' => false,
                    'message' => 'Please select at least one staff member for specific staff permission'
                ]);
                return;
            }
            
            $allowed_staff_array = is_array($allowed_staff_ids) ? $allowed_staff_ids : explode(',', $allowed_staff_ids);
            $allowed_staff_array = array_filter(array_map('intval', $allowed_staff_array));
            
            if (empty($allowed_staff_array)) {
                echo json_encode([
                    'success' => false,
                    'message' => 'Invalid staff selection'
                ]);
                return;
            }
        }

        // Update permission
        $success = $this->reporting_dashboard_permissions_model->set_staff_permission(
            $staff_id,
            $permission_type,
            $current_user_id,
            $allowed_staff_array,
            $can_generate_reports
        );

        if ($success) {
            echo json_encode([
                'success' => true,
                'message' => 'Permission updated successfully'
            ]);
        } else {
            echo json_encode([
                'success' => false,
                'message' => 'Failed to update permission'
            ]);
        }
    }

    /**
     * Remove staff permission (revert to default)
     */
    public function remove_permission()
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $staff_id = $this->input->post('staff_id');

        if (empty($staff_id)) {
            echo json_encode([
                'success' => false,
                'message' => 'Invalid staff ID'
            ]);
            return;
        }

        $success = $this->reporting_dashboard_permissions_model->remove_staff_permission($staff_id);

        if ($success) {
            echo json_encode([
                'success' => true,
                'message' => 'Permission reset to default successfully'
            ]);
        } else {
            echo json_encode([
                'success' => false,
                'message' => 'Failed to reset permission'
            ]);
        }
    }

    /**
     * Bulk update permissions
     */
    public function bulk_update()
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $permissions = $this->input->post('permissions');
        $current_user_id = get_staff_user_id();

        if (empty($permissions) || !is_array($permissions)) {
            echo json_encode([
                'success' => false,
                'message' => 'No permissions provided'
            ]);
            return;
        }

        $updated_count = 0;
        $errors = [];

        foreach ($permissions as $staff_id => $permission_data) {
            $permission_type = $permission_data['type'] ?? '';
            $allowed_staff_ids = $permission_data['allowed_staff_ids'] ?? [];
            
            if (in_array($permission_type, ['view_all_staff', 'view_own_only', 'view_specific_staff'])) {
                $success = $this->reporting_dashboard_permissions_model->set_staff_permission(
                    $staff_id,
                    $permission_type,
                    $current_user_id,
                    $allowed_staff_ids
                );
                
                if ($success) {
                    $updated_count++;
                } else {
                    $errors[] = "Failed to update permission for staff ID: $staff_id";
                }
            }
        }

        echo json_encode([
            'success' => empty($errors),
            'message' => empty($errors) ? 
                "Successfully updated $updated_count permissions" :
                "Updated $updated_count permissions with " . count($errors) . " errors",
            'errors' => $errors
        ]);
    }
}
?>
