<?php

defined('BASEPATH') or exit('No direct script access allowed');

/**
 * Cron Job: Autograph Feedback Insights Analysis
 * Run this every 6 hours to automatically analyze feedback
 * 
 * Add to crontab: 0 */6 * * * /usr/bin/php /path/to/crm/index.php autograph_feedback_insights cron run_analysis
 * 
 * Or access via: /admin/autograph_feedback_insights/cron/run_analysis
 */

class Cron extends AdminController
{
    public function __construct()
    {
        parent::__construct();
        $this->load->model('autograph_feedback_insights/autograph_feedback_insights_model');
    }

    /**
     * Run analysis via cron
     */
    public function run_analysis()
    {
        // Check if cron is enabled
        $settings = $this->autograph_feedback_insights_model->get_all_settings();
        $cron_enabled = $settings['cron_enabled'] ?? '0';

        if ($cron_enabled != '1') {
            echo "Cron job is disabled in settings.\n";
            exit(0);
        }

        // Check if API key is configured
        $api_key = $settings['openai_api_key'] ?? '';
        if (empty($api_key)) {
            echo "OpenAI API key is not configured.\n";
            exit(1);
        }

        // Run analysis
        echo "Starting Autograph Feedback Insights Analysis...\n";
        $result = $this->autograph_feedback_insights_model->run_analysis();

        if ($result['success']) {
            echo "Analysis completed successfully!\n";
            echo "Notes analyzed: " . $result['notes_analyzed'] . "\n";
            echo "Leads analyzed: " . $result['leads_analyzed'] . "\n";
            echo "Report ID: " . $result['report_id'] . "\n";
            exit(0);
        } else {
            echo "Analysis failed: " . $result['message'] . "\n";
            exit(1);
        }
    }

    /**
     * Execute scheduled reports
     * Run this every hour via cron: 0 * * * *
     */
    public function execute_schedules()
    {
        echo "Starting scheduled reports execution...\n";
        
        // Get due schedules
        $schedules = $this->autograph_feedback_insights_model->get_due_schedules();
        
        if (empty($schedules)) {
            echo "No schedules due for execution.\n";
            exit(0);
        }

        echo "Found " . count($schedules) . " schedule(s) to execute.\n";

        foreach ($schedules as $schedule) {
            echo "\n--- Executing: " . $schedule->schedule_name . " ---\n";
            
            try {
                $result = $this->autograph_feedback_insights_model->execute_scheduled_report($schedule);
                
                if ($result['success']) {
                    echo "✓ Report generated successfully\n";
                    echo "  Notes analyzed: " . $result['notes_analyzed'] . "\n";
                    echo "  Leads analyzed: " . $result['leads_analyzed'] . "\n";
                    echo "  Sent to " . $result['recipients_count'] . " WhatsApp number(s)\n";
                    echo "  Successful: " . $result['sent_success'] . "\n";
                    echo "  Failed: " . $result['sent_failed'] . "\n";
                } else {
                    echo "✗ Execution failed: " . $result['message'] . "\n";
                }
            } catch (Exception $e) {
                echo "✗ Error: " . $e->getMessage() . "\n";
            }
        }
        
        echo "\nScheduled reports execution completed.\n";
        exit(0);
    }
}

