<?php
// Idempotent SQL import: inserts only missing rows by using INSERT IGNORE
// Usage: open this file in the browser, upload/paste the SQL dump generated by export_data_oct30_31.php

// Safety: require a simple secret key or localhost
$allow_from_localhost = in_array($_SERVER['REMOTE_ADDR'] ?? '', ['127.0.0.1', '::1']);
$secret = isset($_GET['key']) ? $_GET['key'] : (isset($_POST['key']) ? $_POST['key'] : '');
$authorized = $allow_from_localhost || ($secret === 'import-ok');

// Remove output buffering conflicts
if (function_exists('ob_get_level')) {
    while (ob_get_level() > 0) { ob_end_clean(); }
}

// Load DB config constants used by other debug scripts
define('BASEPATH', __DIR__);
require_once __DIR__ . '/application/config/app-config.php';

if (!defined('APP_DB_HOSTNAME')) {
    die('Database config not loaded.');
}

// Connect
$mysqli = @new mysqli(APP_DB_HOSTNAME, APP_DB_USERNAME, APP_DB_PASSWORD, APP_DB_NAME);
if ($mysqli->connect_errno) {
    http_response_code(500);
    echo '<h3>DB connection failed</h3><pre>' . htmlspecialchars($mysqli->connect_error) . '</pre>';
    exit;
}
$mysqli->set_charset('utf8mb4');

// Increase limits
@set_time_limit(0);
@ini_set('memory_limit', '512M');

function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }

$is_post = ($_SERVER['REQUEST_METHOD'] ?? 'GET') === 'POST';
$dry_run = isset($_POST['dry_run']);

// HTML header
echo '<!doctype html><meta charset="utf-8"><title>Import SQL (insert-missing only)</title>';
echo '<style>body{font-family:system-ui,Segoe UI,Arial;margin:20px;max-width:1000px} pre{background:#f7f7f7;padding:10px;overflow:auto} .ok{color:#2e7d32}.warn{color:#ef6c00}.err{color:#c62828} table{border-collapse:collapse} td,th{border:1px solid #ddd;padding:6px 8px}</style>';
echo '<h2>Import SQL (only new rows)</h2>';
echo '<p>This tool executes INSERT IGNORE for each INSERT in your dump, so existing rows (duplicate keys) are skipped. Delete this file after use.</p>';

// Gate: require key when not localhost
if (!$authorized) {
    echo '<div class="warn"><strong>Access restricted.</strong> Provide the access key to continue (or run from localhost).</div>';
    echo '<form method="get">';
    echo '<label>Key: <input type="text" name="key" autofocus></label> ';
    echo '<button type="submit">Continue</button>';
    echo '</form>';
    echo '<p class="warn">Tip: If you are following our default, append ?key=import-ok to the URL. Remove this file after use.</p>';
    exit;
}

if (!$is_post) {
    echo '<form method="post" enctype="multipart/form-data">';
    echo '<div><label>Upload .sql file: <input type="file" name="sqlfile" accept=".sql,.txt"></label></div>';
    echo '<div style="color:#666;font-size:12px;margin:-6px 0 10px 0">If upload fails due to size limits, use the server file option below.</div>';
    echo '<div><label>Or paste SQL:</label><br><textarea name="sqltext" rows="12" style="width:100%" placeholder="Paste the SQL here..."></textarea></div>';
    echo '<div><label>Or read server file (place .sql in this folder): <input type="text" name="server_path" placeholder="e.g. mydump.sql"></label></div>';
    echo '<div><label><input type="checkbox" name="dry_run" value="1"> Dry run (parse only, don\'t execute)</label></div>';
    echo '<div><label>Key (optional): <input type="text" name="key" value="' . h($secret) . '"></label></div>';
    echo '<div><button type="submit">Import</button></div>';
    echo '<hr><p class="warn">Tip: You can also navigate with ?key=import-ok to bypass localhost restriction.</p>';
    echo '<h3>DB</h3><ul>';
    echo '<li>Host: ' . h(APP_DB_HOSTNAME) . '</li>';
    echo '<li>DB: ' . h(APP_DB_NAME) . '</li>';
    echo '</ul>';
    exit;
}

// Read input
$content = '';
// 1) Uploaded file
if (!empty($_FILES['sqlfile']) && isset($_FILES['sqlfile']['error']) && $_FILES['sqlfile']['error'] !== UPLOAD_ERR_NO_FILE) {
    $uploadErr = (int)$_FILES['sqlfile']['error'];
    if ($uploadErr === UPLOAD_ERR_OK && !empty($_FILES['sqlfile']['tmp_name']) && is_uploaded_file($_FILES['sqlfile']['tmp_name'])) {
        $content = file_get_contents($_FILES['sqlfile']['tmp_name']);
    } else {
        echo '<div class="err"><strong>Upload error:</strong> ' . h($uploadErr) . '</div>';
        if (in_array($uploadErr, [UPLOAD_ERR_INI_SIZE, UPLOAD_ERR_FORM_SIZE])) {
            echo '<div class="warn">File too large. Server limits: upload_max_filesize=' . h(ini_get('upload_max_filesize')) . ', post_max_size=' . h(ini_get('post_max_size')) . '.</div>';
            echo '<div>Workarounds: (a) Use the "server file" option below after uploading via File Manager/FTP. (b) Paste SQL in the textarea (smaller chunks).</div>';
        }
    }
}
// 2) Textarea
if (!$content && !empty($_POST['sqltext'])) {
    $content = (string)$_POST['sqltext'];
}
// 3) Server-side file path (same folder by default)
if (!$content && !empty($_POST['server_path'])) {
    $sp = trim((string)$_POST['server_path']);
    $base = realpath(__DIR__);
    $candidate = $sp;
    // If only a filename, use current dir
    if (strpos($sp, DIRECTORY_SEPARATOR) === false) {
        $candidate = __DIR__ . DIRECTORY_SEPARATOR . $sp;
    }
    $real = realpath($candidate);
    if ($real && strpos($real, $base) === 0 && is_file($real) && is_readable($real)) {
        $content = file_get_contents($real);
        echo '<div class="ok">Loaded server file: ' . h(str_replace($base . DIRECTORY_SEPARATOR, '', $real)) . '</div>';
    } else {
        echo '<div class="err">Server file not found or not readable: ' . h($sp) . '</div>';
    }
}
if (!$content) {
    echo '<p class="err">No SQL provided.</p>';
    echo '<ul>';
    echo '<li>Upload a .sql file (Choose File above) and click Import.</li>';
    echo '<li>Paste the SQL into the textarea and click Import.</li>';
    echo '<li>Or upload the .sql to this folder via File Manager/FTP and enter its name in "server file".</li>';
    echo '</ul>';
    echo '<p><a href="' . h($_SERVER['PHP_SELF']) . (isset($_GET['key'])? ('?key=' . urlencode($_GET['key'])) : '') . '">Go back</a></p>';
    exit;
}

// Normalize line endings and strip BOM
$content = preg_replace("/\r\n?|\n/", "\n", $content);
if (substr($content, 0, 3) === "\xEF\xBB\xBF") {
    $content = substr($content, 3);
}

// Parser: accumulate INSERT statements safely (don't split by semicolon inside strings)
$lines = explode("\n", $content);
$statements = [];
$buffer = '';
$capturing = false;

foreach ($lines as $rawLine) {
    $line = trim($rawLine);
    if ($line === '') continue;
    // Skip comments and warnings and meta
    if (preg_match('/^(--|#|\/\*|SET\s|Warning:|No records found|EXPORT COMPLETE|RELATED DATA)/i', $line)) {
        continue;
    }
    if (!$capturing) {
        if (stripos($line, 'INSERT INTO ') === 0) {
            $capturing = true;
            $buffer = $line;
        } else {
            // ignore everything else (e.g., CREATE, etc.)
            continue;
        }
    } else {
        // continuation line (in case an INSERT spans multiple lines)
        $buffer .= ' ' . $line;
    }

    // Heuristic: statement ends with ");" or ") ;" ignoring trailing spaces
    if ($capturing && preg_match('/\)\s*;\s*$/', $buffer)) {
        $statements[] = $buffer;
        $buffer = '';
        $capturing = false;
    }
}

// Process statements
$total = 0; $executed = 0; $inserted = 0; $skipped = 0; $errors = 0;
$perTable = [];

foreach ($statements as $stmt) {
    $total++;
    // Find table name
    $table = null;
    if (preg_match('/^INSERT\s+INTO\s+`?([a-z0-9_]+)`?/i', $stmt, $m)) {
        $table = $m[1];
    } else {
        $table = 'unknown';
    }
    if (!isset($perTable[$table])) {
        $perTable[$table] = ['total'=>0,'inserted'=>0,'skipped'=>0,'errors'=>0];
    }
    $perTable[$table]['total']++;

    // Rewrite to INSERT IGNORE
    $execSql = preg_replace('/^INSERT\s+INTO/i', 'INSERT IGNORE INTO', $stmt, 1);

    if ($dry_run) { continue; }

    $ok = $mysqli->query($execSql);
    $executed++;
    if ($ok) {
        $aff = $mysqli->affected_rows;
        if ($aff > 0) { $inserted++; $perTable[$table]['inserted']++; }
        else { $skipped++; $perTable[$table]['skipped']++; }
    } else {
        $errno = $mysqli->errno;
        $err   = $mysqli->error;
        // Duplicate key safety: some engines may still throw on specific modes
        if ($errno == 1062) {
            $skipped++; $perTable[$table]['skipped']++;
        } else {
            $errors++; $perTable[$table]['errors']++;
            // Show a brief snippet for debugging
            echo '<div class="err"><strong>Error:</strong> ['.h($errno).'] '.h($err).'</div>';
            echo '<pre>'.h(mb_strimwidth($execSql, 0, 800, '...')).'</pre>';
        }
    }
}

// Summary
echo '<h3>Summary</h3>';
echo '<ul>';
echo '<li>Total INSERTs parsed: <strong>'.h($total).'</strong></li>';
echo '<li>Executed: <strong>'.h($executed).'</strong> '.($dry_run? '(dry-run)':'').'</li>';
echo '<li class="ok">Inserted: <strong>'.h($inserted).'</strong></li>';
echo '<li class="warn">Skipped (already present): <strong>'.h($skipped).'</strong></li>';
echo '<li class="err">Errors: <strong>'.h($errors).'</strong></li>';
echo '</ul>';

echo '<h3>Per-table</h3>';
echo '<table><tr><th>Table</th><th>Total</th><th>Inserted</th><th>Skipped</th><th>Errors</th></tr>';
foreach ($perTable as $t=>$row) {
    echo '<tr>'; 
    echo '<td>'.h($t).'</td>'; 
    echo '<td>'.h($row['total']).'</td>'; 
    echo '<td class="ok">'.h($row['inserted']).'</td>'; 
    echo '<td class="warn">'.h($row['skipped']).'</td>'; 
    echo '<td class="err">'.h($row['errors']).'</td>'; 
    echo '</tr>';
}
echo '</table>';

echo '<p><a href="'.h($_SERVER['PHP_SELF']).'">Run again</a></p>';

// Close
$mysqli->close();

?>
