/**
 * Reminder Sound Generator - FIXED VERSION
 * Creates audio notifications for reminder alerts
 * Eliminates AudioContext warnings and handles user interaction properly
 */

// Global flag to track if user has interacted with the page
let userHasInteracted = false;

// Audio context for generated sounds (only created after user interaction)
let audioContext = null;

/**
 * Initialize user interaction detection on script load
 */
function initUserInteractionDetection() {
    if (userHasInteracted) return;
    
    const events = ['click', 'touchstart', 'keydown', 'mousedown'];
    const handleFirstInteraction = () => {
        userHasInteracted = true;
        console.log('User interaction detected - audio now available');
        
        // Remove event listeners
        events.forEach(event => {
            document.removeEventListener(event, handleFirstInteraction, true);
        });
        
        // Initialize AudioContext now that we have user interaction
        initAudioContextSafely();
    };
    
    events.forEach(event => {
        document.addEventListener(event, handleFirstInteraction, true);
    });
}

/**
 * Initialize audio context safely (only after user interaction)
 */
function initAudioContextSafely() {
    if (!audioContext && userHasInteracted) {
        try {
            audioContext = new (window.AudioContext || window.webkitAudioContext)();
        } catch (e) {
            // AudioContext not supported, will use HTML5 audio fallback
            audioContext = null;
        }
    }
    return audioContext && audioContext.state === 'running';
}

/**
 * Play reminder notification sound
 * @param {number} frequency - Sound frequency (default: 800)
 * @param {number} duration - Sound duration in milliseconds (default: 200)
 * @param {number} volume - Sound volume 0-1 (default: 0.3)
 */
function playReminderSound(frequency = 800, duration = 200, volume = 0.3) {
    // Only try AudioContext if user has interacted
    if (userHasInteracted && initAudioContextSafely()) {
        try {
            // Create oscillator for the beep sound
            const oscillator = audioContext.createOscillator();
            const gainNode = audioContext.createGain();
            
            // Connect nodes
            oscillator.connect(gainNode);
            gainNode.connect(audioContext.destination);
            
            // Configure sound
            oscillator.frequency.value = frequency;
            oscillator.type = 'sine';
            
            // Configure volume envelope
            gainNode.gain.setValueAtTime(0, audioContext.currentTime);
            gainNode.gain.linearRampToValueAtTime(volume, audioContext.currentTime + 0.01);
            gainNode.gain.exponentialRampToValueAtTime(0.001, audioContext.currentTime + duration / 1000);
            
            // Play sound
            oscillator.start(audioContext.currentTime);
            oscillator.stop(audioContext.currentTime + duration / 1000);
            
            return; // Success, exit early
        } catch (e) {
            // Fall through to HTML5 audio fallback
        }
    }
    
    // Fallback: HTML5 audio (only if user has interacted)
    if (userHasInteracted) {
        try {
            // Create a simple beep using HTML5 audio with data URI
            const beepDataUri = `data:audio/wav;base64,UklGRnoGAABXQVZFZm10IBAAAAABAAEAQB8AAEAfAAABAAgAZGF0YQoGAACBhYqFbF1fdJivrJBhNjVgodDbq2EcBj+a2/LDciUFLIHO8tiJNwgZaLvt559NEAxQp+PwtmMcBjiR1/LMeSwFJHfH8N2QQAoUXrTp66hVFApGn+DyvmEaAi2Fzfa/cCIBHnfF89OZSA0PVqzn77BdGAhAl+L0txQELHfL9cN1KgQk`;
            const audio = new Audio(beepDataUri);
            audio.volume = Math.min(volume, 0.5);
            
            const playPromise = audio.play();
            if (playPromise !== undefined) {
                playPromise.catch(() => {
                    // Silent fail - expected in some browsers
                });
            }
        } catch (e) {
            // Silent fail - audio is not critical functionality
        }
    }
    // If no user interaction, sound won't play (expected browser behavior)
}

/**
 * Play multiple beeps for urgent reminders
 * @param {number} count - Number of beeps
 * @param {number} interval - Interval between beeps in milliseconds
 */
function playUrgentReminderSound(count = 3, interval = 300) {
    for (let i = 0; i < count; i++) {
        setTimeout(() => {
            playReminderSound(900, 150, 0.4);
        }, i * interval);
    }
}

/**
 * Play a gentle notification sound
 */
function playGentleReminderSound() {
    playReminderSound(600, 200, 0.2);
    setTimeout(() => {
        playReminderSound(800, 200, 0.2);
    }, 250);
}

/**
 * Play a bell notification sound
 */
function playBellReminderSound() {
    playReminderSound(523.25, 300, 0.25); // C5
    setTimeout(() => {
        playReminderSound(659.25, 400, 0.2); // E5
    }, 100);
    setTimeout(() => {
        playReminderSound(783.99, 500, 0.15); // G5
    }, 200);
}

/**
 * Play a ding notification sound
 */
function playDingReminderSound() {
    playReminderSound(1000, 100, 0.3);
    setTimeout(() => {
        playReminderSound(800, 150, 0.2);
    }, 120);
}

/**
 * Play reminder sound based on type
 * @param {string} soundType - Type of sound to play (gentle, standard, urgent, bell, ding, none, custom)
 */
function playReminderSoundByType(soundType = 'standard') {
    switch(soundType) {
        case 'gentle':
            playGentleReminderSound();
            break;
        case 'urgent':
            playUrgentReminderSound();
            break;
        case 'bell':
            playBellReminderSound();
            break;
        case 'ding':
            playDingReminderSound();
            break;
        case 'custom':
            playCustomReminderSound();
            break;
        case 'none':
            // No sound
            break;
        case 'standard':
        default:
            playReminderSound();
            break;
    }
}

/**
 * Play standard reminder sound (alias for playReminderSound)
 */
function playStandardReminderSound() {
    playReminderSound();
}

/**
 * Play custom uploaded reminder sound
 */
function playCustomReminderSound() {
    // Get custom sound file from global settings
    var customSoundFile = window.custom_reminder_sound_file || null;
    
    if (!customSoundFile) {
        console.warn('No custom sound file configured, falling back to standard sound');
        playReminderSound();
        return;
    }

    // Only attempt to play if user has interacted
    if (!userHasInteracted) {
        console.log('Cannot play custom sound - no user interaction yet');
        return;
    }

    try {
        // Create audio element and play custom sound
        var audio = new Audio();
        audio.volume = 0.3;
        audio.src = (window.site_url || '') + 'uploads/custom_sounds/' + customSoundFile;
        
        // Error handling
        audio.onerror = function(e) {
            var errorMessage = 'Failed to load custom sound file';
            if (e.target && e.target.error) {
                var error = e.target.error;
                switch(error.code) {
                    case error.MEDIA_ERR_ABORTED:
                        errorMessage = 'Custom sound playback was aborted';
                        break;
                    case error.MEDIA_ERR_NETWORK:
                        errorMessage = 'Custom sound network error';
                        break;
                    case error.MEDIA_ERR_DECODE:
                        errorMessage = 'Custom sound format/decode error';
                        break;
                    case error.MEDIA_ERR_SRC_NOT_SUPPORTED:
                        errorMessage = 'Custom sound format not supported';
                        break;
                    default:
                        errorMessage = 'Custom sound error (code: ' + error.code + ')';
                }
            }
            console.warn(errorMessage + ', falling back to standard sound');
            playReminderSound();
        };
        
        // Play the custom sound
        const playPromise = audio.play();
        if (playPromise !== undefined) {
            playPromise.catch(function(error) {
                console.warn('Failed to play custom sound: ' + error.message + ', falling back to standard sound');
                playReminderSound();
            });
        }
        
    } catch (error) {
        console.warn('Error with custom sound playback: ' + error.message + ', falling back to standard sound');
        playReminderSound();
    }
}

// Initialize user interaction detection when script loads
initUserInteractionDetection();

// Export functions if using modules
if (typeof module !== 'undefined' && module.exports) {
    module.exports = {
        playReminderSound,
        playStandardReminderSound,
        playUrgentReminderSound,
        playGentleReminderSound,
        playBellReminderSound,
        playDingReminderSound,
        playCustomReminderSound,
        playReminderSoundByType,
        initUserInteractionDetection
    };
}
