<?php defined('BASEPATH') or exit('No direct script access allowed'); ?>
<?php init_head(); ?>
<div id="wrapper">
    <div class="content">
        <div class="row">
            <div class="col-md-12">
                <div class="panel panel-primary">
                    <div class="panel-heading">
                        <h3 class="panel-title">
                            <i class="fa fa-bell"></i> Reminder System - Quick Test
                        </h3>
                    </div>
                    <div class="panel-body">
                        <div class="alert alert-info">
                            <h4><i class="fa fa-info-circle"></i> Quick Reminder System Test</h4>
                            <p>This page will automatically test the reminder system and show you if everything is working correctly.</p>
                        </div>

                        <div class="row">
                            <div class="col-md-12">
                                <div class="panel panel-success">
                                    <div class="panel-heading">
                                        <h4><i class="fa fa-check-circle"></i> Automatic Test Results</h4>
                                    </div>
                                    <div class="panel-body">
                                        <div id="auto-test-results">
                                            <p><i class="fa fa-spinner fa-spin"></i> Running automatic tests...</p>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="panel panel-default">
                                    <div class="panel-heading">
                                        <h4><i class="fa fa-play"></i> Manual Actions</h4>
                                    </div>
                                    <div class="panel-body">
                                        <button class="btn btn-primary btn-block" onclick="forceInitReminders()">
                                            <i class="fa fa-power-off"></i> Force Initialize Reminders
                                        </button>
                                        <button class="btn btn-info btn-block" onclick="checkRemindersNow()">
                                            <i class="fa fa-refresh"></i> Check for Reminders Now
                                        </button>                                        <button class="btn btn-warning btn-block" onclick="showTestModal()">
                                            <i class="fa fa-bell"></i> Show Test Notification
                                        </button>                                        <button class="btn btn-default btn-block" onclick="debugScripts()">
                                            <i class="fa fa-bug"></i> Debug Script Loading
                                        </button>
                                        <button class="btn btn-danger btn-block" onclick="enableMinifiedFiles()">
                                            <i class="fa fa-compress"></i> Enable Minified Files
                                        </button>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="panel panel-default">
                                    <div class="panel-heading">
                                        <h4><i class="fa fa-info"></i> System Information</h4>
                                    </div>
                                    <div class="panel-body">
                                        <div id="system-info">
                                            <p>Loading system information...</p>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="panel panel-default">
                            <div class="panel-heading">
                                <h4><i class="fa fa-terminal"></i> Browser Console Instructions</h4>
                            </div>
                            <div class="panel-body">
                                <p>You can also test the reminder system manually using the browser console:</p>
                                <ol>
                                    <li>Open Developer Tools (F12)</li>
                                    <li>Go to the Console tab</li>
                                    <li>Paste and run the verification script below:</li>
                                </ol>
                                <div style="background: #f8f9fa; border: 1px solid #dee2e6; padding: 10px; border-radius: 4px; margin-top: 10px;">
                                    <a href="<?php echo base_url('reminder_console_test.js'); ?>" target="_blank" class="btn btn-info btn-sm">
                                        <i class="fa fa-download"></i> Download Console Test Script
                                    </a>
                                    <p style="margin-top: 10px; margin-bottom: 0; font-size: 12px; color: #666;">
                                        Right-click the link above and "Save As" to download the test script, then copy and paste it into the browser console.
                                    </p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
let testResults = [];

function addResult(message, type = 'info', icon = 'fa-info') {
    const icons = {
        'success': 'fa-check-circle text-success',
        'error': 'fa-times-circle text-danger', 
        'warning': 'fa-exclamation-triangle text-warning',
        'info': 'fa-info-circle text-info'
    };
    
    testResults.push({
        message: message,
        type: type,
        icon: icons[type] || icon,
        timestamp: new Date().toLocaleTimeString()
    });
    
    updateTestDisplay();
}

function updateTestDisplay() {
    const container = document.getElementById('auto-test-results');
    let html = '';
    
    testResults.forEach(result => {
        html += `<div class="alert alert-${result.type}" style="margin-bottom: 5px; padding: 8px;">
                    <i class="fa ${result.icon}"></i> 
                    <strong>[${result.timestamp}]</strong> ${result.message}
                 </div>`;
    });
    
    container.innerHTML = html;
}

function updateSystemInfo() {
    const container = document.getElementById('system-info');
    let html = '<ul class="list-unstyled">';
    
    // User info
    if (typeof window.app !== 'undefined' && window.app.user_id) {
        html += '<li><i class="fa fa-user text-success"></i> <strong>User ID:</strong> ' + window.app.user_id + '</li>';
    } else {
        html += '<li><i class="fa fa-user text-danger"></i> <strong>User ID:</strong> Not available</li>';
    }
    
    // Admin URL
    if (typeof admin_url !== 'undefined') {
        html += '<li><i class="fa fa-link text-success"></i> <strong>Admin URL:</strong> ' + admin_url + '</li>';
    } else {
        html += '<li><i class="fa fa-link text-danger"></i> <strong>Admin URL:</strong> Not available</li>';
    }
    
    // Current page
    html += '<li><i class="fa fa-globe text-info"></i> <strong>Current Page:</strong> ' + window.location.href + '</li>';
    
    // System status
    if (typeof reminderCheckInterval !== 'undefined' && reminderCheckInterval !== null) {
        html += '<li><i class="fa fa-play text-success"></i> <strong>Reminder System:</strong> Active</li>';
    } else {
        html += '<li><i class="fa fa-pause text-warning"></i> <strong>Reminder System:</strong> Not Active</li>';
    }
    
    html += '</ul>';
    container.innerHTML = html;
}

function runAutoTest() {
    testResults = [];
    
    addResult('Starting automatic reminder system test...', 'info');
    
    // Test 1: Check window.app
    if (typeof window.app !== 'undefined') {
        addResult('window.app object exists', 'success');
        
        if (window.app.user_id && window.app.user_id !== "" && window.app.user_id !== "0") {
            addResult('User ID is valid: ' + window.app.user_id, 'success');
        } else {
            addResult('User ID is missing or invalid: ' + JSON.stringify(window.app.user_id), 'error');
        }
    } else {
        addResult('window.app object not available - you may not be on a CRM admin page', 'error');
    }
    
    // Test 2: Check admin_url
    if (typeof admin_url !== 'undefined') {
        addResult('admin_url is available: ' + admin_url, 'success');
    } else {
        addResult('admin_url is not defined', 'error');
    }
    
    // Test 3: Check functions
    const functions = ['initReminderNotifications', 'checkForDueReminders', 'showReminderNotification'];
    let availableFunctions = 0;
    
    functions.forEach(funcName => {
        if (typeof window[funcName] === 'function') {
            availableFunctions++;
        }
    });
    
    if (availableFunctions === functions.length) {
        addResult('All reminder functions are available (' + availableFunctions + '/' + functions.length + ')', 'success');
    } else {
        addResult('Some reminder functions are missing (' + availableFunctions + '/' + functions.length + ')', 'warning');
        
        // If functions are missing, try to load them manually
        if (availableFunctions === 0) {
            addResult('No reminder functions found - attempting manual injection...', 'info');
            loadReminderFunctions();
        }
    }
    
    // Test 4: Check if system is running
    if (typeof reminderCheckInterval !== 'undefined' && reminderCheckInterval !== null) {
        addResult('Reminder system is actively checking for due reminders', 'success');
    } else {
        addResult('Reminder system is not actively running', 'warning');
    }
    
    // Test 5: jQuery check
    if (typeof $ !== 'undefined') {
        addResult('jQuery is available', 'success');
    } else {
        addResult('jQuery is not available', 'error');
    }
    
    addResult('Automatic test completed!', 'success');
    
    // If system is not running but functions are available, try to initialize
    if (typeof initReminderNotifications === 'function' && 
        (typeof reminderCheckInterval === 'undefined' || reminderCheckInterval === null)) {
        addResult('Attempting to initialize reminder system...', 'info');
        try {
            initReminderNotifications();
            addResult('Reminder system initialization attempted', 'success');
        } catch (error) {
            addResult('Error initializing reminder system: ' + error.message, 'error');
        }
    }
}

function forceInitReminders() {
    if (typeof initReminderNotifications === 'function') {
        addResult('Manually initializing reminder system...', 'info');
        try {
            // Stop any existing interval first
            if (typeof stopReminderChecking === 'function') {
                stopReminderChecking();
            }
            
            // Initialize
            initReminderNotifications();
            addResult('Reminder system manually initialized', 'success');
            
            // Update system info
            setTimeout(updateSystemInfo, 1000);
        } catch (error) {
            addResult('Error manually initializing: ' + error.message, 'error');
        }
    } else {
        addResult('initReminderNotifications function not available', 'error');
    }
}

function checkRemindersNow() {
    if (typeof checkForDueReminders === 'function') {
        addResult('Manually checking for due reminders...', 'info');
        try {
            checkForDueReminders();
            addResult('Manual reminder check completed', 'success');
        } catch (error) {
            addResult('Error during manual check: ' + error.message, 'error');
        }
    } else {
        addResult('checkForDueReminders function not available', 'error');
    }
}

function showTestModal() {
    if (typeof showReminderNotification === 'function') {
        addResult('Showing test notification modal...', 'info');
        
        const testReminder = {
            id: 999999,
            description: "🧪 Test Notification - This is a test reminder to verify the notification system is working correctly.",
            formatted_date: new Date().toLocaleString(),
            time_until: "Now (Test Mode)",
            rel_type: "verification_test",
            rel_id: "999"
        };
        
        try {
            showReminderNotification([testReminder]);
            addResult('Test notification modal should now be visible', 'success');
        } catch (error) {
            addResult('Error showing test modal: ' + error.message, 'error');
        }
    } else {
        addResult('showReminderNotification function not available', 'error');
    }
}

// Debug function availability
function debugScripts() {
    console.log('=== Debugging Script Loading ===');
    console.log('window.app:', typeof window.app);
    console.log('admin_url:', typeof admin_url);
    console.log('jQuery:', typeof $);
    console.log('initReminderNotifications:', typeof initReminderNotifications);
    console.log('checkForDueReminders:', typeof checkForDueReminders);
    console.log('showReminderNotification:', typeof showReminderNotification);
    
    // Check if main.js functions exist in global scope
    console.log('Functions in window:', Object.keys(window).filter(key => 
        key.includes('reminder') || key.includes('Reminder')
    ));
    
    // Check for script tags
    const mainJsScripts = Array.from(document.querySelectorAll('script[src*="main.js"]'));
    console.log('Main.js script tags:', mainJsScripts.map(s => s.src));
    
    // Check if scripts loaded successfully
    mainJsScripts.forEach((script, index) => {
        console.log(`Script ${index + 1} loaded:`, !script.error);
        if (script.error) {
            console.error(`Script ${index + 1} error:`, script.error);
        }
    });
    
    // Try to manually load and execute reminder functions
    addResult('Debug info logged to console - check Developer Tools', 'info');
    
    // Check if we can find the functions by trying to access them differently
    if (typeof window['initReminderNotifications'] !== 'undefined') {
        addResult('Found initReminderNotifications via window object', 'success');
    } else {
        addResult('initReminderNotifications not found via window object', 'warning');
    }
    
    // Try to inject the functions manually as a test
    if (typeof initReminderNotifications === 'undefined') {
        addResult('Attempting to manually load reminder functions...', 'info');
        loadReminderFunctions();
    }
}

// Manual function injection for testing
function loadReminderFunctions() {
    try {
        // Define the reminder functions manually
        window.reminderCheckInterval = null;
        window.reminderCheckFrequency = 60000;
        window.currentDueReminders = [];
        window.reminderSoundEnabled = true;
        window.lastReminderCheck = 0;
        
        window.initReminderNotifications = function() {
            if (!window.app || !window.app.user_id || window.app.user_id == "" || window.app.user_id == "0") {
                console.log('Reminder system: Not initializing - user not logged in or missing user ID');
                return;
            }
            
            console.log('Reminder system: Initializing for user ID ' + window.app.user_id);
            window.startReminderChecking();
            console.log('Reminder notification system initialized');
        };
        
        window.startReminderChecking = function() {
            if (window.reminderCheckInterval) {
                clearInterval(window.reminderCheckInterval);
            }
            
            window.reminderCheckInterval = setInterval(function() {
                window.checkForDueReminders();
            }, window.reminderCheckFrequency);
            
            console.log('Reminder system: Checking interval started');
        };
        
        window.checkForDueReminders = function() {
            var now = Date.now();
            if (now - window.lastReminderCheck < 30000) {
                return;
            }
            window.lastReminderCheck = now;
            
            $.ajax({
                url: admin_url + 'misc/check_due_reminders',
                type: 'GET',
                dataType: 'json',
                success: function(response) {
                    console.log('Reminder check response:', response);
                    if (response.success && response.reminders && response.reminders.length > 0) {
                        var newReminders = response.reminders.filter(function(reminder) {
                            return !window.currentDueReminders.some(function(current) {
                                return current.id === reminder.id;
                            });
                        });
                        
                        if (newReminders.length > 0) {
                            window.currentDueReminders = response.reminders;
                            window.showReminderNotification(newReminders);
                        }
                    }
                },
                error: function(xhr, status, error) {
                    console.warn('Error checking for due reminders:', error);
                }
            });
        };
        
        window.showReminderNotification = function(reminders) {
            if (!reminders || reminders.length === 0) return;
            
            var reminder = reminders[0];
            alert('📋 Reminder Due!\n\n' + reminder.description + '\n\nDue: ' + reminder.formatted_date + '\nTime: ' + reminder.time_until);
            
            // Mark as notified
            $.post(admin_url + 'misc/mark_reminder_notified/' + reminder.id);
        };
        
        addResult('Manually injected reminder functions successfully', 'success');
        
        // Try to initialize
        window.initReminderNotifications();
        addResult('Attempted manual initialization', 'info');
          } catch (error) {
        addResult('Error injecting functions: ' + error.message, 'error');
    }
}

function enableMinifiedFiles() {
    addResult('Enabling minified files setting...', 'info');
    
    $.ajax({
        url: admin_url + 'settings?group=misc',
        type: 'POST',
        data: {
            'settings[use_minified_files]': '1'
        },
        success: function(response) {
            addResult('Minified files setting enabled successfully!', 'success');
            addResult('🎉 System is now configured to use minified files with reminder functions', 'success');
            addResult('The reminder system should now work across all CRM pages', 'info');
            
            // Show instructions for verification
            addResult('📋 Next Steps:', 'info');
            addResult('1. Navigate to any CRM page (dashboard, leads, etc.)', 'info');
            addResult('2. The reminder system should automatically initialize', 'info');
            addResult('3. Check browser console for reminder initialization messages', 'info');
        },
        error: function(xhr, status, error) {
            addResult('Error enabling minified files: ' + error, 'error');
            addResult('Settings can also be enabled manually:', 'info');
            addResult('Go to Setup → Settings → Misc → Use minified files → Yes', 'info');
        }
    });
}

// Initialize when page loads
$(document).ready(function() {
    // Wait a moment for all scripts to load
    setTimeout(function() {
        // Add debug function first
        debugScripts();
        
        runAutoTest();
        updateSystemInfo();
        
        // Auto-refresh system info every 10 seconds
        setInterval(updateSystemInfo, 10000);
    }, 2000);
});
</script>

<?php init_tail(); ?>
