<?php

use app\services\AbstractKanban;
use app\services\leads\LeadsKanban;

defined('BASEPATH') or exit('No direct script access allowed');

class Leads_model extends App_Model
{
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Get lead
     * @param  string $id Optional - leadid
     * @return mixed
     */
    public function get($id = '', $where = [])
    {
        $this->db->select('
            ' . db_prefix() . 'leads.*,
            ' . db_prefix() . 'leads.city,
            ' . db_prefix() . 'leads.state,
            ' . db_prefix() . 'leads.country,
            ' . db_prefix() . 'leads.zip,
            ' . db_prefix() . 'leads_status.name as status_name,
            ' . db_prefix() . 'leads_status.color,
            ' . db_prefix() . 'leads_sources.name as source_name,
            ' . db_prefix() . 'items.description as item_description
        ');
        $this->db->join(db_prefix() . 'leads_status', db_prefix() . 'leads_status.id=' . db_prefix() . 'leads.status', 'left');
        $this->db->join(db_prefix() . 'leads_sources', db_prefix() . 'leads_sources.id=' . db_prefix() . 'leads.source', 'left');
        $this->db->join(db_prefix() . 'items', db_prefix() . 'items.id=' . db_prefix() . 'leads.item_id', 'left');

        $this->db->where($where);
        if (is_numeric($id)) {
            $this->db->where(db_prefix() . 'leads.id', $id);
            $lead = $this->db->get(db_prefix() . 'leads')->row();
            if ($lead) {
                if ($lead->from_form_id != 0) {
                    $lead->form_data = $this->get_form([
                        'id' => $lead->from_form_id,
                    ]);
                }
                $lead->attachments = $this->get_lead_attachments($id);
                $lead->public_url  = leads_public_url($id);
            }

            return $lead;
        }

        return $this->db->get(db_prefix() . 'leads')->result_array();
    }

    /**
     * Get lead by given email
     *
     * @since 2.8.0
     *
     * @param  string $email
     *
     * @return \strClass|null
     */
    public function get_lead_by_email($email)
    {
        $this->db->where('email', $email);
        $this->db->limit(1);

        return $this->db->get('leads')->row();
    }

    /**
     * Add new lead to database
     * @param mixed $data lead data
     * @return mixed false || leadid
     */    public function add($data)
    {
        // Handle phone numbers
        if (isset($data['phonenumber'])) {
            // Convert array to string if needed and clean
            $data['phonenumber'] = is_array($data['phonenumber']) ? 
                (isset($data['phonenumber'][0]) ? trim((string)$data['phonenumber'][0]) : '') : 
                trim((string)$data['phonenumber']);

            app_log("Processing primary phone number", 'debug', [
                'original' => $data['phonenumber'],
                'cleaned' => $data['phonenumber']
            ]);
        }

        // Additional phone numbers
        if (isset($data['additional_phones']) && is_array($data['additional_phones'])) {
            // Filter out empty values and reindex array
            $additional_phones = array_values(array_filter($data['additional_phones'], function($phone) {
                return !empty(trim($phone));
            }));
            
            $data['phone_numbers'] = !empty($additional_phones) ? json_encode($additional_phones) : null;
            unset($data['additional_phones']);
            
            // Log phone numbers being saved
            app_log("Saving lead phone numbers", 'debug', [
                'primary' => $data['phonenumber'] ?? null,
                'additional' => $additional_phones
            ]);
        }
        
        // Handle item_id field - ensure it's null if empty or doesn't exist
        if (isset($data['item_id']) && (empty($data['item_id']) || $data['item_id'] == '')) {
            $data['item_id'] = null;
        }
        
        // Handle address fields
        $address_fields = ['address', 'city', 'state', 'country', 'zip'];
        foreach ($address_fields as $field) {
            if (isset($data[$field])) {
                // Clean the input and ensure it's a string (except for country)
                $data[$field] = is_string($data[$field]) ? trim($data[$field]) : $data[$field];
                
                // Special handling for country field which is stored as integer
                if ($field === 'country') {
                    $data[$field] = empty($data[$field]) ? 0 : (int)$data[$field];
                }
                
                // Convert empty strings to null for all fields except country
                if ($data[$field] === '' && $field !== 'country') {
                    $data[$field] = null;
                }
            } else if ($field === 'country') {
                // Ensure country is 0 and not null when not set
                $data[$field] = 0;
            }
        }

        // Log the start of lead creation
        app_log("Starting lead creation process", 'info', ['data' => array_intersect_key($data, array_flip(['name', 'email', 'status', 'source']))]);
        
        if (isset($data['custom_contact_date']) || isset($data['custom_contact_date'])) {
            if (isset($data['contacted_today'])) {
                $data['lastcontact'] = date('Y-m-d H:i:s');
                unset($data['contacted_today']);
            } else {
                $data['lastcontact'] = to_sql_date($data['custom_contact_date'], true);
            }
        }

        if (isset($data['is_public']) && ($data['is_public'] == 1 || $data['is_public'] === 'on')) {
            $data['is_public'] = 1;
        } else {
            $data['is_public'] = 0;
        }

        if (!isset($data['country']) || isset($data['country']) && $data['country'] == '') {
            $data['country'] = 0;
        }

        if (isset($data['custom_contact_date'])) {
            unset($data['custom_contact_date']);
        }

        $data['description'] = nl2br($data['description']);
        $data['dateadded']   = date('Y-m-d H:i:s');
        $data['addedfrom']   = get_staff_user_id();

        $data = hooks()->apply_filters('before_lead_added', $data);

        $tags = '';
        if (isset($data['tags'])) {
            $tags = $data['tags'];
            unset($data['tags']);
        }

        if (isset($data['custom_fields'])) {
            $custom_fields = $data['custom_fields'];
            unset($data['custom_fields']);
        }

        $data['address'] = trim($data['address']);
        $data['address'] = nl2br($data['address']);

        $data['email'] = trim($data['email']);
        $this->db->insert(db_prefix() . 'leads', $data);
        $insert_id = $this->db->insert_id();        if ($insert_id) {
            log_activity('New Lead Added [ID: ' . $insert_id . ']');
            $this->log_lead_activity($insert_id, 'not_lead_activity_created');

            // Log lead creation success to our custom event log            // Prepare source and status data safely
            $source_info = null;
            if (isset($data['source'])) {
                $source_obj = $this->get_source($data['source']);
                if ($source_obj && is_object($source_obj) && isset($source_obj->name)) {
                    $source_info = $source_obj->name;
                } else {
                    $source_info = $data['source'];
                }
            }
            
            $status_info = null;
            if (isset($data['status'])) {
                $status_obj = $this->get_status($data['status']);
                if ($status_obj && !is_array($status_obj) && isset($status_obj->name)) {
                    $status_info = $status_obj->name;
                } elseif (is_array($status_obj) && isset($status_obj[0]['name'])) {
                    $status_info = $status_obj[0]['name'];
                } else {
                    $status_info = $data['status'];
                }
            }
            
            app_log("Lead created successfully", 'info', [
                'lead_id' => $insert_id,
                'name' => $data['name'],
                'email' => $data['email'] ?? null,
                'phone' => $data['phonenumber'] ?? null,
                'source' => $source_info,
                'status' => $status_info,
                'assigned_to' => isset($data['assigned']) ? get_staff_full_name($data['assigned']) : null
            ]);

            handle_tags_save($tags, $insert_id, 'lead');

            if (isset($custom_fields)) {
                handle_custom_fields_post($insert_id, $custom_fields);
            }

            $this->lead_assigned_member_notification($insert_id, $data['assigned']);
            hooks()->do_action('lead_created', $insert_id);

            return $insert_id;
        }

        return false;
    }

    public function lead_assigned_member_notification($lead_id, $assigned, $integration = false)
    {
        if (empty($assigned) || $assigned == 0) {
            return;
        }

        if ($integration == false) {
            if ($assigned == get_staff_user_id()) {
                return false;
            }
        }

        $name = $this->db->select('name')->from(db_prefix() . 'leads')->where('id', $lead_id)->get()->row()->name;

        $notification_data = [
            'description'     => ($integration == false) ? 'not_assigned_lead_to_you' : 'not_lead_assigned_from_form',
            'touserid'        => $assigned,
            'link'            => '#leadid=' . $lead_id,
            'additional_data' => ($integration == false ? serialize([
                $name,
            ]) : serialize([])),
        ];

        if ($integration != false) {
            $notification_data['fromcompany'] = 1;
        }

        if (add_notification($notification_data)) {
            pusher_trigger_notification([$assigned]);
        }

        $this->db->select('email');
        $this->db->where('staffid', $assigned);
        $email = $this->db->get(db_prefix() . 'staff')->row()->email;

        send_mail_template('lead_assigned', $lead_id, $email);

        $this->db->where('id', $lead_id);
        $this->db->update(db_prefix() . 'leads', [
            'dateassigned' => date('Y-m-d'),
        ]);

        $not_additional_data = [
            e(get_staff_full_name()),
            '<a href="' . admin_url('profile/' . $assigned) . '" target="_blank">' . e(get_staff_full_name($assigned)) . '</a>',
        ];

        if ($integration == true) {
            unset($not_additional_data[0]);
            array_values(($not_additional_data));
        }

        $not_additional_data = serialize($not_additional_data);

        $not_desc = ($integration == false ? 'not_lead_activity_assigned_to' : 'not_lead_activity_assigned_from_form');
        $this->log_lead_activity($lead_id, $not_desc, $integration, $not_additional_data);

        hooks()->do_action('after_lead_assigned_member_notification_sent', $lead_id);
    }    /**
     * Update a lead
     * @param  array $data lead data
     * @param  mixed $id   lead id
     * @return boolean
     */    public function update($data, $id)
    {
        $current_lead_data = $this->get($id);
        $affectedRows = 0;
        
        app_log("Lead update - Original data", 'debug', [
            'lead_id' => $id,
            'data' => $data
        ]);

        if (isset($data['custom_fields'])) {
            $custom_fields = $data['custom_fields'];
            if (handle_custom_fields_post($id, $custom_fields)) {
                $affectedRows++;
            }
            unset($data['custom_fields']);
        }

        // Handle item_id field - ensure it's null if empty or doesn't exist
        if (isset($data['item_id']) && (empty($data['item_id']) || $data['item_id'] == '')) {
            $data['item_id'] = null;
        }

        // Handle address fields - simplified approach
        if (isset($data['address'])) {
            $data['address'] = trim($data['address']);
            $data['address'] = nl2br($data['address']);
        }
        
        if (isset($data['city'])) {
            $data['city'] = trim($data['city']);
        }
        
        if (isset($data['state'])) {
            $data['state'] = trim($data['state']);
        }
        
        if (isset($data['zip'])) {
            $data['zip'] = trim($data['zip']);
        }
        
        // Handle country field specifically
        if (isset($data['country'])) {
            if ($data['country'] === '' || $data['country'] === null) {
                $data['country'] = 0;
            } else {
                $data['country'] = (int)$data['country'];
            }
        }

        if (isset($data['status_id']) && $data['status_id'] == "") {
            $data['status_id'] = 0;
        }

        if (isset($data['phonenumber'])) {
            $data['phonenumber'] = is_array($data['phonenumber']) ? 
                (isset($data['phonenumber'][0]) ? trim((string)$data['phonenumber'][0]) : '') : 
                trim((string)$data['phonenumber']);
        }

        if (isset($data['additional_phones'])) {
            $additional_phones = [];
            if (is_array($data['additional_phones'])) {
                foreach ($data['additional_phones'] as $phone) {
                    $phone_value = is_array($phone) ? '' : trim((string)$phone);
                    if (!empty($phone_value)) {
                        $additional_phones[] = $phone_value;
                    }
                }
                $data['phone_numbers'] = !empty($additional_phones) ? json_encode($additional_phones) : null;
            } else {
                $phone = trim((string)$data['additional_phones']);
                $data['phone_numbers'] = !empty($phone) ? json_encode([$phone]) : null;
            }
            unset($data['additional_phones']);
        }

        if (!defined('API')) {
            $data['is_public'] = isset($data['is_public']) ? 1 : 0;
            if (isset($data['description'])) {
                $data['description'] = nl2br($data['description']);
            }
        }

        if (isset($data['lastcontact'])) {
            $data['lastcontact'] = empty($data['lastcontact']) ? null : to_sql_date($data['lastcontact'], true);
        }

        if (isset($data['tags'])) {
            if (handle_tags_save($data['tags'], $id, 'lead')) {
                $affectedRows++;
            }
            unset($data['tags']);
        }

        if (isset($data['remove_attachments'])) {
            $removed_attachments = false;
            if (isset($data['remove_attachments'])) {
                $removed_attachments = true;
                unset($data['remove_attachments']);
            }
            foreach ($current_lead_data->attachments as $attachment) {
                if ($removed_attachments == true || array_key_exists($attachment['id'], $data['remove_attachments'])) {
                    $this->delete_lead_attachment($attachment['id']);
                }
            }
            unset($data['remove_attachments']);
        }        $data = hooks()->apply_filters('before_lead_updated', $data, $id);

        // Log the final data being saved
        app_log("Lead update - Final data for database update", 'debug', [
            'lead_id' => $id,
            'data' => array_intersect_key($data, array_flip(['address', 'city', 'state', 'country', 'zip']))
        ]);

        $this->db->where('id', $id);
        $this->db->update(db_prefix() . 'leads', $data);
        
        if ($this->db->affected_rows() > 0) {
            $affectedRows++;
            app_log("Lead update - Address fields saved successfully", 'info', [
                'lead_id' => $id
            ]);
        } else {
            app_log("Lead update - No changes to address fields or update failed", 'debug', [
                'lead_id' => $id,
                'last_query' => $this->db->last_query()
            ]);
        }

        if (isset($data['is_public'])) {
            $current = $current_lead_data->is_public;
            $updated = $data['is_public'];
            if ((int) $current != (int) $updated) {
                $this->log_lead_activity(
                    $id,
                    'not_lead_activity_' . ($updated == '1' ? 'made_public' : 'made_not_public'),
                    false,
                    serialize([
                        get_staff_full_name(),
                    ])
                );
            }
        }

        // Check if assignment has changed and send notification
        if (isset($data['assigned'])) {
            $current_assigned = $current_lead_data->assigned;
            $new_assigned = $data['assigned'];
            
            // Only send notification if assignment actually changed and new assignment is not empty
            if ($current_assigned != $new_assigned && !empty($new_assigned) && $new_assigned != 0) {
                $this->lead_assigned_member_notification($id, $new_assigned);
            }
        }

        hooks()->do_action('after_lead_updated', $id);

        return (bool) $affectedRows;
    }

    /**
     * Delete lead from database and all connections
     * @param  mixed $id leadid
     * @return boolean
     */    public function delete($id)
    {
        $affectedRows = 0;

        hooks()->do_action('before_lead_deleted', $id);

        $lead = $this->get($id);
        
        // Log lead deletion attempt
        app_log("Lead deletion initiated", 'warning', [
            'lead_id' => $id,
            'name' => $lead->name ?? 'Unknown',
            'staff' => get_staff_full_name()
        ]);

        $this->db->where('id', $id);
        $this->db->delete(db_prefix() . 'leads');
        if ($this->db->affected_rows() > 0) {
            log_activity('Lead Deleted [Deleted by: ' . get_staff_full_name() . ', ID: ' . $id . ']');
            
            // Log lead deletion success
            app_log("Lead deleted successfully", 'warning', [
                'lead_id' => $id,
                'name' => $lead->name ?? 'Unknown',
                'staff' => get_staff_full_name()
            ]);            $attachments = $this->get_lead_attachments($id);
            foreach ($attachments as $attachment) {
                $this->delete_lead_attachment($attachment['id']);
            }

            // Delete the custom field values
            $this->db->where('relid', $id);
            $this->db->where('fieldto', 'leads');
            $this->db->delete(db_prefix() . 'customfieldsvalues');

            $this->db->where('leadid', $id);
            $this->db->delete(db_prefix() . 'lead_activity_log');

            $this->db->where('leadid', $id);
            $this->db->delete(db_prefix() . 'lead_integration_emails');

            $this->db->where('rel_id', $id);
            $this->db->where('rel_type', 'lead');
            $this->db->delete(db_prefix() . 'notes');

            $this->db->where('rel_type', 'lead');
            $this->db->where('rel_id', $id);
            $this->db->delete(db_prefix() . 'reminders');

            $this->db->where('rel_type', 'lead');
            $this->db->where('rel_id', $id);
            $this->db->delete(db_prefix() . 'taggables');

            $this->load->model('proposals_model');
            $this->db->where('rel_id', $id);
            $this->db->where('rel_type', 'lead');
            $proposals = $this->db->get(db_prefix() . 'proposals')->result_array();

            foreach ($proposals as $proposal) {
                $this->proposals_model->delete($proposal['id']);
            }

            // Get related tasks
            $this->db->where('rel_type', 'lead');
            $this->db->where('rel_id', $id);
            $tasks = $this->db->get(db_prefix() . 'tasks')->result_array();
            foreach ($tasks as $task) {
                $this->tasks_model->delete_task($task['id']);
            }

            if (is_gdpr()) {
                $this->db->where('(description LIKE "%' . $lead->email . '%" OR description LIKE "%' . $lead->name . '%" OR description LIKE "%' . $lead->phonenumber . '%")');
                $this->db->delete(db_prefix() . 'activity_log');
            }

            $affectedRows++;
        }
        if ($affectedRows > 0) {
            hooks()->do_action('after_lead_deleted', $id);
            return true;
        }

        return false;
    }

    /**
     * Mark lead as lost
     * @param  mixed $id lead id
     * @return boolean
     */
    public function mark_as_lost($id)
    {
        $this->db->select('status');
        $this->db->from(db_prefix() . 'leads');
        $this->db->where('id', $id);
        $last_lead_status = $this->db->get()->row()->status;

        $this->db->where('id', $id);
        $this->db->update(db_prefix() . 'leads', [
            'lost'               => 1,
            'status'             => 0,
            'last_status_change' => date('Y-m-d H:i:s'),
            'last_lead_status'   => $last_lead_status,
        ]);

        if ($this->db->affected_rows() > 0) {
            $this->log_lead_activity($id, 'not_lead_activity_marked_lost');

            log_activity('Lead Marked as Lost [ID: ' . $id . ']');

            hooks()->do_action('lead_marked_as_lost', $id);

            return true;
        }

        return false;
    }

    /**
     * Unmark lead as lost
     * @param  mixed $id leadid
     * @return boolean
     */
    public function unmark_as_lost($id)
    {
        $this->db->select('last_lead_status');
        $this->db->from(db_prefix() . 'leads');
        $this->db->where('id', $id);
        $last_lead_status = $this->db->get()->row()->last_lead_status;

        $this->db->where('id', $id);
        $this->db->update(db_prefix() . 'leads', [
            'lost'   => 0,
            'status' => $last_lead_status,
        ]);
        if ($this->db->affected_rows() > 0) {
            $this->log_lead_activity($id, 'not_lead_activity_unmarked_lost');

            log_activity('Lead Unmarked as Lost [ID: ' . $id . ']');

            return true;
        }

        return false;
    }

    /**
     * Mark lead as junk
     * @param  mixed $id lead id
     * @return boolean
     */
    public function mark_as_junk($id)
    {
        $this->db->select('status');
        $this->db->from(db_prefix() . 'leads');
        $this->db->where('id', $id);
        $last_lead_status = $this->db->get()->row()->status;

        $this->db->where('id', $id);
        $this->db->update(db_prefix() . 'leads', [
            'junk'               => 1,
            'status'             => 0,
            'last_status_change' => date('Y-m-d H:i:s'),
            'last_lead_status'   => $last_lead_status,
        ]);

        if ($this->db->affected_rows() > 0) {
            $this->log_lead_activity($id, 'not_lead_activity_marked_junk');

            log_activity('Lead Marked as Junk [ID: ' . $id . ']');

            hooks()->do_action('lead_marked_as_junk', $id);

            return true;
        }

        return false;
    }

    /**
     * Unmark lead as junk
     * @param  mixed $id leadid
     * @return boolean
     */
    public function unmark_as_junk($id)
    {
        $this->db->select('last_lead_status');
        $this->db->from(db_prefix() . 'leads');
        $this->db->where('id', $id);
        $last_lead_status = $this->db->get()->row()->last_lead_status;

        $this->db->where('id', $id);
        $this->db->update(db_prefix() . 'leads', [
            'junk'   => 0,
            'status' => $last_lead_status,
        ]);
        if ($this->db->affected_rows() > 0) {
            $this->log_lead_activity($id, 'not_lead_activity_unmarked_junk');
            log_activity('Lead Unmarked as Junk [ID: ' . $id . ']');

            return true;
        }

        return false;
    }

    /**
     * Get lead attachments
     * @since Version 1.0.4
     * @param  mixed $id lead id
     * @return array
     */
    public function get_lead_attachments($id = '', $attachment_id = '', $where = [])
    {
        $this->db->where($where);
        $idIsHash = !is_numeric($attachment_id) && strlen($attachment_id) == 32;
        if (is_numeric($attachment_id) || $idIsHash) {
            $this->db->where($idIsHash ? 'attachment_key' : 'id', $attachment_id);

            return $this->db->get(db_prefix() . 'files')->row();
        }
        $this->db->where('rel_id', $id);
        $this->db->where('rel_type', 'lead');
        $this->db->order_by('dateadded', 'DESC');
        
        // Ensure we're selecting the description field explicitly
        $this->db->select('*, description'); // Make sure description is included

        $result = $this->db->get(db_prefix() . 'files')->result_array();
        
        // Log the first result to check if descriptions are coming through
        if (!empty($result)) {
            error_log("First lead attachment: " . json_encode($result[0]));
        }
        
        return $result;
    }

    public function add_attachment_to_database($lead_id, $attachment, $external = false, $form_activity = false)
    {
        $this->misc_model->add_attachment_to_database($lead_id, 'lead', $attachment, $external);

        if ($form_activity == false) {
            $this->leads_model->log_lead_activity($lead_id, 'not_lead_activity_added_attachment');
        } else {
            $this->leads_model->log_lead_activity($lead_id, 'not_lead_activity_log_attachment', true, serialize([
                $form_activity,
            ]));
        }

        // No notification when attachment is imported from web to lead form
        if ($form_activity == false) {
            $lead         = $this->get($lead_id);
            $not_user_ids = [];
            if ($lead->addedfrom != get_staff_user_id()) {
                array_push($not_user_ids, $lead->addedfrom);
            }
            if ($lead->assigned != get_staff_user_id() && $lead->assigned != 0) {
                array_push($not_user_ids, $lead->assigned);
            }
            $notifiedUsers = [];
            foreach ($not_user_ids as $uid) {
                $notified = add_notification([
                    'description'     => 'not_lead_added_attachment',
                    'touserid'        => $uid,
                    'link'            => '#leadid=' . $lead_id,
                    'additional_data' => serialize([
                        $lead->name,
                    ]),
                ]);
                if ($notified) {
                    array_push($notifiedUsers, $uid);
                }
            }
            pusher_trigger_notification($notifiedUsers);
        }
    }

    /**
     * Delete lead attachment
     * @param  mixed $id attachment id
     * @return boolean
     */    public function delete_lead_attachment($id)
    {
        // Get the attachment by attachment ID (second parameter)
        $attachment = $this->get_lead_attachments('', $id);
        $deleted    = false;

        if ($attachment) {
            if (empty($attachment->external)) {
                unlink(get_upload_path_by_type('lead') . $attachment->rel_id . '/' . $attachment->file_name);
            }
            $this->db->where('id', $attachment->id);
            $this->db->delete(db_prefix() . 'files');
            if ($this->db->affected_rows() > 0) {
                $deleted = true;
                log_activity('Lead Attachment Deleted [ID: ' . $attachment->rel_id . ']');
            }

            if (is_dir(get_upload_path_by_type('lead') . $attachment->rel_id)) {
                // Check if no attachments left, so we can delete the folder also
                $other_attachments = list_files(get_upload_path_by_type('lead') . $attachment->rel_id);
                if (count($other_attachments) == 0) {
                    // okey only index.html so we can delete the folder also
                    delete_dir(get_upload_path_by_type('lead') . $attachment->rel_id);
                }
            }
        }

        return $deleted;
    }    /**
     * Get leads sources
     * @param  mixed $id Optional - Source ID
     * @return mixed object if id passed else array
     */
    public function get_source($id = false)
    {
        try {
            if (is_numeric($id)) {
                $this->db->where('id', $id);
                
                $result = $this->db->get(db_prefix() . 'leads_sources')->row();
                
                if (function_exists('app_log') && empty($result)) {
                    app_log("Source not found for ID: {$id}", 'warning', [
                        'id' => $id
                    ]);
                }
                
                return $result;
            }

            $this->db->order_by('name', 'asc');
            
            return $this->db->get(db_prefix() . 'leads_sources')->result_array();
        } catch (Exception $e) {
            if (function_exists('app_log')) {
                app_log("Error in get_source: " . $e->getMessage(), 'error', [
                    'id' => $id,
                    'trace' => $e->getTraceAsString()
                ]);
            }
            
            // Return empty result based on what was requested
            return is_numeric($id) ? null : [];
        }
    }

    /**
     * Add new lead source
     * @param mixed $data source data
     */
    public function add_source($data)
    {
        $this->db->insert(db_prefix() . 'leads_sources', $data);
        $insert_id = $this->db->insert_id();
        if ($insert_id) {
            log_activity('New Leads Source Added [SourceID: ' . $insert_id . ', Name: ' . $data['name'] . ']');
        }

        return $insert_id;
    }

    /**
     * Update lead source
     * @param  mixed $data source data
     * @param  mixed $id   source id
     * @return boolean
     */
    public function update_source($data, $id)
    {
        $this->db->where('id', $id);
        $this->db->update(db_prefix() . 'leads_sources', $data);
        if ($this->db->affected_rows() > 0) {
            log_activity('Leads Source Updated [SourceID: ' . $id . ', Name: ' . $data['name'] . ']');

            return true;
        }

        return false;
    }

    /**
     * Delete lead source from database
     * @param  mixed $id source id
     * @return mixed
     */
    public function delete_source($id)
    {
        $current = $this->get_source($id);
        // Check if is already using in table
        if (is_reference_in_table('source', db_prefix() . 'leads', $id) || is_reference_in_table('lead_source', db_prefix() . 'leads_email_integration', $id)) {
            return [
                'referenced' => true,
            ];
        }
        $this->db->where('id', $id);
        $this->db->delete(db_prefix() . 'leads_sources');
        if ($this->db->affected_rows() > 0) {
            if (get_option('leads_default_source') == $id) {
                update_option('leads_default_source', '');
            }
            log_activity('Leads Source Deleted [SourceID: ' . $id . ']');

            return true;
        }

        return false;
    }

    // Statuses

    /**
     * Get lead statuses
     * @param  mixed $id status id
     * @return mixed      object if id passed else array
     */    /**
     * Get lead statuses
     * @param  mixed $id status id
     * @param  array $where additional where conditions
     * @return mixed object if id passed else array
     */
    public function get_status($id = '', $where = [])
    {
        try {
            if (is_numeric($id)) {
                $this->db->where($where);
                $this->db->where('id', $id);
                
                $result = $this->db->get(db_prefix() . 'leads_status')->row();
                
                if (function_exists('app_log') && empty($result)) {
                    app_log("Status not found for ID: {$id}", 'warning', [
                        'id' => $id,
                        'where' => $where
                    ]);
                }
                
                return $result;
            }

            $whereKey = md5(serialize($where));
          
            $statuses = $this->app_object_cache->get('leads-all-statuses-'.$whereKey);

            if (!$statuses) {
                $this->db->where($where);
                $this->db->order_by('statusorder', 'asc');

                $statuses = $this->db->get(db_prefix() . 'leads_status')->result_array();
                $this->app_object_cache->add('leads-all-statuses-'.$whereKey, $statuses);
            }
            
            return $statuses;
        } catch (Exception $e) {
            if (function_exists('app_log')) {
                app_log("Error in get_status: " . $e->getMessage(), 'error', [
                    'id' => $id,
                    'where' => $where,
                    'trace' => $e->getTraceAsString()
                ]);
            }
            
            // Return empty result based on what was requested
            return is_numeric($id) ? null : [];
        }
    }

    /**
     * Add new lead status
     * @param array $data lead status data
     */
    public function add_status($data)
    {
        if (isset($data['color']) && $data['color'] == '') {
            $data['color'] = hooks()->apply_filters('default_lead_status_color', '#757575');
        }

        if (!isset($data['statusorder'])) {
            $data['statusorder'] = total_rows(db_prefix() . 'leads_status') + 1;
        }

        $this->db->insert(db_prefix() . 'leads_status', $data);
        $insert_id = $this->db->insert_id();
        if ($insert_id) {
            log_activity('New Leads Status Added [StatusID: ' . $insert_id . ', Name: ' . $data['name'] . ']');

            return $insert_id;
        }

        return false;
    }

    public function update_status($data, $id)
    {
        $this->db->where('id', $id);
        $this->db->update(db_prefix() . 'leads_status', $data);
        if ($this->db->affected_rows() > 0) {
            log_activity('Leads Status Updated [StatusID: ' . $id . ', Name: ' . $data['name'] . ']');

            return true;
        }

        return false;
    }

    /**
     * Delete lead status from database
     * @param  mixed $id status id
     * @return boolean
     */    public function delete_status($id)
    {
        $current = $this->get_status($id);
        // Double check that $current is actually an object, not an array
        if (is_array($current)) {
            // If it's an array, try to get the first item if it exists
            $current = !empty($current) && isset($current[0]) ? (object)$current[0] : null;
        }
        
        // Check if is already using in table
        if (is_reference_in_table('status', db_prefix() . 'leads', $id) || is_reference_in_table('lead_status', db_prefix() . 'leads_email_integration', $id)) {
            return [
                'referenced' => true,
            ];
        }

        $this->db->where('id', $id);
        $this->db->delete(db_prefix() . 'leads_status');
        if ($this->db->affected_rows() > 0) {
            if (get_option('leads_default_status') == $id) {
                update_option('leads_default_status', '');
            }
            log_activity('Leads Status Deleted [StatusID: ' . $id . ']');

            return true;
        }

        return false;
    }

    /**
     * Update canban lead status when drag and drop
     * @param  array $data lead data
     * @return boolean
         /**
     * Update lead status
     * @param array $data Data containing leadid and status
     * @return bool
     */
    public function update_lead_status($data)
    {
        $affectedRows = 0;
        $_log_message = '';
        $additional_data = null;
        
        try {
            // Ensure we have a valid leadid
            if (!isset($data['leadid']) || empty($data['leadid'])) {
                app_log("Missing lead ID in update_lead_status", 'error', ['data' => $data]);
                return false;
            }
            
            // Handle cases where status might be an array
            if (isset($data['status']) && is_array($data['status'])) {
                app_log("Status was an array in update_lead_status", 'warning', ['status' => $data['status']]);
                
                // If status is an array, try to extract a valid ID
                if (isset($data['status']['id'])) {
                    $data['status'] = $data['status']['id'];
                } else {
                    // Get first item if it's numeric
                    $first_value = reset($data['status']);
                    if (is_numeric($first_value)) {
                        $data['status'] = $first_value;
                    } else {
                        app_log("Could not determine valid status from array", 'error', ['status' => $data['status']]);
                        return false;
                    }
                }
            }
            
            // Ensure we have a valid status now
            if (!isset($data['status']) || empty($data['status'])) {
                app_log("Missing status after conversion in update_lead_status", 'error', ['data' => $data]);
                return false;
            }
            
            // Get the old status info
            $this->db->select('status');
            $this->db->where('id', $data['leadid']);
            $_old = $this->db->get(db_prefix() . 'leads')->row();
            $old_status = '';

            if ($_old) {
                $status_obj = $this->get_status($_old->status);
                if ($status_obj && !is_array($status_obj)) {
                    $old_status = $status_obj->name;
                } elseif (is_array($status_obj) && isset($status_obj[0]['name'])) {
                    $old_status = $status_obj[0]['name'];
                }
            }

            // Get the current status info
            $current_status = '';
            $status_id = $data['status']; // Already validated and normalized above
            $status_obj = $this->get_status($status_id);
            
            if ($status_obj && !is_array($status_obj)) {
                $current_status = $status_obj->name;
            } elseif (is_array($status_obj) && isset($status_obj[0]['name'])) {
                $current_status = $status_obj[0]['name'];
            }

            // Update the lead status
            $this->db->where('id', $data['leadid']);
            $this->db->update(db_prefix() . 'leads', [
                'status' => $status_id,
            ]);

            // Process the update if it was successful
            if ($this->db->affected_rows() > 0) {
                $affectedRows++;
                if ($current_status != $old_status && $old_status != '') {
                    $_log_message = 'not_lead_activity_status_updated';
                    $additional_data = serialize([
                        get_staff_full_name(),
                        $old_status,
                        $current_status,
                    ]);

                    // Log the status change to our custom event log
                    app_log("Lead status updated - Lead ID: " . $data['leadid'] . ", Old status: " . $old_status . ", New status: " . $current_status, 
                            'info', 
                            [
                                'lead_id' => $data['leadid'],
                                'old_status' => $old_status,
                                'new_status' => $current_status,
                                'staff' => get_staff_full_name()
                            ]);

                    hooks()->do_action('lead_status_changed', [
                        'lead_id'    => $data['leadid'],
                        'old_status' => $old_status,
                        'new_status' => $current_status,
                    ]);
                }
                
                $this->db->where('id', $data['leadid']);
                $this->db->update(db_prefix() . 'leads', [
                    'last_status_change' => date('Y-m-d H:i:s'),
                ]);
            }

            // Handle kanban order updates
            if (isset($data['order'])) {
                AbstractKanban::updateOrder($data['order'], 'leadorder', 'leads', $data['status']);
            }
        } catch (Exception $e) {
            app_log("Exception in update_lead_status: " . $e->getMessage(), 'error', [
                'data' => $data,
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }

        // Return the result
        if ($affectedRows > 0) {
            if ($_log_message != '' && $additional_data) {
                $this->log_lead_activity($data['leadid'], $_log_message, false, $additional_data);
            }
            return true;
        }
        
        return false;
    }

    /* Ajax */

    /**
     * All lead activity by staff
     * @param  mixed $id lead id
     * @return array
     */
    public function get_lead_activity_log($id)
    {
        $sorting = hooks()->apply_filters('lead_activity_log_default_sort', 'ASC');

        $this->db->where('leadid', $id);
        $this->db->order_by('date', $sorting);

        return $this->db->get(db_prefix() . 'lead_activity_log')->result_array();
    }

    public function staff_can_access_lead($id, $staff_id = '')
    {
        $staff_id = $staff_id == '' ? get_staff_user_id() : $staff_id;

        if (has_permission('leads', $staff_id, 'view')) {
            return true;
        }

        $CI = &get_instance();

        if (total_rows(db_prefix() . 'leads', 'id="' . $CI->db->escape_str($id) . '" AND (assigned=' . $CI->db->escape_str($staff_id) . ' OR is_public=1 OR addedfrom=' . $CI->db->escape_str($staff_id) . ')') > 0) {
            return true;
        }

        return false;
    }

    /**
     * Add lead activity from staff
     * @param  mixed  $id          lead id
     * @param  string  $description activity description
     */
    public function log_lead_activity($id, $description, $integration = false, $additional_data = '')
    {
        $log = [
            'date'            => date('Y-m-d H:i:s'),
            'description'     => $description,
            'leadid'          => $id,
            'staffid'         => get_staff_user_id(),
            'additional_data' => $additional_data,
            'full_name'       => get_staff_full_name(get_staff_user_id()),
        ];
        if ($integration == true) {
            $log['staffid']   = 0;
            $log['full_name'] = '[CRON]';
        }

        $this->db->insert(db_prefix() . 'lead_activity_log', $log);

        return $this->db->insert_id();
    }

    /**
     * Get email integration config
     * @return object
     */
    public function get_email_integration()
    {
        $this->db->where('id', 1);

        return $this->db->get(db_prefix() . 'leads_email_integration')->row();
    }

    /**
     * Get lead imported email activity
     * @param  mixed $id leadid
     * @return array
     */
    public function get_mail_activity($id)
    {
        $this->db->where('leadid', $id);
        $this->db->order_by('dateadded', 'asc');

        return $this->db->get(db_prefix() . 'lead_integration_emails')->result_array();
    }

    /**
     * Update email integration config
     * @param  mixed $data All $_POST data
     * @return boolean
     */
    public function update_email_integration($data)
    {
        $this->db->where('id', 1);
        $original_settings = $this->db->get(db_prefix() . 'leads_email_integration')->row();

        $data['create_task_if_customer']        = isset($data['create_task_if_customer']) ? 1 : 0;
        $data['active']                         = isset($data['active']) ? 1 : 0;
        $data['delete_after_import']            = isset($data['delete_after_import']) ? 1 : 0;
        $data['notify_lead_imported']           = isset($data['notify_lead_imported']) ? 1 : 0;
        $data['only_loop_on_unseen_emails']     = isset($data['only_loop_on_unseen_emails']) ? 1 : 0;
        $data['notify_lead_contact_more_times'] = isset($data['notify_lead_contact_more_times']) ? 1 : 0;
        $data['mark_public']                    = isset($data['mark_public']) ? 1 : 0;
        $data['responsible']                    = !isset($data['responsible']) ? 0 : $data['responsible'];

        if ($data['notify_lead_contact_more_times'] != 0 || $data['notify_lead_imported'] != 0) {
            if (isset($data['notify_type']) && $data['notify_type'] == 'specific_staff') {
                if (isset($data['notify_ids_staff'])) {
                    $data['notify_ids'] = serialize($data['notify_ids_staff']);
                    unset($data['notify_ids_staff']);
                } else {
                    $data['notify_ids'] = serialize([]);
                    unset($data['notify_ids_staff']);
                }
                if (isset($data['notify_ids_roles'])) {
                    unset($data['notify_ids_roles']);
                }
            } else {
                if (isset($data['notify_ids_roles'])) {
                    $data['notify_ids'] = serialize($data['notify_ids_roles']);
                    unset($data['notify_ids_roles']);
                } else {
                    $data['notify_ids'] = serialize([]);
                    unset($data['notify_ids_roles']);
                }
                if (isset($data['notify_ids_staff'])) {
                    unset($data['notify_ids_staff']);
                }
            }
        } else {
            $data['notify_ids']  = serialize([]);
            $data['notify_type'] = null;
            if (isset($data['notify_ids_staff'])) {
                unset($data['notify_ids_staff']);
            }
            if (isset($data['notify_ids_roles'])) {
                unset($data['notify_ids_roles']);
            }
        }

        // Check if not empty $data['password']
        // Get original
        // Decrypt original
        // Compare with $data['password']
        // If equal unset
        // If not encrypt and save
        if (!empty($data['password'])) {
            $or_decrypted = $this->encryption->decrypt($original_settings->password);
            if ($or_decrypted == $data['password']) {
                unset($data['password']);
            } else {
                $data['password'] = $this->encryption->encrypt($data['password']);
            }
        }

        $this->db->where('id', 1);
        $this->db->update(db_prefix() . 'leads_email_integration', $data);
        if ($this->db->affected_rows() > 0) {
            return true;
        }

        return false;
    }

    public function change_status_color($data)
    {
        $this->db->where('id', $data['status_id']);
        $this->db->update(db_prefix() . 'leads_status', [
            'color' => $data['color'],
        ]);
    }

    public function update_status_order($data)
    {
        foreach ($data['order'] as $status) {
            $this->db->where('id', $status[0]);
            $this->db->update(db_prefix() . 'leads_status', [
                'statusorder' => $status[1],
            ]);
        }
    }

    public function get_form($where)
    {
        $this->db->where($where);

        return $this->db->get(db_prefix() . 'web_to_lead')->row();
    }

    public function add_form($data)
    {
        $data                       = $this->_do_lead_web_to_form_responsibles($data);
        $data['success_submit_msg'] = nl2br($data['success_submit_msg']);
        $data['form_key']           = app_generate_hash();

        $data['create_task_on_duplicate'] = (int) isset($data['create_task_on_duplicate']);
        $data['mark_public']              = (int) isset($data['mark_public']);

        if (isset($data['allow_duplicate'])) {
            $data['allow_duplicate']           = 1;
            $data['track_duplicate_field']     = '';
            $data['track_duplicate_field_and'] = '';
            $data['create_task_on_duplicate']  = 0;
        } else {
            $data['allow_duplicate'] = 0;
        }

        $data['dateadded'] = date('Y-m-d H:i:s');

        $this->db->insert(db_prefix() . 'web_to_lead', $data);
        $insert_id = $this->db->insert_id();
        if ($insert_id) {
            log_activity('New Web to Lead Form Added [' . $data['name'] . ']');

            return $insert_id;
        }

        return false;
    }

    public function update_form($id, $data)
    {
        $data                       = $this->_do_lead_web_to_form_responsibles($data);
        $data['success_submit_msg'] = nl2br($data['success_submit_msg']);

        $data['create_task_on_duplicate'] = (int) isset($data['create_task_on_duplicate']);
        $data['mark_public']              = (int) isset($data['mark_public']);

        if (isset($data['allow_duplicate'])) {
            $data['allow_duplicate']           = 1;
            $data['track_duplicate_field']     = '';
            $data['track_duplicate_field_and'] = '';
            $data['create_task_on_duplicate']  = 0;
        } else {
            $data['allow_duplicate'] = 0;
        }

        $this->db->where('id', $id);
        $this->db->update(db_prefix() . 'web_to_lead', $data);

        return ($this->db->affected_rows() > 0 ? true : false);
    }

    public function delete_form($id)
    {
        $this->db->where('id', $id);
        $this->db->delete(db_prefix() . 'web_to_lead');

        $this->db->where('from_form_id', $id);
        $this->db->update(db_prefix() . 'leads', [
            'from_form_id' => 0,
        ]);

        if ($this->db->affected_rows() > 0) {
            log_activity('Lead Form Deleted [' . $id . ']');

            return true;
        }

        return false;
    }

    private function _do_lead_web_to_form_responsibles($data)
    {
        if (isset($data['notify_lead_imported'])) {
            $data['notify_lead_imported'] = 1;
        } else {
            $data['notify_lead_imported'] = 0;
        }

        if ($data['responsible'] == '') {
            $data['responsible'] = 0;
        }
        if ($data['notify_lead_imported'] != 0) {
            if ($data['notify_type'] == 'specific_staff') {
                if (isset($data['notify_ids_staff'])) {
                    $data['notify_ids'] = serialize($data['notify_ids_staff']);
                    unset($data['notify_ids_staff']);
                } else {
                    $data['notify_ids'] = serialize([]);
                    unset($data['notify_ids_staff']);
                }
                if (isset($data['notify_ids_roles'])) {
                    unset($data['notify_ids_roles']);
                }
            } else {
                if (isset($data['notify_ids_roles'])) {
                    $data['notify_ids'] = serialize($data['notify_ids_roles']);
                    unset($data['notify_ids_roles']);
                } else {
                    $data['notify_ids'] = serialize([]);
                    unset($data['notify_ids_roles']);
                }
                if (isset($data['notify_ids_staff'])) {
                    unset($data['notify_ids_staff']);
                }
            }
        } else {
            $data['notify_ids']  = serialize([]);
            $data['notify_type'] = null;
            if (isset($data['notify_ids_staff'])) {
                unset($data['notify_ids_staff']);
            }
            if (isset($data['notify_ids_roles'])) {
                unset($data['notify_ids_roles']);
            }
        }

        return $data;    }

    /**
     * Do kanban query
     * @param  mixed  $status
     * @param  string $search
     * @param  integer $page
     * @param  array  $sort
     * @param  boolean $count
     * @return mixed
     */
    public function do_kanban_query($status, $search = '', $page = 1, $sort = [], $count = false)
    {
        _deprecated_function('Leads_model::do_kanban_query', '2.9.2', 'LeadsKanban class');

        $kanBan = (new LeadsKanban($status))
            ->search($search)
            ->page($page)
            ->sortBy($sort['sort'] ?? null, $sort['sort_by'] ?? null);

        if ($count) {
            return $kanBan->countAll();
        }

        return $kanBan->get();
    }

    /**
     * Get note with attachment
     * @param  mixed $id  note id
     * @param  mixed $rel_id  lead id
     * @return array
     */
    public function get_lead_note_with_attachment($id = '', $rel_id = '')
    {
        $this->db->select('notes.*, files.id as file_id, files.file_name, files.filetype');
        $this->db->from(db_prefix() . 'notes as notes');
        $this->db->join(db_prefix() . 'files as files', 'files.rel_id = notes.id AND files.rel_type = "lead_note"', 'left');
        
        if (is_numeric($id)) {
            $this->db->where('notes.id', $id);
        }
        
        if (is_numeric($rel_id)) {
            $this->db->where('notes.rel_id', $rel_id);
            $this->db->where('notes.rel_type', 'lead');
        }
        
        $this->db->order_by('notes.dateadded', 'desc');
        
        return $this->db->get()->result_array();
    }
    
    /**
     * Add attachment to note
     * @param array $data note and attachment data
     * @return boolean
     */
    public function add_note_attachments($data) {
        if (isset($_FILES['note_attachment']) && _perfex_upload_error($_FILES['note_attachment']['error'])) {
            return false;
        }

        if (isset($_FILES['note_attachment'])) {
            // Create directory if it doesn't exist
            $path = get_upload_path_by_type('lead_note') . $data['rel_id'] . '/';
            if (!is_dir($path)) {
                mkdir($path, 0755, true);
            }
            
            $config = [
                'upload_path' => $path,
                'allowed_types' => '*',
                'max_size' => get_option('file_max_size'),
                'file_name' => unique_filename($path, $_FILES['note_attachment']['name'])
            ];
            
            $this->load->library('upload', $config);
            
            if ($this->upload->do_upload('note_attachment')) {
                $upload_data = $this->upload->data();
                
                $this->db->insert(db_prefix() . 'files', [
                    'rel_id' => $data['id'],
                    'rel_type' => 'lead_note',
                    'file_name' => $upload_data['file_name'],
                    'filetype' => $upload_data['file_type'],
                    'path' => $path,
                    'created_at' => date('Y-m-d H:i:s')
                ]);

                $insert_id = $this->db->insert_id();
                
                if ($insert_id) {
                    $this->db->where('id', $data['id']);
                    $this->db->update(db_prefix() . 'notes', ['has_attachment' => 1]);
                    
                    // Log the attachment
                    app_log("Note attachment added", 'info', [
                        'note_id' => $data['id'],
                        'lead_id' => $data['rel_id'],
                        'file_name' => $upload_data['file_name']
                    ]);
                    
                    return true;
                }
            } else {
                app_log("Failed to upload note attachment: " . $this->upload->display_errors(), 'error', [
                    'note_id' => $data['id'],
                    'lead_id' => $data['rel_id']
                ]);
            }
        }
        
        return false;
    }
    
    /**
     * Delete note attachment
     * @param  mixed $note_id  note id
     * @return boolean
     */
    public function delete_note_attachment($note_id) {
        $this->db->where('rel_id', $note_id);
        $this->db->where('rel_type', 'note');
        $attachment = $this->db->get(db_prefix() . 'files')->row();
        
        if ($attachment) {
            if (empty($attachment->external)) {
                unlink(get_upload_path_by_type('note') . $attachment->rel_id . '/' . $attachment->file_name);
            }
            
            $this->db->where('id', $attachment->id);
            $this->db->delete(db_prefix() . 'files');
            
            $this->db->where('id', $note_id);
            $this->db->update(db_prefix() . 'notes', ['has_attachment' => 0]);
            
            return true;
        }
        
        return false;
    }

    /**
     * Get lead notes from lead_notes table with next_followup_date support
     * @param string $id note id
     * @param string $rel_id lead id
     * @return array
     */
    public function get_lead_notes_with_followup($id = '', $rel_id = '')
    {
        $this->db->select('notes.*, files.id as file_id, files.file_name, files.filetype');
        $this->db->from(db_prefix() . 'lead_notes as notes');
        $this->db->join(db_prefix() . 'files as files', 'files.rel_id = notes.id AND files.rel_type = "lead_note"', 'left');
        
        if (is_numeric($id)) {
            $this->db->where('notes.id', $id);
        }
        
        if (is_numeric($rel_id)) {
            $this->db->where('notes.leadid', $rel_id);
        }
        
        $this->db->order_by('notes.dateadded', 'desc');
        
        return $this->db->get()->result_array();
    }
}
