<?php

defined('BASEPATH') or exit('No direct script access allowed');

/**
 * AI Analysis Service
 * 
 * Service layer wrapper for AI Analysis module
 * Provides simplified interface for common AI analysis tasks
 * 
 * @package    Application
 * @subpackage Services
 * @category   AI Analysis
 * @author     Ibraya Group
 * @since      Version 1.0.0
 * 
 * Usage Examples:
 * 
 * // Initialize service
 * $this->load->service('AI_Analysis_Service');
 * 
 * // Quick lead analysis
 * $result = $this->ai_analysis_service->quick_lead_analysis($lead_id);
 * 
 * // Custom analysis with files
 * $result = $this->ai_analysis_service->custom_analysis($prompt, $data, $files);
 * 
 * // Batch analysis
 * $results = $this->ai_analysis_service->batch_analysis($items, $prompt_template);
 */
class AI_Analysis_Service
{
    protected $CI;
    protected $ai_analysis;
    
    /**
     * Constructor
     */
    public function __construct()
    {
        $this->CI =& get_instance();
        $this->CI->load->library('AI_Analysis');
        $this->ai_analysis = $this->CI->ai_analysis;
        
        log_message('info', 'AI_Analysis_Service initialized');
    }
    
    /**
     * Quick lead analysis with all lead data
     * 
     * @param  int   $lead_id Lead ID
     * @param  array $options Optional API options
     * @return array Analysis result with formatted data
     */
    public function quick_lead_analysis($lead_id, $options = [])
    {
        try {
            // Load lead data
            $this->CI->load->model('leads_model');
            $lead = $this->CI->leads_model->get($lead_id);
            
            if (!$lead) {
                return [
                    'success' => false,
                    'error' => 'Lead not found'
                ];
            }
            
            // Get follow-up notes
            $notes = $this->CI->leads_model->get_notes($lead_id);
            
            // Get activity log
            $activity_log = $this->CI->leads_model->get_lead_activity_log($lead_id);
            
            // Prepare lead data array
            $lead_data = $this->_prepare_lead_data($lead, $notes, $activity_log);
            
            // Check for uploaded files
            $file_paths = $this->_get_lead_files($lead_id);
            
            // Perform analysis
            if (!empty($file_paths)) {
                $result = $this->ai_analysis->analyze_lead($lead_data, $file_paths, $options);
            } else {
                $result = $this->ai_analysis->analyze_lead($lead_data, [], $options);
            }
            
            // Save result if successful
            if ($result['success']) {
                $this->_save_lead_analysis_result($lead_id, $result);
            }
            
            return $result;
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => 'Lead analysis failed: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Analyze customer data
     * 
     * @param  int   $customer_id Customer ID
     * @param  array $options     Optional API options
     * @return array Analysis result
     */
    public function quick_customer_analysis($customer_id, $options = [])
    {
        try {
            // Load customer data
            $this->CI->load->model('clients_model');
            $customer = $this->CI->clients_model->get($customer_id);
            
            if (!$customer) {
                return [
                    'success' => false,
                    'error' => 'Customer not found'
                ];
            }
            
            // Prepare customer data
            $customer_data = $this->_prepare_customer_data($customer);
            
            // Perform analysis
            $result = $this->ai_analysis->analyze_customer($customer_data, [], $options);
            
            return $result;
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => 'Customer analysis failed: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Analyze project data
     * 
     * @param  int   $project_id Project ID
     * @param  array $options    Optional API options
     * @return array Analysis result
     */
    public function quick_project_analysis($project_id, $options = [])
    {
        try {
            // Load project data
            $this->CI->load->model('projects_model');
            $project = $this->CI->projects_model->get($project_id);
            
            if (!$project) {
                return [
                    'success' => false,
                    'error' => 'Project not found'
                ];
            }
            
            // Prepare project data
            $project_data = $this->_prepare_project_data($project);
            
            // Perform analysis
            $result = $this->ai_analysis->analyze_project($project_data, [], $options);
            
            return $result;
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => 'Project analysis failed: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Custom analysis with user-provided data
     * 
     * @param  string $prompt     Analysis prompt
     * @param  array  $data       Data for analysis
     * @param  array  $file_paths Optional file paths
     * @param  array  $options    Optional API options
     * @return array  Analysis result
     */
    public function custom_analysis($prompt, $data = [], $file_paths = [], $options = [])
    {
        try {
            if (!empty($file_paths)) {
                return $this->ai_analysis->analyze_with_files($prompt, $data, $file_paths, $options);
            } else {
                return $this->ai_analysis->analyze_text($prompt, $data, $options);
            }
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => 'Custom analysis failed: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Batch analysis of multiple items
     * 
     * @param  array  $items          Array of items to analyze
     * @param  string $prompt_template Prompt template with placeholders
     * @param  array  $options        Optional API options
     * @return array  Batch analysis results
     */
    public function batch_analysis($items, $prompt_template, $options = [])
    {
        $results = [];
        $successful = 0;
        $failed = 0;
        
        foreach ($items as $index => $item) {
            try {
                $result = $this->ai_analysis->analyze_text($prompt_template, $item, $options);
                
                $results[] = [
                    'index' => $index,
                    'item_id' => $item['id'] ?? $index,
                    'result' => $result
                ];
                
                if ($result['success']) {
                    $successful++;
                } else {
                    $failed++;
                }
                
                // Add small delay to avoid rate limiting
                usleep(100000); // 0.1 second delay
                
            } catch (Exception $e) {
                $results[] = [
                    'index' => $index,
                    'item_id' => $item['id'] ?? $index,
                    'result' => [
                        'success' => false,
                        'error' => $e->getMessage()
                    ]
                ];
                $failed++;
            }
        }
        
        return [
            'success' => true,
            'total_items' => count($items),
            'successful' => $successful,
            'failed' => $failed,
            'results' => $results
        ];
    }
    
    /**
     * Get analysis statistics
     * 
     * @param  array $filters Optional filters
     * @return array Statistics
     */
    public function get_statistics($filters = [])
    {
        return $this->ai_analysis->get_api_statistics($filters);
    }
    
    /**
     * Get recent analysis logs
     * 
     * @param  int   $limit   Number of logs to retrieve
     * @param  array $filters Optional filters
     * @return array Recent logs
     */
    public function get_recent_logs($limit = 50, $filters = [])
    {
        return $this->ai_analysis->get_recent_logs($limit, $filters);
    }
    
    /**
     * Validate files for analysis
     * 
     * @param  array $file_paths File paths to validate
     * @return array Validation result
     */
    public function validate_files($file_paths)
    {
        return $this->ai_analysis->validate_files($file_paths);
    }
    
    /**
     * Parse analysis response to structured format
     * 
     * @param  string $response Raw AI response
     * @return array  Parsed response
     */
    public function parse_response($response)
    {
        return $this->ai_analysis->parse_ai_response($response);
    }
    
    /**
     * Clean up old logs
     * 
     * @param  int $days_old Days to keep logs
     * @return int Number of cleaned logs
     */
    public function cleanup_logs($days_old = 30)
    {
        return $this->ai_analysis->cleanup_old_logs($days_old);
    }
    
    /**
     * Test API connection
     * 
     * @return array Test result
     */
    public function test_api_connection()
    {
        $test_prompt = "Respond with 'API connection successful' if you receive this message.";
        
        $result = $this->ai_analysis->analyze_text($test_prompt, [], [
            'max_tokens' => 50,
            'temperature' => 0
        ]);
        
        return [
            'success' => $result['success'],
            'message' => $result['success'] ? 'API connection successful' : $result['error'],
            'response_time' => $result['response_time'] ?? null
        ];
    }
    
    // ===========================================
    // PRIVATE HELPER METHODS
    // ===========================================
    
    /**
     * Prepare lead data for analysis
     * 
     * @param  object $lead         Lead object
     * @param  array  $notes        Lead notes
     * @param  array  $activity_log Activity log
     * @return array  Prepared data
     */
    private function _prepare_lead_data($lead, $notes = [], $activity_log = [])
    {
        return [
            'id' => $lead->id,
            'name' => $lead->name ?? '',
            'email' => $lead->email ?? '',
            'phonenumber' => $lead->phonenumber ?? '',
            'company' => $lead->company ?? '',
            'source' => $lead->source_name ?? $lead->source ?? '',
            'lead_value' => $lead->lead_value ?? '',
            'description' => $lead->description ?? '',
            'dateadded' => $lead->dateadded ?? '',
            'last_contact' => $lead->last_contact ?? '',
            'status' => $lead->status ?? '',
            'notes' => $notes,
            'activity_log' => $activity_log
        ];
    }
    
    /**
     * Prepare customer data for analysis
     * 
     * @param  object $customer Customer object
     * @return array  Prepared data
     */
    private function _prepare_customer_data($customer)
    {
        return [
            'id' => $customer->userid,
            'company' => $customer->company ?? '',
            'firstname' => $customer->firstname ?? '',
            'lastname' => $customer->lastname ?? '',
            'email' => $customer->email ?? '',
            'phonenumber' => $customer->phonenumber ?? '',
            'address' => trim(($customer->address ?? '') . ' ' . ($customer->city ?? '') . ' ' . ($customer->state ?? '') . ' ' . ($customer->country ?? '')),
            'datecreated' => $customer->datecreated ?? '',
            'active' => $customer->active ?? ''
        ];
    }
    
    /**
     * Prepare project data for analysis
     * 
     * @param  object $project Project object
     * @return array  Prepared data
     */
    private function _prepare_project_data($project)
    {
        return [
            'id' => $project->id,
            'name' => $project->name ?? '',
            'description' => $project->description ?? '',
            'status' => $project->status ?? '',
            'start_date' => $project->start_date ?? '',
            'deadline' => $project->deadline ?? '',
            'project_cost' => $project->project_cost ?? '',
            'progress' => $project->progress ?? 0,
            'client_id' => $project->clientid ?? ''
        ];
    }
    
    /**
     * Get uploaded files for a lead
     * 
     * @param  int   $lead_id Lead ID
     * @return array File paths
     */
    private function _get_lead_files($lead_id)
    {
        $files = [];
        $upload_path = FCPATH . 'uploads/ai_analysis/lead_' . $lead_id . '/';
        
        if (is_dir($upload_path)) {
            $file_list = scandir($upload_path);
            foreach ($file_list as $file) {
                if ($file !== '.' && $file !== '..' && is_file($upload_path . $file)) {
                    $files[] = $upload_path . $file;
                }
            }
        }
        
        return $files;
    }
    
    /**
     * Save lead analysis result to database
     * 
     * @param  int   $lead_id Lead ID
     * @param  array $result  Analysis result
     * @return bool  Success status
     */
    private function _save_lead_analysis_result($lead_id, $result)
    {
        try {
            if (!$result['success']) {
                return false;
            }
            
            // Parse the response
            $parsed = $this->ai_analysis->parse_ai_response($result['response']);
            
            if ($parsed['type'] === 'json' && isset($parsed['data'])) {
                $data = $parsed['data'];
                
                $score = (int)($data['confidence_score'] ?? $data['score'] ?? 50);
                $verdict = $data['verdict'] ?? 'Cold';
                $insights = $data['key_insights'] ?? $data['rationale'] ?? 'Analysis completed';
                $recommendations = $data['recommended_actions'] ?? $data['recommendations'] ?? '';
                
                // Use the Leads_ai_model to save the report
                $this->CI->load->model('Leads_ai_model');
                
                $uploaded_files = isset($result['parsed_files']) ? array_keys($result['parsed_files']) : [];
                
                $extra_data = [
                    'communication_analysis' => $data['communication_analysis'] ?? '',
                    'urgency_level' => $data['urgency_level'] ?? 'Medium',
                    'risk_factors' => $data['risk_factors'] ?? [],
                    'positive_signals' => $data['positive_signals'] ?? []
                ];
                
                return $this->CI->leads_ai_model->save_report(
                    $lead_id,
                    $score,
                    $verdict,
                    $insights,
                    $recommendations,
                    $uploaded_files,
                    $extra_data
                );
            }
            
            return false;
            
        } catch (Exception $e) {
            log_message('error', 'Failed to save lead analysis result: ' . $e->getMessage());
            return false;
        }
    }
}
