<?php

defined('BASEPATH') or exit('No direct script access allowed');

/**
 * AI Lead Analysis Helper Functions
 * 
 * Helper functions for AI-driven lead evaluation
 * 
 * @package    Application
 * @subpackage Helpers
 * @category   Lead Management
 * @author     Ibraya Group
 * @since      Version 1.0.0
 */

if (!function_exists('get_upload_path_by_type')) {
    /**
     * Get upload path by type
     * Enhanced to support AI lead analysis uploads
     * 
     * @param string $type Upload type
     * @return string Upload path
     */
    function get_upload_path_by_type($type)
    {
        $upload_path = FCPATH . 'uploads/';
        
        switch ($type) {
            case 'lead_ai':
                $upload_path .= 'lead_ai/';
                break;
            case 'lead':
                $upload_path .= 'leads/';
                break;
            default:
                $upload_path .= $type . '/';
                break;
        }
        
        // Ensure directory exists
        if (!is_dir($upload_path)) {
            mkdir($upload_path, 0755, true);
        }
        
        return $upload_path;
    }
}

if (!function_exists('ai_analysis_score_badge_class')) {
    /**
     * Get Bootstrap badge class based on AI analysis score
     * 
     * @param int $score Analysis score (0-100)
     * @return string Bootstrap badge class
     */
    function ai_analysis_score_badge_class($score)
    {
        if ($score >= 70) {
            return 'label-success';
        } elseif ($score >= 50) {
            return 'label-warning';
        } else {
            return 'label-danger';
        }
    }
}

if (!function_exists('ai_analysis_verdict_icon')) {
    /**
     * Get icon class for AI analysis verdict
     * 
     * @param string $verdict Analysis verdict (Likely/Unlikely)
     * @return string Icon class
     */
    function ai_analysis_verdict_icon($verdict)
    {
        return $verdict === 'Likely' ? 'fa-thumbs-up text-success' : 'fa-thumbs-down text-danger';
    }
}

if (!function_exists('format_ai_analysis_date')) {
    /**
     * Format AI analysis date for display
     * 
     * @param string $date Date string
     * @return string Formatted date
     */
    function format_ai_analysis_date($date)
    {
        return date('M j, Y \a\t H:i', strtotime($date));
    }
}

if (!function_exists('can_perform_ai_analysis')) {
    /**
     * Check if current staff member can perform AI analysis
     * 
     * @return bool True if allowed, false otherwise
     */
    function can_perform_ai_analysis()
    {
        // Only admin and staff with leads create permission can perform AI analysis
        return is_admin() || has_permission('leads', '', 'create');
    }
}

if (!function_exists('get_ai_analysis_upload_limit')) {
    /**
     * Get upload limits for AI analysis
     * 
     * @return array Upload limits configuration
     */
    function get_ai_analysis_upload_limit()
    {
        return [
            'max_files' => 10,
            'max_size_mb' => 5,
            'allowed_types' => ['jpg', 'jpeg', 'png'],
            'allowed_types_str' => 'JPG, JPEG, PNG'
        ];
    }
}
