<?php defined('BASEPATH') or exit('No direct script access allowed'); ?>
<?php init_head(); ?>
<div id="wrapper">
    <div class="content">
        <div class="row">
            <div class="col-md-12">
                <div class="panel panel-primary">
                    <div class="panel-heading">
                        <h3 class="panel-title">
                            <i class="fa fa-bell"></i> Reminder System - WORKING IMPLEMENTATION
                        </h3>
                    </div>
                    <div class="panel-body">
                        <div class="alert alert-success">
                            <h4><i class="fa fa-check-circle"></i> Complete Working Reminder System</h4>
                            <p>This implementation includes all reminder functions and will work regardless of the main.js compilation status.</p>
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="panel panel-success">
                                    <div class="panel-heading">
                                        <h4><i class="fa fa-play"></i> System Controls</h4>
                                    </div>
                                    <div class="panel-body">
                                        <button class="btn btn-success btn-block" onclick="startReminderSystem()">
                                            <i class="fa fa-play"></i> Start Reminder System
                                        </button>
                                        <button class="btn btn-warning btn-block" onclick="stopReminderSystem()">
                                            <i class="fa fa-stop"></i> Stop Reminder System
                                        </button>
                                        <button class="btn btn-info btn-block" onclick="checkRemindersManually()">
                                            <i class="fa fa-refresh"></i> Check Reminders Now
                                        </button>
                                        <button class="btn btn-primary btn-block" onclick="createTestReminder()">
                                            <i class="fa fa-plus"></i> Create Test Reminder (2 min)
                                        </button>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="panel panel-info">
                                    <div class="panel-heading">
                                        <h4><i class="fa fa-info"></i> System Status</h4>
                                    </div>
                                    <div class="panel-body">
                                        <div id="system-status">
                                            <p><i class="fa fa-spinner fa-spin"></i> Initializing...</p>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="panel panel-default">
                            <div class="panel-heading">
                                <h4><i class="fa fa-list"></i> Activity Log</h4>
                            </div>
                            <div class="panel-body">
                                <div id="activity-log" style="height: 300px; overflow-y: auto; background: #f8f9fa; border: 1px solid #dee2e6; padding: 15px; font-family: monospace;"></div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Reminder Modal (standalone) -->
<div class="modal fade" id="standalone-reminder-modal" tabindex="-1" role="dialog">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <div class="modal-header bg-warning">
                <button type="button" class="close" data-dismiss="modal">
                    <span aria-hidden="true">&times;</span>
                </button>
                <h4 class="modal-title">
                    <i class="fa fa-bell text-warning"></i> Reminder Notification
                </h4>
            </div>
            <div class="modal-body">
                <div id="standalone-reminder-content"></div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-default" data-dismiss="modal">
                    <i class="fa fa-clock-o"></i> Snooze 5 min
                </button>
                <button type="button" class="btn btn-success" onclick="markCurrentReminderComplete()">
                    <i class="fa fa-check"></i> Mark as Complete
                </button>
            </div>
        </div>
    </div>
</div>

<script>
// Standalone Reminder System - Complete Implementation
var standaloneReminderSystem = {
    checkInterval: null,
    checkFrequency: 60000, // 60 seconds
    currentReminders: [],
    lastCheck: 0,
    isRunning: false,
    
    log: function(message, type = 'info') {
        var logDiv = document.getElementById('activity-log');
        var timestamp = new Date().toLocaleTimeString();
        var prefix = type === 'success' ? '✅' : type === 'error' ? '❌' : type === 'warning' ? '⚠️' : '📋';
        logDiv.innerHTML += '[' + timestamp + '] ' + prefix + ' ' + message + '\n';
        logDiv.scrollTop = logDiv.scrollHeight;
    },
    
    updateStatus: function() {
        var statusDiv = document.getElementById('system-status');
        var html = '<ul class="list-unstyled">';
        
        if (this.isRunning) {
            html += '<li><i class="fa fa-play text-success"></i> <strong>Status:</strong> Running</li>';
        } else {
            html += '<li><i class="fa fa-pause text-warning"></i> <strong>Status:</strong> Stopped</li>';
        }
        
        html += '<li><i class="fa fa-clock-o text-info"></i> <strong>Check Frequency:</strong> ' + (this.checkFrequency / 1000) + ' seconds</li>';
        html += '<li><i class="fa fa-user text-success"></i> <strong>User ID:</strong> ' + (window.app ? window.app.user_id : 'N/A') + '</li>';
        html += '<li><i class="fa fa-bell text-info"></i> <strong>Current Reminders:</strong> ' + this.currentReminders.length + '</li>';
        
        if (this.lastCheck > 0) {
            var lastCheckTime = new Date(this.lastCheck).toLocaleTimeString();
            html += '<li><i class="fa fa-refresh text-muted"></i> <strong>Last Check:</strong> ' + lastCheckTime + '</li>';
        }
        
        html += '</ul>';
        statusDiv.innerHTML = html;
    },
    
    start: function() {
        if (this.isRunning) {
            this.log('System is already running', 'warning');
            return;
        }
        
        if (!window.app || !window.app.user_id || window.app.user_id === "" || window.app.user_id === "0") {
            this.log('Cannot start - user not logged in or missing user ID', 'error');
            return;
        }
        
        this.log('Starting reminder system for user ID: ' + window.app.user_id, 'success');
        this.isRunning = true;
        
        // Start checking interval
        this.checkInterval = setInterval(() => {
            this.checkForReminders();
        }, this.checkFrequency);
        
        // Do initial check
        this.checkForReminders();
        this.updateStatus();
    },
    
    stop: function() {
        if (!this.isRunning) {
            this.log('System is already stopped', 'warning');
            return;
        }
        
        this.log('Stopping reminder system', 'info');
        this.isRunning = false;
        
        if (this.checkInterval) {
            clearInterval(this.checkInterval);
            this.checkInterval = null;
        }
        
        this.updateStatus();
    },
    
    checkForReminders: function() {
        // Throttle checks
        var now = Date.now();
        if (now - this.lastCheck < 30000) { // Minimum 30 seconds
            return;
        }
        this.lastCheck = now;
        
        this.log('Checking for due reminders...', 'info');
        
        $.ajax({
            url: admin_url + 'misc/check_due_reminders',
            type: 'GET',
            dataType: 'json',
            success: (response) => {
                this.log('API response: ' + response.count + ' reminders found', 'info');
                
                if (response.success && response.reminders && response.reminders.length > 0) {
                    // Filter new reminders
                    var newReminders = response.reminders.filter(reminder => {
                        return !this.currentReminders.some(current => current.id === reminder.id);
                    });
                    
                    if (newReminders.length > 0) {
                        this.log('Displaying ' + newReminders.length + ' new reminder(s)', 'success');
                        this.currentReminders = response.reminders;
                        this.showNotification(newReminders);
                    }
                }
                
                this.updateStatus();
            },
            error: (xhr, status, error) => {
                this.log('Error checking reminders: ' + error, 'error');
            }
        });
    },
    
    showNotification: function(reminders) {
        if (!reminders || reminders.length === 0) return;
        
        var reminder = reminders[0]; // Show first reminder
        
        // Build content
        var html = '<div class="alert alert-warning">';
        html += '<h4><strong>📋 Reminder Due Now!</strong></h4>';
        html += '<div style="margin-top: 15px;">';
        html += '<p><strong>Description:</strong></p>';
        html += '<div style="background: #f8f9fa; padding: 10px; border-radius: 4px; margin-bottom: 10px;">' + this.escapeHtml(reminder.description) + '</div>';
        html += '<p><strong>Due Date:</strong> ' + reminder.formatted_date + '</p>';
        html += '<p><strong>Time:</strong> <span class="text-danger">' + reminder.time_until + '</span></p>';
        
        if (reminder.rel_type && reminder.rel_id) {
            html += '<p><strong>Related to:</strong> ' + this.capitalizeFirst(reminder.rel_type) + ' #' + reminder.rel_id + '</p>';
        }
        
        html += '</div>';
        html += '</div>';
        
        // Update modal and show
        document.getElementById('standalone-reminder-content').innerHTML = html;
        $('#standalone-reminder-modal').data('current-reminder', reminder);
        $('#standalone-reminder-modal').modal({
            backdrop: 'static',
            keyboard: false
        });
        
        // Play sound notification
        this.playNotificationSound();
        
        this.log('Showing notification for: ' + reminder.description.substring(0, 50) + '...', 'success');
    },
    
    markComplete: function(reminderId) {
        this.log('Marking reminder ' + reminderId + ' as complete...', 'info');
        
        $.ajax({
            url: admin_url + 'misc/mark_reminder_notified/' + reminderId,
            type: 'POST',
            success: (response) => {
                this.log('Reminder marked as complete', 'success');
                
                // Remove from current reminders
                this.currentReminders = this.currentReminders.filter(r => r.id != reminderId);
                this.updateStatus();
            },
            error: (xhr, status, error) => {
                this.log('Error marking reminder complete: ' + error, 'error');
            }
        });
    },
    
    createTestReminder: function() {
        var testDate = new Date();
        testDate.setMinutes(testDate.getMinutes() + 2);
        var formattedDate = testDate.getFullYear() + '-' + 
                           String(testDate.getMonth() + 1).padStart(2, '0') + '-' + 
                           String(testDate.getDate()).padStart(2, '0') + ' ' + 
                           String(testDate.getHours()).padStart(2, '0') + ':' + 
                           String(testDate.getMinutes()).padStart(2, '0') + ':00';
        
        this.log('Creating test reminder for ' + formattedDate + '...', 'info');
        
        $.ajax({
            url: admin_url + 'misc/add_reminder/1/test',
            type: 'POST',
            data: {
                description: '🧪 Test reminder created by standalone system - should appear in 2 minutes',
                date: formattedDate,
                staff: window.app.user_id,
                rel_type: 'standalone_test',
                rel_id: '1'
            },
            success: (response) => {
                this.log('Test reminder created successfully!', 'success');
                this.log('You should receive a notification in 2 minutes', 'info');
            },
            error: (xhr, status, error) => {
                this.log('Error creating test reminder: ' + error, 'error');
            }
        });
    },
    
    playNotificationSound: function() {
        try {
            // Create audio context and play notification sound
            var audioContext = new (window.AudioContext || window.webkitAudioContext)();
            var oscillator = audioContext.createOscillator();
            var gainNode = audioContext.createGain();
            
            oscillator.connect(gainNode);
            gainNode.connect(audioContext.destination);
            
            oscillator.frequency.setValueAtTime(800, audioContext.currentTime);
            oscillator.frequency.setValueAtTime(600, audioContext.currentTime + 0.1);
            oscillator.frequency.setValueAtTime(800, audioContext.currentTime + 0.2);
            
            gainNode.gain.setValueAtTime(0.3, audioContext.currentTime);
            gainNode.gain.exponentialRampToValueAtTime(0.01, audioContext.currentTime + 0.3);
            
            oscillator.start(audioContext.currentTime);
            oscillator.stop(audioContext.currentTime + 0.3);
            
            this.log('Notification sound played', 'info');
        } catch (error) {
            this.log('Could not play notification sound: ' + error.message, 'warning');
        }
    },
    
    escapeHtml: function(text) {
        var map = {
            '&': '&amp;',
            '<': '&lt;',
            '>': '&gt;',
            '"': '&quot;',
            "'": '&#039;'
        };
        return text.replace(/[&<>"']/g, function(m) { return map[m]; });
    },
    
    capitalizeFirst: function(str) {
        return str.charAt(0).toUpperCase() + str.slice(1);
    }
};

// Global functions for button clicks
function startReminderSystem() {
    standaloneReminderSystem.start();
}

function stopReminderSystem() {
    standaloneReminderSystem.stop();
}

function checkRemindersManually() {
    standaloneReminderSystem.checkForReminders();
}

function createTestReminder() {
    standaloneReminderSystem.createTestReminder();
}

function markCurrentReminderComplete() {
    var reminder = $('#standalone-reminder-modal').data('current-reminder');
    if (reminder) {
        standaloneReminderSystem.markComplete(reminder.id);
        $('#standalone-reminder-modal').modal('hide');
    }
}

// Initialize when page loads
$(document).ready(function() {
    standaloneReminderSystem.log('Standalone reminder system loaded', 'success');
    standaloneReminderSystem.log('User ID: ' + (window.app ? window.app.user_id : 'N/A'), 'info');
    standaloneReminderSystem.log('Admin URL: ' + (typeof admin_url !== 'undefined' ? admin_url : 'N/A'), 'info');
    standaloneReminderSystem.updateStatus();
    
    // Auto-start the system
    setTimeout(function() {
        standaloneReminderSystem.start();
    }, 2000);
});

// Cleanup when page unloads
$(window).on('beforeunload', function() {
    standaloneReminderSystem.stop();
});
</script>

<?php init_tail(); ?>
