<?php defined('BASEPATH') or exit('No direct script access allowed'); ?>
<?php init_head(); ?>
<div id="wrapper">
    <div class="content">
        <div class="row">
            <div class="col-md-12">
                <div class="panel panel-primary">
                    <div class="panel-heading">
                        <h3 class="panel-title">
                            <i class="fa fa-bell"></i> Reminder System Verification
                        </h3>
                    </div>
                    <div class="panel-body">
                        <div class="alert alert-info">
                            <strong>Reminder System Test Tool</strong><br>
                            This tool verifies that the reminder notification system is working correctly across all CRM pages.
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="panel panel-default">
                                    <div class="panel-heading">
                                        <h4>System Status</h4>
                                    </div>
                                    <div class="panel-body">
                                        <div id="system-status">
                                            <p>Loading system status...</p>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="panel panel-default">
                                    <div class="panel-heading">
                                        <h4>Test Actions</h4>
                                    </div>
                                    <div class="panel-body">
                                        <button class="btn btn-primary btn-block" onclick="runSystemTest()">
                                            <i class="fa fa-play"></i> Run System Test
                                        </button>
                                        <button class="btn btn-info btn-block" onclick="testReminderAPI()">
                                            <i class="fa fa-globe"></i> Test Reminder API
                                        </button>
                                        <button class="btn btn-warning btn-block" onclick="showTestNotification()">
                                            <i class="fa fa-bell"></i> Show Test Notification
                                        </button>
                                        <button class="btn btn-success btn-block" onclick="createTestReminder()">
                                            <i class="fa fa-plus"></i> Create Test Reminder
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="panel panel-default">
                            <div class="panel-heading">
                                <h4>Test Output</h4>
                            </div>
                            <div class="panel-body">
                                <div id="test-output" style="height: 300px; overflow-y: auto; background: #f8f9fa; border: 1px solid #dee2e6; padding: 15px; font-family: monospace; white-space: pre-wrap;"></div>
                            </div>
                        </div>

                        <div class="panel panel-default">
                            <div class="panel-heading">
                                <h4>App Object Debug</h4>
                            </div>
                            <div class="panel-body">
                                <pre id="app-debug" style="max-height: 200px; overflow: auto;"></pre>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
function log(message, type = 'info') {
    const output = document.getElementById('test-output');
    const timestamp = new Date().toLocaleTimeString();
    const prefix = type === 'success' ? '✅' : type === 'error' ? '❌' : type === 'warning' ? '⚠️' : '📋';
    output.textContent += `[${timestamp}] ${prefix} ${message}\n`;
    output.scrollTop = output.scrollHeight;
}

function updateSystemStatus() {
    const statusDiv = document.getElementById('system-status');
    let html = '';
    
    // Check window.app
    if (typeof window.app !== 'undefined') {
        html += '<div class="alert alert-success"><i class="fa fa-check"></i> window.app object available</div>';
        
        if (window.app.user_id && window.app.user_id !== "" && window.app.user_id !== "0") {
            html += '<div class="alert alert-success"><i class="fa fa-user"></i> User ID: ' + window.app.user_id + '</div>';
        } else {
            html += '<div class="alert alert-danger"><i class="fa fa-times"></i> User ID missing or invalid</div>';
        }
    } else {
        html += '<div class="alert alert-danger"><i class="fa fa-times"></i> window.app object not available</div>';
    }
    
    // Check admin_url
    if (typeof admin_url !== 'undefined') {
        html += '<div class="alert alert-success"><i class="fa fa-check"></i> admin_url available: ' + admin_url + '</div>';
    } else {
        html += '<div class="alert alert-danger"><i class="fa fa-times"></i> admin_url not defined</div>';
    }
    
    // Check reminder functions
    const functions = ['initReminderNotifications', 'checkForDueReminders', 'showReminderNotification'];
    let funcCount = 0;
    functions.forEach(func => {
        if (typeof window[func] === 'function') {
            funcCount++;
        }
    });
    
    if (funcCount === functions.length) {
        html += '<div class="alert alert-success"><i class="fa fa-check"></i> All reminder functions available</div>';
    } else {
        html += '<div class="alert alert-warning"><i class="fa fa-exclamation-triangle"></i> ' + funcCount + '/' + functions.length + ' reminder functions available</div>';
    }
    
    // Check if system is running
    if (typeof reminderCheckInterval !== 'undefined' && reminderCheckInterval !== null) {
        html += '<div class="alert alert-success"><i class="fa fa-play"></i> Reminder system actively running</div>';
    } else {
        html += '<div class="alert alert-warning"><i class="fa fa-pause"></i> Reminder system not actively running</div>';
    }
    
    statusDiv.innerHTML = html;
}

function runSystemTest() {
    document.getElementById('test-output').textContent = '';
    log('Starting comprehensive system test...', 'info');
    
    // Test 1: App object
    if (typeof window.app !== 'undefined') {
        log('✓ window.app object exists', 'success');
        
        if (window.app.user_id && window.app.user_id !== "" && window.app.user_id !== "0") {
            log('✓ User ID is valid: ' + window.app.user_id, 'success');
        } else {
            log('✗ User ID is missing or invalid: ' + JSON.stringify(window.app.user_id), 'error');
        }
    } else {
        log('✗ window.app object not available', 'error');
    }
    
    // Test 2: Admin URL
    if (typeof admin_url !== 'undefined') {
        log('✓ admin_url is available: ' + admin_url, 'success');
    } else {
        log('✗ admin_url is not defined', 'error');
    }
    
    // Test 3: Functions
    const functions = [
        'initReminderNotifications',
        'startReminderChecking',
        'checkForDueReminders',
        'showReminderNotification',
        'markRemindersAsComplete'
    ];
    
    functions.forEach(funcName => {
        if (typeof window[funcName] === 'function') {
            log('✓ ' + funcName + ' function available', 'success');
        } else {
            log('✗ ' + funcName + ' function missing', 'error');
        }
    });
    
    // Test 4: jQuery
    if (typeof $ !== 'undefined') {
        log('✓ jQuery is available', 'success');
    } else {
        log('✗ jQuery is not available', 'error');
    }
    
    // Test 5: System status
    if (typeof reminderCheckInterval !== 'undefined') {
        if (reminderCheckInterval !== null) {
            log('✓ Reminder system is actively checking', 'success');
        } else {
            log('⚠ Reminder system is not active', 'warning');
        }
    } else {
        log('✗ reminderCheckInterval variable not found', 'error');
    }
    
    log('System test completed!', 'success');
    updateSystemStatus();
    updateAppDebug();
}

function testReminderAPI() {
    if (typeof admin_url === 'undefined') {
        log('Cannot test API - admin_url not available', 'error');
        return;
    }
    
    log('Testing reminder API...', 'info');
    
    $.ajax({
        url: admin_url + 'misc/check_due_reminders',
        type: 'GET',
        dataType: 'json',
        success: function(data) {
            log('✓ API response received', 'success');
            log('Response: ' + JSON.stringify(data, null, 2), 'info');
            
            if (data.success) {
                log('✓ API success: true', 'success');
                log('Found ' + data.count + ' due reminders', 'info');
                
                if (data.reminders && data.reminders.length > 0) {
                    data.reminders.forEach((reminder, index) => {
                        log('  ' + (index + 1) + '. ' + reminder.description + ' (Due: ' + reminder.time_until + ')', 'info');
                    });
                }
            } else {
                log('⚠ API returned success: false', 'warning');
            }
        },
        error: function(xhr, status, error) {
            log('✗ API request failed: ' + error, 'error');
            log('Status: ' + status + ', Response: ' + xhr.responseText, 'error');
        }
    });
}

function showTestNotification() {
    if (typeof showReminderNotification !== 'function') {
        log('Cannot show test notification - showReminderNotification function not available', 'error');
        return;
    }
    
    log('Showing test notification...', 'info');
    
    const testReminder = {
        id: 999999,
        description: "🧪 Test reminder notification - This is a demonstration of the reminder notification system working correctly.",
        formatted_date: new Date().toLocaleString(),
        time_until: "Now (Test Mode)",
        rel_type: "system_test",
        rel_id: "999"
    };
    
    try {
        showReminderNotification([testReminder]);
        log('✓ Test notification should now be visible', 'success');
    } catch (error) {
        log('✗ Error showing test notification: ' + error.message, 'error');
    }
}

function createTestReminder() {
    if (typeof admin_url === 'undefined') {
        log('Cannot create test reminder - admin_url not available', 'error');
        return;
    }
    
    log('Creating test reminder for 2 minutes from now...', 'info');
    
    const testDate = new Date();
    testDate.setMinutes(testDate.getMinutes() + 2);
    const formattedDate = testDate.getFullYear() + '-' + 
                         String(testDate.getMonth() + 1).padStart(2, '0') + '-' + 
                         String(testDate.getDate()).padStart(2, '0') + ' ' + 
                         String(testDate.getHours()).padStart(2, '0') + ':' + 
                         String(testDate.getMinutes()).padStart(2, '0') + ':00';
    
    const reminderData = {
        description: 'Test reminder created by verification tool - should appear in 2 minutes',
        date: formattedDate,
        staff: window.app.user_id,
        rel_type: 'verification_test',
        rel_id: '1'
    };
    
    $.ajax({
        url: admin_url + 'misc/add_reminder/1/verification_test',
        type: 'POST',
        data: reminderData,
        dataType: 'json',
        success: function(data) {
            log('✓ Test reminder creation response received', 'success');
            log('Response: ' + JSON.stringify(data), 'info');
            
            if (data.alert_type === 'success') {
                log('✓ Test reminder created successfully for ' + formattedDate, 'success');
                log('You should receive a notification in 2 minutes', 'info');
            } else {
                log('⚠ Reminder creation may have failed: ' + data.message, 'warning');
            }
        },
        error: function(xhr, status, error) {
            log('✗ Failed to create test reminder: ' + error, 'error');
            log('Status: ' + status + ', Response: ' + xhr.responseText, 'error');
        }
    });
}

function updateAppDebug() {
    const debugDiv = document.getElementById('app-debug');
    if (typeof window.app !== 'undefined') {
        debugDiv.textContent = JSON.stringify(window.app, null, 2);
    } else {
        debugDiv.textContent = 'window.app object not available';
    }
}

// Initialize when page loads
$(document).ready(function() {
    log('Reminder verification tool loaded', 'info');
    log('Current page: ' + window.location.href, 'info');
    
    setTimeout(function() {
        updateSystemStatus();
        updateAppDebug();
    }, 1000);
    
    // Auto-refresh status every 10 seconds
    setInterval(updateSystemStatus, 10000);
});
</script>

<?php init_tail(); ?>
