<?php

defined('BASEPATH') or exit('No direct script access allowed');

class Reporting_dashboard_model extends App_Model
{
    public function __construct()
    {
        parent::__construct();
    }

    public function get_kpis(array $filters)
    {
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        
        $today = date('Y-m-d');
        $weekStart = date('Y-m-d', strtotime('monday this week'));
        $monthStart = date('Y-m-01');
        
        return [
            'today' => $this->count_lead_activities(['from' => $today, 'to' => $today, 'staff' => $filters['staff'] ?? []]),
            'week' => $this->count_lead_activities(['from' => $weekStart, 'to' => $today, 'staff' => $filters['staff'] ?? []]),
            'month' => $this->count_lead_activities(['from' => $monthStart, 'to' => $today, 'staff' => $filters['staff'] ?? []]),
            'logins_today' => $this->count_logins_today($filters['staff'] ?? []),
            'top_staff' => $this->get_top_staff_simple($filters),
            'dealer_stats' => $this->get_dealer_statistics($filters)
        ];
    }

    public function get_staff_activity_chart(array $filters)
    {
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        
        // Try to find the correct activity table
        $table = $this->get_activity_table_name();
        if (!$table) {
            return ['labels' => [], 'data' => []];
        }
        
        $this->db->select('s.staffid, CONCAT(s.firstname, " ", s.lastname) as staff_name, COUNT(a.id) as total');
        $this->db->from($table . ' a');
        $this->db->join(db_prefix() . 'staff s', 's.staffid = a.staffid', 'left');
        
        if ($from) $this->db->where('DATE(a.date) >=', $from);
        if ($to) $this->db->where('DATE(a.date) <=', $to);
        if (!empty($filters['staff'])) {
            $this->db->where_in('a.staffid', $filters['staff']);
        }
        
        $this->db->group_by('s.staffid');
        $this->db->order_by('total', 'DESC');
        $this->db->limit(10);
        
        $result = $this->db->get()->result_array();
        
        return [
            'labels' => array_column($result, 'staff_name'),
            'data' => array_column($result, 'total')
        ];
    }

    public function get_activity_over_time_chart(array $filters)
    {
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        
        $table = $this->get_activity_table_name();
        if (!$table) {
            return ['labels' => [], 'data' => []];
        }
        
        $this->db->select('DATE(date) as activity_date, COUNT(id) as total');
        $this->db->from($table);
        $this->db->where('DATE(date) >=', $from);
        $this->db->where('DATE(date) <=', $to);
        
        if (!empty($filters['staff'])) {
            $this->db->where_in('staffid', $filters['staff']);
        }
        
        $this->db->group_by('DATE(date)');
        $this->db->order_by('activity_date', 'ASC');
        
        $result = $this->db->get()->result_array();
        
        return [
            'labels' => array_column($result, 'activity_date'),
            'data' => array_column($result, 'total')
        ];
    }

    /**
     * Enhanced activity table with modern features
     */
    public function get_activity_table(array $filters)
    {
        $table = $this->get_activity_table_name();
        if (!$table) {
            error_log('No activity table found');
            return ['data' => [], 'recordsTotal' => 0, 'recordsFiltered' => 0];
        }

        error_log('Using activity table: ' . $table);

        // DataTables parameters - check POST first, then GET as fallback
        $draw = (int)($this->input->post('draw') ?: $this->input->get('draw'));
        $start = (int)($this->input->post('start') ?: $this->input->get('start'));
        $length = (int)($this->input->post('length') ?: $this->input->get('length'));
        
        // For client-side DataTables, return all data (up to reasonable limit)
        // Frontend will handle pagination
        if ($length <= 0) {
            $length = 1000; // Default to getting all data for client-side pagination
        } elseif ($length == -1) {
            $length = 1000; // Set a reasonable maximum
        }
        
        error_log("Activity table pagination - Start: $start, Length: $length, Draw: $draw");

        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');

        error_log('Activity table query - From: ' . $from . ', To: ' . $to);

        // Enhanced followup type extraction from actual notes table
        $this->db->select("a.*, s.firstname, s.lastname, l.name as lead_name, l.company as lead_company, l.phonenumber as lead_phone, l.email as lead_email, COALESCE(ln.followup_type, 'General') as followup_type, COALESCE(ln.description, a.description) as original_description");
        $this->db->from($table . ' a');
        $this->db->join(db_prefix() . 'staff s', 's.staffid = a.staffid', 'left');
        $this->db->join(db_prefix() . 'leads l', 'l.id = a.leadid', 'left');
        $this->db->join(db_prefix() . 'lead_notes ln', 'ln.leadid = a.leadid AND ABS(TIMESTAMPDIFF(MINUTE, ln.dateadded, a.date)) <= 5', 'left');
        
        // Only show actual followup activities, not system notifications
        $this->db->where("(a.description = 'not_lead_activity_added_followup' OR a.description NOT LIKE 'not_%')");
        
        if ($from) $this->db->where('DATE(a.date) >=', $from);
        if ($to) $this->db->where('DATE(a.date) <=', $to);
        
        // Staff filtering
        if (!empty($filters['staff']) && is_array($filters['staff'])) {
            $this->db->where_in('a.staffid', $filters['staff']);
        }

        // Search functionality
        $search = $this->input->post('search');
        if (!empty($search['value'])) {
            $search_term = $search['value'];
            $this->db->group_start();
            $this->db->like('a.description', $search_term);
            $this->db->or_like('CONCAT(s.firstname, " ", s.lastname)', $search_term);
            $this->db->or_like('l.name', $search_term);
            $this->db->or_like('l.company', $search_term);
            $this->db->group_end();
        }

        // Get total count with a fresh query to avoid issues with JOINs
        $count_sql = "SELECT COUNT(DISTINCT a.id) as total_count FROM " . $table . " a ";
        $count_sql .= "LEFT JOIN " . db_prefix() . "staff s ON s.staffid = a.staffid ";
        $count_sql .= "LEFT JOIN " . db_prefix() . "leads l ON l.id = a.leadid ";
        $count_sql .= "LEFT JOIN " . db_prefix() . "lead_notes ln ON ln.leadid = a.leadid AND ABS(TIMESTAMPDIFF(MINUTE, ln.dateadded, a.date)) <= 5 ";
        $count_sql .= "WHERE (a.description = 'not_lead_activity_added_followup' OR a.description NOT LIKE 'not_%') ";
        
        $count_params = [];
        if ($from) {
            $count_sql .= "AND DATE(a.date) >= ? ";
            $count_params[] = $from;
        }
        if ($to) {
            $count_sql .= "AND DATE(a.date) <= ? ";
            $count_params[] = $to;
        }
        
        if (!empty($filters['staff']) && is_array($filters['staff'])) {
            $placeholders = str_repeat('?,', count($filters['staff']) - 1) . '?';
            $count_sql .= "AND a.staffid IN ($placeholders) ";
            $count_params = array_merge($count_params, $filters['staff']);
        }
        
        // Search filter for count
        $search = $this->input->post('search');
        if (!empty($search['value'])) {
            $search_term = '%' . $search['value'] . '%';
            $count_sql .= "AND (a.description LIKE ? OR CONCAT(s.firstname, ' ', s.lastname) LIKE ? OR l.name LIKE ? OR l.company LIKE ?) ";
            $count_params[] = $search_term;
            $count_params[] = $search_term;
            $count_params[] = $search_term;
            $count_params[] = $search_term;
        }
        
        try {
            $count_result = $this->db->query($count_sql, $count_params)->row();
            $total = $count_result ? (int)$count_result->total_count : 0;
            error_log('Activity table total count: ' . $total);
        } catch (Exception $e) {
            error_log('Count query error: ' . $e->getMessage());
            $total = 0;
        }
        
        $this->db->order_by('a.date', 'DESC');
        
        // For client-side DataTables, don't apply LIMIT - get all data
        // Apply LIMIT only if specifically requested for server-side pagination
        if ($start > 0 || ($length < 1000 && $length > 0)) {
            $this->db->limit($length, $start);
            error_log("Applying LIMIT: $length OFFSET: $start");
        } else {
            error_log("Getting all records for client-side pagination (no LIMIT applied)");
        }
        
        try {
            $query_result = $this->db->get();
            $rows = $query_result->result_array();
            error_log('Activity table query successful: ' . count($rows) . ' rows');
        } catch (Exception $e) {
            error_log('Activity table query error: ' . $e->getMessage());
            return ['data' => [], 'recordsTotal' => 0, 'recordsFiltered' => 0];
        }

        $data = [];
        foreach ($rows as $r) {
            $staff_name = trim(($r['firstname'] ?? '') . ' ' . ($r['lastname'] ?? '')) ?: 'System';
            
            $lead_id = (int)($r['leadid'] ?? 0);
            $lead_name = trim($r['lead_name'] ?? '');
            $lead_company = trim($r['lead_company'] ?? '');
            $lead_phone = trim($r['lead_phone'] ?? '');
            $lead_email = trim($r['lead_email'] ?? '');
            
            $lead_display = $this->build_lead_display($lead_name, $lead_company, $lead_phone, $lead_email);
            $followup_type = $r['followup_type'] ?? 'General';
            
            // Use original description from notes if available, otherwise use activity description
            $description = !empty($r['original_description']) ? $r['original_description'] : ($r['description'] ?? '');
            
            $data[] = [
                'id' => $r['id'],
                'date' => date('Y-m-d H:i', strtotime($r['date'] ?? 'now')),
                'staff' => $staff_name,
                'staff_id' => $r['staffid'] ?? 0,
                'lead' => $lead_display,
                'lead_id' => $lead_id,
                'lead_name' => $lead_name,
                'lead_company' => $lead_company,
                'lead_phone' => $lead_phone,
                'lead_email' => $lead_email,
                'followup_type' => $followup_type,
                'description' => $description,
                'priority' => $this->determine_priority($r)
            ];
        }

        return [
            'draw' => $draw,
            'recordsTotal' => $total,
            'recordsFiltered' => $total,
            'data' => $data
        ];
    }

    /**
     * Find the correct activity table name
     */
    private function get_activity_table_name()
    {
        $table_variations = ['lead_activity_log', 'activity_log', 'activities'];
        
        foreach ($table_variations as $variation) {
            if ($this->db->table_exists($variation)) {
                return db_prefix() . $variation;
            }
        }
        
        return null;
    }

    /**
     * Get SQL for followup type detection
     */
    private function get_followup_type_sql()
    {
        return "(
            CASE 
                WHEN LOWER(a.description) REGEXP '(whatsapp|whats.app|wa )' THEN 'WhatsApp'
                WHEN LOWER(a.description) REGEXP '(call|phone|dial|rang|calling)' THEN 'Phone Call'
                WHEN LOWER(a.description) REGEXP '(email|mail|sent.*email)' THEN 'Email'
                WHEN LOWER(a.description) REGEXP '(sms|text|message)' THEN 'SMS'
                WHEN LOWER(a.description) REGEXP '(meeting|appointment|visit)' THEN 'Meeting'
                WHEN LOWER(a.description) REGEXP '(reminder|follow.?up|followup)' THEN 'Reminder'
                WHEN LOWER(a.description) REGEXP '(status|updated|update)' THEN 'Status Update'
                WHEN LOWER(a.description) REGEXP '(proposal|quote|quotation)' THEN 'Proposal'
                WHEN LOWER(a.description) REGEXP '(note|comment|added.*note)' THEN 'Note'
                WHEN LOWER(a.description) REGEXP '(activity.*new|new.*activity|activity.*created|created)' THEN 'Activity Created'
                WHEN LOWER(a.description) REGEXP '(activity.*added|added.*activity)' THEN 'Activity Added'
                ELSE 'General'
            END
        )";
    }

    /**
     * Build lead display with contact info
     */
    private function build_lead_display($name, $company, $phone, $email)
    {
        $parts = [];
        if ($name) $parts[] = $name;
        if ($company) $parts[] = '(' . $company . ')';
        
        $main = !empty($parts) ? implode(' ', $parts) : 'Unknown Lead';
        
        $contacts = [];
        if ($phone) $contacts[] = $phone;
        // Show company in contact details instead of email
        if ($company && !in_array('(' . $company . ')', $parts)) {
            $contacts[] = $company;
        }
        
        if (!empty($contacts)) {
            $main .= ' • ' . implode(' • ', $contacts);
        }
        
        return $main;
    }

    /**
     * Determine activity priority
     */
    private function determine_priority($row)
    {
        $date = $row['date'] ?? '';
        $hours_ago = $date ? (time() - strtotime($date)) / 3600 : 999;
        
        if ($hours_ago <= 2) return 'high';
        if ($hours_ago <= 24) return 'medium';
        return 'normal';
    }

    /**
     * Helper functions for KPI calculation
     */
    private function count_lead_activities($filters) 
    {
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        $staff = $filters['staff'] ?? [];
        
        $table = $this->get_activity_table_name();
        if (!$table) return 0;
        
        $this->db->from($table);
        $this->db->where('DATE(date) >=', $from);
        $this->db->where('DATE(date) <=', $to);
        
        if (!empty($staff)) {
            $this->db->where_in('staffid', $staff);
        }
        
        return $this->db->count_all_results();
    }
    
    private function count_logins_today($staff) 
    {
        return 0; // Placeholder
    }
    
    private function get_top_staff_simple($filters) 
    {
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        
        $table = $this->get_activity_table_name();
        if (!$table) return [];
        
        $this->db->select('s.staffid, CONCAT(s.firstname, " ", s.lastname) as staff_name, COUNT(a.id) as total');
        $this->db->from($table . ' a');
        $this->db->join(db_prefix() . 'staff s', 's.staffid = a.staffid', 'left');
        $this->db->where('DATE(a.date) >=', $from);
        $this->db->where('DATE(a.date) <=', $to);
        $this->db->group_by('s.staffid');
        $this->db->order_by('total', 'DESC');
        $this->db->limit(5);
        
        return $this->db->get()->result_array();
    }

    // Method for email summary
    public function get_email_summary_data()
    {
        return [
            'kpis' => $this->get_kpis([])
        ];
    }

    /**
     * Return reminders grouped for dashboard
     */
    public function get_reminders(array $filters)
    {
        return ['todays' => [], 'due' => [], 'upcoming' => []];
    }

    /**
     * Get activity type pie chart data
     */
    public function get_activity_type_pie(array $filters)
    {
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        
        $table = $this->get_activity_table_name();
        if (!$table) {
            return ['labels' => [], 'data' => []];
        }
        
        $this->db->select("COALESCE(ln.followup_type, 'General') as followup_type, COUNT(a.id) as count");
        $this->db->from($table . ' a');
        $this->db->join(db_prefix() . 'lead_notes ln', 'ln.leadid = a.leadid AND ABS(TIMESTAMPDIFF(MINUTE, ln.dateadded, a.date)) <= 5', 'left');
        
        // Only show actual followup activities, not system notifications
        $this->db->where("(a.description = 'not_lead_activity_added_followup' OR a.description NOT LIKE 'not_%')");
        
        if ($from) $this->db->where('DATE(a.date) >=', $from);
        if ($to) $this->db->where('DATE(a.date) <=', $to);
        
        if (!empty($filters['staff'])) {
            $this->db->where_in('a.staffid', $filters['staff']);
        }
        
        $this->db->group_by('followup_type');
        $this->db->order_by('count', 'DESC');
        
        $result = $this->db->get()->result_array();
        
        return [
            'labels' => array_column($result, 'followup_type'),
            'data' => array_column($result, 'count')
        ];
    }

    /**
     * Get followup types chart data
     */
    public function get_followup_types_chart(array $filters)
    {
        return $this->get_activity_type_pie($filters);
    }

    /**
     * Get dealer statistics
     */
public function get_dealer_statistics(array $filters)
    {
        try {
            // NOTE: Dealer statistics show ALL dealers regardless of date range
            // because dealer interest is tracked over time, not limited to when leads were created
            
            // Get total dealers (no date filter - we want all dealers)
            $this->db->select('COUNT(*) as total_dealers');
            $this->db->from(db_prefix() . 'leads l');
            $this->db->join(db_prefix() . 'leads_status ls', 'ls.id = l.status', 'left');
            $this->db->where('LOWER(ls.name)', 'dealer');
            
            // No date filtering for total dealers count
            
            $result = $this->db->get();
            if (!$result) {
                error_log('Dealer Statistics Error: Failed to get total dealers - ' . $this->db->last_query());
                return ['total_dealers' => 0, 'interested_dealers' => 0, 'not_interested_dealers' => 0, 'no_response_dealers' => 0];
            }
            
            $total_dealers = $result->row()->total_dealers;
            
            // Get interested dealers - count distinct leads with interested status
            $this->db->select('COUNT(DISTINCT l.id) as interested_dealers');
            $this->db->from(db_prefix() . 'leads l');
            $this->db->join(db_prefix() . 'leads_status ls', 'ls.id = l.status', 'left');
            $this->db->join(db_prefix() . 'lead_notes ln', 'ln.leadid = l.id', 'left');
            $this->db->where('LOWER(ls.name)', 'dealer');
            $this->db->where('ln.dealer_interest', 'interested');
            
            // No date filtering - we want all interested dealers
            
            $interested_result = $this->db->get();
            $interested_dealers = $interested_result ? $interested_result->row()->interested_dealers : 0;
            
            // Get not interested dealers - count distinct leads with not_interested status
            $this->db->select('COUNT(DISTINCT l.id) as not_interested_dealers');
            $this->db->from(db_prefix() . 'leads l');
            $this->db->join(db_prefix() . 'leads_status ls', 'ls.id = l.status', 'left');
            $this->db->join(db_prefix() . 'lead_notes ln', 'ln.leadid = l.id', 'left');
            $this->db->where('LOWER(ls.name)', 'dealer');
            $this->db->where('ln.dealer_interest', 'not_interested');
            
            // No date filtering - we want all not interested dealers
            
            $not_interested_result = $this->db->get();
            $not_interested_dealers = $not_interested_result ? $not_interested_result->row()->not_interested_dealers : 0;
            
            // Calculate no response dealers (total - interested - not_interested)
            $no_response_dealers = $total_dealers - $interested_dealers - $not_interested_dealers;
            
            // Ensure no negative values
            if ($no_response_dealers < 0) {
                $no_response_dealers = 0;
            }
            
            // Debug logging
            error_log('Dealer Statistics Debug: Total=' . $total_dealers . ', Interested=' . $interested_dealers . ', Not Interested=' . $not_interested_dealers . ', No Response=' . $no_response_dealers);
        
            return [
                'total_dealers' => (int)$total_dealers,
                'interested_dealers' => (int)$interested_dealers,
                'not_interested_dealers' => (int)$not_interested_dealers,
                'no_response_dealers' => (int)$no_response_dealers
            ];
        } catch (Exception $e) {
            error_log('Dealer Statistics Error: ' . $e->getMessage());
            error_log('Dealer Statistics Error Trace: ' . $e->getTraceAsString());
            return [
                'total_dealers' => 0,
                'interested_dealers' => 0,
                'not_interested_dealers' => 0,
                'no_response_dealers' => 0
            ];
        }
    }

    /**
     * Get dealer list by interest status
     */
    public function get_dealer_list_by_interest($interest_status, array $filters)
    {
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        
        // Debug logging
        error_log('Dealer List Query Debug: Interest Status=' . $interest_status . ', From=' . $from . ', To=' . $to);
        
        try {
            if ($interest_status === 'no_response') {
                // For no response dealers, use LEFT JOIN with IS NULL
                $this->db->select('l.id, l.name, l.email, l.phonenumber, l.company, l.dateadded, NULL as interest_date');
                $this->db->from(db_prefix() . 'leads l');
                $this->db->join(db_prefix() . 'leads_status ls', 'ls.id = l.status', 'left');
                $this->db->join(db_prefix() . 'lead_notes ln', 'ln.leadid = l.id AND ln.dealer_interest IS NOT NULL', 'left');
                $this->db->where('LOWER(ls.name)', 'dealer');
                $this->db->where('ln.leadid IS NULL');
                
                if ($from) $this->db->where('DATE(l.dateadded) >=', $from);
                if ($to) $this->db->where('DATE(l.dateadded) <=', $to);
                
                $this->db->order_by('l.dateadded', 'DESC');
            } else {
                // For interested/not_interested dealers, find those with specific dealer_interest values
                $this->db->select('l.id, l.name, l.email, l.phonenumber, l.company, l.dateadded, ln.dateadded as interest_date');
                $this->db->from(db_prefix() . 'leads l');
                $this->db->join(db_prefix() . 'leads_status ls', 'ls.id = l.status', 'left');
                $this->db->join(db_prefix() . 'lead_notes ln', 'ln.leadid = l.id', 'left');
                $this->db->where('LOWER(ls.name)', 'dealer');
                $this->db->where('ln.dealer_interest', $interest_status);
                
                if ($from) $this->db->where('DATE(l.dateadded) >=', $from);
                if ($to) $this->db->where('DATE(l.dateadded) <=', $to);
                
                $this->db->order_by('ln.dateadded', 'DESC');
            }
            
            $query = $this->db->get();
            error_log('Dealer List Query: ' . $this->db->last_query());
            
            if (!$query) {
                error_log('Dealer List Query Error: ' . $this->db->error()['message']);
                return [];
            }
            
            return $query->result_array();
            
        } catch (Exception $e) {
            error_log('Dealer List Query Exception: ' . $e->getMessage());
            error_log('Dealer List Query Exception Trace: ' . $e->getTraceAsString());
            return [];
        }
    }

    /**
     * Get activity log for PDF reports
     */
    public function get_activity_log(array $filters, $limit = 50)
    {
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        
        $table = $this->get_activity_table_name();
        if (!$table) {
            return [];
        }
        
        $this->db->select("a.*, s.firstname, s.lastname, l.name as lead_name, l.company as lead_company, l.phonenumber as lead_phone, l.email as lead_email, COALESCE(ln.followup_type, 'General') as followup_type, COALESCE(ln.description, a.description) as original_description");
        $this->db->from($table . ' a');
        $this->db->join(db_prefix() . 'staff s', 's.staffid = a.staffid', 'left');
        $this->db->join(db_prefix() . 'leads l', 'l.id = a.leadid', 'left');
        $this->db->join(db_prefix() . 'lead_notes ln', 'ln.leadid = a.leadid AND ABS(TIMESTAMPDIFF(MINUTE, ln.dateadded, a.date)) <= 5', 'left');
        
        // Only show actual followup activities, not system notifications
        $this->db->where("(a.description = 'not_lead_activity_added_followup' OR a.description NOT LIKE 'not_%')");
        
        if ($from) $this->db->where('DATE(a.date) >=', $from);
        if ($to) $this->db->where('DATE(a.date) <=', $to);
        
        if (!empty($filters['staff']) && is_array($filters['staff'])) {
            $this->db->where_in('a.staffid', $filters['staff']);
        }
        
        $this->db->order_by('a.date', 'DESC');
        $this->db->limit($limit);
        
        $result = $this->db->get()->result_array();
        
        // Format the data for PDF display
        $formatted_data = [];
        foreach ($result as $row) {
            $staff_name = trim(($row['firstname'] ?? '') . ' ' . ($row['lastname'] ?? '')) ?: 'System';
            $lead_name = trim($row['lead_name'] ?? '');
            $lead_company = trim($row['lead_company'] ?? '');
            $lead_display = $this->build_lead_display($lead_name, $lead_company, $row['lead_phone'] ?? '', $row['lead_email'] ?? '');
            
            $formatted_data[] = [
                'id' => $row['id'],
                'date' => date('Y-m-d H:i', strtotime($row['date'] ?? 'now')),
                'staff' => $staff_name,
                'lead' => $lead_display,
                'followup_type' => $row['followup_type'] ?? 'General',
                'description' => !empty($row['original_description']) ? $row['original_description'] : ($row['description'] ?? ''),
                'priority' => $this->determine_priority($row)
            ];
        }
        
        return $formatted_data;
    }

    /**
     * Get followup types by staff for PDF reports
     */
    public function get_followup_types_by_staff(array $filters)
    {
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        
        $table = $this->get_activity_table_name();
        if (!$table) {
            return [];
        }
        
        $this->db->select("s.staffid, CONCAT(s.firstname, ' ', s.lastname) as staff_name, COALESCE(ln.followup_type, 'General') as followup_type, COUNT(a.id) as count");
        $this->db->from($table . ' a');
        $this->db->join(db_prefix() . 'staff s', 's.staffid = a.staffid', 'left');
        $this->db->join(db_prefix() . 'lead_notes ln', 'ln.leadid = a.leadid AND ABS(TIMESTAMPDIFF(MINUTE, ln.dateadded, a.date)) <= 5', 'left');
        
        // Only show actual followup activities, not system notifications
        $this->db->where("(a.description = 'not_lead_activity_added_followup' OR a.description NOT LIKE 'not_%')");
        
        if ($from) $this->db->where('DATE(a.date) >=', $from);
        if ($to) $this->db->where('DATE(a.date) <=', $to);
        
        if (!empty($filters['staff']) && is_array($filters['staff'])) {
            $this->db->where_in('a.staffid', $filters['staff']);
        }
        
        $this->db->group_by(['s.staffid', 'followup_type']);
        $this->db->order_by('staff_name, count', 'DESC');
        
        $result = $this->db->get()->result_array();
        
        // Group by staff for better presentation
        $grouped_data = [];
        foreach ($result as $row) {
            $staff_name = $row['staff_name'] ?: 'System';
            if (!isset($grouped_data[$staff_name])) {
                $grouped_data[$staff_name] = [];
            }
            $grouped_data[$staff_name][] = [
                'followup_type' => $row['followup_type'],
                'count' => (int)$row['count']
            ];
        }
        
        return $grouped_data;
    }
}
?>
