<?php
/*
 Module Name: Reporting Dashboard
 Module URI: https://example.com/reporting-dashboard
 Description: Centralized staff activity tracking and reporting with KPIs, charts, and detailed logs.
 Version: 1.0.0
 Requires at least: 3.0.*
*/

defined('BASEPATH') or exit('No direct script access allowed');

// Module constants
define('REPORTING_DASHBOARD_MODULE_NAME', 'reporting_dashboard');
define('REPORTING_DASHBOARD_MODULE', 'reporting_dashboard');

// Register language files (none for now but keep for future)
if (function_exists('register_language_files')) {
    register_language_files(REPORTING_DASHBOARD_MODULE, [REPORTING_DASHBOARD_MODULE]);
}

// Register hooks
if (function_exists('register_activation_hook')) {
    register_activation_hook(REPORTING_DASHBOARD_MODULE, 'reporting_dashboard_activate');
}

function reporting_dashboard_activate()
{
    // Default options for scheduled emails
    add_option('reporting_dashboard_schedule_enabled', 0);
    add_option('reporting_dashboard_schedule_frequency', 'daily'); // daily|weekly
    add_option('reporting_dashboard_schedule_recipients', '');
    add_option('reporting_dashboard_openai_api_key', '');
    add_option('reporting_dashboard_openai_model', 'gpt-4o-mini');
}

// Include module pieces
require_once __DIR__ . '/includes/staff_permissions.php';
require_once __DIR__ . '/includes/sidebar_menu_links.php';
require_once __DIR__ . '/includes/assets.php';

// Cron hook for scheduled email reports
hooks()->add_action('after_cron_run', function () {
    $CI = &get_instance();
    if ((int) get_option('reporting_dashboard_schedule_enabled') !== 1) {
        return;
    }

    $recipients = trim((string) get_option('reporting_dashboard_schedule_recipients'));
    if ($recipients === '') {
        return;
    }

    // Run once per day/week based on frequency using a transient option
    $frequency = get_option('reporting_dashboard_schedule_frequency');
    $last_run  = (int) get_option('reporting_dashboard_last_email_ts');
    $now       = time();

    $interval = $frequency === 'weekly' ? 7 * 24 * 60 * 60 : 24 * 60 * 60;

    if ($last_run && ($now - $last_run) < $interval) {
        return;
    }

    update_option('reporting_dashboard_last_email_ts', $now);

    $CI->load->model('reporting_dashboard/reporting_dashboard_model');
    $summary = $CI->reporting_dashboard_model->get_email_summary_data();

    // Build a simple HTML summary (keep light-weight)
    $html = '<h2>Activity Summary</h2>';
    $html .= '<p>Total Activities Today: ' . (int) $summary['kpis']['today'] . '</p>';
    $html .= '<p>Total Activities This Week: ' . (int) $summary['kpis']['week'] . '</p>';
    $html .= '<p>Total Activities This Month: ' . (int) $summary['kpis']['month'] . '</p>';
    $html .= '<p>Total Logins Today: ' . (int) $summary['kpis']['logins_today'] . '</p>';

    $subject = 'Reporting Dashboard - Scheduled Summary';

    $emails = array_filter(array_map('trim', explode(',', $recipients)));
    foreach ($emails as $email) {
        send_mail_template('simple', $email, $subject, $html);
    }
});

/**
 * Cron handler: run automated staff report schedules.
 */
function reporting_dashboard_run_staff_schedules()
{
    $CI = &get_instance();

    $CI->load->model('reporting_dashboard/reporting_dashboard_model');
    
    // Log cron execution for debugging
    log_activity('Reporting Dashboard Cron: Checking for due staff report schedules at ' . date('Y-m-d H:i:s'));
    
    $dueSchedules = $CI->reporting_dashboard_model->get_due_staff_report_schedules();

    if (empty($dueSchedules)) {
        log_activity('Reporting Dashboard Cron: No due schedules found');
        return;
    }
    
    log_activity('Reporting Dashboard Cron: Found ' . count($dueSchedules) . ' due schedule(s)');

    $CI->load->library('reporting_dashboard/staff_report_service');

    foreach ($dueSchedules as $schedule) {
        $startTime = microtime(true);
        $logData = [
            'status' => 'failed',
            'report_generated' => false,
            'whatsapp_sent' => false,
            'whatsapp_success_count' => 0,
            'whatsapp_failed_count' => 0,
            'whatsapp_numbers' => [],
            'error_message' => null,
            'staff_count' => 0,
            'activity_count' => 0,
        ];

        try {
            $filters = $CI->reporting_dashboard_model->build_filters_for_schedule($schedule);
            $summary = $CI->reporting_dashboard_model->get_staff_activity_summary($filters);

            if (empty($summary['staff'])) {
                $logData['error_message'] = 'No staff activity data for schedule window.';
                $CI->reporting_dashboard_model->mark_staff_schedule_run($schedule['id'], [
                    'success' => false,
                    'message' => $logData['error_message']
                ]);
                $logData['execution_duration_ms'] = (int)((microtime(true) - $startTime) * 1000);
                $CI->reporting_dashboard_model->log_schedule_execution($schedule['id'], $schedule['name'], $logData);
                continue;
            }

            $logData['staff_count'] = count($summary['staff']);
            $logData['activity_count'] = $summary['totals']['total_activities'] ?? 0;

            $analysisResult = $CI->staff_report_service->analyze_summary($summary, [
                'notes' => 'Scheduled automation: ' . ($schedule['name'] ?? '')
            ]);
            $analysis = $analysisResult['analysis'] ?? null;

            $numbers = $schedule['whatsapp_numbers_list'] ?? [];
            if (empty($numbers)) {
                $logData['error_message'] = 'Schedule has no WhatsApp numbers configured.';
                $CI->reporting_dashboard_model->mark_staff_schedule_run($schedule['id'], [
                    'success' => false,
                    'message' => $logData['error_message']
                ]);
                $logData['execution_duration_ms'] = (int)((microtime(true) - $startTime) * 1000);
                $CI->reporting_dashboard_model->log_schedule_execution($schedule['id'], $schedule['name'], $logData);
                continue;
            }

            $logData['whatsapp_numbers'] = $numbers;
            $message = $CI->staff_report_service->format_whatsapp_message($summary, $analysis);

            $context = [
                'generated_by' => $schedule['created_by'] ?? null,
                'whatsapp_numbers' => $numbers,
                'ai_model' => $analysisResult['model'] ?? null,
                'message_preview' => $message,
                'notes' => 'Scheduled run: ' . ($schedule['name'] ?? 'Staff Report'),
            ];

            if (!$analysisResult['success'] && !empty($analysisResult['error'])) {
                $context['notes'] .= ' (AI warning: ' . $analysisResult['error'] . ')';
            }

            $reportId = $CI->reporting_dashboard_model->save_staff_report($summary, $analysis, $filters, $context);
            $logData['report_generated'] = true;
            $logData['report_id'] = $reportId;

            $whatsappStatus = $CI->staff_report_service->send_whatsapp_message($message, $numbers);
            $logData['whatsapp_sent'] = $whatsappStatus['success'] ?? false;
            $logData['whatsapp_success_count'] = $whatsappStatus['sent'] ?? 0;
            $logData['whatsapp_failed_count'] = $whatsappStatus['failed'] ?? 0;

            $CI->reporting_dashboard_model->update_staff_report($reportId, [
                'whatsapp_status' => $whatsappStatus,
                'whatsapp_numbers' => $numbers,
                'message_preview' => $message,
            ]);

            $success = $whatsappStatus['success'] ?? false;
            $logData['status'] = $success ? 'success' : ($logData['report_generated'] ? 'partial' : 'failed');

            $CI->reporting_dashboard_model->mark_staff_schedule_run($schedule['id'], [
                'success' => $success,
                'report_id' => $reportId,
                'analysis_success' => $analysisResult['success'] ?? false,
                'analysis_error' => $analysisResult['error'] ?? null,
                'whatsapp' => $whatsappStatus,
            ]);

            $logData['execution_duration_ms'] = (int)((microtime(true) - $startTime) * 1000);
            $CI->reporting_dashboard_model->log_schedule_execution($schedule['id'], $schedule['name'], $logData);

        } catch (Exception $e) {
            log_message('error', 'Reporting Dashboard schedule execution failed: ' . $e->getMessage());
            $logData['error_message'] = $e->getMessage();
            $logData['execution_duration_ms'] = (int)((microtime(true) - $startTime) * 1000);
            
            $CI->reporting_dashboard_model->mark_staff_schedule_run($schedule['id'], [
                'success' => false,
                'error' => $e->getMessage()
            ]);
            
            $CI->reporting_dashboard_model->log_schedule_execution($schedule['id'], $schedule['name'], $logData);
        }
    }
}

hooks()->add_action('after_cron_run', 'reporting_dashboard_run_staff_schedules');
