# Staff Report Generation Permission System

## Overview
This feature adds granular permission control for staff report generation and scheduling. Only authorized users can generate AI-powered staff reports and schedule automatic delivery via WhatsApp.

## Features Implemented

### 1. **Database Schema Enhancement**
**File**: `modules/reporting_dashboard/install/create_permissions_table.php`

**New Column**: `can_generate_reports` (tinyint(1), default: 0)
- Added to `tblreporting_dashboard_permissions` table
- Controls permission to generate and schedule staff reports
- Separate from view permissions (view_all_staff, view_own_only, view_specific_staff)

**Migration**: Automatically adds column if it doesn't exist during module initialization

### 2. **Permission Model Enhancement**
**File**: `modules/reporting_dashboard/models/Reporting_dashboard_permissions_model.php`

**Updated Methods**:
- `set_staff_permission()` - Now accepts `$can_generate_reports` parameter
- Added `can_generate_reports()` - Checks if a staff member has report generation permission

**Logic**:
```php
public function can_generate_reports($staff_id)
{
    // Admins always can generate reports
    if (is_admin($staff_id)) {
        return true;
    }

    $permission = $this->db->get(...)- >row();
    return (bool)$permission->can_generate_reports;
}
```

### 3. **Controller Permission Checks**
**File**: `modules/reporting_dashboard/controllers/Reporting_dashboard.php`

**Protected Endpoints**:
1. `generate_staff_report()` - Generates AI report with WhatsApp delivery
2. `save_staff_report_schedule()` - Creates/updates report schedules
3. `delete_staff_report_schedule()` - Deletes report schedules
4. `toggle_staff_report_schedule()` - Enables/disables schedules

**Permission Check Logic**:
```php
$current_staff_id = get_staff_user_id();
if (!$this->reporting_dashboard_permissions_model->can_generate_reports($current_staff_id)) {
    $this->json_response([
        'success' => false,
        'error' => 'You do not have permission to generate staff reports.'
    ], 403);
    return;
}
```

**Response**: HTTP 403 Forbidden with descriptive error message

### 4. **Dashboard UI Conditional Rendering**
**File**: `modules/reporting_dashboard/views/dashboard.php`

**Hidden Panels** (for users without permission):
- Staff AI Report & WhatsApp Delivery panel
- Saved Staff Reports panel
- Report scheduling interface

**Implementation**:
```php
<?php if ($can_generate_reports ?? false): ?>
    <!-- Staff Report Panels -->
<?php endif; ?>
```

**JavaScript Variable**: `canGenerateReports` (boolean) injected into page

### 5. **Permissions Management UI**
**File**: `modules/reporting_dashboard/views/permissions_management.php`

**New UI Element**: Checkbox for report generation permission

**Features**:
- Checkbox: "Can Generate & Schedule Reports"
- Description: "(Allow this user to generate AI reports and schedule automatic delivery)"
- Shown for all non-admin staff
- Admins always have this permission (shown as text: "Admin - Always 'View All Staff' & Can Generate Reports")

**Visual Layout**:
```
┌─────────────────────────────────────────┐
│ Permission Type: [View Own Only ▼]     │
│                                         │
│ ☑ Can Generate & Schedule Reports      │
│   (Allow this user to generate AI...)  │
│                                         │
│ Select staff members:                   │
│ ┌─────────────────────────────────┐   │
│ │ □ John Doe                       │   │
│ │ □ Jane Smith                     │   │
│ └─────────────────────────────────┘   │
└─────────────────────────────────────────┘
```

### 6. **Permission Controller Enhancement**
**File**: `modules/reporting_dashboard/controllers/Reporting_dashboard_permissions.php`

**Updated Method**: `update_permission()`

**Changes**:
- Accepts `can_generate_reports` POST parameter
- Passes it to model's `set_staff_permission()` method
- Saves permission to database

**AJAX Request Data**:
```javascript
{
    staff_id: 123,
    permission_type: 'view_all_staff',
    allowed_staff_ids: [],
    can_generate_reports: 1,  // NEW
    csrf_token: '...'
}
```

### 7. **Dashboard Index Enhancement**
**File**: `modules/reporting_dashboard/controllers/Reporting_dashboard.php` - `index()` method

**Changes**:
- Queries `can_generate_reports` field from permissions table
- Sets to `true` for admins
- Passes `$data['can_generate_reports']` to view
- Injects JavaScript variable `canGenerateReports`

## Permission Matrix

| User Type | View Data | Generate Reports | Schedule Reports | Manage Permissions |
|-----------|-----------|------------------|------------------|--------------------|
| **Admin** | All Staff | ✅ Always | ✅ Always | ✅ Always |
| **Staff with Permission** | As configured | ✅ Yes | ✅ Yes | ❌ No |
| **Staff without Permission** | As configured | ❌ No | ❌ No | ❌ No |

## User Experience

### For Users WITH Permission:
1. **Dashboard**: Can see and use all staff report features
   - Staff AI Report & WhatsApp Delivery panel
   - Preview AI Report button
   - Send via WhatsApp button
   - Manage Schedules button
   - Saved Staff Reports history

2. **Actions Available**:
   - Generate instant reports with AI analysis
   - Send reports to WhatsApp numbers
   - Create recurring schedules (daily/weekly/monthly)
   - Edit/delete/toggle schedules
   - View historical reports

### For Users WITHOUT Permission:
1. **Dashboard**: Report features are hidden
   - Only see: KPIs, Activity Log, Reminders, Charts
   - No report generation or scheduling options

2. **API Behavior**:
   - Direct API calls return HTTP 403
   - Error message: "You do not have permission to generate staff reports."

### For Administrators:
1. **Dashboard**: Full access to all features (always)
2. **Permissions Management**: Can grant/revoke report permissions for other staff
3. **Cannot lose permission**: Admin status always grants report permission

## Security

### Backend Protection:
- ✅ Permission check on all report-related endpoints
- ✅ HTTP 403 response for unauthorized access
- ✅ CSRF protection on all POST requests
- ✅ Staff ID validation
- ✅ Admin-only access to permission management

### Frontend Protection:
- ✅ UI elements hidden for unauthorized users
- ✅ JavaScript variable for conditional logic
- ✅ No API endpoints exposed in UI for unauthorized users

### Database Protection:
- ✅ Foreign key constraints
- ✅ Default value: 0 (no permission)
- ✅ Column check during table creation/migration

## Installation/Upgrade

### For New Installations:
- Column created automatically during module activation
- Default value: `can_generate_reports = 0` for all users
- Admins get permission through code logic (not database)

### For Existing Installations:
- Run the installation script: `modules/reporting_dashboard/install/create_permissions_table.php`
- Or manually add column:
  ```sql
  ALTER TABLE `tblreporting_dashboard_permissions` 
  ADD COLUMN `can_generate_reports` tinyint(1) NOT NULL DEFAULT '0' 
  COMMENT 'Permission to generate and schedule staff reports' 
  AFTER `allowed_staff_ids`;
  ```

## Granting Permission

### Via UI (Recommended):
1. Navigate to: **Admin → Reporting Dashboard → Permissions**
2. Find the staff member
3. Check "Can Generate & Schedule Reports"
4. Click "Update" button

### Via Database:
```sql
UPDATE tblreporting_dashboard_permissions 
SET can_generate_reports = 1 
WHERE staff_id = ?;
```

### Via Model (Programmatic):
```php
$this->load->model('reporting_dashboard/reporting_dashboard_permissions_model');
$this->reporting_dashboard_permissions_model->set_staff_permission(
    $staff_id,
    'view_all_staff',  // or view_own_only, view_specific_staff
    get_staff_user_id(),
    [],  // allowed_staff_ids (for view_specific_staff)
    true  // can_generate_reports
);
```

## Testing

### Test Cases:

#### 1. Admin User:
- ✅ Can always see report panels
- ✅ Can generate reports
- ✅ Can create/edit/delete schedules
- ✅ `canGenerateReports` is `true` in JavaScript

#### 2. Staff with Permission:
- ✅ Can see report panels
- ✅ Can generate reports (success)
- ✅ Can manage schedules (success)
- ✅ Direct API calls succeed

#### 3. Staff without Permission:
- ✅ Cannot see report panels
- ✅ `canGenerateReports` is `false` in JavaScript
- ✅ Direct API calls return 403
- ✅ Error message shown: "You do not have permission..."

#### 4. Permission Toggle:
- ✅ Granting permission shows panels on refresh
- ✅ Revoking permission hides panels on refresh
- ✅ Changes persist in database

### Manual Test Script:
```javascript
// In browser console on dashboard:
console.log('Can generate reports:', canGenerateReports);

// Try to call API directly (should fail if no permission):
fetch('/admin/reporting_dashboard/generate_staff_report', {
    method: 'POST',
    headers: {'Content-Type': 'application/json'},
    body: JSON.stringify({})
}).then(r => r.json()).then(console.log);
```

## Files Modified

1. ✅ `modules/reporting_dashboard/install/create_permissions_table.php` - Schema
2. ✅ `modules/reporting_dashboard/models/Reporting_dashboard_permissions_model.php` - Model logic
3. ✅ `modules/reporting_dashboard/controllers/Reporting_dashboard.php` - Endpoint protection
4. ✅ `modules/reporting_dashboard/controllers/Reporting_dashboard_permissions.php` - Permission update
5. ✅ `modules/reporting_dashboard/views/dashboard.php` - UI conditional rendering
6. ✅ `modules/reporting_dashboard/views/permissions_management.php` - Permission UI

## Future Enhancements

1. **Role-based Templates**: Predefined permission sets (Manager, Viewer, Analyst)
2. **Audit Log**: Track who generated/scheduled reports and when
3. **Report Limits**: Max reports per day/month per user
4. **Email Notifications**: Alert admins when reports are generated
5. **Advanced Scheduling**: More granular cron expressions
6. **Report Templates**: Pre-configured report formats
7. **Export Permissions**: Separate permission for exporting data

## Support

For issues or questions:
1. Check browser console for JavaScript errors
2. Check `application/logs/log-YYYY-MM-DD.php` for PHP errors
3. Verify database column exists: `DESCRIBE tblreporting_dashboard_permissions;`
4. Verify permission value: `SELECT * FROM tblreporting_dashboard_permissions WHERE staff_id = ?;`

## Version History

- **v2.0.0** (Nov 15, 2025) - Added report generation permission system
- **v1.0.0** - Initial release with basic permissions

---

**Last Updated**: November 15, 2025  
**Module Version**: 2.0.0  
**Compatible with**: Perfex CRM 3.0+




