# Schedule Execution Logging Feature

## Overview
This document describes the comprehensive logging system for scheduled staff reports in the Reporting Dashboard module. The logging system tracks every execution of scheduled reports, including success/failure status, WhatsApp delivery results, and detailed execution metrics.

## Database Schema

### Table: `tblreporting_dashboard_schedule_logs`

```sql
CREATE TABLE IF NOT EXISTS `tblreporting_dashboard_schedule_logs` (
    `id` INT(11) NOT NULL AUTO_INCREMENT,
    `schedule_id` INT(11) NOT NULL,
    `schedule_name` VARCHAR(255) NULL,
    `execution_time` DATETIME NOT NULL,
    `status` ENUM('success','failed','partial') NOT NULL DEFAULT 'success',
    `report_generated` TINYINT(1) NOT NULL DEFAULT 0,
    `whatsapp_sent` TINYINT(1) NOT NULL DEFAULT 0,
    `whatsapp_success_count` INT(11) DEFAULT 0,
    `whatsapp_failed_count` INT(11) DEFAULT 0,
    `whatsapp_numbers` TEXT NULL,
    `error_message` TEXT NULL,
    `execution_duration_ms` INT(11) NULL,
    `report_id` INT(11) NULL,
    `staff_count` INT(11) DEFAULT 0,
    `activity_count` INT(11) DEFAULT 0,
    `details` TEXT NULL COMMENT 'JSON data with additional details',
    PRIMARY KEY (`id`),
    KEY `idx_schedule_id` (`schedule_id`),
    KEY `idx_execution_time` (`execution_time`),
    KEY `idx_status` (`status`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8;
```

### Field Descriptions

- **id**: Auto-incrementing primary key
- **schedule_id**: Foreign key linking to the schedule
- **schedule_name**: Name of the schedule for easy reference
- **execution_time**: When the schedule was executed
- **status**: Execution status (success/failed/partial)
  - `success`: Report generated AND WhatsApp sent successfully
  - `partial`: Report generated BUT WhatsApp failed
  - `failed`: Report generation failed
- **report_generated**: Whether the report was successfully created
- **whatsapp_sent**: Whether WhatsApp messages were sent
- **whatsapp_success_count**: Number of successfully sent WhatsApp messages
- **whatsapp_failed_count**: Number of failed WhatsApp messages
- **whatsapp_numbers**: JSON array of recipient phone numbers
- **error_message**: Error message if execution failed
- **execution_duration_ms**: Total execution time in milliseconds
- **report_id**: ID of the generated report (if successful)
- **staff_count**: Number of staff members in the report
- **activity_count**: Total number of activities in the report
- **details**: Additional JSON data for debugging

## Backend Implementation

### Model Methods (`Reporting_dashboard_model.php`)

#### 1. `log_schedule_execution($scheduleId, $scheduleName, array $logData)`
Saves a new log entry for a schedule execution.

**Parameters:**
```php
$logData = [
    'status' => 'success|failed|partial',
    'report_generated' => true|false,
    'whatsapp_sent' => true|false,
    'whatsapp_success_count' => 0,
    'whatsapp_failed_count' => 0,
    'whatsapp_numbers' => ['123', '456'],
    'error_message' => 'Error details',
    'execution_duration_ms' => 1234,
    'report_id' => 42,
    'staff_count' => 5,
    'activity_count' => 100,
    'details' => ['key' => 'value']
];
```

**Returns:** Log ID (integer)

#### 2. `get_schedule_logs($scheduleId = null, $limit = 100)`
Retrieves execution logs for a specific schedule or all schedules.

**Parameters:**
- `$scheduleId`: Schedule ID (optional, null = all schedules)
- `$limit`: Maximum number of logs to retrieve (default: 100)

**Returns:** Array of log entries with decoded JSON fields

#### 3. `get_schedule_stats($scheduleId)`
Calculates aggregate statistics for a schedule.

**Returns:**
```php
[
    'total_executions' => 10,
    'successful' => 8,
    'failed' => 1,
    'partial' => 1,
    'total_whatsapp_sent' => 80,
    'total_whatsapp_failed' => 2,
    'last_execution' => '2025-11-15 10:30:00',
    'avg_duration_ms' => 2345
]
```

### Controller Endpoints (`Reporting_dashboard.php`)

#### 1. `GET /admin/reporting_dashboard/get_schedule_logs`
Fetches execution logs for a specific schedule.

**Query Parameters:**
- `schedule_id` (required): Schedule ID
- `limit` (optional): Maximum number of logs (default: 100)

**Response:**
```json
{
    "success": true,
    "logs": [
        {
            "id": 1,
            "schedule_id": 5,
            "schedule_name": "Daily Sales Report",
            "execution_time": "2025-11-15 10:00:00",
            "status": "success",
            "report_generated": 1,
            "whatsapp_sent": 1,
            "whatsapp_success_count": 3,
            "whatsapp_failed_count": 0,
            "whatsapp_numbers": ["+1234567890", "+9876543210"],
            "error_message": null,
            "execution_duration_ms": 2340,
            "report_id": 42,
            "staff_count": 5,
            "activity_count": 120
        }
    ],
    "stats": {
        "total_executions": 10,
        "successful": 9,
        "failed": 0,
        "partial": 1,
        "total_whatsapp_sent": 27,
        "total_whatsapp_failed": 3,
        "last_execution": "2025-11-15 10:00:00",
        "avg_duration_ms": 2234
    }
}
```

#### 2. `GET /admin/reporting_dashboard/get_schedules_with_stats`
Fetches all schedules with their execution statistics.

**Response:**
```json
{
    "success": true,
    "schedules": [
        {
            "id": 5,
            "name": "Daily Sales Report",
            "frequency": "daily",
            "active": 1,
            "next_run": "2025-11-16 10:00:00",
            "stats": {
                "total_executions": 10,
                "successful": 9,
                "failed": 0,
                "partial": 1
            }
        }
    ]
}
```

### Cron Job Integration (`reporting_dashboard.php`)

The cron function `reporting_dashboard_run_staff_schedules()` has been enhanced to log every execution:

```php
foreach ($dueSchedules as $schedule) {
    $startTime = microtime(true);
    $logData = [
        'status' => 'failed',
        'report_generated' => false,
        'whatsapp_sent' => false,
        // ... other fields
    ];

    try {
        // Generate report
        // Update logData['report_generated'] = true
        // Send WhatsApp
        // Update logData['whatsapp_sent'] and counts
        // Calculate duration
        
        $CI->reporting_dashboard_model->log_schedule_execution(
            $schedule['id'], 
            $schedule['name'], 
            $logData
        );
    } catch (Exception $e) {
        $logData['error_message'] = $e->getMessage();
        $CI->reporting_dashboard_model->log_schedule_execution(
            $schedule['id'], 
            $schedule['name'], 
            $logData
        );
    }
}
```

## Frontend Implementation

### UI Components

#### 1. Logs Button in Schedules Table
Each schedule now has a "View Logs" button (blue icon button) that opens the logs modal.

#### 2. Schedule Logs Modal (`#scheduleLogsModal`)
A comprehensive modal displaying:
- **Statistics Cards**: Total executions, successful, partial, failed
- **WhatsApp Stats**: Total messages sent and failed
- **Logs Table**: Detailed execution history with:
  - Execution time
  - Status (success/failed/partial badge)
  - Report generated (yes/no)
  - WhatsApp delivery status and counts
  - Recipients (first 3, with +X more if applicable)
  - Execution duration
  - Error messages (if any)

### JavaScript Functions

#### 1. `viewScheduleLogs(scheduleId, scheduleName)`
Opens the logs modal for a specific schedule.

**Parameters:**
- `scheduleId`: Schedule ID
- `scheduleName`: Schedule name for display

#### 2. `loadScheduleLogs(scheduleId)`
Fetches and renders logs for a schedule via AJAX.

**Parameters:**
- `scheduleId`: Schedule ID

#### 3. `renderScheduleLogs(logs, stats)`
Renders the logs table and statistics cards.

**Parameters:**
- `logs`: Array of log entries
- `stats`: Aggregate statistics object

## Usage Guide

### For Users

1. **View Execution Logs:**
   - Navigate to: Reporting Dashboard → Staff AI Report section
   - Click "Manage Schedules"
   - Click the blue icon (📄) button next to any schedule
   - View detailed execution history, statistics, and WhatsApp delivery status

2. **Understand Log Status:**
   - **Success (green)**: Report generated AND WhatsApp sent successfully
   - **Partial (yellow)**: Report generated BUT WhatsApp delivery failed
   - **Failed (red)**: Report generation failed

3. **Troubleshoot Issues:**
   - Check the "Error / Notes" column for error messages
   - Verify WhatsApp numbers are correct
   - Ensure cron jobs are running (check "Last Execution" time)
   - Contact admin if errors persist

### For Administrators

1. **Monitor Schedule Health:**
   - Review statistics cards for overall performance
   - Identify schedules with high failure rates
   - Check execution durations for performance issues

2. **Debug Failed Executions:**
   - Review error messages in the logs table
   - Check server logs for detailed stack traces
   - Verify OpenAI API key and WhatsApp API credentials
   - Ensure database tables exist and are properly structured

3. **Optimize Performance:**
   - Monitor `execution_duration_ms` for slow schedules
   - Adjust staff filters to reduce data volume
   - Consider off-peak execution times for large reports

## Permissions

The logging feature respects the existing permission system:
- Only users with `can_generate_reports` permission can view logs
- Admins always have access to all logs
- Permission checks are enforced at the controller level

## Best Practices

1. **Regular Monitoring:**
   - Check logs weekly to ensure schedules are running
   - Address failed executions promptly
   - Monitor WhatsApp delivery rates

2. **Error Handling:**
   - All errors are logged for troubleshooting
   - Critical errors are also logged to server error logs
   - Failed executions don't stop subsequent schedule runs

3. **Data Retention:**
   - Logs are retained indefinitely by default
   - Consider archiving old logs (>90 days) for performance
   - Limit set to 100 logs per schedule by default (adjustable)

## Troubleshooting

### Issue: No logs appearing
**Solution:**
1. Verify cron jobs are running: Check "Next Run" times
2. Manually trigger cron: Visit `/admin/misc/cron` (if enabled)
3. Check server cron configuration
4. Review server error logs for PHP errors

### Issue: "Partial" status on all executions
**Solution:**
1. Verify WhatsApp API credentials in AutographFeedbackInsights module
2. Check WhatsApp phone number format (must be international)
3. Review WhatsApp API logs for delivery errors
4. Test WhatsApp API with a simple test message

### Issue: High execution durations
**Solution:**
1. Reduce date range scope (use "last_7_days" instead of "custom")
2. Filter by specific staff members instead of "all"
3. Optimize database queries (add indexes if needed)
4. Consider upgrading server resources

## Future Enhancements

1. **Log Cleanup Job:** Automatic archiving of logs older than X days
2. **Email Notifications:** Alert admins on repeated failures
3. **Dashboard Widget:** Summary of schedule health on main dashboard
4. **Export Logs:** Download logs as CSV/PDF
5. **Retry Failed Executions:** Manual retry button for failed schedules
6. **Advanced Filtering:** Filter logs by date range, status, etc.

## Related Files

- Model: `modules/reporting_dashboard/models/Reporting_dashboard_model.php`
- Controller: `modules/reporting_dashboard/controllers/Reporting_dashboard.php`
- View: `modules/reporting_dashboard/views/dashboard.php`
- Cron: `modules/reporting_dashboard/reporting_dashboard.php`
- Library: `modules/reporting_dashboard/libraries/Staff_report_service.php`

## Support

For issues or questions about the logging feature:
1. Check this documentation first
2. Review server error logs
3. Check browser console for JavaScript errors
4. Contact system administrator

---

**Version:** 1.0.0  
**Date:** November 15, 2025  
**Author:** Reporting Dashboard Module




