<?php
if (!defined('BASEPATH')) { exit('No direct script access allowed');}

class Reminder_management extends AdminController
{
    public function __construct()
    {
        parent::__construct();
        $this->load->model('whatsapp_api_model');
        $this->load->library('whatsapp_api_lib');
    }    public function index()
    {
        // Basic admin check - remove complex permissions for now
        if (!is_admin() && !is_staff_member()) {
            access_denied('reminder_management');
        }
        
        $data['title'] = _l('whatsapp_reminder_management', 'WhatsApp Reminder Notifications');
        $data['bodyclass'] = 'whatsapp-reminder-management';
        
        // Use admin layout wrapper
        $this->load->view('admin/reminder_management', $data);
    }

    public function check_status()
    {
        if (!$this->input->is_ajax_request()) {
            exit('No direct script access allowed');
        }
        
        $status = [
            'module_active' => true, // Module is loaded if we're here
            'api_configured' => $this->check_api_configuration(),
            'template_mapped' => $this->check_template_mapping(),
            'db_ready' => $this->check_database_structure()
        ];
        
        header('Content-Type: application/json');
        echo json_encode($status);
    }

    public function upcoming_reminders()
    {
        if (!$this->input->is_ajax_request()) {
            exit('No direct script access allowed');
        }
        
        $reminders = $this->whatsapp_api_model->get_upcoming_reminders();
        
        // Format the reminders for display
        foreach ($reminders as &$reminder) {
            $reminder['formatted_date'] = date('F j, Y g:i A', strtotime($reminder['date']));
        }
        
        header('Content-Type: application/json');
        echo json_encode(['reminders' => $reminders]);
    }    public function activity_log()
    {
        if (!$this->input->is_ajax_request()) {
            exit('No direct script access allowed');
        }
        
        // Get recent WhatsApp reminder logs - simplified query since debug log doesn't have rel_id
        $this->db->select('*');
        $this->db->from(db_prefix() . 'whatsapp_api_debug_log');
        $this->db->where('message_category', 'staff_reminder');
        $this->db->order_by('recorded_at', 'DESC');
        $this->db->limit(20);
        
        $logs = $this->db->get()->result_array();
        
        // Format the logs
        foreach ($logs as &$log) {
            $log['staff_name'] = 'Staff Member';
            $log['status'] = ($log['response_code'] == '200') ? 'SUCCESS' : 'FAILED';
            $log['reminder_description'] = 'Staff reminder notification';
            $log['created_at'] = $log['recorded_at'];
            
            // Try to extract staff name from merge_field_data if available
            if (!empty($log['merge_field_data'])) {
                $merge_data = json_decode($log['merge_field_data'], true);
                if (is_array($merge_data)) {
                    if (isset($merge_data['{staff_firstname}']) && isset($merge_data['{staff_lastname}'])) {
                        $log['staff_name'] = $merge_data['{staff_firstname}'] . ' ' . $merge_data['{staff_lastname}'];
                    }
                    if (isset($merge_data['{reminder_description}'])) {
                        $log['reminder_description'] = $merge_data['{reminder_description}'];
                    }
                }
            }
        }
        
        header('Content-Type: application/json');
        echo json_encode(['logs' => $logs]);
    }    public function check_now()
    {
        if (!$this->input->is_ajax_request()) {
            exit('No direct script access allowed');
        }
        
        try {
            $upcoming_reminders = $this->whatsapp_api_model->get_upcoming_reminders();
            $sent_count = 0;
            $skipped_count = 0;
            $errors = [];
            
            foreach ($upcoming_reminders as $reminder) {
                // Double-check that this reminder hasn't been WhatsApp notified
                if (!$this->whatsapp_api_model->is_reminder_whatsapp_notified($reminder['id'])) {
                    try {
                        // Send using the mapped template approach
                        $this->whatsapp_api_lib->send_mapped_template('staff_reminder', false, 'MANUAL', $reminder['id']);
                        
                        // Mark as WhatsApp notified (atomic operation)
                        $marked = $this->whatsapp_api_model->mark_reminder_whatsapp_notified($reminder['id']);
                        
                        if ($marked) {
                            $sent_count++;
                        } else {
                            $skipped_count++;
                        }
                    } catch (Exception $e) {
                        $errors[] = "Failed to send to " . $reminder['firstname'] . " " . $reminder['lastname'] . ": " . $e->getMessage();
                    }
                } else {
                    $skipped_count++;
                }
            }
            
            $message = "Processed " . count($upcoming_reminders) . " upcoming reminders. ";
            $message .= "Sent " . $sent_count . " WhatsApp notifications. ";
            if ($skipped_count > 0) {
                $message .= "Skipped " . $skipped_count . " (already notified). ";
            }
            
            if (!empty($errors)) {
                $message .= " Errors: " . implode(', ', $errors);
            }
            
            header('Content-Type: application/json');
            echo json_encode(['success' => true, 'message' => $message]);
            
        } catch (Exception $e) {
            header('Content-Type: application/json');
            echo json_encode(['success' => false, 'message' => 'Error: ' . $e->getMessage()]);
        }
    }

    public function test_notification()
    {
        if (!$this->input->is_ajax_request()) {
            exit('No direct script access allowed');
        }
        
        try {
            // Create a test reminder for the current user
            $current_staff_id = get_staff_user_id();
            $current_staff = $this->db->get_where(db_prefix() . 'staff', ['staffid' => $current_staff_id])->row();
            
            if (!$current_staff || empty($current_staff->phonenumber)) {
                header('Content-Type: application/json');
                echo json_encode(['success' => false, 'message' => 'Your staff profile needs a phone number to test notifications']);
                return;
            }
            
            // Create test reminder
            $test_reminder_data = [
                'description' => 'Test WhatsApp Reminder Notification - ' . date('Y-m-d H:i:s'),
                'date' => date('Y-m-d H:i:s', strtotime('+2 minutes')),
                'staff' => $current_staff_id,
                'creator' => $current_staff_id,
                'rel_type' => 'test',
                'rel_id' => 0
            ];
            
            $this->db->insert(db_prefix() . 'reminders', $test_reminder_data);
            $test_reminder_id = $this->db->insert_id();
            
            // Send test notification using the mapped template approach
            $this->whatsapp_api_lib->send_mapped_template('staff_reminder', false, 'MANUAL', $test_reminder_id);
                
            // Clean up test reminder
            $this->db->delete(db_prefix() . 'reminders', ['id' => $test_reminder_id]);
            
            header('Content-Type: application/json');
            echo json_encode(['success' => true, 'message' => 'Test notification sent successfully to ' . $current_staff->phonenumber]);
            
        } catch (Exception $e) {
            header('Content-Type: application/json');
            echo json_encode(['success' => false, 'message' => 'Error: ' . $e->getMessage()]);
        }
    }

    public function trigger_cron_test()
    {
        if (!$this->input->is_ajax_request()) {
            exit('No direct script access allowed');
        }
          try {
            // Manually trigger the cron hooks to test
            wa_check_due_reminders_hook();
            // wa_check_processed_reminders_hook(); // DISABLED: Text-based system with phone number issues
            
            header('Content-Type: application/json');
            echo json_encode(['success' => true, 'message' => 'Cron hooks triggered manually for testing']);
            
        } catch (Exception $e) {
            header('Content-Type: application/json');
            echo json_encode(['success' => false, 'message' => 'Error: ' . $e->getMessage()]);
        }
    }

    public function manual_cron()    {
        // Allow this to be called without AJAX for web-based cron services
        try {
            // Trigger our reminder hooks
            wa_check_due_reminders_hook();
            // wa_check_processed_reminders_hook(); // DISABLED: Text-based system with phone number issues
            wa_admin_reminder_check();
            
            $message = 'Manual cron executed successfully at ' . date('Y-m-d H:i:s');
            log_activity($message);
            
            header('Content-Type: application/json');
            echo json_encode(['success' => true, 'message' => $message]);
            
        } catch (Exception $e) {
            header('Content-Type: application/json');
            echo json_encode(['success' => false, 'message' => 'Error: ' . $e->getMessage()]);
        }
    }

    private function check_api_configuration()
    {
        $required_settings = ['phone_number_id', 'whatsapp_access_token', 'whatsapp_business_account_id'];
        
        foreach ($required_settings as $setting) {
            if (!get_option($setting)) {
                return false;
            }
        }
        
        return true;
    }

    private function check_template_mapping()
    {
        $template_count = $this->db->get_where(db_prefix() . 'whatsapp_templates_mapping', ['category' => 'staff_reminder'])->num_rows();
        return $template_count > 0;
    }

    private function check_database_structure()
    {
        return $this->db->field_exists('whatsapp_notified', db_prefix() . 'reminders') &&
               $this->db->field_exists('whatsapp_notified_at', db_prefix() . 'reminders');
    }
}
