<?php

defined('BASEPATH') or exit('No direct script access allowed');

class Reporting_dashboard_model extends App_Model
{
    public function __construct()
    {
        parent::__construct();
    }

    public function get_kpis(array $filters)
    {
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        
        $today = date('Y-m-d');
        $weekStart = date('Y-m-d', strtotime('monday this week'));
        $monthStart = date('Y-m-01');
        
        return [
            'today' => $this->count_lead_activities(['from' => $today, 'to' => $today, 'staff' => $filters['staff'] ?? []]),
            'week' => $this->count_lead_activities(['from' => $weekStart, 'to' => $today, 'staff' => $filters['staff'] ?? []]),
            'month' => $this->count_lead_activities(['from' => $monthStart, 'to' => $today, 'staff' => $filters['staff'] ?? []]),
            'logins_today' => $this->count_logins_today($filters['staff'] ?? []),
            'top_staff' => $this->get_top_staff_simple($filters)
        ];
    }

    public function get_staff_activity_chart(array $filters)
    {
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        
        // Try to find the correct activity table
        $table = $this->get_activity_table_name();
        if (!$table) {
            return ['labels' => [], 'data' => []];
        }
        
        $this->db->select('s.staffid, CONCAT(s.firstname, " ", s.lastname) as staff_name, COUNT(a.id) as total');
        $this->db->from($table . ' a');
        $this->db->join(db_prefix() . 'staff s', 's.staffid = a.staffid', 'left');
        
        if ($from) $this->db->where('DATE(a.date) >=', $from);
        if ($to) $this->db->where('DATE(a.date) <=', $to);
        if (!empty($filters['staff'])) {
            $this->db->where_in('a.staffid', $filters['staff']);
        }
        
        $this->db->group_by('s.staffid');
        $this->db->order_by('total', 'DESC');
        $this->db->limit(10);
        
        $result = $this->db->get()->result_array();
        
        return [
            'labels' => array_column($result, 'staff_name'),
            'data' => array_column($result, 'total')
        ];
    }

    public function get_activity_over_time_chart(array $filters)
    {
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        
        $table = $this->get_activity_table_name();
        if (!$table) {
            return ['labels' => [], 'data' => []];
        }
        
        $this->db->select('DATE(date) as activity_date, COUNT(id) as total');
        $this->db->from($table);
        $this->db->where('DATE(date) >=', $from);
        $this->db->where('DATE(date) <=', $to);
        
        if (!empty($filters['staff'])) {
            $this->db->where_in('staffid', $filters['staff']);
        }
        
        $this->db->group_by('DATE(date)');
        $this->db->order_by('activity_date', 'ASC');
        
        $result = $this->db->get()->result_array();
        
        return [
            'labels' => array_column($result, 'activity_date'),
            'data' => array_column($result, 'total')
        ];
    }

    /**
     * Enhanced activity table with modern features
     */
    public function get_activity_table(array $filters)
    {
        $table = $this->get_activity_table_name();
        if (!$table) {
            error_log('No activity table found');
            return ['data' => [], 'recordsTotal' => 0, 'recordsFiltered' => 0];
        }

        error_log('Using activity table: ' . $table);

        // DataTables parameters
        $draw = (int)$this->input->post('draw');
        $start = (int)$this->input->post('start');
        $length = (int)$this->input->post('length');
        if ($length <= 0 || $length > 100) { $length = 25; }

        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');

        error_log('Activity table query - From: ' . $from . ', To: ' . $to);

        // Enhanced followup type extraction
        $followup_type_case = $this->get_followup_type_sql();
        
        $this->db->select("a.*, s.firstname, s.lastname, l.name as lead_name, l.company as lead_company, l.phonenumber as lead_phone, l.email as lead_email, $followup_type_case as followup_type");
        $this->db->from($table . ' a');
        $this->db->join(db_prefix() . 'staff s', 's.staffid = a.staffid', 'left');
        $this->db->join(db_prefix() . 'leads l', 'l.id = a.leadid', 'left');
        
        if ($from) $this->db->where('DATE(a.date) >=', $from);
        if ($to) $this->db->where('DATE(a.date) <=', $to);
        
        // Staff filtering
        if (!empty($filters['staff']) && is_array($filters['staff'])) {
            $this->db->where_in('a.staffid', $filters['staff']);
        }

        // Search functionality
        $search = $this->input->post('search');
        if (!empty($search['value'])) {
            $search_term = $search['value'];
            $this->db->group_start();
            $this->db->like('a.description', $search_term);
            $this->db->or_like('CONCAT(s.firstname, " ", s.lastname)', $search_term);
            $this->db->or_like('l.name', $search_term);
            $this->db->or_like('l.company', $search_term);
            $this->db->group_end();
        }

        // Get total count
        $total_query = clone $this->db;
        $total = $total_query->count_all_results();
        
        $this->db->order_by('a.date', 'DESC');
        $this->db->limit($length, $start);
        
        try {
            $query_result = $this->db->get();
            $rows = $query_result->result_array();
            error_log('Activity table query successful: ' . count($rows) . ' rows');
        } catch (Exception $e) {
            error_log('Activity table query error: ' . $e->getMessage());
            return ['data' => [], 'recordsTotal' => 0, 'recordsFiltered' => 0];
        }

        $data = [];
        foreach ($rows as $r) {
            $staff_name = trim(($r['firstname'] ?? '') . ' ' . ($r['lastname'] ?? '')) ?: 'System';
            
            $lead_id = (int)($r['leadid'] ?? 0);
            $lead_name = trim($r['lead_name'] ?? '');
            $lead_company = trim($r['lead_company'] ?? '');
            $lead_phone = trim($r['lead_phone'] ?? '');
            $lead_email = trim($r['lead_email'] ?? '');
            
            $lead_display = $this->build_lead_display($lead_name, $lead_company, $lead_phone, $lead_email);
            $followup_type = $r['followup_type'] ?? 'General';
            
            $data[] = [
                'id' => $r['id'],
                'date' => date('Y-m-d H:i', strtotime($r['date'] ?? 'now')),
                'staff' => $staff_name,
                'staff_id' => $r['staffid'] ?? 0,
                'lead' => $lead_display,
                'lead_id' => $lead_id,
                'lead_name' => $lead_name,
                'lead_company' => $lead_company,
                'lead_phone' => $lead_phone,
                'lead_email' => $lead_email,
                'followup_type' => $followup_type,
                'description' => $r['description'] ?? '',
                'priority' => $this->determine_priority($r)
            ];
        }

        return [
            'draw' => $draw,
            'recordsTotal' => $total,
            'recordsFiltered' => $total,
            'data' => $data
        ];
    }

    /**
     * Find the correct activity table name
     */
    private function get_activity_table_name()
    {
        $table_variations = ['lead_activity_log', 'activity_log', 'activities'];
        
        foreach ($table_variations as $variation) {
            if ($this->db->table_exists($variation)) {
                return db_prefix() . $variation;
            }
        }
        
        return null;
    }

    /**
     * Get SQL for followup type detection
     */
    private function get_followup_type_sql()
    {
        return "(
            CASE 
                WHEN LOWER(a.description) REGEXP '(whatsapp|whats.app|wa )' THEN 'WhatsApp'
                WHEN LOWER(a.description) REGEXP '(call|phone|dial|rang)' THEN 'Phone Call'
                WHEN LOWER(a.description) REGEXP '(email|mail)' THEN 'Email'
                WHEN LOWER(a.description) REGEXP '(sms|text)' THEN 'SMS'
                WHEN LOWER(a.description) REGEXP '(meeting|appointment)' THEN 'Meeting'
                WHEN LOWER(a.description) REGEXP '(reminder|follow.?up)' THEN 'Reminder'
                WHEN LOWER(a.description) REGEXP '(status|updated)' THEN 'Status Update'
                WHEN LOWER(a.description) REGEXP '(proposal|quote)' THEN 'Proposal'
                ELSE 'General'
            END
        )";
    }

    /**
     * Build lead display with contact info
     */
    private function build_lead_display($name, $company, $phone, $email)
    {
        $parts = [];
        if ($name) $parts[] = $name;
        if ($company) $parts[] = '(' . $company . ')';
        
        $main = !empty($parts) ? implode(' ', $parts) : 'Unknown Lead';
        
        $contacts = [];
        if ($phone) $contacts[] = $phone;
        if ($email) $contacts[] = $email;
        
        if (!empty($contacts)) {
            $main .= ' • ' . implode(' • ', $contacts);
        }
        
        return $main;
    }

    /**
     * Determine activity priority
     */
    private function determine_priority($row)
    {
        $date = $row['date'] ?? '';
        $hours_ago = $date ? (time() - strtotime($date)) / 3600 : 999;
        
        if ($hours_ago <= 2) return 'high';
        if ($hours_ago <= 24) return 'medium';
        return 'normal';
    }

    /**
     * Helper functions for KPI calculation
     */
    private function count_lead_activities($filters) 
    {
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        $staff = $filters['staff'] ?? [];
        
        $table = $this->get_activity_table_name();
        if (!$table) return 0;
        
        $this->db->from($table);
        $this->db->where('DATE(date) >=', $from);
        $this->db->where('DATE(date) <=', $to);
        
        if (!empty($staff)) {
            $this->db->where_in('staffid', $staff);
        }
        
        return $this->db->count_all_results();
    }
    
    private function count_logins_today($staff) 
    {
        return 0; // Placeholder
    }
    
    private function get_top_staff_simple($filters) 
    {
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        
        $table = $this->get_activity_table_name();
        if (!$table) return [];
        
        $this->db->select('s.staffid, CONCAT(s.firstname, " ", s.lastname) as staff_name, COUNT(a.id) as total');
        $this->db->from($table . ' a');
        $this->db->join(db_prefix() . 'staff s', 's.staffid = a.staffid', 'left');
        $this->db->where('DATE(a.date) >=', $from);
        $this->db->where('DATE(a.date) <=', $to);
        $this->db->group_by('s.staffid');
        $this->db->order_by('total', 'DESC');
        $this->db->limit(5);
        
        return $this->db->get()->result_array();
    }

    // Method for email summary
    public function get_email_summary_data()
    {
        return [
            'kpis' => $this->get_kpis([])
        ];
    }

    /**
     * Return reminders grouped for dashboard
     */
    public function get_reminders(array $filters)
    {
        return ['todays' => [], 'due' => [], 'upcoming' => []];
    }
}
?>
