<?php defined('BASEPATH') or exit('No direct script access allowed');

class Reporting_dashboard extends AdminController
{
    public function __construct()
    {
        parent::__construct();
        
        // Load models using CodeIgniter approach
        $this->load->model('reporting_dashboard/reporting_dashboard_model');
        $this->load->model('reporting_dashboard/reporting_dashboard_permissions_model');
        $this->load->model('staff_model');
        
        // Check if user is logged in
        if (!is_staff_logged_in()) {
            redirect(admin_url('authentication'));
        }
    }
    
    /**
     * Check if user has access to reporting dashboard and get allowed staff IDs
     */
    private function check_access_and_get_filters()
    {
        // Everyone can access the dashboard, but data is filtered based on permissions
        $current_staff_id = get_staff_user_id();
        
        // Direct database query for permissions
        $this->db->select('*');
        $this->db->from(db_prefix() . 'reporting_dashboard_permissions');
        $this->db->where('staff_id', $current_staff_id);
        $permission_query = $this->db->get();
        
        $allowed_staff_ids = [$current_staff_id]; // default: own only
        $can_view_all = false;
        
        if ($permission_query->num_rows() > 0) {
            $permission = $permission_query->row();
            if ($permission->permission_type === 'view_all_staff') {
                $allowed_staff_ids = [];
                $can_view_all = true;
            } elseif ($permission->permission_type === 'view_specific_staff') {
                $allowed_staff_ids = json_decode($permission->allowed_staff_ids, true) ?: [];
                if (!in_array($current_staff_id, $allowed_staff_ids)) {
                    $allowed_staff_ids[] = $current_staff_id;
                }
            }
        }
        
        // Admins always can view all
        if (is_admin($current_staff_id)) {
            $allowed_staff_ids = [];
            $can_view_all = true;
        }
        
        return [
            'has_access' => true,
            'allowed_staff_ids' => $allowed_staff_ids,
            'can_view_all' => $can_view_all
        ];
    }

    public function index()
    {
        if (!is_staff_logged_in()) {
            redirect(admin_url('authentication'));
        }
        
        $data['title'] = 'Reporting Dashboard';
        
        // Get current user info
        $current_staff_id = get_staff_user_id();
        $data['user_staff_id'] = $current_staff_id;
        
        // Check permissions using direct database queries (bypassing model issues)
        $permission_type = 'view_own_only'; // default
        $allowed_staff_ids = [$current_staff_id]; // default
        
        // Check if user has custom permissions
        $this->db->select('*');
        $this->db->from(db_prefix() . 'reporting_dashboard_permissions');
        $this->db->where('staff_id', $current_staff_id);
        $permission_query = $this->db->get();
        
        if ($permission_query->num_rows() > 0) {
            $permission = $permission_query->row();
            $permission_type = $permission->permission_type;
            
            if ($permission_type === 'view_specific_staff') {
                $allowed_staff_ids = json_decode($permission->allowed_staff_ids, true) ?: [];
                // Always include current user
                if (!in_array($current_staff_id, $allowed_staff_ids)) {
                    $allowed_staff_ids[] = $current_staff_id;
                }
            } elseif ($permission_type === 'view_all_staff') {
                $allowed_staff_ids = []; // empty means all
            }
        }
        
        // Admins always have view_all_staff permission
        if (is_admin($current_staff_id)) {
            $permission_type = 'view_all_staff';
            $allowed_staff_ids = [];
        }
        
        // Get staff list based on permissions
        $this->db->select('staffid, firstname, lastname, email, admin, active');
        $this->db->from(db_prefix() . 'staff');
        $this->db->where('active', 1);
        
        if (!empty($allowed_staff_ids)) {
            // Filter to specific staff only
            $this->db->where_in('staffid', $allowed_staff_ids);
        }
        
        $staff_query = $this->db->get();
        $data['staff_list'] = $staff_query->result_array();
        
        $data['can_view_all_staff'] = ($permission_type === 'view_all_staff');
        
        // Debug logging
        log_message('debug', 'Reporting Dashboard - Current staff ID: ' . $current_staff_id);
        log_message('debug', 'Reporting Dashboard - Permission type: ' . $permission_type);
        log_message('debug', 'Reporting Dashboard - Allowed staff IDs: ' . json_encode($allowed_staff_ids));
        log_message('debug', 'Reporting Dashboard - Staff list count: ' . count($data['staff_list']));
        
        // Add permission management link for admins
        if (is_admin()) {
            $data['show_permission_link'] = true;
        }
        
        $this->load->view('dashboard', $data);
    }

    public function get_kpis()
    {
        // Check access permissions
        $access_info = $this->check_access_and_get_filters();
        if (!$access_info['has_access']) {
            header('HTTP/1.1 403 Forbidden');
            echo json_encode(['error' => 'Access denied']);
            return;
        }
        
        try {
            $filters = $this->get_filters();
            $kpis = $this->reporting_dashboard_model->get_kpis($filters);
            
            header('Content-Type: application/json');
            echo json_encode($kpis);
        } catch (Exception $e) {
            error_log('Reporting Dashboard KPI Error: ' . $e->getMessage());
            error_log('Reporting Dashboard KPI Error Trace: ' . $e->getTraceAsString());
            
            header('Content-Type: application/json');
            http_response_code(500);
            echo json_encode([
                'error' => true,
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
        } catch (Error $e) {
            error_log('Reporting Dashboard KPI Fatal Error: ' . $e->getMessage());
            error_log('Reporting Dashboard KPI Fatal Error Trace: ' . $e->getTraceAsString());
            
            header('Content-Type: application/json');
            http_response_code(500);
            echo json_encode([
                'error' => true,
                'message' => 'Fatal error: ' . $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
        }
    }

    public function get_staff_activity_chart()
    {
        // Check access permissions
        $access_info = $this->check_access_and_get_filters();
        if (!$access_info['has_access']) {
            header('HTTP/1.1 403 Forbidden');
            echo json_encode(['error' => 'Access denied']);
            return;
        }
        
        try {
            $filters = $this->get_filters();
            $data = $this->reporting_dashboard_model->get_staff_activity_chart($filters);
            header('Content-Type: application/json');
            echo json_encode($data);
        } catch (Exception $e) {
            error_log('Reporting Dashboard Staff Chart Error: ' . $e->getMessage());
            header('Content-Type: application/json');
            http_response_code(500);
            echo json_encode([
                'error' => true,
                'message' => $e->getMessage()
            ]);
        }
    }

    public function get_activity_over_time_chart()
    {
        // Check access permissions
        $access_info = $this->check_access_and_get_filters();
        if (!$access_info['has_access']) {
            header('HTTP/1.1 403 Forbidden');
            echo json_encode(['error' => 'Access denied']);
            return;
        }
        
        try {
            $filters = $this->get_filters();
            $data = $this->reporting_dashboard_model->get_activity_over_time_chart($filters);
            header('Content-Type: application/json');
            echo json_encode($data);
        } catch (Exception $e) {
            error_log('Reporting Dashboard Activity Over Time Error: ' . $e->getMessage());
            header('Content-Type: application/json');
            http_response_code(500);
            echo json_encode([
                'error' => true,
                'message' => $e->getMessage()
            ]);
        }
    }

    public function get_activity_type_pie()
    {
        // Check access permissions
        $access_info = $this->check_access_and_get_filters();
        if (!$access_info['has_access']) {
            header('HTTP/1.1 403 Forbidden');
            echo json_encode(['error' => 'Access denied']);
            return;
        }
        
        try {
            $filters = $this->get_filters();
            $data = $this->reporting_dashboard_model->get_activity_type_pie($filters);
            header('Content-Type: application/json');
            echo json_encode($data);
        } catch (Exception $e) {
            error_log('Reporting Dashboard Activity Type Pie Error: ' . $e->getMessage());
            header('Content-Type: application/json');
            http_response_code(500);
            echo json_encode([
                'error' => true,
                'message' => $e->getMessage()
            ]);
        }
    }

    public function get_activity_table()
    {
        // DEBUG: Log all received data
        error_log('POST data: ' . print_r($_POST, true));
        error_log('GET data: ' . print_r($_GET, true));
        
        // Check access permissions
        $access_info = $this->check_access_and_get_filters();
        if (!$access_info['has_access']) {
            header('HTTP/1.1 403 Forbidden');
            echo json_encode(['error' => 'Access denied']);
            return;
        }
        
        try {
            $filters = $this->get_filters();
            
            // Get pagination parameters
            $start = (int)($this->input->post('start') ?: $this->input->get('start') ?: 0);
            $length = (int)($this->input->post('length') ?: $this->input->get('length') ?: 25);
            $draw = (int)($this->input->post('draw') ?: $this->input->get('draw') ?: 1);
            
            // Handle "All" entries (DataTables sends -1 for "All")
            if ($length == -1) {
                $length = 1000; // Set a reasonable maximum
            } elseif ($length <= 0) {
                $length = 25; // Default
            }
            
            error_log("Activity Table Controller - Start: $start, Length: $length, Draw: $draw");
            
            // Add pagination parameters to filters
            $filters['start'] = $start;
            $filters['length'] = $length;
            $filters['draw'] = $draw;
            
            // Load enhanced model if available, fallback to standard
            if (file_exists(APPPATH . 'modules/reporting_dashboard/models/Enhanced_activity_model.php')) {
                $this->load->model('reporting_dashboard/enhanced_activity_model');
                $data = $this->enhanced_activity_model->get_enhanced_activity_table($filters);
                error_log('Using enhanced activity table model');
            } else {
                $data = $this->reporting_dashboard_model->get_activity_table($filters);
                error_log('Using standard activity table model');
            }
            
            // Ensure DataTables format is returned
            if (!isset($data['draw'])) {
                $data['draw'] = $draw;
            }
            if (!isset($data['recordsTotal'])) {
                $data['recordsTotal'] = count($data['data'] ?? []);
            }
            if (!isset($data['recordsFiltered'])) {
                $data['recordsFiltered'] = $data['recordsTotal'];
            }
            
            error_log('Returning data: ' . print_r($data, true));
            
            header('Content-Type: application/json');
            echo json_encode($data);
        } catch (Exception $e) {
            error_log('Reporting Dashboard Activity Table Error: ' . $e->getMessage());
            error_log('Error trace: ' . $e->getTraceAsString());
            header('Content-Type: application/json');
            http_response_code(500);
            echo json_encode([
                'error' => true,
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
        }
    }

    /**
     * Enhanced activity table with modern features
     */
    public function get_enhanced_activity_table()
    {
        $access_info = $this->check_access_and_get_filters();
        if (!$access_info['has_access']) {
            header('HTTP/1.1 403 Forbidden');
            echo json_encode(['error' => 'Access denied']);
            return;
        }
        
        try {
            $this->load->model('reporting_dashboard/enhanced_activity_model');
            $filters = $this->get_filters();
            $data = $this->enhanced_activity_model->get_enhanced_activity_table($filters);
            
            header('Content-Type: application/json');
            echo json_encode($data);
        } catch (Exception $e) {
            error_log('Enhanced Activity Table Error: ' . $e->getMessage());
            header('Content-Type: application/json');
            http_response_code(500);
            echo json_encode([
                'error' => true,
                'message' => $e->getMessage()
            ]);
        }
    }

    public function get_followup_types_chart()
    {
        // Check access permissions
        $access_info = $this->check_access_and_get_filters();
        if (!$access_info['has_access']) {
            header('HTTP/1.1 403 Forbidden');
            echo json_encode(['error' => 'Access denied']);
            return;
        }
        
        try {
            $filters = $this->get_filters();
            $data = $this->reporting_dashboard_model->get_followup_types_chart($filters);
            header('Content-Type: application/json');
            echo json_encode($data);
        } catch (Exception $e) {
            error_log('Reporting Dashboard Follow-up Types Error: ' . $e->getMessage());
            header('Content-Type: application/json');
            http_response_code(500);
            echo json_encode([
                'error' => true,
                'message' => $e->getMessage()
            ]);
        }
    }

    public function get_reminders()
    {
        // Check access permissions
        $access_info = $this->check_access_and_get_filters();
        if (!$access_info['has_access']) {
            header('HTTP/1.1 403 Forbidden');
            echo json_encode(['error' => 'Access denied']);
            return;
        }
        
        try {
            $filters = $this->get_filters();
            
            // Use direct database query to avoid model loading issues
            $data = $this->get_reminders_direct($filters);
            
            header('Content-Type: application/json');
            echo json_encode($data);
        } catch (Exception $e) {
            error_log('Reporting Dashboard Reminders Error: ' . $e->getMessage());
            header('Content-Type: application/json');
            http_response_code(500);
            echo json_encode([
                'error' => true,
                'message' => $e->getMessage(),
                'todays' => [],
                'due' => [],
                'upcoming' => []
            ]);
        }
    }
    
    private function get_reminders_direct($filters)
    {
        // Get staff filter for permission-based access
        $access_info = $this->check_access_and_get_filters();
        $allowed_staff_ids = $access_info['allowed_staff_ids'];
        $can_view_all = $access_info['can_view_all'];

        // Simple query to get basic reminders
        $this->db->select('*');
        $this->db->from(db_prefix() . 'reminders');

        // Exclude reminders that are marked as read/notified (isnotified = 1 means "Yes" (read/completed), so exclude those strictly)
        $this->db->where('(isnotified IS NULL OR isnotified = 0)');

        // Apply date filters if provided - make them more inclusive for reminders
        // For reminders, we want to include a broader date range to capture today's and upcoming reminders
        if (!empty($filters['from'])) {
            // Include reminders from 30 days before the start date to ensure we don't miss any
            $extended_from = date('Y-m-d', strtotime($filters['from'] . ' -30 days'));
            $this->db->where('DATE(date) >=', $extended_from);
            error_log('Extended reminder date filter from: ' . $extended_from . ' (original: ' . $filters['from'] . ')');
        }
        if (!empty($filters['to'])) {
            // Include reminders up to 90 days after the end date to capture upcoming reminders
            $extended_to = date('Y-m-d', strtotime($filters['to'] . ' +90 days'));
            $this->db->where('DATE(date) <=', $extended_to);
            error_log('Extended reminder date filter to: ' . $extended_to . ' (original: ' . $filters['to'] . ')');
        }

        // Apply staff filter (this already includes permission-based filtering from get_filters())
        if (!empty($filters['staff']) && is_array($filters['staff'])) {
            $this->db->where_in('staff', $filters['staff']);
            error_log('Applying staff filter to reminders: ' . implode(',', $filters['staff']));
        } elseif (!$can_view_all && !empty($allowed_staff_ids)) {
            // Fallback: if no specific staff selected but user has limited permissions
            $this->db->where_in('staff', $allowed_staff_ids);
            error_log('Applying permission-based staff filter: ' . implode(',', $allowed_staff_ids));
        }
        // If can_view_all is true and no specific staff filter, don't apply any staff restriction

        $this->db->order_by('date', 'ASC');

        // Debug: Log the SQL query
        $sql = $this->db->get_compiled_select();
        error_log('Reminders Query: ' . $sql);

        // Reset query and execute
        $this->db->select('*');
        $this->db->from(db_prefix() . 'reminders');

        // Apply the same extended date filters as used in the debug query
        if (!empty($filters['from'])) {
            $extended_from = date('Y-m-d', strtotime($filters['from'] . ' -30 days'));
            $this->db->where('DATE(date) >=', $extended_from);
        }
        if (!empty($filters['to'])) {
            $extended_to = date('Y-m-d', strtotime($filters['to'] . ' +90 days'));
            $this->db->where('DATE(date) <=', $extended_to);
        }
        if (!empty($filters['staff']) && is_array($filters['staff'])) {
            $this->db->where_in('staff', $filters['staff']);
        } elseif (!$can_view_all && !empty($allowed_staff_ids)) {
            $this->db->where_in('staff', $allowed_staff_ids);
        }
        $this->db->order_by('date', 'ASC');

        $query = $this->db->get();
        $reminders = $query->result_array();
        error_log('Found ' . count($reminders) . ' reminders after SQL filtering');

        // Post-process: strictly exclude any isnotified=1 (notified/read) and lost/junk/closed/cancelled/dead leads
        $filtered_reminders = [];
        foreach ($reminders as &$reminder) {
            // Exclude if reminder is notified/read (should not be present, but double check)
            if (isset($reminder['isnotified']) && $reminder['isnotified'] == 1) {
                error_log("Excluding reminder ID {$reminder['id']} as isnotified=1");
                continue;
            }
            $should_include = true;
            if ($reminder['rel_type'] === 'lead' && !empty($reminder['rel_id'])) {
                $this->db->select('status');
                $this->db->from(db_prefix() . 'leads');
                $this->db->where('id', $reminder['rel_id']);
                $lead_query = $this->db->get();
                if ($lead_query->num_rows() > 0) {
                    $lead = $lead_query->row();
                    if (in_array(strtolower($lead->status), ['lost', 'junk', 'closed', 'cancelled', 'dead'])) {
                        $should_include = false;
                        error_log("Excluding reminder for lead {$reminder['rel_id']} with status: {$lead->status}");
                    }
                } else {
                    $should_include = false;
                    error_log("Excluding reminder for non-existent lead {$reminder['rel_id']}");
                }
            }
            if (!$should_include) {
                continue;
            }
            // Get staff name
            $this->db->select('firstname, lastname');
            $this->db->from(db_prefix() . 'staff');
            $this->db->where('staffid', $reminder['staff']);
            $staff_query = $this->db->get();
            if ($staff_query->num_rows() > 0) {
                $staff = $staff_query->row();
                $reminder['staff_name'] = trim($staff->firstname . ' ' . $staff->lastname);
            } else {
                $reminder['staff_name'] = 'Unknown Staff';
            }
            $reminder['related_name'] = $this->get_related_name($reminder['rel_type'], $reminder['rel_id']);
            $reminder['related_email'] = $this->get_related_email($reminder['rel_type'], $reminder['rel_id']);
            $filtered_reminders[] = $reminder;
        }
        $reminders = $filtered_reminders;
        error_log('After post-processing: ' . count($reminders) . ' reminders remaining');
        if (count($reminders) > 0) {
            $sample = $reminders[0];
            error_log('Sample reminder: ID=' . $sample['id'] . ', rel_type=' . $sample['rel_type'] . ', rel_id=' . $sample['rel_id'] . ', isnotified=' . $sample['isnotified'] . ', date=' . $sample['date']);
        }
        
        // Categorize reminders with more robust date handling
        $today = date('Y-m-d');
        $result = [
            'todays' => [],
            'due' => [],
            'upcoming' => []
        ];

        foreach ($reminders as $reminder) {
            // Extract just the date part in case the reminder date includes time
            $reminder_date = date('Y-m-d', strtotime($reminder['date']));

            // Debug: Log reminder categorization
            error_log("Categorizing reminder ID {$reminder['id']}: date={$reminder_date}, today={$today}");

            if ($reminder_date === $today) {
                $result['todays'][] = $reminder;
                error_log("Added to todays: reminder ID {$reminder['id']}");
            } elseif ($reminder_date < $today) {
                // Only add to due if it's actually missed (past date)
                $result['due'][] = $reminder;
                error_log("Added to due: reminder ID {$reminder['id']}");
            } else {
                // Future reminders go to upcoming
                $result['upcoming'][] = $reminder;
                error_log("Added to upcoming: reminder ID {$reminder['id']}");
            }
        }
        
        // Log the filtering results
        error_log('Reminders filtered - Total: ' . count($reminders) . 
                 ', Today: ' . count($result['todays']) . 
                 ', Due (missed): ' . count($result['due']) . 
                 ', Upcoming: ' . count($result['upcoming']));
        
        return $result;
    }
    
    private function get_related_name($rel_type, $rel_id)
    {
        if (empty($rel_type) || empty($rel_id)) {
            return $rel_type . ' #' . $rel_id;
        }
        
        switch ($rel_type) {
            case 'lead':
                $this->db->select('name');
                $this->db->from(db_prefix() . 'leads');
                $this->db->where('id', $rel_id);
                $query = $this->db->get();
                if ($query->num_rows() > 0) {
                    return $query->row()->name;
                }
                break;
                
            case 'customer':
                $this->db->select('company');
                $this->db->from(db_prefix() . 'clients');
                $this->db->where('userid', $rel_id);
                $query = $this->db->get();
                if ($query->num_rows() > 0) {
                    return $query->row()->company;
                }
                break;
                
            case 'project':
                $this->db->select('name');
                $this->db->from(db_prefix() . 'projects');
                $this->db->where('id', $rel_id);
                $query = $this->db->get();
                if ($query->num_rows() > 0) {
                    return $query->row()->name;
                }
                break;
                
            case 'proposal':
                $this->db->select('subject');
                $this->db->from(db_prefix() . 'proposals');
                $this->db->where('id', $rel_id);
                $query = $this->db->get();
                if ($query->num_rows() > 0) {
                    return $query->row()->subject;
                }
                break;
                
            case 'invoice':
                $this->db->select('number');
                $this->db->from(db_prefix() . 'invoices');
                $this->db->where('id', $rel_id);
                $query = $this->db->get();
                if ($query->num_rows() > 0) {
                    return 'Invoice #' . $query->row()->number;
                }
                break;
                
            case 'estimate':
                $this->db->select('number');
                $this->db->from(db_prefix() . 'estimates');
                $this->db->where('id', $rel_id);
                $query = $this->db->get();
                if ($query->num_rows() > 0) {
                    return 'Estimate #' . $query->row()->number;
                }
                break;
                
            case 'contract':
                $this->db->select('subject');
                $this->db->from(db_prefix() . 'contracts');
                $this->db->where('id', $rel_id);
                $query = $this->db->get();
                if ($query->num_rows() > 0) {
                    return $query->row()->subject;
                }
                break;
        }
        
        return $rel_type . ' #' . $rel_id;
    }
    
    private function get_related_email($rel_type, $rel_id)
    {
        if (empty($rel_type) || empty($rel_id)) {
            return null;
        }
        
        switch ($rel_type) {
            case 'lead':
                $this->db->select('email');
                $this->db->from(db_prefix() . 'leads');
                $this->db->where('id', $rel_id);
                $query = $this->db->get();
                if ($query->num_rows() > 0) {
                    return $query->row()->email;
                }
                break;
                
            case 'customer':
                $this->db->select('email');
                $this->db->from(db_prefix() . 'clients');
                $this->db->where('userid', $rel_id);
                $query = $this->db->get();
                if ($query->num_rows() > 0) {
                    return $query->row()->email;
                }
                break;
        }
        
        return null;
    }

    private function get_filters()
    {
        // Accept both POST and GET for flexibility
        $input = array_merge($_GET, $_POST);
        
        $filters = [
            'from' => $input['from'] ?? '',
            'to' => $input['to'] ?? '',
            'staff' => $input['staff'] ?? [],
            'activity_type' => $input['activity_type'] ?? [],
        ];
        
        // Ensure staff is always an array
        if (!is_array($filters['staff'])) {
            if (empty($filters['staff'])) {
                $filters['staff'] = [];
            } else {
                // Convert string to array (in case it's a single ID)
                $filters['staff'] = [$filters['staff']];
            }
        }
        
        // Ensure activity_type is always an array
        if (!is_array($filters['activity_type'])) {
            if (empty($filters['activity_type'])) {
                $filters['activity_type'] = [];
            } else {
                $filters['activity_type'] = [$filters['activity_type']];
            }
        }
        
        // Apply permission-based filtering
        $current_staff_id = get_staff_user_id();
        $allowed_staff_ids = $this->reporting_dashboard_permissions_model->get_allowed_staff_ids($current_staff_id);
        
        // Debug logging
        error_log('Current staff ID: ' . $current_staff_id);
        error_log('Allowed staff IDs: ' . print_r($allowed_staff_ids, true));
        error_log('Requested staff filter: ' . print_r($filters['staff'], true));
        
        // If user has limited permissions, override the staff filter
        if (!empty($allowed_staff_ids)) {
            // If user requested specific staff, filter to only those they can see
            if (!empty($filters['staff'])) {
                $filters['staff'] = array_intersect($filters['staff'], $allowed_staff_ids);
            } else {
                // If no specific staff requested, limit to allowed staff
                $filters['staff'] = $allowed_staff_ids;
            }
        } else {
            // If no permission restrictions (superadmin), allow all staff
            error_log('No permission restrictions - allowing all staff');
            // Keep the original staff filter or leave empty for all staff
        }
        
        error_log('Final staff filter: ' . print_r($filters['staff'], true));
        
        return $filters;
    }

    public function download_pdf()
    {
        // Check access permissions
        $access_info = $this->check_access_and_get_filters();
        if (!$access_info['has_access']) {
            header('HTTP/1.1 403 Forbidden');
            show_error('Access denied');
            return;
        }
        
        try {
            // Clear any output buffer and suppress warnings that could break PDF generation
            ob_clean();
            $old_error_reporting = error_reporting(E_ALL & ~E_WARNING);
            
            $filters = $this->get_filters();
            
            // Load required dependencies first
            if (!class_exists('App_pdf')) {
                require_once(APPPATH . 'libraries/pdf/App_pdf.php');
            }
            
            // Now load our custom PDF class
            $pdfClassPath = FCPATH . 'modules/reporting_dashboard/libraries/Reporting_dashboard_pdf.php';
            if (!file_exists($pdfClassPath)) {
                error_reporting($old_error_reporting);
                show_error('PDF class file not found at: ' . $pdfClassPath);
                return;
            }
            
            require_once($pdfClassPath);
            
            // Check if class exists after include
            if (!class_exists('Reporting_dashboard_pdf')) {
                error_reporting($old_error_reporting);
                show_error('Reporting_dashboard_pdf class not found after include');
                return;
            }
            
            // Create instance manually
            $pdf = new Reporting_dashboard_pdf($filters);
            
            // Generate the PDF
            $result = $pdf->prepare();
            
            // Set filename
            $filename = 'Staff_Activity_Report_' . date('Y-m-d_H-i-s') . '.pdf';
            
            // Restore error reporting
            error_reporting($old_error_reporting);
            
            // Output the PDF for download
            $pdf->Output($filename, 'D');
            
        } catch (Exception $e) {
            error_reporting($old_error_reporting);
            show_error('Error generating PDF report: ' . $e->getMessage());
        }
    }

    /**
     * Permissions management page (Admin only)
     */
    public function permissions()
    {
        // Only admin can access permission management
        if (!is_admin()) {
            access_denied('Reporting Dashboard Permissions');
        }

        $data['title'] = 'Reporting Dashboard - Permissions Management';
        $data['staff_permissions'] = $this->reporting_dashboard_permissions_model->get_all_staff_permissions();
        $data['reporting_dashboard_permissions_model'] = $this->reporting_dashboard_permissions_model;
        $this->load->view('permissions_management', $data);
    }

    /**
     * Update staff permission via AJAX
     */
    public function update_permission()
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        // Only admin can update permissions
        if (!is_admin()) {
            echo json_encode([
                'success' => false,
                'message' => 'Access denied'
            ]);
            return;
        }

        $staff_id = $this->input->post('staff_id');
        $permission_type = $this->input->post('permission_type');
        $allowed_staff_ids = $this->input->post('allowed_staff_ids');
        $current_user_id = get_staff_user_id();

        // Validate inputs
        if (empty($staff_id) || empty($permission_type)) {
            echo json_encode([
                'success' => false,
                'message' => 'Invalid parameters'
            ]);
            return;
        }

        // Validate permission type
        if (!in_array($permission_type, ['view_all_staff', 'view_own_only', 'view_specific_staff'])) {
            echo json_encode([
                'success' => false,
                'message' => 'Invalid permission type'
            ]);
            return;
        }

        // For view_specific_staff, validate allowed_staff_ids
        $allowed_staff_array = [];
        if ($permission_type === 'view_specific_staff') {
            if (empty($allowed_staff_ids)) {
                echo json_encode([
                    'success' => false,
                    'message' => 'Please select at least one staff member for specific staff permission'
                ]);
                return;
            }
            
            $allowed_staff_array = is_array($allowed_staff_ids) ? $allowed_staff_ids : explode(',', $allowed_staff_ids);
            $allowed_staff_array = array_filter(array_map('intval', $allowed_staff_array));
            
            if (empty($allowed_staff_array)) {
                echo json_encode([
                    'success' => false,
                    'message' => 'Invalid staff selection'
                ]);
                return;
            }
        }

        // Update permission
        $success = $this->reporting_dashboard_permissions_model->set_staff_permission(
            $staff_id,
            $permission_type,
            $current_user_id,
            $allowed_staff_array
        );

        if ($success) {
            echo json_encode([
                'success' => true,
                'message' => 'Permission updated successfully'
            ]);
        } else {
            echo json_encode([
                'success' => false,
                'message' => 'Failed to update permission'
            ]);
        }
    }

    /**
     * Remove staff permission (revert to default)
     */
    public function remove_permission()
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        if (!is_admin()) {
            echo json_encode([
                'success' => false,
                'message' => 'Access denied'
            ]);
            return;
        }

        $staff_id = $this->input->post('staff_id');

        if (empty($staff_id)) {
            echo json_encode([
                'success' => false,
                'message' => 'Invalid staff ID'
            ]);
            return;
        }

        $success = $this->reporting_dashboard_permissions_model->remove_staff_permission($staff_id);

        if ($success) {
            echo json_encode([
                'success' => true,
                'message' => 'Permission reset to default successfully'
            ]);
        } else {
            echo json_encode([
                'success' => false,
                'message' => 'Failed to reset permission'
            ]);
        }
    }

    /**
     * Get available staff for selection (AJAX)
     */
    public function get_available_staff()
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        if (!is_admin()) {
            echo json_encode([
                'success' => false,
                'message' => 'Access denied'
            ]);
            return;
        }

        $exclude_staff_id = $this->input->post('exclude_staff_id');
        $staff_list = $this->reporting_dashboard_permissions_model->get_available_staff_for_permission($exclude_staff_id);

        echo json_encode([
            'success' => true,
            'staff_list' => $staff_list
        ]);
    }

    /**
     * Bulk update permissions
     */
    public function bulk_update()
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        if (!is_admin()) {
            echo json_encode([
                'success' => false,
                'message' => 'Access denied'
            ]);
            return;
        }

        $permissions = $this->input->post('permissions');
        $current_user_id = get_staff_user_id();

        if (empty($permissions) || !is_array($permissions)) {
            echo json_encode([
                'success' => false,
                'message' => 'No permissions provided'
            ]);
            return;
        }

        $updated_count = 0;
        $errors = [];

        foreach ($permissions as $staff_id => $permission_data) {
            $permission_type = $permission_data['type'] ?? '';
            $allowed_staff_ids = $permission_data['allowed_staff_ids'] ?? [];
            
            if (in_array($permission_type, ['view_all_staff', 'view_own_only', 'view_specific_staff'])) {
                $success = $this->reporting_dashboard_permissions_model->set_staff_permission(
                    $staff_id,
                    $permission_type,
                    $current_user_id,
                    $allowed_staff_ids
                );
                
                if ($success) {
                    $updated_count++;
                } else {
                    $errors[] = "Failed to update permission for staff ID: $staff_id";
                }
            }
        }

        echo json_encode([
            'success' => empty($errors),
            'message' => empty($errors) ? 
                "Successfully updated $updated_count permissions" :
                "Updated $updated_count permissions with " . count($errors) . " errors",
            'errors' => $errors
        ]);
    }

    /**
     * Get dealer list by interest status
     */
    public function get_dealer_list()
    {
        // Check access permissions
        $access_info = $this->check_access_and_get_filters();
        if (!$access_info['has_access']) {
            header('HTTP/1.1 403 Forbidden');
            echo json_encode(['error' => 'Access denied']);
            return;
        }

        $interest_status = $this->input->get('interest_status');
        if (!in_array($interest_status, ['interested', 'not_interested'])) {
            header('HTTP/1.1 400 Bad Request');
            echo json_encode(['error' => 'Invalid interest status']);
            return;
        }

        try {
            $filters = $this->get_filters();
            $dealers = $this->reporting_dashboard_model->get_dealer_list_by_interest($interest_status, $filters);
            
            header('Content-Type: application/json');
            echo json_encode([
                'success' => true,
                'data' => $dealers,
                'count' => count($dealers)
            ]);
        } catch (Exception $e) {
            error_log('Reporting Dashboard Dealer List Error: ' . $e->getMessage());
            
            header('Content-Type: application/json');
            http_response_code(500);
            echo json_encode([
                'error' => true,
                'message' => $e->getMessage()
            ]);
        }
    }

    /**
     * Download dealer list as CSV
     */

    public function download_dealers_list()
    {
        // Log the request for debugging
        error_log('Download dealers list called');
        
        // Check access permissions
        $access_info = $this->check_access_and_get_filters();
        if (!$access_info['has_access']) {
            error_log('Download access denied');
            header('HTTP/1.1 403 Forbidden');
            echo 'Access denied';
            return;
        }

        $interest_status = $this->input->get('interest_status');
        error_log('Interest status: ' . ($interest_status ?: 'NULL'));
        
        if (!in_array($interest_status, ['interested', 'not_interested'])) {
            error_log('Invalid interest status: ' . $interest_status);
            header('HTTP/1.1 400 Bad Request');
            echo 'Invalid interest status';
            return;
        }

        try {
            $filters = $this->get_filters();
            $dealers = $this->reporting_dashboard_model->get_dealer_list_by_interest($interest_status, $filters);
            
            // Clear any existing output
            if (ob_get_level()) {
                ob_clean();
            }
            
            // Set proper CSV headers
            $filename = $interest_status . '_dealers_' . date('Y-m-d') . '.csv';
            header('Content-Type: text/csv; charset=UTF-8');
            header('Content-Disposition: attachment; filename="' . $filename . '"');
            header('Cache-Control: no-cache, must-revalidate');
            header('Expires: Sat, 26 Jul 1997 05:00:00 GMT');
            header('Pragma: no-cache');
            
            // Output UTF-8 BOM for Excel compatibility
            echo "\xEF\xBB\xBF";
            
            // Open output stream
            $output = fopen('php://output', 'w');
            
            // Write CSV header
            fputcsv($output, [
                'ID',
                'Name',
                'Email',
                'Phone',
                'Company',
                'Lead Date',
                'Interest Date',
                'Interest Status'
            ]);
            
            // Write dealer data
            foreach ($dealers as $dealer) {
                fputcsv($output, [
                    $dealer['id'],
                    $dealer['name'] ?: 'N/A',
                    $dealer['email'] ?: 'N/A',
                    $dealer['phonenumber'] ?: 'N/A',
                    $dealer['company'] ?: 'N/A',
                    $dealer['dateadded'],
                    $dealer['interest_date'],
                    ucfirst(str_replace('_', ' ', $interest_status))
                ]);
            }
            
            fclose($output);
            exit(); // Ensure no additional output
            
        } catch (Exception $e) {
            error_log('Reporting Dashboard Download Error: ' . $e->getMessage());
            
            // Clear any existing output
            if (ob_get_level()) {
                ob_clean();
            }
            
            header('Content-Type: text/plain');
            http_response_code(500);
            echo 'Error generating download: ' . $e->getMessage();
            exit();
        }
    }
}

