<?php defined('BASEPATH') or exit('No direct script access allowed');

class Reporting_dashboard extends AdminController
{
    public function __construct()
    {
        parent::__construct();
        
        // Load models using CodeIgniter approach
        $this->load->model('reporting_dashboard/reporting_dashboard_model');
        $this->load->model('reporting_dashboard/reporting_dashboard_permissions_model');
        $this->load->model('staff_model');
        
        // Check if user is logged in
        if (!is_staff_logged_in()) {
            redirect(admin_url('authentication'));
        }
    }
    
    /**
     * Check if user has access to reporting dashboard and get allowed staff IDs
     */
    private function check_access_and_get_filters()
    {
        // Everyone can access the dashboard, but data is filtered based on permissions
        $current_staff_id = get_staff_user_id();
        
        // Direct database query for permissions
        $this->db->select('*');
        $this->db->from(db_prefix() . 'reporting_dashboard_permissions');
        $this->db->where('staff_id', $current_staff_id);
        $permission_query = $this->db->get();
        
        $allowed_staff_ids = [$current_staff_id]; // default: own only
        $can_view_all = false;
        
        if ($permission_query->num_rows() > 0) {
            $permission = $permission_query->row();
            if ($permission->permission_type === 'view_all_staff') {
                $allowed_staff_ids = [];
                $can_view_all = true;
            } elseif ($permission->permission_type === 'view_specific_staff') {
                $allowed_staff_ids = json_decode($permission->allowed_staff_ids, true) ?: [];
                if (!in_array($current_staff_id, $allowed_staff_ids)) {
                    $allowed_staff_ids[] = $current_staff_id;
                }
            }
        }
        
        // Admins always can view all
        if (is_admin($current_staff_id)) {
            $allowed_staff_ids = [];
            $can_view_all = true;
        }
        
        return [
            'has_access' => true,
            'allowed_staff_ids' => $allowed_staff_ids,
            'can_view_all' => $can_view_all
        ];
    }

    public function index()
    {
        if (!is_staff_logged_in()) {
            redirect(admin_url('authentication'));
        }
        
        $data['title'] = 'Reporting Dashboard';
        
        // Get current user info
        $current_staff_id = get_staff_user_id();
        $data['user_staff_id'] = $current_staff_id;
        
        // Check permissions using direct database queries (bypassing model issues)
        $permission_type = 'view_own_only'; // default
        $allowed_staff_ids = [$current_staff_id]; // default
        
        // Check if user has custom permissions
        $this->db->select('*');
        $this->db->from(db_prefix() . 'reporting_dashboard_permissions');
        $this->db->where('staff_id', $current_staff_id);
        $permission_query = $this->db->get();
        
        $can_generate_reports = false;
        
        if ($permission_query->num_rows() > 0) {
            $permission = $permission_query->row();
            $permission_type = $permission->permission_type;
            $can_generate_reports = (bool)($permission->can_generate_reports ?? 0);
            
            if ($permission_type === 'view_specific_staff') {
                $allowed_staff_ids = json_decode($permission->allowed_staff_ids, true) ?: [];
                // Always include current user
                if (!in_array($current_staff_id, $allowed_staff_ids)) {
                    $allowed_staff_ids[] = $current_staff_id;
                }
            } elseif ($permission_type === 'view_all_staff') {
                $allowed_staff_ids = []; // empty means all
            }
        }
        
        // Admins always have view_all_staff and report generation permission
        if (is_admin($current_staff_id)) {
            $permission_type = 'view_all_staff';
            $allowed_staff_ids = [];
            $can_generate_reports = true;
        }
        
        // Get staff list based on permissions
        $this->db->select('staffid, firstname, lastname, email, admin, active');
        $this->db->from(db_prefix() . 'staff');
        $this->db->where('active', 1);
        
        if (!empty($allowed_staff_ids)) {
            // Filter to specific staff only
            $this->db->where_in('staffid', $allowed_staff_ids);
        }
        
        $staff_query = $this->db->get();
        $data['staff_list'] = $staff_query->result_array();
        
        $data['can_view_all_staff'] = ($permission_type === 'view_all_staff');
        $data['can_generate_reports'] = $can_generate_reports;
        
        // Debug logging
        log_message('debug', 'Reporting Dashboard - Current staff ID: ' . $current_staff_id);
        log_message('debug', 'Reporting Dashboard - Permission type: ' . $permission_type);
        log_message('debug', 'Reporting Dashboard - Allowed staff IDs: ' . json_encode($allowed_staff_ids));
        log_message('debug', 'Reporting Dashboard - Staff list count: ' . count($data['staff_list']));
        
        // Add permission management link for admins
        if (is_admin()) {
            $data['show_permission_link'] = true;
        }
        
        $this->load->view('dashboard', $data);
    }

    public function get_kpis()
    {
        // Check access permissions
        $access_info = $this->check_access_and_get_filters();
        if (!$access_info['has_access']) {
            header('HTTP/1.1 403 Forbidden');
            echo json_encode(['error' => 'Access denied']);
            return;
        }
        
        try {
            $filters = $this->get_filters();
            $kpis = $this->reporting_dashboard_model->get_kpis($filters);
            
            header('Content-Type: application/json');
            echo json_encode($kpis);
        } catch (Exception $e) {
            error_log('Reporting Dashboard KPI Error: ' . $e->getMessage());
            error_log('Reporting Dashboard KPI Error Trace: ' . $e->getTraceAsString());
            
            header('Content-Type: application/json');
            http_response_code(500);
            echo json_encode([
                'error' => true,
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
        } catch (Error $e) {
            error_log('Reporting Dashboard KPI Fatal Error: ' . $e->getMessage());
            error_log('Reporting Dashboard KPI Fatal Error Trace: ' . $e->getTraceAsString());
            
            header('Content-Type: application/json');
            http_response_code(500);
            echo json_encode([
                'error' => true,
                'message' => 'Fatal error: ' . $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
        }
    }

    public function get_staff_activity_chart()
    {
        // Check access permissions
        $access_info = $this->check_access_and_get_filters();
        if (!$access_info['has_access']) {
            header('HTTP/1.1 403 Forbidden');
            echo json_encode(['error' => 'Access denied']);
            return;
        }
        
        try {
            $filters = $this->get_filters();
            $data = $this->reporting_dashboard_model->get_staff_activity_chart($filters);
            header('Content-Type: application/json');
            echo json_encode($data);
        } catch (Exception $e) {
            error_log('Reporting Dashboard Staff Chart Error: ' . $e->getMessage());
            header('Content-Type: application/json');
            http_response_code(500);
            echo json_encode([
                'error' => true,
                'message' => $e->getMessage()
            ]);
        }
    }

    public function get_activity_over_time_chart()
    {
        // Check access permissions
        $access_info = $this->check_access_and_get_filters();
        if (!$access_info['has_access']) {
            header('HTTP/1.1 403 Forbidden');
            echo json_encode(['error' => 'Access denied']);
            return;
        }
        
        try {
            $filters = $this->get_filters();
            $data = $this->reporting_dashboard_model->get_activity_over_time_chart($filters);
            header('Content-Type: application/json');
            echo json_encode($data);
        } catch (Exception $e) {
            error_log('Reporting Dashboard Activity Over Time Error: ' . $e->getMessage());
            header('Content-Type: application/json');
            http_response_code(500);
            echo json_encode([
                'error' => true,
                'message' => $e->getMessage()
            ]);
        }
    }

    public function get_activity_type_pie()
    {
        // Check access permissions
        $access_info = $this->check_access_and_get_filters();
        if (!$access_info['has_access']) {
            header('HTTP/1.1 403 Forbidden');
            echo json_encode(['error' => 'Access denied']);
            return;
        }
        
        try {
            $filters = $this->get_filters();
            $data = $this->reporting_dashboard_model->get_activity_type_pie($filters);
            header('Content-Type: application/json');
            echo json_encode($data);
        } catch (Exception $e) {
            error_log('Reporting Dashboard Activity Type Pie Error: ' . $e->getMessage());
            header('Content-Type: application/json');
            http_response_code(500);
            echo json_encode([
                'error' => true,
                'message' => $e->getMessage()
            ]);
        }
    }

    public function get_activity_table()
    {
        // DEBUG: Log all received data
        error_log('POST data: ' . print_r($_POST, true));
        error_log('GET data: ' . print_r($_GET, true));
        
        // Check access permissions
        $access_info = $this->check_access_and_get_filters();
        if (!$access_info['has_access']) {
            header('HTTP/1.1 403 Forbidden');
            echo json_encode(['error' => 'Access denied']);
            return;
        }
        
        try {
            $filters = $this->get_filters();
            
            // Get pagination parameters
            $start = (int)($this->input->post('start') ?: $this->input->get('start') ?: 0);
            $length = (int)($this->input->post('length') ?: $this->input->get('length') ?: 25);
            $draw = (int)($this->input->post('draw') ?: $this->input->get('draw') ?: 1);
            
            // Handle "All" entries (DataTables sends -1 for "All")
            if ($length == -1) {
                $length = 1000; // Set a reasonable maximum
            } elseif ($length <= 0) {
                $length = 25; // Default
            }
            
            error_log("Activity Table Controller - Start: $start, Length: $length, Draw: $draw");
            
            // Add pagination parameters to filters
            $filters['start'] = $start;
            $filters['length'] = $length;
            $filters['draw'] = $draw;
            
            // Load enhanced model if available, fallback to standard
            if (file_exists(APPPATH . 'modules/reporting_dashboard/models/Enhanced_activity_model.php')) {
                $this->load->model('reporting_dashboard/enhanced_activity_model');
                $data = $this->enhanced_activity_model->get_enhanced_activity_table($filters);
                error_log('Using enhanced activity table model');
            } else {
                $data = $this->reporting_dashboard_model->get_activity_table($filters);
                error_log('Using standard activity table model');
            }
            
            // Ensure DataTables format is returned
            if (!isset($data['draw'])) {
                $data['draw'] = $draw;
            }
            if (!isset($data['recordsTotal'])) {
                $data['recordsTotal'] = count($data['data'] ?? []);
            }
            if (!isset($data['recordsFiltered'])) {
                $data['recordsFiltered'] = $data['recordsTotal'];
            }
            
            error_log('Returning data: ' . print_r($data, true));
            
            header('Content-Type: application/json');
            echo json_encode($data);
        } catch (Exception $e) {
            error_log('Reporting Dashboard Activity Table Error: ' . $e->getMessage());
            error_log('Error trace: ' . $e->getTraceAsString());
            header('Content-Type: application/json');
            http_response_code(500);
            echo json_encode([
                'error' => true,
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
        }
    }

    /**
     * Enhanced activity table with modern features
     */
    public function get_enhanced_activity_table()
    {
        $access_info = $this->check_access_and_get_filters();
        if (!$access_info['has_access']) {
            header('HTTP/1.1 403 Forbidden');
            echo json_encode(['error' => 'Access denied']);
            return;
        }
        
        try {
            $this->load->model('reporting_dashboard/enhanced_activity_model');
            $filters = $this->get_filters();
            $data = $this->enhanced_activity_model->get_enhanced_activity_table($filters);
            
            header('Content-Type: application/json');
            echo json_encode($data);
        } catch (Exception $e) {
            error_log('Enhanced Activity Table Error: ' . $e->getMessage());
            header('Content-Type: application/json');
            http_response_code(500);
            echo json_encode([
                'error' => true,
                'message' => $e->getMessage()
            ]);
        }
    }

    public function get_followup_types_chart()
    {
        // Check access permissions
        $access_info = $this->check_access_and_get_filters();
        if (!$access_info['has_access']) {
            header('HTTP/1.1 403 Forbidden');
            echo json_encode(['error' => 'Access denied']);
            return;
        }
        
        try {
            $filters = $this->get_filters();
            $data = $this->reporting_dashboard_model->get_followup_types_chart($filters);
            header('Content-Type: application/json');
            echo json_encode($data);
        } catch (Exception $e) {
            error_log('Reporting Dashboard Follow-up Types Error: ' . $e->getMessage());
            header('Content-Type: application/json');
            http_response_code(500);
            echo json_encode([
                'error' => true,
                'message' => $e->getMessage()
            ]);
        }
    }

    public function get_reminders()
    {
        // Check access permissions
        $access_info = $this->check_access_and_get_filters();
        if (!$access_info['has_access']) {
            header('HTTP/1.1 403 Forbidden');
            echo json_encode(['error' => 'Access denied']);
            return;
        }
        
        try {
            $filters = $this->get_filters();
            
            // Use direct database query to avoid model loading issues
            $data = $this->get_reminders_direct($filters);
            
            header('Content-Type: application/json');
            echo json_encode($data);
        } catch (Exception $e) {
            error_log('Reporting Dashboard Reminders Error: ' . $e->getMessage());
            header('Content-Type: application/json');
            http_response_code(500);
            echo json_encode([
                'error' => true,
                'message' => $e->getMessage(),
                'todays' => [],
                'due' => [],
                'upcoming' => []
            ]);
        }
    }
    
    private function get_reminders_direct($filters)
    {
        // Get staff filter for permission-based access
        $access_info = $this->check_access_and_get_filters();
        $allowed_staff_ids = $access_info['allowed_staff_ids'];
        $can_view_all = $access_info['can_view_all'];

        // Simple query to get basic reminders
        $this->db->select('*');
        $this->db->from(db_prefix() . 'reminders');

        // Exclude reminders that are marked as read/notified (isnotified = 1 means "Yes" (read/completed), so exclude those strictly)
        $this->db->where('(isnotified IS NULL OR isnotified = 0)');

        // Apply date filters if provided - make them more inclusive for reminders
        // For reminders, we want to include a broader date range to capture today's and upcoming reminders
        if (!empty($filters['from'])) {
            // Include reminders from 30 days before the start date to ensure we don't miss any
            $extended_from = date('Y-m-d', strtotime($filters['from'] . ' -30 days'));
            $this->db->where('DATE(date) >=', $extended_from);
            error_log('Extended reminder date filter from: ' . $extended_from . ' (original: ' . $filters['from'] . ')');
        }
        if (!empty($filters['to'])) {
            // Include reminders up to 90 days after the end date to capture upcoming reminders
            $extended_to = date('Y-m-d', strtotime($filters['to'] . ' +90 days'));
            $this->db->where('DATE(date) <=', $extended_to);
            error_log('Extended reminder date filter to: ' . $extended_to . ' (original: ' . $filters['to'] . ')');
        }

        // Apply staff filter (this already includes permission-based filtering from get_filters())
        if (!empty($filters['staff']) && is_array($filters['staff'])) {
            $this->db->where_in('staff', $filters['staff']);
            error_log('Applying staff filter to reminders: ' . implode(',', $filters['staff']));
        } elseif (!$can_view_all && !empty($allowed_staff_ids)) {
            // Fallback: if no specific staff selected but user has limited permissions
            $this->db->where_in('staff', $allowed_staff_ids);
            error_log('Applying permission-based staff filter: ' . implode(',', $allowed_staff_ids));
        }
        // If can_view_all is true and no specific staff filter, don't apply any staff restriction

        $this->db->order_by('date', 'ASC');

        // Debug: Log the SQL query
        $sql = $this->db->get_compiled_select();
        error_log('Reminders Query: ' . $sql);

        // Reset query and execute
        $this->db->select('*');
        $this->db->from(db_prefix() . 'reminders');

        // Apply the same extended date filters as used in the debug query
        if (!empty($filters['from'])) {
            $extended_from = date('Y-m-d', strtotime($filters['from'] . ' -30 days'));
            $this->db->where('DATE(date) >=', $extended_from);
        }
        if (!empty($filters['to'])) {
            $extended_to = date('Y-m-d', strtotime($filters['to'] . ' +90 days'));
            $this->db->where('DATE(date) <=', $extended_to);
        }
        if (!empty($filters['staff']) && is_array($filters['staff'])) {
            $this->db->where_in('staff', $filters['staff']);
        } elseif (!$can_view_all && !empty($allowed_staff_ids)) {
            $this->db->where_in('staff', $allowed_staff_ids);
        }
        $this->db->order_by('date', 'ASC');

        $query = $this->db->get();
        $reminders = $query->result_array();
        error_log('Found ' . count($reminders) . ' reminders after SQL filtering');

        // Post-process: strictly exclude any isnotified=1 (notified/read) and lost/junk/closed/cancelled/dead leads
        $filtered_reminders = [];
        foreach ($reminders as &$reminder) {
            // Exclude if reminder is notified/read (should not be present, but double check)
            if (isset($reminder['isnotified']) && $reminder['isnotified'] == 1) {
                error_log("Excluding reminder ID {$reminder['id']} as isnotified=1");
                continue;
            }
            $should_include = true;
            if ($reminder['rel_type'] === 'lead' && !empty($reminder['rel_id'])) {
                $this->db->select('status');
                $this->db->from(db_prefix() . 'leads');
                $this->db->where('id', $reminder['rel_id']);
                $lead_query = $this->db->get();
                if ($lead_query->num_rows() > 0) {
                    $lead = $lead_query->row();
                    if (in_array(strtolower($lead->status), ['lost', 'junk', 'closed', 'cancelled', 'dead'])) {
                        $should_include = false;
                        error_log("Excluding reminder for lead {$reminder['rel_id']} with status: {$lead->status}");
                    }
                } else {
                    $should_include = false;
                    error_log("Excluding reminder for non-existent lead {$reminder['rel_id']}");
                }
            }
            if (!$should_include) {
                continue;
            }
            // Get staff name
            $this->db->select('firstname, lastname');
            $this->db->from(db_prefix() . 'staff');
            $this->db->where('staffid', $reminder['staff']);
            $staff_query = $this->db->get();
            if ($staff_query->num_rows() > 0) {
                $staff = $staff_query->row();
                $reminder['staff_name'] = trim($staff->firstname . ' ' . $staff->lastname);
            } else {
                $reminder['staff_name'] = 'Unknown Staff';
            }
            $reminder['related_name'] = $this->get_related_name($reminder['rel_type'], $reminder['rel_id']);
            $reminder['related_email'] = $this->get_related_email($reminder['rel_type'], $reminder['rel_id']);
            $filtered_reminders[] = $reminder;
        }
        $reminders = $filtered_reminders;
        error_log('After post-processing: ' . count($reminders) . ' reminders remaining');
        if (count($reminders) > 0) {
            $sample = $reminders[0];
            error_log('Sample reminder: ID=' . $sample['id'] . ', rel_type=' . $sample['rel_type'] . ', rel_id=' . $sample['rel_id'] . ', isnotified=' . $sample['isnotified'] . ', date=' . $sample['date']);
        }
        
        // Categorize reminders with more robust date handling
        $today = date('Y-m-d');
        $result = [
            'todays' => [],
            'due' => [],
            'upcoming' => []
        ];

        foreach ($reminders as $reminder) {
            // Extract just the date part in case the reminder date includes time
            $reminder_date = date('Y-m-d', strtotime($reminder['date']));

            // Debug: Log reminder categorization
            error_log("Categorizing reminder ID {$reminder['id']}: date={$reminder_date}, today={$today}");

            if ($reminder_date === $today) {
                $result['todays'][] = $reminder;
                error_log("Added to todays: reminder ID {$reminder['id']}");
            } elseif ($reminder_date < $today) {
                // Only add to due if it's actually missed (past date)
                $result['due'][] = $reminder;
                error_log("Added to due: reminder ID {$reminder['id']}");
            } else {
                // Future reminders go to upcoming
                $result['upcoming'][] = $reminder;
                error_log("Added to upcoming: reminder ID {$reminder['id']}");
            }
        }
        
        // Log the filtering results
        error_log('Reminders filtered - Total: ' . count($reminders) . 
                 ', Today: ' . count($result['todays']) . 
                 ', Due (missed): ' . count($result['due']) . 
                 ', Upcoming: ' . count($result['upcoming']));
        
        return $result;
    }
    
    private function get_related_name($rel_type, $rel_id)
    {
        if (empty($rel_type) || empty($rel_id)) {
            return $rel_type . ' #' . $rel_id;
        }
        
        switch ($rel_type) {
            case 'lead':
                $this->db->select('name');
                $this->db->from(db_prefix() . 'leads');
                $this->db->where('id', $rel_id);
                $query = $this->db->get();
                if ($query->num_rows() > 0) {
                    return $query->row()->name;
                }
                break;
                
            case 'customer':
                $this->db->select('company');
                $this->db->from(db_prefix() . 'clients');
                $this->db->where('userid', $rel_id);
                $query = $this->db->get();
                if ($query->num_rows() > 0) {
                    return $query->row()->company;
                }
                break;
                
            case 'project':
                $this->db->select('name');
                $this->db->from(db_prefix() . 'projects');
                $this->db->where('id', $rel_id);
                $query = $this->db->get();
                if ($query->num_rows() > 0) {
                    return $query->row()->name;
                }
                break;
                
            case 'proposal':
                $this->db->select('subject');
                $this->db->from(db_prefix() . 'proposals');
                $this->db->where('id', $rel_id);
                $query = $this->db->get();
                if ($query->num_rows() > 0) {
                    return $query->row()->subject;
                }
                break;
                
            case 'invoice':
                $this->db->select('number');
                $this->db->from(db_prefix() . 'invoices');
                $this->db->where('id', $rel_id);
                $query = $this->db->get();
                if ($query->num_rows() > 0) {
                    return 'Invoice #' . $query->row()->number;
                }
                break;
                
            case 'estimate':
                $this->db->select('number');
                $this->db->from(db_prefix() . 'estimates');
                $this->db->where('id', $rel_id);
                $query = $this->db->get();
                if ($query->num_rows() > 0) {
                    return 'Estimate #' . $query->row()->number;
                }
                break;
                
            case 'contract':
                $this->db->select('subject');
                $this->db->from(db_prefix() . 'contracts');
                $this->db->where('id', $rel_id);
                $query = $this->db->get();
                if ($query->num_rows() > 0) {
                    return $query->row()->subject;
                }
                break;
        }
        
        return $rel_type . ' #' . $rel_id;
    }
    
    private function get_related_email($rel_type, $rel_id)
    {
        if (empty($rel_type) || empty($rel_id)) {
            return null;
        }
        
        switch ($rel_type) {
            case 'lead':
                $this->db->select('email');
                $this->db->from(db_prefix() . 'leads');
                $this->db->where('id', $rel_id);
                $query = $this->db->get();
                if ($query->num_rows() > 0) {
                    return $query->row()->email;
                }
                break;
                
            case 'customer':
                $this->db->select('email');
                $this->db->from(db_prefix() . 'clients');
                $this->db->where('userid', $rel_id);
                $query = $this->db->get();
                if ($query->num_rows() > 0) {
                    return $query->row()->email;
                }
                break;
        }
        
        return null;
    }

    private function get_filters()
    {
        // Accept both POST and GET for flexibility
        $input = array_merge($_GET, $_POST);
        
        $filters = [
            'from' => $input['from'] ?? '',
            'to' => $input['to'] ?? '',
            'staff' => $input['staff'] ?? [],
            'activity_type' => $input['activity_type'] ?? [],
        ];
        
        // Ensure staff is always an array
        if (!is_array($filters['staff'])) {
            if (empty($filters['staff'])) {
                $filters['staff'] = [];
            } else {
                // Convert string to array (in case it's a single ID)
                $filters['staff'] = [$filters['staff']];
            }
        }
        
        // Ensure activity_type is always an array
        if (!is_array($filters['activity_type'])) {
            if (empty($filters['activity_type'])) {
                $filters['activity_type'] = [];
            } else {
                $filters['activity_type'] = [$filters['activity_type']];
            }
        }
        
        // Apply permission-based filtering
        $current_staff_id = get_staff_user_id();
        $allowed_staff_ids = $this->reporting_dashboard_permissions_model->get_allowed_staff_ids($current_staff_id);
        
        // Debug logging
        error_log('Current staff ID: ' . $current_staff_id);
        error_log('Allowed staff IDs: ' . print_r($allowed_staff_ids, true));
        error_log('Requested staff filter: ' . print_r($filters['staff'], true));
        
        // If user has limited permissions, override the staff filter
        if (!empty($allowed_staff_ids)) {
            // If user requested specific staff, filter to only those they can see
            if (!empty($filters['staff'])) {
                $filters['staff'] = array_intersect($filters['staff'], $allowed_staff_ids);
            } else {
                // If no specific staff requested, limit to allowed staff
                $filters['staff'] = $allowed_staff_ids;
            }
        } else {
            // If no permission restrictions (superadmin), allow all staff
            error_log('No permission restrictions - allowing all staff');
            // Keep the original staff filter or leave empty for all staff
        }
        
        error_log('Final staff filter: ' . print_r($filters['staff'], true));
        
        return $filters;
    }

    /**
     * Normalize WhatsApp numbers from mixed input.
     *
     * @param mixed $input
     * @return array
     */
    private function parse_whatsapp_numbers($input)
    {
        if (empty($input)) {
            return [];
        }

        if (is_string($input)) {
            $numbers = preg_split('/[\s,;]+/', $input);
        } elseif (is_array($input)) {
            $numbers = $input;
        } else {
            return [];
        }

        $normalized = [];
        foreach ($numbers as $number) {
            if ($number === null) {
                continue;
            }

            $clean = preg_replace('/[^0-9+]/', '', (string)$number);
            if ($clean === '') {
                continue;
            }

            if ($clean[0] !== '+' && strlen($clean) >= 10) {
                $clean = '+' . $clean;
            }

            $normalized[] = $clean;
        }

        return array_values(array_unique($normalized));
    }

    /**
     * Output JSON response with appropriate status headers.
     *
     * @param array $payload
     * @param int   $status
     * @return void
     */
    private function json_response(array $payload, $status = 200)
    {
        header('Content-Type: application/json');
        http_response_code($status);
        echo json_encode($payload);
    }

    public function generate_staff_report()
    {
        if (!is_staff_logged_in()) {
            $this->json_response(['success' => false, 'error' => 'Authentication required.'], 401);
            return;
        }

        if ($this->input->method() !== 'post') {
            $this->json_response(['success' => false, 'error' => 'Invalid request method.'], 405);
            return;
        }

        // Check if user has permission to generate reports
        $current_staff_id = get_staff_user_id();
        if (!$this->reporting_dashboard_permissions_model->can_generate_reports($current_staff_id)) {
            $this->json_response([
                'success' => false,
                'error' => 'You do not have permission to generate staff reports. Please contact your administrator.'
            ], 403);
            return;
        }

        $access_info = $this->check_access_and_get_filters();
        if (!$access_info['has_access']) {
            $this->json_response(['success' => false, 'error' => 'Access denied.'], 403);
            return;
        }

        try {
            $filters = $this->get_filters();

            $reportFrom = $this->input->post('report_from');
            $reportTo = $this->input->post('report_to');

            if (!empty($reportFrom)) {
                $filters['from'] = $reportFrom;
            }

            if (!empty($reportTo)) {
                $filters['to'] = $reportTo;
            }

            if (empty($filters['from'])) {
                $filters['from'] = date('Y-m-d', strtotime('-7 days'));
            }

            if (empty($filters['to'])) {
                $filters['to'] = date('Y-m-d');
            }

            $summary = $this->reporting_dashboard_model->get_staff_activity_summary($filters);

            if (empty($summary['staff'])) {
                $this->json_response([
                    'success' => false,
                    'message' => 'No staff activity data found for the selected filters.'
                ]);
                return;
            }

            // Get reminders and follow-ups data
            $remindersData = $this->reporting_dashboard_model->get_staff_reminders_followups($filters);
            $summary['reminders_followups'] = $remindersData;

            $this->load->library('reporting_dashboard/staff_report_service');

            $analysisResult = $this->staff_report_service->analyze_summary($summary);
            $analysis = $analysisResult['analysis'] ?? null;

            $rawNumbers = $this->input->post('whatsapp_numbers');
            $numbers = $this->parse_whatsapp_numbers($rawNumbers);
            $sendWhatsapp = (bool)$this->input->post('send_whatsapp');

            if ($sendWhatsapp && empty($numbers)) {
                $this->json_response([
                    'success' => false,
                    'error' => 'Please provide at least one WhatsApp number.'
                ], 422);
                return;
            }

            $messagePreview = $this->staff_report_service->format_whatsapp_message($summary, $analysis);

            $context = [
                'generated_by' => get_staff_user_id(),
                'whatsapp_numbers' => $numbers,
                'ai_model' => $analysisResult['model'] ?? null,
                'message_preview' => $messagePreview,
                'whatsapp_status' => null,
            ];

            if (!$analysisResult['success'] && !empty($analysisResult['error'])) {
                $context['notes'] = 'AI analysis warning: ' . $analysisResult['error'];
            }

            $reportId = $this->reporting_dashboard_model->save_staff_report($summary, $analysis, $filters, $context);

            $whatsappStatus = null;
            if ($sendWhatsapp) {
                $whatsappStatus = $this->staff_report_service->send_whatsapp_message($messagePreview, $numbers);
                $context['whatsapp_status'] = $whatsappStatus;

                $this->reporting_dashboard_model->update_staff_report($reportId, [
                    'whatsapp_status' => $whatsappStatus,
                    'whatsapp_numbers' => $numbers,
                    'message_preview' => $messagePreview,
                ]);
            }

            $analysisMeta = [
                'success' => $analysisResult['success'] ?? false,
                'error' => $analysisResult['error'] ?? null,
                'model' => $analysisResult['model'] ?? null,
            ];

            $this->json_response([
                'success' => true,
                'report_id' => $reportId,
                'summary' => $summary,
                'analysis' => $analysis,
                'analysis_meta' => $analysisMeta,
                'message_preview' => $messagePreview,
                'whatsapp' => $whatsappStatus,
            ]);
        } catch (Exception $e) {
            log_message('error', 'Reporting Dashboard staff report error: ' . $e->getMessage());
            $this->json_response([
                'success' => false,
                'error' => 'Failed to generate staff report. ' . $e->getMessage()
            ], 500);
        }
    }

    public function get_staff_report_schedules()
    {
        if (!is_staff_logged_in()) {
            $this->json_response(['success' => false, 'error' => 'Authentication required.'], 401);
            return;
        }

        $access_info = $this->check_access_and_get_filters();
        if (!$access_info['has_access']) {
            $this->json_response(['success' => false, 'error' => 'Access denied.'], 403);
            return;
        }

        try {
            $schedules = $this->reporting_dashboard_model->get_staff_report_schedules(true);
            $this->json_response([
                'success' => true,
                'schedules' => $schedules
            ]);
        } catch (Exception $e) {
            log_message('error', 'Reporting Dashboard schedule fetch error: ' . $e->getMessage());
            $this->json_response([
                'success' => false,
                'error' => 'Failed to load schedules: ' . $e->getMessage()
            ], 500);
        }
    }

    public function save_staff_report_schedule()
    {
        if (!is_staff_logged_in()) {
            $this->json_response(['success' => false, 'error' => 'Authentication required.'], 401);
            return;
        }

        if ($this->input->method() !== 'post') {
            $this->json_response(['success' => false, 'error' => 'Invalid request method.'], 405);
            return;
        }

        // Check if user has permission to generate/schedule reports
        $current_staff_id = get_staff_user_id();
        if (!$this->reporting_dashboard_permissions_model->can_generate_reports($current_staff_id)) {
            $this->json_response([
                'success' => false,
                'error' => 'You do not have permission to schedule staff reports.'
            ], 403);
            return;
        }

        $access_info = $this->check_access_and_get_filters();
        if (!$access_info['has_access']) {
            $this->json_response(['success' => false, 'error' => 'Access denied.'], 403);
            return;
        }

        try {
            $scheduleId = $this->input->post('schedule_id');

            $staffIds = $this->input->post('staff_ids');
            if ($staffIds && !is_array($staffIds)) {
                $staffIds = array_filter(array_map('intval', explode(',', $staffIds)));
            }

            $rawNumbers = $this->input->post('whatsapp_numbers');
            $numbers = $this->parse_whatsapp_numbers($rawNumbers);

            if (empty($numbers)) {
                $this->json_response([
                    'success' => false,
                    'error' => 'Please provide at least one WhatsApp number.'
                ], 422);
                return;
            }

            $payload = [
                'name' => $this->input->post('name', true),
                'frequency' => $this->input->post('frequency', true),
                'range_scope' => $this->input->post('range_scope', true),
                'schedule_time' => $this->input->post('schedule_time', true),
                'day_of_week' => $this->input->post('day_of_week'),
                'day_of_month' => $this->input->post('day_of_month'),
                'staff_ids' => $staffIds,
                'whatsapp_numbers' => $numbers,
                'active' => (int)($this->input->post('active') !== null ? (bool)$this->input->post('active') : 1),
                'created_by' => get_staff_user_id(),
            ];

            if (!in_array($payload['frequency'], ['daily', 'weekly', 'monthly'], true)) {
                $payload['frequency'] = 'daily';
            }

            if (empty($payload['schedule_time'])) {
                $payload['schedule_time'] = '09:00';
            }

            if (empty($payload['name'])) {
                $payload['name'] = ucfirst($payload['frequency']) . ' Staff Report';
            }

            if ($payload['frequency'] === 'weekly' && $payload['day_of_week'] === null) {
                $payload['day_of_week'] = date('w');
            }

            if ($payload['frequency'] === 'monthly' && empty($payload['day_of_month'])) {
                $payload['day_of_month'] = date('j');
            }

            if ($scheduleId) {
                $this->reporting_dashboard_model->update_staff_report_schedule((int)$scheduleId, $payload);
            } else {
                $scheduleId = $this->reporting_dashboard_model->create_staff_report_schedule($payload);
            }

            $schedules = $this->reporting_dashboard_model->get_staff_report_schedules(true);

            $this->json_response([
                'success' => true,
                'schedule_id' => $scheduleId,
                'schedules' => $schedules
            ]);
        } catch (Exception $e) {
            log_message('error', 'Reporting Dashboard schedule save error: ' . $e->getMessage());
            $this->json_response([
                'success' => false,
                'error' => 'Failed to save schedule: ' . $e->getMessage()
            ], 500);
        }
    }

    public function delete_staff_report_schedule()
    {
        if (!is_staff_logged_in()) {
            $this->json_response(['success' => false, 'error' => 'Authentication required.'], 401);
            return;
        }

        if ($this->input->method() !== 'post') {
            $this->json_response(['success' => false, 'error' => 'Invalid request method.'], 405);
            return;
        }

        // Check if user has permission to manage schedules
        $current_staff_id = get_staff_user_id();
        if (!$this->reporting_dashboard_permissions_model->can_generate_reports($current_staff_id)) {
            $this->json_response([
                'success' => false,
                'error' => 'You do not have permission to delete report schedules.'
            ], 403);
            return;
        }

        $scheduleId = (int)$this->input->post('schedule_id');

        if ($scheduleId <= 0) {
            $this->json_response(['success' => false, 'error' => 'Schedule ID is required.'], 422);
            return;
        }

        try {
            $this->reporting_dashboard_model->delete_staff_report_schedule($scheduleId);
            $this->json_response(['success' => true]);
        } catch (Exception $e) {
            log_message('error', 'Reporting Dashboard schedule delete error: ' . $e->getMessage());
            $this->json_response([
                'success' => false,
                'error' => 'Failed to delete schedule: ' . $e->getMessage()
            ], 500);
        }
    }

    public function toggle_staff_report_schedule()
    {
        if (!is_staff_logged_in()) {
            $this->json_response(['success' => false, 'error' => 'Authentication required.'], 401);
            return;
        }

        if ($this->input->method() !== 'post') {
            $this->json_response(['success' => false, 'error' => 'Invalid request method.'], 405);
            return;
        }

        // Check if user has permission to manage schedules
        $current_staff_id = get_staff_user_id();
        if (!$this->reporting_dashboard_permissions_model->can_generate_reports($current_staff_id)) {
            $this->json_response([
                'success' => false,
                'error' => 'You do not have permission to toggle report schedules.'
            ], 403);
            return;
        }

        $scheduleId = (int)$this->input->post('schedule_id');
        $active = (bool)$this->input->post('active');

        if ($scheduleId <= 0) {
            $this->json_response(['success' => false, 'error' => 'Schedule ID is required.'], 422);
            return;
        }

        try {
            $this->reporting_dashboard_model->toggle_staff_report_schedule($scheduleId, $active);
            $this->json_response(['success' => true]);
        } catch (Exception $e) {
            log_message('error', 'Reporting Dashboard schedule toggle error: ' . $e->getMessage());
            $this->json_response([
                'success' => false,
                'error' => 'Failed to update schedule: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get schedule execution logs
     */
    public function get_schedule_logs()
    {
        if (!is_staff_logged_in()) {
            $this->json_response(['success' => false, 'error' => 'Authentication required.'], 401);
            return;
        }

        // Check if user has permission to view reports
        $current_staff_id = get_staff_user_id();
        if (!$this->reporting_dashboard_permissions_model->can_generate_reports($current_staff_id)) {
            $this->json_response([
                'success' => false,
                'error' => 'You do not have permission to view schedule logs.'
            ], 403);
            return;
        }

        try {
            $scheduleId = $this->input->get('schedule_id');
            $limit = (int)$this->input->get('limit') ?: 100;
            
            $logs = $this->reporting_dashboard_model->get_schedule_logs(
                $scheduleId ? (int)$scheduleId : null,
                $limit
            );

            // Get stats if schedule_id provided
            $stats = null;
            if ($scheduleId) {
                $stats = $this->reporting_dashboard_model->get_schedule_stats((int)$scheduleId);
            }

            $this->json_response([
                'success' => true,
                'logs' => $logs,
                'stats' => $stats
            ]);
        } catch (Exception $e) {
            log_message('error', 'Reporting Dashboard get logs error: ' . $e->getMessage());
            $this->json_response([
                'success' => false,
                'error' => 'Failed to fetch logs: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get all schedules with their execution stats
     */
    public function get_schedules_with_stats()
    {
        if (!is_staff_logged_in()) {
            $this->json_response(['success' => false, 'error' => 'Authentication required.'], 401);
            return;
        }

        // Check if user has permission to view reports
        $current_staff_id = get_staff_user_id();
        if (!$this->reporting_dashboard_permissions_model->can_generate_reports($current_staff_id)) {
            $this->json_response([
                'success' => false,
                'error' => 'You do not have permission to view schedules.'
            ], 403);
            return;
        }

        try {
            $schedules = $this->reporting_dashboard_model->get_staff_report_schedules(true);
            
            // Add stats for each schedule
            foreach ($schedules as &$schedule) {
                $schedule['stats'] = $this->reporting_dashboard_model->get_schedule_stats($schedule['id']);
            }

            $this->json_response([
                'success' => true,
                'schedules' => $schedules
            ]);
        } catch (Exception $e) {
            log_message('error', 'Reporting Dashboard get schedules error: ' . $e->getMessage());
            $this->json_response([
                'success' => false,
                'error' => 'Failed to fetch schedules: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get staff reminders and follow-ups status (overdue and upcoming)
     */
    public function get_staff_reminders_followups()
    {
        if (!is_staff_logged_in()) {
            $this->json_response(['success' => false, 'error' => 'Authentication required.'], 401);
            return;
        }

        try {
            // Get filters - everyone can view their own reminders
            $current_staff_id = get_staff_user_id();
            $filters = [];
            
            // Check if user has view_all permission
            $this->db->select('permission_type, allowed_staff_ids');
            $this->db->from(db_prefix() . 'reporting_dashboard_permissions');
            $this->db->where('staff_id', $current_staff_id);
            $permission = $this->db->get()->row();
            
            // Admins can see all
            if (is_admin($current_staff_id)) {
                $filters['staff'] = []; // Empty means all staff
            } elseif ($permission && $permission->permission_type === 'view_all_staff') {
                $filters['staff'] = []; // Can view all staff
            } elseif ($permission && $permission->permission_type === 'view_specific_staff') {
                $allowedStaff = json_decode($permission->allowed_staff_ids, true) ?: [];
                if (!in_array($current_staff_id, $allowedStaff)) {
                    $allowedStaff[] = $current_staff_id;
                }
                $filters['staff'] = $allowedStaff;
            } else {
                // Default: only own reminders
                $filters['staff'] = [$current_staff_id];
            }
            
            // Get staff filter from POST body (JSON)
            $input = file_get_contents('php://input');
            if (!empty($input)) {
                $postData = json_decode($input, true);
                if (!empty($postData['staff'])) {
                    $staffFilter = $postData['staff'];
                    if (is_array($staffFilter)) {
                        $filters['staff'] = array_map('intval', array_filter($staffFilter));
                    }
                }
            }

            $data = $this->reporting_dashboard_model->get_staff_reminders_followups($filters);

            $this->json_response([
                'success' => true,
                'data' => $data,
            ]);
        } catch (Exception $e) {
            error_log('Reporting Dashboard reminders/followups error: ' . $e->getMessage());
            $this->json_response([
                'success' => false,
                'error' => 'Failed to fetch reminders and follow-ups: ' . $e->getMessage(),
            ], 500);
        }
    }

    public function get_saved_staff_reports()
    {
        if (!is_staff_logged_in()) {
            $this->json_response(['success' => false, 'error' => 'Authentication required.'], 401);
            return;
        }

        try {
            $reports = $this->reporting_dashboard_model->get_staff_reports(50);
            $formatted = [];

            foreach ($reports as $report) {
                $summary = $report['summary'] ?? [];
                $totals = $summary['totals'] ?? [];
                $period = $summary['period'] ?? [];

                $formatted[] = [
                    'id' => (int)($report['id'] ?? 0),
                    'generated_at' => $report['generated_at'] ?? $report['created_at'] ?? null,
                    'generated_by' => $report['generated_by'] ?? null,
                    'date_from' => $report['date_from'] ?? null,
                    'date_to' => $report['date_to'] ?? null,
                    'summary' => $summary,
                    'analysis' => $report['analysis'] ?? [],
                    'message_preview' => $report['message_preview'] ?? '',
                    'ai_model' => $report['ai_model'] ?? null,
                    'notes' => $report['notes'] ?? null,
                    'totals' => $totals,
                    'period' => $period,
                    'followup_totals' => $summary['followup_totals'] ?? [],
                    'whatsapp_status' => $report['whatsapp_status'] ?? [],
                ];
            }

            $this->json_response([
                'success' => true,
                'reports' => $formatted
            ]);
        } catch (Exception $e) {
            log_message('error', 'Reporting Dashboard saved reports fetch error: ' . $e->getMessage());
            log_message('error', 'Reporting Dashboard saved reports fetch trace: ' . $e->getTraceAsString());
            $this->json_response([
                'success' => false,
                'error' => 'Failed to load saved reports: ' . $e->getMessage()
            ], 500);
        }
    }

    public function download_pdf()
    {
        // Check access permissions
        $access_info = $this->check_access_and_get_filters();
        if (!$access_info['has_access']) {
            header('HTTP/1.1 403 Forbidden');
            show_error('Access denied');
            return;
        }
        
        try {
            // Clear any active output buffers (if any) to avoid corrupting the PDF stream
            while (ob_get_level() > 0) {
                @ob_end_clean();
            }
            $old_error_reporting = error_reporting(E_ALL & ~E_WARNING & ~E_NOTICE);
            
            $filters = $this->get_filters();
            
            // Load required dependencies first
            if (!class_exists('App_pdf')) {
                require_once(APPPATH . 'libraries/pdf/App_pdf.php');
            }
            
            // Now load our custom PDF class
            $pdfClassPath = FCPATH . 'modules/reporting_dashboard/libraries/Reporting_dashboard_pdf.php';
            if (!file_exists($pdfClassPath)) {
                error_reporting($old_error_reporting);
                show_error('PDF class file not found at: ' . $pdfClassPath);
                return;
            }
            
            require_once($pdfClassPath);
            
            // Check if class exists after include
            if (!class_exists('Reporting_dashboard_pdf')) {
                error_reporting($old_error_reporting);
                show_error('Reporting_dashboard_pdf class not found after include');
                return;
            }
            
            // Create instance manually
            $pdf = new Reporting_dashboard_pdf($filters);
            
            // Generate the PDF
            $result = $pdf->prepare();
            
            // Set dynamic filename based on staff selection
            $filename = 'Staff_Activity_Report_' . date('Y-m-d_H-i-s') . '.pdf';
            
            // Check if single staff is selected for personalized filename
            if (isset($filters['staff']) && is_array($filters['staff']) && count($filters['staff']) == 1) {
                $staff_id = $filters['staff'][0];
                
                // Get staff name for filename
                $this->db->select('firstname, lastname');
                $this->db->from(db_prefix() . 'staff');
                $this->db->where('staffid', $staff_id);
                $staff_query = $this->db->get();
                
                if ($staff_query->num_rows() > 0) {
                    $staff_info = $staff_query->row();
                    $staff_name = trim($staff_info->firstname . ' ' . $staff_info->lastname);
                    // Clean filename - remove special characters and spaces
                    $clean_name = preg_replace('/[^a-zA-Z0-9_-]/', '_', $staff_name);
                    $filename = $clean_name . '_Activity_Report_' . date('Y-m-d_H-i-s') . '.pdf';
                }
            } elseif (isset($filters['staff']) && is_array($filters['staff']) && count($filters['staff']) > 1) {
                $staff_count = count($filters['staff']);
                $filename = 'Team_' . $staff_count . '_Staff_Activity_Report_' . date('Y-m-d_H-i-s') . '.pdf';
            }
            
            // Restore error reporting
            error_reporting($old_error_reporting);
            
            // Output the PDF for download
            $pdf->Output($filename, 'D');
            
        } catch (Exception $e) {
            error_reporting($old_error_reporting);
            show_error('Error generating PDF report: ' . $e->getMessage());
        }
    }

    /**
     * Permissions management page (Admin only)
     */
    public function permissions()
    {
        // Only admin can access permission management
        if (!is_admin()) {
            access_denied('Reporting Dashboard Permissions');
        }

        $data['title'] = 'Reporting Dashboard - Permissions Management';
        $data['staff_permissions'] = $this->reporting_dashboard_permissions_model->get_all_staff_permissions();
        $data['reporting_dashboard_permissions_model'] = $this->reporting_dashboard_permissions_model;
        $this->load->view('permissions_management', $data);
    }

    /**
     * Update staff permission via AJAX
     */
    public function update_permission()
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        // Only admin can update permissions
        if (!is_admin()) {
            echo json_encode([
                'success' => false,
                'message' => 'Access denied'
            ]);
            return;
        }

        $staff_id = $this->input->post('staff_id');
        $permission_type = $this->input->post('permission_type');
        $allowed_staff_ids = $this->input->post('allowed_staff_ids');
        $current_user_id = get_staff_user_id();

        // Validate inputs
        if (empty($staff_id) || empty($permission_type)) {
            echo json_encode([
                'success' => false,
                'message' => 'Invalid parameters'
            ]);
            return;
        }

        // Validate permission type
        if (!in_array($permission_type, ['view_all_staff', 'view_own_only', 'view_specific_staff'])) {
            echo json_encode([
                'success' => false,
                'message' => 'Invalid permission type'
            ]);
            return;
        }

        // For view_specific_staff, validate allowed_staff_ids
        $allowed_staff_array = [];
        if ($permission_type === 'view_specific_staff') {
            if (empty($allowed_staff_ids)) {
                echo json_encode([
                    'success' => false,
                    'message' => 'Please select at least one staff member for specific staff permission'
                ]);
                return;
            }
            
            $allowed_staff_array = is_array($allowed_staff_ids) ? $allowed_staff_ids : explode(',', $allowed_staff_ids);
            $allowed_staff_array = array_filter(array_map('intval', $allowed_staff_array));
            
            if (empty($allowed_staff_array)) {
                echo json_encode([
                    'success' => false,
                    'message' => 'Invalid staff selection'
                ]);
                return;
            }
        }

        // Update permission
        $success = $this->reporting_dashboard_permissions_model->set_staff_permission(
            $staff_id,
            $permission_type,
            $current_user_id,
            $allowed_staff_array
        );

        if ($success) {
            echo json_encode([
                'success' => true,
                'message' => 'Permission updated successfully'
            ]);
        } else {
            echo json_encode([
                'success' => false,
                'message' => 'Failed to update permission'
            ]);
        }
    }

    /**
     * Remove staff permission (revert to default)
     */
    public function remove_permission()
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        if (!is_admin()) {
            echo json_encode([
                'success' => false,
                'message' => 'Access denied'
            ]);
            return;
        }

        $staff_id = $this->input->post('staff_id');

        if (empty($staff_id)) {
            echo json_encode([
                'success' => false,
                'message' => 'Invalid staff ID'
            ]);
            return;
        }

        $success = $this->reporting_dashboard_permissions_model->remove_staff_permission($staff_id);

        if ($success) {
            echo json_encode([
                'success' => true,
                'message' => 'Permission reset to default successfully'
            ]);
        } else {
            echo json_encode([
                'success' => false,
                'message' => 'Failed to reset permission'
            ]);
        }
    }

    /**
     * Get available staff for selection (AJAX)
     */
    public function get_available_staff()
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        if (!is_admin()) {
            echo json_encode([
                'success' => false,
                'message' => 'Access denied'
            ]);
            return;
        }

        $exclude_staff_id = $this->input->post('exclude_staff_id');
        $staff_list = $this->reporting_dashboard_permissions_model->get_available_staff_for_permission($exclude_staff_id);

        echo json_encode([
            'success' => true,
            'staff_list' => $staff_list
        ]);
    }

    /**
     * Bulk update permissions
     */
    public function bulk_update()
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        if (!is_admin()) {
            echo json_encode([
                'success' => false,
                'message' => 'Access denied'
            ]);
            return;
        }

        $permissions = $this->input->post('permissions');
        $current_user_id = get_staff_user_id();

        if (empty($permissions) || !is_array($permissions)) {
            echo json_encode([
                'success' => false,
                'message' => 'No permissions provided'
            ]);
            return;
        }

        $updated_count = 0;
        $errors = [];

        foreach ($permissions as $staff_id => $permission_data) {
            $permission_type = $permission_data['type'] ?? '';
            $allowed_staff_ids = $permission_data['allowed_staff_ids'] ?? [];
            
            if (in_array($permission_type, ['view_all_staff', 'view_own_only', 'view_specific_staff'])) {
                $success = $this->reporting_dashboard_permissions_model->set_staff_permission(
                    $staff_id,
                    $permission_type,
                    $current_user_id,
                    $allowed_staff_ids
                );
                
                if ($success) {
                    $updated_count++;
                } else {
                    $errors[] = "Failed to update permission for staff ID: $staff_id";
                }
            }
        }

        echo json_encode([
            'success' => empty($errors),
            'message' => empty($errors) ? 
                "Successfully updated $updated_count permissions" :
                "Updated $updated_count permissions with " . count($errors) . " errors",
            'errors' => $errors
        ]);
    }

    /**
     * Get dealer list by interest status
     */
    public function get_dealer_statistics()
    {
        // Check access permissions
        $access_info = $this->check_access_and_get_filters();
        if (!$access_info['has_access']) {
            header('HTTP/1.1 403 Forbidden');
            echo json_encode(['error' => 'Access denied']);
            return;
        }

        try {
            $filters = $this->get_filters();
            $dealer_stats = $this->reporting_dashboard_model->get_dealer_statistics($filters);
            
            header('Content-Type: application/json');
            echo json_encode($dealer_stats);
        } catch (Exception $e) {
            error_log('Reporting Dashboard Dealer Statistics Error: ' . $e->getMessage());
            
            header('Content-Type: application/json');
            http_response_code(500);
            echo json_encode([
                'error' => true,
                'message' => $e->getMessage()
            ]);
        }
    }

    public function get_followup_types_by_staff()
    {
        // Check access permissions
        $access_info = $this->check_access_and_get_filters();
        if (!$access_info['has_access']) {
            header('HTTP/1.1 403 Forbidden');
            echo json_encode(['error' => 'Access denied']);
            return;
        }

        try {
            $filters = $this->get_filters();
            $followup_data = $this->reporting_dashboard_model->get_followup_types_by_staff($filters);
            
            header('Content-Type: application/json');
            echo json_encode($followup_data);
        } catch (Exception $e) {
            error_log('Reporting Dashboard Follow-up Types Error: ' . $e->getMessage());
            
            header('Content-Type: application/json');
            http_response_code(500);
            echo json_encode([
                'error' => true,
                'message' => $e->getMessage()
            ]);
        }
    }

    public function get_dealer_list()
    {
        // Debug logging
        error_log('Dealer List Controller: Request received');
        error_log('Dealer List Controller: GET params: ' . print_r($_GET, true));
        
        // Check access permissions
        $access_info = $this->check_access_and_get_filters();
        if (!$access_info['has_access']) {
            error_log('Dealer List Controller: Access denied');
            header('HTTP/1.1 403 Forbidden');
            echo json_encode(['error' => 'Access denied']);
            return;
        }

        $interest_status = $this->input->get('interest_status');
        error_log('Dealer List Controller: Interest status: ' . $interest_status);
        
        if (!in_array($interest_status, ['interested', 'not_interested', 'no_response'])) {
            error_log('Dealer List Controller: Invalid interest status: ' . $interest_status);
            header('HTTP/1.1 400 Bad Request');
            echo json_encode(['error' => 'Invalid interest status']);
            return;
        }

        try {
            $filters = $this->get_filters();
            error_log('Dealer List Controller: Filters: ' . print_r($filters, true));
            
            // Ensure model is loaded
            if (!isset($this->reporting_dashboard_model)) {
                $this->load->model('reporting_dashboard/reporting_dashboard_model');
            }
            
            // Validate model exists
            if (!isset($this->reporting_dashboard_model)) {
                throw new Exception('Reporting dashboard model not loaded');
            }
            
            $dealers = $this->reporting_dashboard_model->get_dealer_list_by_interest($interest_status, $filters);
            error_log('Dealer List Controller: Found ' . count($dealers) . ' dealers');
            
            // Ensure we have an array
            if (!is_array($dealers)) {
                $dealers = [];
            }
            
            header('Content-Type: application/json');
            echo json_encode([
                'success' => true,
                'data' => $dealers,
                'count' => count($dealers)
            ]);
        } catch (Exception $e) {
            error_log('Reporting Dashboard Dealer List Error: ' . $e->getMessage());
            error_log('Reporting Dashboard Dealer List Error Trace: ' . $e->getTraceAsString());
            
            header('Content-Type: application/json');
            http_response_code(500);
            echo json_encode([
                'error' => true,
                'message' => $e->getMessage(),
                'debug_info' => [
                    'interest_status' => $interest_status,
                    'filters' => $filters ?? 'not_set'
                ]
            ]);
        }
    }

    /**
     * Download dealer list as CSV
     */

    public function download_dealers_list()
    {
        // Log the request for debugging
        error_log('Download dealers list called');
        
        // Check access permissions
        $access_info = $this->check_access_and_get_filters();
        if (!$access_info['has_access']) {
            error_log('Download access denied');
            header('HTTP/1.1 403 Forbidden');
            echo 'Access denied';
            return;
        }

        $interest_status = $this->input->get('interest_status');
        error_log('Interest status: ' . ($interest_status ?: 'NULL'));
        
        if (!in_array($interest_status, ['interested', 'not_interested', 'no_response'])) {
            error_log('Invalid interest status: ' . $interest_status);
            header('HTTP/1.1 400 Bad Request');
            echo 'Invalid interest status';
            return;
        }

        try {
            $filters = $this->get_filters();
            $dealers = $this->reporting_dashboard_model->get_dealer_list_by_interest($interest_status, $filters);
            
            // Clear any existing output
            if (ob_get_level()) {
                ob_clean();
            }
            
            // Set proper CSV headers
            $filename = $interest_status . '_dealers_' . date('Y-m-d') . '.csv';
            header('Content-Type: text/csv; charset=UTF-8');
            header('Content-Disposition: attachment; filename="' . $filename . '"');
            header('Cache-Control: no-cache, must-revalidate');
            header('Expires: Sat, 26 Jul 1997 05:00:00 GMT');
            header('Pragma: no-cache');
            
            // Output UTF-8 BOM for Excel compatibility
            echo "\xEF\xBB\xBF";
            
            // Open output stream
            $output = fopen('php://output', 'w');
            
            // Write CSV header
            fputcsv($output, [
                'ID',
                'Name',
                'Email',
                'Phone',
                'Company',
                'Lead Date',
                'Interest Date',
                'Interest Status'
            ]);
            
            // Write dealer data
            foreach ($dealers as $dealer) {
                fputcsv($output, [
                    $dealer['id'],
                    $dealer['name'] ?: 'N/A',
                    $dealer['email'] ?: 'N/A',
                    $dealer['phonenumber'] ?: 'N/A',
                    $dealer['company'] ?: 'N/A',
                    $dealer['dateadded'],
                    $dealer['interest_date'],
                    ucfirst(str_replace('_', ' ', $interest_status))
                ]);
            }
            
            fclose($output);
            exit(); // Ensure no additional output
            
        } catch (Exception $e) {
            error_log('Reporting Dashboard Download Error: ' . $e->getMessage());
            
            // Clear any existing output
            if (ob_get_level()) {
                ob_clean();
            }
            
            header('Content-Type: text/plain');
            http_response_code(500);
            echo 'Error generating download: ' . $e->getMessage();
            exit();
        }
    }
}


/* Location: ./modules/reporting_dashboard/controllers/Reporting_dashboard.php */