<?php

defined('BASEPATH') or exit('No direct script access allowed');

/**
 * Autograph Feedback Insights Model
 * Handles database operations and OpenAI API interactions
 */
class Autograph_feedback_insights_model extends App_Model
{
    private $settings_table;
    private $report_table;
    private $notes_table;
    private $schedules_table;

    public function __construct()
    {
        parent::__construct();
        $this->settings_table = db_prefix() . 'autograph_feedback_settings';
        $this->report_table = db_prefix() . 'autograph_feedback_report';
        $this->notes_table = db_prefix() . 'lead_notes';
        $this->schedules_table = db_prefix() . 'autograph_feedback_schedules';
    }

    /**
     * Get a single setting value
     * @param string $name Setting name
     * @return string|null
     */
    public function get_setting($name)
    {
        $result = $this->db->select('setting_value')
            ->where('setting_name', $name)
            ->get($this->settings_table)
            ->row();

        return $result ? $result->setting_value : null;
    }

    /**
     * Get all settings as associative array
     * @return array
     */
    public function get_all_settings()
    {
        $results = $this->db->get($this->settings_table)->result();
        $settings = [];

        foreach ($results as $row) {
            $settings[$row->setting_name] = $row->setting_value;
        }

        return $settings;
    }

    /**
     * Update a setting value
     * @param string $name Setting name
     * @param string $value Setting value
     * @return bool
     */
    public function update_setting($name, $value)
    {
        $data = [
            'setting_value' => $value,
            'updated_at' => date('Y-m-d H:i:s')
        ];

        $this->db->where('setting_name', $name);
        $exists = $this->db->get($this->settings_table)->row();

        if ($exists) {
            $this->db->where('setting_name', $name);
            return $this->db->update($this->settings_table, $data);
        } else {
            $data['setting_name'] = $name;
            return $this->db->insert($this->settings_table, $data);
        }
    }

    /**
     * Create schedule
     */
    public function create_schedule($data)
    {
        return $this->db->insert($this->schedules_table, $data);
    }

    /**
     * Update schedule
     */
    public function update_schedule($schedule_id, $data)
    {
        $this->db->where('id', $schedule_id);
        return $this->db->update($this->schedules_table, $data);
    }

    /**
     * Delete schedule
     */
    public function delete_schedule($schedule_id)
    {
        $this->db->where('id', $schedule_id);
        return $this->db->delete($this->schedules_table);
    }

    /**
     * Get schedule by ID
     */
    public function get_schedule($schedule_id)
    {
        $this->db->where('id', $schedule_id);
        return $this->db->get($this->schedules_table)->row();
    }

    /**
     * Get all schedules
     */
    public function get_all_schedules()
    {
        $this->db->order_by('created_at', 'DESC');
        return $this->db->get($this->schedules_table)->result();
    }

    /**
     * Get schedules due for execution
     */
    public function get_due_schedules()
    {
        $this->db->where('active', 1);
        $this->db->where('next_run <=', date('Y-m-d H:i:s'));
        return $this->db->get($this->schedules_table)->result();
    }

    /**
     * Fetch lead notes related to Autograph Serviced Apartments
     * @param array $status_filter Optional array of status IDs to filter
     * @param array $date_range Optional date range filter
     * @return array
     */
    public function get_autograph_notes($status_filter = [], $date_range = [])
    {
        $this->db->select('ln.*, l.name as lead_name, l.company, l.status as lead_status');
        $this->db->from($this->notes_table . ' ln');
        $this->db->join(db_prefix() . 'leads l', 'l.id = ln.leadid', 'left');
        $this->db->where('ln.description IS NOT NULL');
        $this->db->where('ln.description !=', '');
        
        // Apply status filter if provided
        if (!empty($status_filter)) {
            // If status filter is set, get all notes from leads with those statuses
            $this->db->where_in('l.status', $status_filter);
        } else {
            // If no status filter, search for notes containing "autograph"
            $this->db->like('ln.description', 'autograph', 'both');
        }
        
        // Apply date range filter if provided
        if (!empty($date_range['start'])) {
            $this->db->where('ln.dateadded >=', $date_range['start']);
        }
        if (!empty($date_range['end'])) {
            $this->db->where('ln.dateadded <=', $date_range['end']);
        }
        
        $this->db->order_by('ln.dateadded', 'DESC');
        
        return $this->db->get()->result_array();
    }

    /**
     * Clean and prepare text for analysis
     * @param string $text Raw text
     * @return string Cleaned text
     */
    private function clean_text($text)
    {
        // Remove HTML tags
        $text = strip_tags($text);
        
        // Decode HTML entities
        $text = html_entity_decode($text, ENT_QUOTES | ENT_HTML5, 'UTF-8');
        
        // Normalize whitespace
        $text = preg_replace('/\s+/', ' ', $text);
        
        // Trim
        $text = trim($text);
        
        return $text;
    }

    /**
     * Group notes by lead ID and clean them
     * @param array $notes Raw notes from database
     * @return array Grouped and cleaned notes
     */
    public function prepare_notes_for_analysis($notes)
    {
        $grouped = [];
        $lead_ids = [];
        
        foreach ($notes as $note) {
            $lead_id = $note['leadid'];
            $lead_ids[$lead_id] = true;
            
            $cleaned_description = $this->clean_text($note['description']);
            
            // Skip empty notes
            if (empty($cleaned_description)) {
                continue;
            }
            
            if (!isset($grouped[$lead_id])) {
                $grouped[$lead_id] = [
                    'lead_id' => $lead_id,
                    'lead_name' => $note['lead_name'] ?? '',
                    'company' => $note['company'] ?? '',
                    'notes' => []
                ];
            }
            
            $grouped[$lead_id]['notes'][] = [
                'description' => $cleaned_description,
                'dateadded' => $note['dateadded'],
                'addedfrom' => $note['addedfrom']
            ];
        }
        
        return [
            'grouped_notes' => $grouped,
            'total_leads' => count($lead_ids),
            'total_notes' => count($notes)
        ];
    }

    /**
     * Compile notes into a single text for OpenAI analysis
     * @param array $prepared_data Prepared notes data
     * @return string Compiled text
     */
    public function compile_notes_text($prepared_data)
    {
        $text = "Customer Feedback Analysis for Autograph Serviced Apartments\n\n";
        $text .= "Total Leads: " . $prepared_data['total_leads'] . "\n";
        $text .= "Total Notes: " . $prepared_data['total_notes'] . "\n\n";
        $text .= "=== FEEDBACK BY LEAD ===\n\n";
        
        $max_chars_per_note = 500; // Limit each note to 500 characters
        $note_count = 0;
        $max_total_notes = 200; // Limit total notes to prevent token overflow
        
        foreach ($prepared_data['grouped_notes'] as $lead_id => $lead_data) {
            if ($note_count >= $max_total_notes) {
                $text .= "\n[Note: Additional leads were omitted due to size constraints]\n";
                break;
            }
            
            $text .= "Lead: " . ($lead_data['lead_name'] ?: 'ID ' . $lead_id);
            if (!empty($lead_data['company'])) {
                $text .= " (" . $lead_data['company'] . ")";
            }
            $text .= "\n";
            
            foreach ($lead_data['notes'] as $index => $note) {
                if ($note_count >= $max_total_notes) {
                    break;
                }
                
                $description = $note['description'];
                // Truncate long notes
                if (strlen($description) > $max_chars_per_note) {
                    $description = substr($description, 0, $max_chars_per_note) . '...';
                }
                
                $text .= "  - " . $description . "\n";
                $note_count++;
            }
            
            $text .= "\n";
        }
        
        return $text;
    }

    /**
     * Call OpenAI API to analyze feedback
     * @param string $feedback_text Compiled feedback text
     * @return array Analysis results
     */
    public function analyze_with_openai($feedback_text)
    {
        $api_key = $this->get_setting('openai_api_key');
        $model = $this->get_setting('openai_model') ?: 'gpt-4-turbo-preview';
        
        if (empty($api_key)) {
            return [
                'success' => false,
                'error' => 'OpenAI API key is not configured'
            ];
        }
        
        // Set max tokens based on model
        $max_tokens = 2000;
        if (strpos($model, 'turbo-preview') !== false || strpos($model, '128k') !== false) {
            $max_tokens = 4000; // Allow longer responses for larger context models
        }
        
        $prompt = "You are a market sentiment and consumer behavior analyst. 

Analyze the following customer feedback from sales calls, follow-ups, and lead notes.

Provide your response in JSON format with the following structure:
{
    \"sentiment\": \"positive\" or \"neutral\" or \"negative\",
    \"sentiment_score\": 0.0 to 1.0 (where 1.0 is most positive),
    \"summary\": \"Brief summary of overall feedback\",
    \"positive_highlights\": [\"highlight 1\", \"highlight 2\", ...],
    \"negative_concerns\": [\"concern 1\", \"concern 2\", ...],
    \"market_narrative\": \"How customers perceive Autograph Serviced Apartments\",
    \"action_suggestions\": [\"suggestion 1\", \"suggestion 2\", ...]
}

Feedback:
" . $feedback_text;
        
        $data = [
            'model' => $model,
            'messages' => [
                [
                    'role' => 'system',
                    'content' => 'You are a market sentiment and consumer behavior analyst. Always respond with valid JSON.'
                ],
                [
                    'role' => 'user',
                    'content' => $prompt
                ]
            ],
            'temperature' => 0.7,
            'max_tokens' => $max_tokens
        ];
        
        $ch = curl_init('https://api.openai.com/v1/chat/completions');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Authorization: Bearer ' . $api_key
        ]);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        if ($error) {
            return [
                'success' => false,
                'error' => 'CURL Error: ' . $error
            ];
        }
        
        if ($http_code !== 200) {
            $error_data = json_decode($response, true);
            return [
                'success' => false,
                'error' => 'OpenAI API Error: ' . ($error_data['error']['message'] ?? 'Unknown error'),
                'http_code' => $http_code
            ];
        }
        
        $response_data = json_decode($response, true);
        
        if (!isset($response_data['choices'][0]['message']['content'])) {
            return [
                'success' => false,
                'error' => 'Invalid response from OpenAI API'
            ];
        }
        
        $content = $response_data['choices'][0]['message']['content'];
        
        // Try to extract JSON from the response
        // Sometimes OpenAI wraps JSON in markdown code blocks
        if (preg_match('/```(?:json)?\s*(\{.*?\})\s*```/s', $content, $matches)) {
            $content = $matches[1];
        } elseif (preg_match('/\{.*\}/s', $content, $matches)) {
            $content = $matches[0];
        }
        
        $analysis = json_decode($content, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            return [
                'success' => false,
                'error' => 'Failed to parse JSON response: ' . json_last_error_msg(),
                'raw_response' => $content
            ];
        }
        
        return [
            'success' => true,
            'analysis' => $analysis
        ];
    }

    /**
     * Save analysis report to database
     * @param array $analysis Analysis data from OpenAI
     * @param int $notes_count Number of notes analyzed
     * @param int $leads_count Number of leads analyzed
     * @return int Report ID
     */
    public function save_report($analysis, $notes_count, $leads_count)
    {
        $data = [
            'sentiment' => strtolower($analysis['sentiment'] ?? 'neutral'),
            'sentiment_score' => isset($analysis['sentiment_score']) ? (float)$analysis['sentiment_score'] : null,
            'summary' => $analysis['summary'] ?? '',
            'positive_highlights' => isset($analysis['positive_highlights']) ? json_encode($analysis['positive_highlights']) : null,
            'negative_concerns' => isset($analysis['negative_concerns']) ? json_encode($analysis['negative_concerns']) : null,
            'market_narrative' => $analysis['market_narrative'] ?? '',
            'action_suggestions' => isset($analysis['action_suggestions']) ? json_encode($analysis['action_suggestions']) : null,
            'notes_analyzed' => $notes_count,
            'leads_analyzed' => $leads_count,
            'analysis_date' => date('Y-m-d H:i:s'),
            'created_at' => date('Y-m-d H:i:s'),
            'updated_at' => date('Y-m-d H:i:s')
        ];
        
        $this->db->insert($this->report_table, $data);
        return $this->db->insert_id();
    }

    /**
     * Get latest report
     * @return object|null
     */
    public function get_latest_report()
    {
        $this->db->order_by('analysis_date', 'DESC');
        $this->db->limit(1);
        return $this->db->get($this->report_table)->row();
    }

    /**
     * Get a specific report by ID
     * @param int $id Report ID
     * @return object|null
     */
    public function get_report($id)
    {
        $this->db->where('id', $id);
        return $this->db->get($this->report_table)->row();
    }

    /**
     * Get all reports
     * @param int $limit
     * @return array
     */
    public function get_all_reports($limit = 10)
    {
        $this->db->order_by('analysis_date', 'DESC');
        $this->db->limit($limit);
        return $this->db->get($this->report_table)->result();
    }

    /**
     * Calculate date range from type
     */
    private function get_date_range_from_type($type, $custom_start = null, $custom_end = null)
    {
        $range = [];
        
        switch ($type) {
            case 'last_7_days':
                $range['start'] = date('Y-m-d 00:00:00', strtotime('-7 days'));
                $range['end'] = date('Y-m-d 23:59:59');
                break;
            case 'last_30_days':
                $range['start'] = date('Y-m-d 00:00:00', strtotime('-30 days'));
                $range['end'] = date('Y-m-d 23:59:59');
                break;
            case 'last_90_days':
                $range['start'] = date('Y-m-d 00:00:00', strtotime('-90 days'));
                $range['end'] = date('Y-m-d 23:59:59');
                break;
            case 'this_month':
                $range['start'] = date('Y-m-01 00:00:00');
                $range['end'] = date('Y-m-d 23:59:59');
                break;
            case 'last_month':
                $range['start'] = date('Y-m-01 00:00:00', strtotime('first day of last month'));
                $range['end'] = date('Y-m-t 23:59:59', strtotime('last day of last month'));
                break;
            case 'custom':
                if ($custom_start && $custom_end) {
                    $range['start'] = $custom_start . ' 00:00:00';
                    $range['end'] = $custom_end . ' 23:59:59';
                }
                break;
        }
        
        return $range;
    }

    /**
     * Execute scheduled report
     */
    public function execute_scheduled_report($schedule)
    {
        // Parse status filter
        $status_filter = json_decode($schedule->status_filter, true) ?: [];
        
        // Parse date range
        $date_range = $this->get_date_range_from_type(
            $schedule->date_range_type,
            $schedule->custom_start_date,
            $schedule->custom_end_date
        );
        
        // Get notes with filters
        $notes = $this->get_autograph_notes($status_filter, $date_range);
        
        if (empty($notes)) {
            // Update next run anyway
            $this->update_schedule_after_execution($schedule);
            return [
                'success' => false,
                'message' => 'No notes found with the specified filters'
            ];
        }
        
        // Prepare and analyze
        $prepared_data = $this->prepare_notes_for_analysis($notes);
        $feedback_text = $this->compile_notes_text($prepared_data);
        $analysis_result = $this->analyze_with_openai($feedback_text);
        
        if (!$analysis_result['success']) {
            $this->update_schedule_after_execution($schedule);
            return [
                'success' => false,
                'message' => $analysis_result['error'] ?? 'Analysis failed'
            ];
        }
        
        // Save report
        $report_id = $this->save_report(
            $analysis_result['analysis'],
            $prepared_data['total_notes'],
            $prepared_data['total_leads']
        );
        
        // Get the report
        $report = $this->get_report($report_id);
        
        // Send to WhatsApp numbers
        $whatsapp_numbers = json_decode($schedule->whatsapp_numbers, true) ?: [];
        $sent_success = 0;
        $sent_failed = 0;
        
        foreach ($whatsapp_numbers as $phone) {
            $send_result = $this->send_report_to_whatsapp($report, $phone);
            if ($send_result['success']) {
                $sent_success++;
            } else {
                $sent_failed++;
            }
        }
        
        // Update schedule
        $this->update_schedule_after_execution($schedule);
        
        return [
            'success' => true,
            'report_id' => $report_id,
            'notes_analyzed' => $prepared_data['total_notes'],
            'leads_analyzed' => $prepared_data['total_leads'],
            'recipients_count' => count($whatsapp_numbers),
            'sent_success' => $sent_success,
            'sent_failed' => $sent_failed
        ];
    }

    /**
     * Update schedule after execution
     */
    private function update_schedule_after_execution($schedule)
    {
        $next_run = $this->calculate_next_run_from_schedule($schedule);
        
        $this->db->where('id', $schedule->id);
        $this->db->update($this->schedules_table, [
            'last_run' => date('Y-m-d H:i:s'),
            'next_run' => $next_run
        ]);
    }

    /**
     * Calculate next run from schedule object
     */
    private function calculate_next_run_from_schedule($schedule)
    {
        $time = $schedule->schedule_time;
        $frequency = $schedule->schedule_frequency;
        $day = $schedule->schedule_day;

        $now = time();
        $current_datetime = strtotime($schedule->last_run ?: date('Y-m-d H:i:s'));

        switch ($frequency) {
            case 'daily':
                $next_run = date('Y-m-d H:i:s', strtotime('+1 day', strtotime(date('Y-m-d', $current_datetime) . ' ' . $time)));
                break;
            case 'weekly':
                $next_run = date('Y-m-d H:i:s', strtotime('+1 week', strtotime(date('Y-m-d', $current_datetime) . ' ' . $time)));
                break;
            case 'monthly':
                $next_run = date('Y-m-d H:i:s', strtotime('+1 month', strtotime(date('Y-m-d', $current_datetime) . ' ' . $time)));
                break;
            default:
                $next_run = date('Y-m-d H:i:s', strtotime('+1 day', $current_datetime));
        }

        return $next_run;
    }

    /**
     * Send report to WhatsApp
     */
    public function send_report_to_whatsapp($report, $phone_number)
    {
        // Get WhatsApp credentials
        $phone_number_id = get_option('phone_number_id');
        $whatsapp_access_token = get_option('whatsapp_access_token');
        
        if (empty($phone_number_id) || empty($whatsapp_access_token)) {
            return [
                'success' => false,
                'error' => 'WhatsApp API is not configured'
            ];
        }

        // Format report
        $message_text = $this->format_report_for_whatsapp($report);
        
        // Prepare send data
        $send_data = [
            'messaging_product' => 'whatsapp',
            'recipient_type' => 'individual',
            'type' => 'text',
            'text' => ['body' => $message_text],
            'to' => $phone_number
        ];

        // Send via API
        $url = "https://graph.facebook.com/v18.0/{$phone_number_id}/messages";
        
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($send_data));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Authorization: Bearer ' . $whatsapp_access_token
        ]);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code == 200) {
            return ['success' => true];
        } else {
            $error_data = json_decode($response, true);
            return [
                'success' => false,
                'error' => $error_data['error']['message'] ?? 'Unknown error'
            ];
        }
    }

    /**
     * Format report for WhatsApp
     */
    private function format_report_for_whatsapp($report)
    {
        $positive_highlights = json_decode($report->positive_highlights, true) ?: [];
        $negative_concerns = json_decode($report->negative_concerns, true) ?: [];
        $action_suggestions = json_decode($report->action_suggestions, true) ?: [];
        
        $sentiment_emoji = '😐';
        if ($report->sentiment == 'positive') {
            $sentiment_emoji = '😊';
        } elseif ($report->sentiment == 'negative') {
            $sentiment_emoji = '😞';
        }
        
        $message = "📊 *Autograph Feedback Insights Report*\n";
        $message .= "━━━━━━━━━━━━━━━━━━━━\n\n";
        
        $message .= "📅 *Analysis Date:* " . date('F j, Y g:i A', strtotime($report->analysis_date)) . "\n";
        $message .= "📝 *Notes Analyzed:* " . $report->notes_analyzed . "\n";
        $message .= "👥 *Leads Analyzed:* " . $report->leads_analyzed . "\n\n";
        
        $message .= "{$sentiment_emoji} *Overall Sentiment:* " . strtoupper($report->sentiment) . "\n";
        if ($report->sentiment_score !== null) {
            $message .= "📈 *Score:* " . number_format($report->sentiment_score * 100, 1) . "%\n";
        }
        $message .= "\n━━━━━━━━━━━━━━━━━━━━\n\n";
        
        if (!empty($report->summary)) {
            $message .= "*📋 Summary:*\n" . substr($report->summary, 0, 400) . "\n\n";
        }
        
        if (!empty($positive_highlights)) {
            $message .= "*✅ Key Positive Highlights:*\n";
            foreach (array_slice($positive_highlights, 0, 5) as $highlight) {
                $message .= "• " . $highlight . "\n";
            }
            $message .= "\n";
        }
        
        if (!empty($negative_concerns)) {
            $message .= "*⚠️ Key Concerns:*\n";
            foreach (array_slice($negative_concerns, 0, 5) as $concern) {
                $message .= "• " . $concern . "\n";
            }
            $message .= "\n";
        }
        
        if (!empty($action_suggestions)) {
            $message .= "*💡 Top Suggestions:*\n";
            foreach (array_slice($action_suggestions, 0, 3) as $suggestion) {
                $message .= "• " . $suggestion . "\n";
            }
        }
        
        $message .= "\n━━━━━━━━━━━━━━━━━━━━\n";
        $message .= "_Automated Report by Autograph Insights_";
        
        // WhatsApp limit
        if (strlen($message) > 4000) {
            $message = substr($message, 0, 3990) . "...\n\n_Full report in CRM_";
        }
        
        return $message;
    }

    /**
     * Run complete analysis process
     * @return array Result with success status and message
     */
    public function run_analysis()
    {
        // Get status filter from settings
        $status_filter_setting = $this->get_setting('status_filter');
        $status_filter = [];
        
        if (!empty($status_filter_setting)) {
            $status_filter = json_decode($status_filter_setting, true);
            if (!is_array($status_filter)) {
                $status_filter = [];
            }
        }
        
        // Get notes
        $notes = $this->get_autograph_notes($status_filter);
        
        if (empty($notes)) {
            return [
                'success' => false,
                'message' => 'No notes found related to Autograph Serviced Apartments'
            ];
        }
        
        // Prepare notes
        $prepared_data = $this->prepare_notes_for_analysis($notes);
        
        if ($prepared_data['total_notes'] == 0) {
            return [
                'success' => false,
                'message' => 'No valid notes found after cleaning'
            ];
        }
        
        // Compile text
        $feedback_text = $this->compile_notes_text($prepared_data);
        
        // Analyze with OpenAI
        $analysis_result = $this->analyze_with_openai($feedback_text);
        
        if (!$analysis_result['success']) {
            return [
                'success' => false,
                'message' => $analysis_result['error'] ?? 'Analysis failed'
            ];
        }
        
        // Save report
        $report_id = $this->save_report(
            $analysis_result['analysis'],
            $prepared_data['total_notes'],
            $prepared_data['total_leads']
        );
        
        // Update last analysis date
        $this->update_setting('last_analysis_date', date('Y-m-d H:i:s'));
        
        return [
            'success' => true,
            'message' => 'Analysis completed successfully',
            'report_id' => $report_id,
            'notes_analyzed' => $prepared_data['total_notes'],
            'leads_analyzed' => $prepared_data['total_leads']
        ];
    }
}

