// Handles adding/removing phone numbers in the leads form
function initPhoneNumbersHandler() {
    // Remove any existing click handlers to prevent duplicates
    $('#add-phone-number').off('click');
    
    $('#add-phone-number').on('click', function() {
        const phoneCount = $('#additional-phone-numbers .extra-phone-number').length;
        
        const template = `
            <div class="extra-phone-number mtop10">
                <div class="input-group">
                    <input type="text" name="additional_phones[${phoneCount}]" class="form-control" placeholder="Additional Phone Number" />
                    <div class="input-group-btn">
                        <button type="button" class="btn btn-danger remove-phone" onclick="removePhone(this)">
                            <i class="fa fa-minus"></i>
                        </button>
                    </div>
                </div>
            </div>
        `;
        const $newPhone = $(template);
        $('#additional-phone-numbers').append($newPhone);
        const $input = $newPhone.find('input');
        $input.focus();
        
        // Set up duplicate check on blur
        $input.on('blur', function() {
            checkDuplicatePhone($(this));
        });
        
        // After adding a new phone, reindex all phone inputs to ensure they're sequential
        reindexPhoneInputs();
    });

    // Use event delegation for remove button clicks (as a backup to the onclick handler)
    $(document).off('click', '.remove-phone').on('click', '.remove-phone', function() {
        $(this).closest('.extra-phone-number').remove();
        reindexPhoneInputs();
    });
    
    // Add blur handler to existing phone fields too
    $('.phone-numbers input[name="phonenumber"]').off('blur').on('blur', function() {
        checkDuplicatePhone($(this));
    });
    
    $('#additional-phone-numbers input').each(function() {
        $(this).off('blur').on('blur', function() {
            checkDuplicatePhone($(this));
        });
    });
}

function stylePhoneNumberInput($element) {
    // Add any additional styling if needed
    $element.find('input').focus();
}

function removePhone(element) {
    $(element).closest('.extra-phone-number').remove();
    reindexPhoneInputs(); // Reindex remaining phone numbers after removal
}

function reindexPhoneInputs() {
    $('#additional-phone-numbers .extra-phone-number').each(function(index) {
        $(this).find('input').attr('name', `additional_phones[${index}]`);
    });
}

/**
 * Check if a phone number already exists in the database
 * Shows a notification with lead details if the number exists
 */
function checkDuplicatePhone($input) {
    const phone = $input.val().trim();
    if (!phone) return; // Skip empty values
    
    // Get current lead ID if we're editing a lead
    let leadId = $('input[name="leadid"]').val();
    
    $.ajax({
        url: admin_url + 'leads/check_phone_exists',
        type: 'POST',
        data: {
            phone: phone,
            lead_id: leadId
        },
        dataType: 'json',
        success: function(response) {
            if (response.exists && response.lead) {
                const lead = response.lead;
                
                // Create a detailed notification
                const statusClass = lead.status === 'Lost' ? 'danger' : 
                                   (lead.status === 'Customer' ? 'success' : 'warning');
                
                // Remove any existing duplicate warnings for this input
                $input.closest('.input-group').find('.duplicate-phone-warning').remove();
                
                // Show a detailed warning next to the input
                const $warning = $(`
                    <div class="duplicate-phone-warning text-${statusClass}" style="margin-top:5px;">
                        <i class="fa fa-exclamation-triangle"></i> 
                        <strong>Duplicate phone number!</strong> 
                        <div class="duplicate-lead-details">
                            <div><strong>Name:</strong> ${lead.name}</div>
                            ${lead.email ? `<div><strong>Email:</strong> ${lead.email}</div>` : ''}
                            ${lead.company ? `<div><strong>Company:</strong> ${lead.company}</div>` : ''}
                            <div><strong>Status:</strong> <span class="label label-${statusClass}">${lead.status}</span></div>
                            <div><strong><a href="${admin_url}leads/index/${lead.id}" target="_blank">View Lead</a></strong></div>
                        </div>
                    </div>
                `);
                
                $input.closest('.input-group').after($warning);
                
                // Also show an alert
                alert_float('warning', 'This phone number already exists in the database for lead: ' + lead.name);
            } else {
                // Remove any existing warnings if no duplicate
                $input.closest('.input-group').siblings('.duplicate-phone-warning').remove();
            }
        },
        error: function() {
            console.error('Error checking for duplicate phone number');
        }
    });
}

// First remove any existing handlers to prevent duplicates
$(document).off('submit', '#lead-note-form, #lead-modal-note-form');

// Track if we're currently submitting to prevent multiple submissions
// Use window-level variable to ensure it persists across modal refreshes
window.isSubmitting = window.isSubmitting || false;

$(document).on('submit', '#lead-note-form, #lead-modal-note-form', function(e) {
    console.log('Form submission detected:', this.id, 'isSubmitting:', window.isSubmitting);
    
    e.preventDefault();
    
    // If already submitting, don't allow another submission
    if (window.isSubmitting) {
        console.log('Preventing duplicate submission');
        return false;
    }
    
    var form = this; // Use the raw DOM element!
    var formId = form.id;
    window.isSubmitting = true;
    
    console.log('Starting form submission for:', formId);
    
    var data = new FormData(form);
    
    // Prevent multiple submissions
    var $submitBtn = $(form).find('button[type="submit"]');
    $submitBtn.addClass('disabled').prop('disabled', true);
    
    $.ajax({
        url: $(form).attr('action'),
        type: 'POST',
        data: data,
        processData: false,
        contentType: false,
        success: function(response) {
            // Re-enable the submit button
            $submitBtn.removeClass('disabled').prop('disabled', false);
            
            // Clear the form
            form.reset();
            
            try {
                if (typeof response === 'string') {
                    response = JSON.parse(response);
                }
                
                if (response.leadView) {
                    // Update the entire modal with fresh data
                    _lead_init_data(response, response.id);
                    
                    // Show success/error message from server
                    if (response.success) {
                        alert_float('success', response.message || 'Note added successfully');
                    } else {
                        alert_float('danger', response.message || 'Failed to add note');
                    }
                } else {
                    alert_float('danger', response.message || 'Failed to add note');
                }
            } catch (e) {
                console.error('Error processing response:', e);
                alert_float('danger', 'Error processing server response');
            }
            
            // Reset the submission flag so subsequent submissions are possible
            window.isSubmitting = false;
        },
        error: function() {
            $submitBtn.removeClass('disabled').prop('disabled', false);
            alert_float('danger', 'Failed to add note');
            
            // Reset the submission flag on error
            window.isSubmitting = false;
        },
        complete: function() {
            // Ensure the flag is reset in all cases including network issues
            window.isSubmitting = false;
        }
    });
});

// Delete note
function delete_lead_note(note_id) {
    if (confirm_delete()) {
        $.get(admin_url + 'leads/delete_note/' + note_id, function(response) {
            if (response.success) {
                $('[data-note-id="' + note_id + '"]').remove();
                alert_float('success', 'Note deleted successfully');
            }
        }, 'json');
    }
}

// Delete note attachment - DISABLED
// function delete_note_attachment(note_id) {
//     if (confirm_delete()) {
//         $.get(admin_url + 'leads/delete_note_attachment/' + note_id, function(response) {
//             if (response.success) {
//                 $('[data-note-id="' + note_id + '"] .note-attachment').fadeOut(400, function() {
//                     $(this).remove();
//                 });
//                 alert_float('success', 'Attachment deleted successfully');
//             }
//         }, 'json');
//     }
// }

// Initialize when document is ready
$(document).ready(function() {
    initPhoneNumbersHandler();
    
    // Debug: Log the initialization of duplicate prevention
    console.log('🔧 Lead notes duplicate prevention initialized');
    console.log('🔧 window.isSubmitting:', window.isSubmitting);
});

// Handle dynamic loading in modal
$(document).on('app.form-filled app.lead-modal-loaded', function(e, data) {
    initPhoneNumbersHandler();
    
    // Debug: Log modal reload events
    console.log('🔧 Lead modal reloaded, resetting submission state');
    window.isSubmitting = false;
});
