// This script should be added to the lead detail page for testing
// It monitors the AJAX requests to check if descriptions are being sent properly

// Wait for page to fully load
document.addEventListener('DOMContentLoaded', function() {
    console.log('Lead attachment description monitoring initialized');
    
    // Create a UI panel for monitoring
    const monitorPanel = document.createElement('div');
    monitorPanel.style.position = 'fixed';
    monitorPanel.style.bottom = '10px';
    monitorPanel.style.right = '10px';
    monitorPanel.style.width = '350px';
    monitorPanel.style.maxHeight = '300px';
    monitorPanel.style.overflowY = 'auto';
    monitorPanel.style.backgroundColor = 'rgba(0,0,0,0.8)';
    monitorPanel.style.color = 'white';
    monitorPanel.style.padding = '10px';
    monitorPanel.style.borderRadius = '5px';
    monitorPanel.style.zIndex = '9999';
    monitorPanel.style.fontSize = '12px';
    monitorPanel.innerHTML = '<h3 style="margin-top:0;">Description Monitor</h3><div id="monitor-log"></div>';
    
    document.body.appendChild(monitorPanel);
    
    const logDiv = document.getElementById('monitor-log');
    
    // Function to log messages
    function logMessage(message, type = 'info') {
        const now = new Date();
        const timestamp = now.toLocaleTimeString();
        const logEntry = document.createElement('div');
        logEntry.style.marginBottom = '5px';
        logEntry.style.padding = '3px';
        logEntry.style.borderLeft = '3px solid ' + (type === 'info' ? 'blue' : 'red');
        logEntry.innerHTML = `<strong>${timestamp}</strong>: ${message}`;
        logDiv.appendChild(logEntry);
        logDiv.scrollTop = logDiv.scrollHeight;
    }
    
    // Override the XMLHttpRequest object to monitor AJAX requests
    const originalXHROpen = XMLHttpRequest.prototype.open;
    const originalXHRSend = XMLHttpRequest.prototype.send;
    
    XMLHttpRequest.prototype.open = function(method, url) {
        this._method = method;
        this._url = url;
        return originalXHROpen.apply(this, arguments);
    };
    
    XMLHttpRequest.prototype.send = function(data) {
        // Check if it's a lead attachment request
        if (this._url.includes('leads/add_lead_attachment') || 
            this._url.includes('upload_file')) {
            
            logMessage(`AJAX Request to ${this._url}`);
            
            // Check for description in FormData
            if (data instanceof FormData) {
                const description = data.get('description');
                if (description) {
                    logMessage(`✅ Found description: "${description}"`);
                } else {
                    logMessage(`❌ No description found in FormData`, 'error');
                    // List all form fields for debugging
                    logMessage('Form fields: ' + Array.from(data.keys()).join(', '));
                }
            } else {
                logMessage('❌ Not using FormData for upload', 'error');
            }
        }
        
        return originalXHRSend.apply(this, arguments);
    };
    
    // Monitor Dropzone initialization
    const originalDropzone = window.Dropzone;
    if (originalDropzone) {
        logMessage('Monitoring Dropzone initialization');
        
        window.Dropzone = function(...args) {
            const instance = new originalDropzone(...args);
            
            logMessage('Dropzone initialized');
            
            // Monitor options
            if (instance.options) {
                const options = instance.options;
                if (options.params && options.params.description) {
                    logMessage(`✅ Dropzone has description param: "${options.params.description}"`);
                } else if (!options.params || !options.params.description) {
                    logMessage(`❌ No description param in Dropzone options`, 'error');
                }
            }
            
            // Monitor sending event
            instance.on('sending', function(file, xhr, formData) {
                logMessage(`Sending file: ${file.name}`);
                
                setTimeout(() => {
                    // Check if description is in formData
                    let hasDescription = false;
                    for (const pair of formData.entries()) {
                        if (pair[0] === 'description') {
                            hasDescription = true;
                            logMessage(`✅ Sending description: "${pair[1]}"`);
                        }
                    }
                    
                    if (!hasDescription) {
                        logMessage(`❌ No description found in sending data`, 'error');
                    }
                }, 10);
            });
            
            return instance;
        };
        
        for (const key in originalDropzone) {
            window.Dropzone[key] = originalDropzone[key];
        }
    } else {
        logMessage('❌ Dropzone not found', 'error');
    }
    
    // Monitor the description fields
    setInterval(() => {
        const descInputs = document.querySelectorAll('input[name="description"]');
        if (descInputs.length > 0) {
            for (let i = 0; i < descInputs.length; i++) {
                const value = descInputs[i].value;
                const isVisible = descInputs[i].offsetParent !== null;
                
                if (isVisible && !descInputs[i]._monitored) {
                    logMessage(`Found description field with value: "${value}"`);
                    descInputs[i]._monitored = true;
                    
                    // Monitor changes
                    descInputs[i].addEventListener('change', function() {
                        logMessage(`Description changed to: "${this.value}"`);
                    });
                }
            }
        }
    }, 2000);
    
    logMessage('Monitoring system ready');
});
