<?php

defined('BASEPATH') or exit('No direct script access allowed');

/**
 * AI Analysis Library
 * 
 * Universal AI analysis module for OpenAI integration across the CRM
 * Provides multimodal analysis, file processing, logging, and reporting
 * 
 * @package    Application
 * @subpackage Libraries
 * @category   AI Analysis
 * @author     Ibraya Group
 * @since      Version 1.0.0
 * 
 * Usage Examples:
 * 
 * // Load the library
 * $this->load->library('AI_Analysis');
 * 
 * // Basic text analysis
 * $result = $this->ai_analysis->analyze_text("Analyze this lead behavior", $data);
 * 
 * // Multimodal analysis with files
 * $result = $this->ai_analysis->analyze_with_files($prompt, $data, $file_paths);
 * 
 * // Customer analysis with predefined template
 * $result = $this->ai_analysis->analyze_customer($customer_data, $files);
 * 
 * // Lead analysis with predefined template
 * $result = $this->ai_analysis->analyze_lead($lead_data, $files);
 */
class AI_Analysis
{
    protected $CI;
    protected $api_key;
    protected $default_model = 'gpt-4o';
    protected $default_temperature = 0.3;
    protected $default_max_tokens = 1000;
    protected $logs_table;
    
    // Supported file types
    protected $image_types = ['jpg', 'jpeg', 'png', 'gif', 'webp', 'bmp'];
    protected $document_types = ['pdf', 'doc', 'docx', 'txt', 'csv', 'xls', 'xlsx'];
    
    // File size limits (bytes)
    protected $max_image_size = 20 * 1024 * 1024; // 20MB
    protected $max_pdf_size = 32 * 1024 * 1024;   // 32MB
    protected $max_doc_size = 10 * 1024 * 1024;   // 10MB
    protected $max_payload_size = 25 * 1024 * 1024; // 25MB total payload
    
    /**
     * Constructor
     */
    public function __construct($config = [])
    {
        $this->CI =& get_instance();
        $this->CI->load->helper(['ai_file_parser', 'file']);
        
        // Initialize configuration
        $this->api_key = get_option('openai_api_key') ?: ($config['api_key'] ?? '');
        $this->default_model = get_option('ai_model') ?: ($config['model'] ?? $this->default_model);
        $this->default_temperature = get_option('ai_temperature') ?: ($config['temperature'] ?? $this->default_temperature);
        $this->default_max_tokens = get_option('ai_max_tokens') ?: ($config['max_tokens'] ?? $this->default_max_tokens);
        
        $this->logs_table = db_prefix() . 'ai_api_logs';
        
        // Create logs table if it doesn't exist
        $this->_ensure_logs_table();
        
        log_message('info', 'AI_Analysis library initialized');
    }
    
    /**
     * Analyze text with OpenAI
     * 
     * @param  string $prompt Analysis prompt
     * @param  array  $data   Additional data for context
     * @param  array  $options API options (model, temperature, etc.)
     * @return array  Analysis result
     */
    public function analyze_text($prompt, $data = [], $options = [])
    {
        try {
            // Validate API key
            if (empty($this->api_key)) {
                return $this->_error_response('OpenAI API key not configured');
            }
            
            // Merge options with defaults
            $api_options = array_merge([
                'model' => $this->default_model,
                'temperature' => $this->default_temperature,
                'max_tokens' => $this->default_max_tokens,
                'response_format' => null
            ], $options);
            
            // Build the prompt with data if provided
            $final_prompt = $this->_build_prompt($prompt, $data);
            
            // Make API call
            $api_result = $this->_make_openai_api_call($final_prompt, $api_options);
            
            // Log the API call
            $this->_log_api_call([
                'endpoint' => 'https://api.openai.com/v1/chat/completions',
                'method' => 'POST',
                'request_data' => json_encode(['prompt' => $prompt, 'options' => $api_options]),
                'response_data' => json_encode($api_result),
                'status_code' => $api_result['success'] ? 200 : 400,
                'response_time' => $api_result['response_time'] ?? null,
                'success' => $api_result['success'] ? 1 : 0,
                'error_message' => $api_result['success'] ? null : $api_result['error']
            ]);
            
            return $api_result;
            
        } catch (Exception $e) {
            $error_response = $this->_error_response('Analysis failed: ' . $e->getMessage());
            $this->_log_api_call([
                'endpoint' => 'https://api.openai.com/v1/chat/completions',
                'request_data' => 'Error before API call',
                'error_message' => $e->getMessage(),
                'success' => 0
            ]);
            return $error_response;
        }
    }
    
    /**
     * Analyze with multimodal content (text + images/files)
     * 
     * @param  string $prompt     Analysis prompt
     * @param  array  $data       Additional data for context
     * @param  array  $file_paths Array of file paths to include
     * @param  array  $options    API options
     * @return array  Analysis result
     */
    public function analyze_with_files($prompt, $data = [], $file_paths = [], $options = [])
    {
        try {
            // Validate API key
            if (empty($this->api_key)) {
                return $this->_error_response('OpenAI API key not configured');
            }
            
            // Validate files
            $file_validation = $this->validate_files($file_paths);
            if (!$file_validation['valid']) {
                return $this->_error_response('File validation failed: ' . implode(', ', $file_validation['errors']));
            }
            
            // Parse files
            $parsed_files = $this->parse_files($file_paths);
            
            // Prepare multimodal content
            $content_array = $this->_prepare_multimodal_content($prompt, $data, $parsed_files, $file_paths);
            
            // Merge options with defaults
            $api_options = array_merge([
                'model' => $this->default_model,
                'temperature' => $this->default_temperature,
                'max_tokens' => $this->default_max_tokens,
                'response_format' => ['type' => 'json_object']
            ], $options);
            
            // Make enhanced API call
            $api_result = $this->_make_enhanced_openai_api_call($content_array, $api_options);
            
            // If enhanced call fails, try text-only fallback
            if (!$api_result['success']) {
                log_message('error', 'Multimodal API call failed, trying text-only fallback: ' . $api_result['error']);
                
                $text_prompt = $this->_build_prompt($prompt, $data, $parsed_files);
                $text_prompt = $this->_sanitize_text($text_prompt);
                
                unset($api_options['response_format']); // Remove JSON format for fallback
                $api_result = $this->_make_openai_api_call($text_prompt, $api_options);
            }
            
            // Log the API call
            $this->_log_api_call([
                'endpoint' => 'https://api.openai.com/v1/chat/completions',
                'method' => 'POST',
                'request_data' => json_encode([
                    'prompt' => $prompt,
                    'files_count' => count($file_paths),
                    'options' => $api_options
                ]),
                'response_data' => json_encode($api_result),
                'status_code' => $api_result['success'] ? 200 : 400,
                'response_time' => $api_result['response_time'] ?? null,
                'success' => $api_result['success'] ? 1 : 0,
                'error_message' => $api_result['success'] ? null : $api_result['error']
            ]);
            
            if ($api_result['success']) {
                $api_result['parsed_files'] = $parsed_files;
            }
            
            return $api_result;
            
        } catch (Exception $e) {
            $error_response = $this->_error_response('Multimodal analysis failed: ' . $e->getMessage());
            $this->_log_api_call([
                'endpoint' => 'https://api.openai.com/v1/chat/completions',
                'request_data' => 'Error before API call',
                'error_message' => $e->getMessage(),
                'success' => 0
            ]);
            return $error_response;
        }
    }
    
    /**
     * Analyze lead with predefined template
     * 
     * @param  array $lead_data   Lead data
     * @param  array $file_paths  Optional file paths
     * @param  array $options     API options
     * @return array Analysis result
     */
    public function analyze_lead($lead_data, $file_paths = [], $options = [])
    {
        $prompt_template = get_option('ai_lead_analysis_prompt') ?: $this->_get_default_lead_prompt();
        
        return $this->analyze_with_files($prompt_template, $lead_data, $file_paths, $options);
    }
    
    /**
     * Analyze customer with predefined template
     * 
     * @param  array $customer_data Customer data
     * @param  array $file_paths    Optional file paths
     * @param  array $options       API options
     * @return array Analysis result
     */
    public function analyze_customer($customer_data, $file_paths = [], $options = [])
    {
        $prompt_template = get_option('ai_customer_analysis_prompt') ?: $this->_get_default_customer_prompt();
        
        return $this->analyze_with_files($prompt_template, $customer_data, $file_paths, $options);
    }
    
    /**
     * Analyze project with predefined template
     * 
     * @param  array $project_data Project data
     * @param  array $file_paths   Optional file paths
     * @param  array $options      API options
     * @return array Analysis result
     */
    public function analyze_project($project_data, $file_paths = [], $options = [])
    {
        $prompt_template = get_option('ai_project_analysis_prompt') ?: $this->_get_default_project_prompt();
        
        return $this->analyze_with_files($prompt_template, $project_data, $file_paths, $options);
    }
    
    /**
     * Validate uploaded files
     * 
     * @param  array $file_paths Array of file paths to validate
     * @return array Validation result ['valid' => bool, 'errors' => array]
     */
    public function validate_files($file_paths)
    {
        $errors = [];
        $allowed_types = array_merge($this->image_types, $this->document_types);
        $max_files = 10;
        
        if (empty($file_paths)) {
            return ['valid' => true, 'errors' => []]; // No files is valid
        }
        
        if (count($file_paths) > $max_files) {
            $errors[] = "Maximum {$max_files} files allowed";
        }
        
        foreach ($file_paths as $file_path) {
            if (!file_exists($file_path)) {
                $errors[] = "File not found: " . basename($file_path);
                continue;
            }
            
            // Check file type
            $file_ext = strtolower(pathinfo($file_path, PATHINFO_EXTENSION));
            if (!in_array($file_ext, $allowed_types)) {
                $errors[] = "File " . basename($file_path) . " has invalid type. Allowed: " . implode(', ', $allowed_types);
                continue;
            }
            
            // Check file size based on type
            $file_size = filesize($file_path);
            $max_size = $this->max_doc_size; // Default
            $type_desc = "document";
            
            if (in_array($file_ext, $this->image_types)) {
                $max_size = $this->max_image_size;
                $type_desc = "image";
            } elseif ($file_ext === 'pdf') {
                $max_size = $this->max_pdf_size;
                $type_desc = "PDF";
            }
            
            if ($file_size > $max_size) {
                $size_mb = round($max_size / (1024 * 1024));
                $errors[] = "File " . basename($file_path) . " is too large. Maximum {$size_mb}MB for {$type_desc} files.";
            }
        }
        
        return [
            'valid' => empty($errors),
            'errors' => $errors
        ];
    }
    
    /**
     * Parse uploaded files for AI analysis
     * 
     * @param  array $file_paths Array of file paths to parse
     * @return array Array of parsed file contents
     */
    public function parse_files($file_paths)
    {
        $parsed_files = [];
        
        foreach ($file_paths as $file_path) {
            $filename = basename($file_path);
            
            if (!file_exists($file_path)) {
                $parsed_files[$filename] = [
                    'success' => false,
                    'error' => 'File not found: ' . $file_path,
                    'content' => '',
                    'metadata' => []
                ];
                continue;
            }
            
            // Get file MIME type
            $file_type = mime_content_type($file_path);
            if (!$file_type) {
                $file_type = strtolower(pathinfo($file_path, PATHINFO_EXTENSION));
            }
            
            // Parse the file using helper function
            if (function_exists('parse_file_for_ai_analysis')) {
                $parsed_result = parse_file_for_ai_analysis($file_path, $file_type);
            } else {
                $parsed_result = $this->_basic_file_parse($file_path, $file_type);
            }
            
            $parsed_files[$filename] = $parsed_result;
            
            log_message('info', "Parsed file: {$filename} - Success: " . ($parsed_result['success'] ? 'Yes' : 'No'));
        }
        
        return $parsed_files;
    }
    
    /**
     * Get API call statistics
     * 
     * @param  array $filters Optional filters (date_from, date_to, success_only, etc.)
     * @return array Statistics data
     */
    public function get_api_statistics($filters = [])
    {
        $this->CI->db->from($this->logs_table);
        
        // Apply filters
        if (!empty($filters['date_from'])) {
            $this->CI->db->where('created_at >=', $filters['date_from']);
        }
        if (!empty($filters['date_to'])) {
            $this->CI->db->where('created_at <=', $filters['date_to']);
        }
        if (isset($filters['success_only']) && $filters['success_only']) {
            $this->CI->db->where('success', 1);
        }
        if (!empty($filters['endpoint'])) {
            $this->CI->db->like('endpoint', $filters['endpoint']);
        }
        
        // Get total calls
        $total_calls = $this->CI->db->count_all_results($this->logs_table);
        
        // Reset query builder and apply filters again for other stats
        $this->CI->db->from($this->logs_table);
        if (!empty($filters['date_from'])) {
            $this->CI->db->where('created_at >=', $filters['date_from']);
        }
        if (!empty($filters['date_to'])) {
            $this->CI->db->where('created_at <=', $filters['date_to']);
        }
        if (!empty($filters['endpoint'])) {
            $this->CI->db->like('endpoint', $filters['endpoint']);
        }
        
        $this->CI->db->where('success', 1);
        $successful_calls = $this->CI->db->count_all_results($this->logs_table);
        
        // Get average response time for successful calls
        $this->CI->db->select_avg('response_time');
        $this->CI->db->from($this->logs_table);
        $this->CI->db->where('success', 1);
        $this->CI->db->where('response_time IS NOT NULL');
        
        if (!empty($filters['date_from'])) {
            $this->CI->db->where('created_at >=', $filters['date_from']);
        }
        if (!empty($filters['date_to'])) {
            $this->CI->db->where('created_at <=', $filters['date_to']);
        }
        
        $avg_response_time = $this->CI->db->get()->row()->response_time ?? 0;
        
        return [
            'total_calls' => $total_calls,
            'successful_calls' => $successful_calls,
            'failed_calls' => $total_calls - $successful_calls,
            'success_rate' => $total_calls > 0 ? round(($successful_calls / $total_calls) * 100, 1) : 0,
            'average_response_time' => round($avg_response_time, 2)
        ];
    }
    
    /**
     * Get recent API call logs
     * 
     * @param  int   $limit   Number of logs to retrieve
     * @param  array $filters Optional filters
     * @return array Array of log entries
     */
    public function get_recent_logs($limit = 50, $filters = [])
    {
        $this->CI->db->from($this->logs_table);
        $this->CI->db->order_by('created_at', 'DESC');
        $this->CI->db->limit($limit);
        
        // Apply filters
        if (isset($filters['success_only']) && $filters['success_only']) {
            $this->CI->db->where('success', 1);
        }
        if (isset($filters['errors_only']) && $filters['errors_only']) {
            $this->CI->db->where('success', 0);
        }
        if (!empty($filters['endpoint'])) {
            $this->CI->db->like('endpoint', $filters['endpoint']);
        }
        
        return $this->CI->db->get()->result();
    }
    
    /**
     * Parse structured AI response (expects JSON)
     * 
     * @param  string $response Raw AI response
     * @return array  Parsed response data
     */
    public function parse_ai_response($response)
    {
        // Try to parse as JSON first
        $json_data = json_decode(trim($response), true);
        
        if (json_last_error() === JSON_ERROR_NONE && is_array($json_data)) {
            return [
                'success' => true,
                'type' => 'json',
                'data' => $json_data,
                'raw_response' => $response
            ];
        } else {
            // Return as text response
            return [
                'success' => true,
                'type' => 'text',
                'data' => ['content' => trim($response)],
                'raw_response' => $response
            ];
        }
    }
    
    /**
     * Clean up old API logs
     * 
     * @param  int $days_old Logs older than this many days will be deleted
     * @return int Number of logs cleaned up
     */
    public function cleanup_old_logs($days_old = 30)
    {
        $cutoff_date = date('Y-m-d H:i:s', strtotime("-{$days_old} days"));
        
        $this->CI->db->where('created_at <', $cutoff_date);
        $count = $this->CI->db->count_all_results($this->logs_table);
        
        $this->CI->db->where('created_at <', $cutoff_date);
        $this->CI->db->delete($this->logs_table);
        
        return $count;
    }
    
    // ===========================================
    // PRIVATE METHODS
    // ===========================================
    
    /**
     * Build prompt with data replacement
     * 
     * @param  string $template Prompt template
     * @param  array  $data     Data for replacement
     * @param  array  $files    Optional parsed files
     * @return string Final prompt
     */
    private function _build_prompt($template, $data = [], $files = [])
    {
        $prompt = $template;
        
        // Replace placeholders with data
        if (!empty($data)) {
            foreach ($data as $key => $value) {
                if (is_string($value) || is_numeric($value)) {
                    $placeholder = '{' . $key . '}';
                    $prompt = str_replace($placeholder, $this->_sanitize_text($value), $prompt);
                }
            }
        }
        
        // Add file content if provided
        if (!empty($files)) {
            $file_content = "\n\n=== FILE CONTENT ===\n";
            foreach ($files as $filename => $file_data) {
                if ($file_data['success'] && !empty($file_data['content'])) {
                    $safe_filename = $this->_sanitize_text($filename);
                    $safe_content = $this->_sanitize_text(substr($file_data['content'], 0, 1000));
                    $file_content .= "File: {$safe_filename}\nContent: {$safe_content}\n\n";
                }
            }
            $prompt .= $file_content;
        }
        
        return $prompt;
    }
    
    /**
     * Prepare multimodal content array for OpenAI API
     * 
     * @param  string $prompt      Analysis prompt
     * @param  array  $data        Data for context
     * @param  array  $parsed_files Parsed file contents
     * @param  array  $file_paths   Original file paths
     * @return array  Content array for OpenAI API
     */
    private function _prepare_multimodal_content($prompt, $data, $parsed_files, $file_paths)
    {
        $content = [];
        
        // Start with text prompt
        $prompt_text = $this->_build_prompt($prompt, $data, $parsed_files);
        
        // Validate prompt text
        if (!mb_check_encoding($prompt_text, 'UTF-8')) {
            $prompt_text = mb_convert_encoding($prompt_text, 'UTF-8', 'UTF-8');
        }
        
        $content[] = [
            'type' => 'text',
            'text' => $prompt_text
        ];
        
        // Add images (limited to prevent payload issues)
        $images_added = 0;
        $max_images = 3;
        
        foreach ($file_paths as $file_path) {
            if (!file_exists($file_path) || $images_added >= $max_images) {
                continue;
            }
            
            $file_ext = strtolower(pathinfo($file_path, PATHINFO_EXTENSION));
            $filename = basename($file_path);
            
            if (in_array($file_ext, $this->image_types)) {
                try {
                    $file_size = filesize($file_path);
                    
                    // Only process small images
                    if ($file_size > 2 * 1024 * 1024) { // 2MB limit
                        continue;
                    }
                    
                    $image_data = file_get_contents($file_path);
                    if ($image_data === false) {
                        continue;
                    }
                    
                    $base64_image = base64_encode($image_data);
                    if ($base64_image === false) {
                        continue;
                    }
                    
                    $mime_type = ($file_ext === 'png') ? 'image/png' : 'image/jpeg';
                    
                    $content[] = [
                        'type' => 'image_url',
                        'image_url' => [
                            'url' => "data:{$mime_type};base64,{$base64_image}",
                            'detail' => 'low'
                        ]
                    ];
                    
                    $images_added++;
                    
                } catch (Exception $e) {
                    log_message('error', "Error processing image {$filename}: " . $e->getMessage());
                    continue;
                }
            }
        }
        
        return $content;
    }
    
    /**
     * Make OpenAI API call (text-only)
     * 
     * @param  string $prompt  AI prompt
     * @param  array  $options API options
     * @return array  API response
     */
    private function _make_openai_api_call($prompt, $options)
    {
        $data = [
            'model' => $options['model'] ?? $this->default_model,
            'messages' => [
                [
                    'role' => 'user',
                    'content' => $prompt
                ]
            ],
            'temperature' => (float)($options['temperature'] ?? $this->default_temperature),
            'max_tokens' => (int)($options['max_tokens'] ?? $this->default_max_tokens)
        ];
        
        // Add response format if specified
        if (!empty($options['response_format'])) {
            $data['response_format'] = $options['response_format'];
        }
        
        return $this->_execute_api_call($data);
    }
    
    /**
     * Make enhanced OpenAI API call (multimodal)
     * 
     * @param  array $content_array Array of content items
     * @param  array $options       API options
     * @return array API response
     */
    private function _make_enhanced_openai_api_call($content_array, $options)
    {
        $data = [
            'model' => $options['model'] ?? $this->default_model,
            'messages' => [
                [
                    'role' => 'user',
                    'content' => $content_array
                ]
            ],
            'temperature' => (float)($options['temperature'] ?? $this->default_temperature),
            'max_tokens' => (int)($options['max_tokens'] ?? $this->default_max_tokens)
        ];
        
        // Add response format if specified
        if (!empty($options['response_format'])) {
            $data['response_format'] = $options['response_format'];
        }
        
        // Validate payload before sending
        $json_payload = json_encode($data, JSON_UNESCAPED_SLASHES);
        if (json_last_error() !== JSON_ERROR_NONE) {
            return [
                'success' => false,
                'error' => 'JSON encoding error: ' . json_last_error_msg()
            ];
        }
        
        $payload_size = strlen($json_payload);
        if ($payload_size > $this->max_payload_size) {
            return [
                'success' => false,
                'error' => 'Request payload too large: ' . round($payload_size / (1024 * 1024), 2) . 'MB'
            ];
        }
        
        return $this->_execute_api_call($data);
    }
    
    /**
     * Execute the actual API call to OpenAI
     * 
     * @param  array $data Request data
     * @return array API response
     */
    private function _execute_api_call($data)
    {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, 'https://api.openai.com/v1/chat/completions');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Authorization: Bearer ' . $this->api_key
        ]);
        curl_setopt($ch, CURLOPT_TIMEOUT, 120);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        
        $start_time = microtime(true);
        $response = curl_exec($ch);
        $response_time = microtime(true) - $start_time;
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curl_error = curl_error($ch);
        
        curl_close($ch);
        
        if ($response === false) {
            return [
                'success' => false,
                'error' => 'Failed to connect to OpenAI API: ' . $curl_error
            ];
        }
        
        $response_data = json_decode($response, true);
        
        if ($http_code !== 200) {
            $error = isset($response_data['error']['message']) ? 
                     $response_data['error']['message'] : 'Unknown API error';
            
            return [
                'success' => false,
                'error' => $error,
                'http_code' => $http_code
            ];
        }
        
        if (!isset($response_data['choices'][0]['message']['content'])) {
            return [
                'success' => false,
                'error' => 'Invalid response from OpenAI API'
            ];
        }
        
        return [
            'success' => true,
            'response' => $response_data['choices'][0]['message']['content'],
            'response_time' => $response_time,
            'usage' => $response_data['usage'] ?? null
        ];
    }
    
    /**
     * Log API call
     * 
     * @param  array $log_data API call data to log
     * @return bool  Success status
     */
    private function _log_api_call($log_data)
    {
        try {
            $default_data = [
                'endpoint' => '',
                'method' => 'POST',
                'request_data' => '',
                'response_data' => '',
                'status_code' => null,
                'response_time' => null,
                'success' => 0,
                'error_message' => null,
                'staff_id' => get_staff_user_id(),
                'created_at' => date('Y-m-d H:i:s')
            ];
            
            $log_data = array_merge($default_data, $log_data);
            
            // Truncate large data
            if (isset($log_data['request_data']) && strlen($log_data['request_data']) > 50000) {
                $log_data['request_data'] = substr($log_data['request_data'], 0, 50000) . '... [TRUNCATED]';
            }
            if (isset($log_data['response_data']) && strlen($log_data['response_data']) > 50000) {
                $log_data['response_data'] = substr($log_data['response_data'], 0, 50000) . '... [TRUNCATED]';
            }
            
            return $this->CI->db->insert($this->logs_table, $log_data);
            
        } catch (Exception $e) {
            log_message('error', 'Failed to log API call: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Ensure logs table exists
     * 
     * @return void
     */
    private function _ensure_logs_table()
    {
        if (!$this->CI->db->table_exists($this->logs_table)) {
            $sql = "CREATE TABLE IF NOT EXISTS `{$this->logs_table}` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `lead_id` int(11) DEFAULT NULL,
                `endpoint` varchar(255) NOT NULL,
                `method` varchar(10) NOT NULL DEFAULT 'POST',
                `request_data` longtext,
                `response_data` longtext,
                `status_code` int(3) DEFAULT NULL,
                `response_time` float DEFAULT NULL,
                `success` tinyint(1) DEFAULT 0,
                `error_message` text,
                `staff_id` int(11) DEFAULT NULL,
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                KEY `lead_id` (`lead_id`),
                KEY `status_code` (`status_code`),
                KEY `success` (`success`),
                KEY `created_at` (`created_at`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;";
            
            $this->CI->db->query($sql);
        }
    }
    
    /**
     * Basic file parsing fallback
     * 
     * @param  string $file_path File path
     * @param  string $file_type File type/MIME
     * @return array  Parse result
     */
    private function _basic_file_parse($file_path, $file_type)
    {
        $file_ext = strtolower(pathinfo($file_path, PATHINFO_EXTENSION));
        
        try {
            if ($file_ext === 'txt') {
                $content = file_get_contents($file_path);
                return [
                    'success' => true,
                    'content' => $content,
                    'metadata' => ['type' => 'text', 'size' => strlen($content)]
                ];
            } elseif (in_array($file_ext, $this->image_types)) {
                return [
                    'success' => true,
                    'content' => '[Image file: ' . basename($file_path) . ']',
                    'metadata' => ['type' => 'image', 'format' => $file_ext]
                ];
            } else {
                return [
                    'success' => false,
                    'error' => 'File parsing not available for this type',
                    'content' => '',
                    'metadata' => ['type' => $file_ext]
                ];
            }
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage(),
                'content' => '',
                'metadata' => []
            ];
        }
    }
    
    /**
     * Sanitize text content for JSON encoding
     * 
     * @param  string $text Text to sanitize
     * @return string Sanitized text
     */
    private function _sanitize_text($text)
    {
        if (empty($text)) {
            return '';
        }
        
        $text = (string)$text;
        $text = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F]/', '', $text);
        $text = mb_convert_encoding($text, 'UTF-8', 'UTF-8');
        $text = trim($text);
        $text = str_replace(['"', "'", '`', "\r\n", "\r", "\n"], ['\\"', "\\'", "'", " ", " ", " "], $text);
        
        return $text;
    }
    
    /**
     * Return error response format
     * 
     * @param  string $message Error message
     * @return array  Error response
     */
    private function _error_response($message)
    {
        return [
            'success' => false,
            'error' => $message
        ];
    }
    
    /**
     * Get default lead analysis prompt template
     * 
     * @return string Default prompt
     */
    private function _get_default_lead_prompt()
    {
        return "Analyze this lead information and provide a JSON response with purchase likelihood assessment:

Lead Details:
- Name: {name}
- Email: {email}
- Phone: {phonenumber}
- Company: {company}
- Source: {source}
- Value: {lead_value}
- Description: {description}

Respond with valid JSON in this format:
{
  \"confidence_score\": 75,
  \"verdict\": \"Warm\",
  \"key_insights\": \"Analysis of lead behavior and potential\",
  \"recommended_actions\": \"Specific next steps\",
  \"urgency_level\": \"Medium\",
  \"risk_factors\": [\"potential concerns\"],
  \"positive_signals\": [\"encouraging signs\"]
}";
    }
    
    /**
     * Get default customer analysis prompt template
     * 
     * @return string Default prompt
     */
    private function _get_default_customer_prompt()
    {
        return "Analyze this customer information and provide insights:

Customer Details:
- Name: {company}
- Contact: {firstname} {lastname}
- Email: {email}
- Phone: {phonenumber}
- Address: {address}
- Registration: {datecreated}

Provide analysis focusing on customer value, engagement potential, and relationship opportunities.

Respond with valid JSON format.";
    }
    
    /**
     * Get default project analysis prompt template
     * 
     * @return string Default prompt
     */
    private function _get_default_project_prompt()
    {
        return "Analyze this project information and provide insights:

Project Details:
- Name: {name}
- Description: {description}
- Status: {status}
- Start Date: {start_date}
- Deadline: {deadline}
- Budget: {project_cost}

Provide analysis focusing on project success factors, risks, and recommendations.

Respond with valid JSON format.";
    }
}
