<?php
defined('BASEPATH') or exit('No direct script access allowed');

class Ai_setup extends AdminController
{
    public function __construct()
    {
        parent::__construct();
        $this->load->database();
    }

    public function index()
    {
        if (!is_admin()) {
            access_denied('AI Setup');
        }

        $data['title'] = 'AI Analysis Setup';
        $data['table_status'] = $this->_check_table_status();
        
        // Check OpenAI configuration
        $openai_key = get_option('openai_api_key');
        $data['openai_configured'] = !empty($openai_key);
        $data['openai_key_masked'] = !empty($openai_key) ? substr($openai_key, 0, 7) . '...' . substr($openai_key, -4) : 'Not set';
        
        // Ensure default prompt is set if none exists
        $current_prompt = get_option('ai_analysis_prompt');
        if (empty($current_prompt)) {
            $default_prompt = $this->_get_default_prompt();
            $this->_update_option('ai_analysis_prompt', $default_prompt);
            $current_prompt = $default_prompt;
        }
        
        // Make default prompt available to view
        $data['default_prompt'] = $this->_get_default_prompt();
        
        // Get current AI configuration
        $current_model = get_option('ai_model') ?: 'gpt-4o';
        
        // Auto-update deprecated models
        $deprecated_models = [
            'gpt-4-vision-preview' => 'gpt-4o',
            'gpt-4-32k' => 'gpt-4-turbo',
            'gpt-3.5-turbo-16k' => 'gpt-3.5-turbo'
        ];
        
        if (isset($deprecated_models[$current_model])) {
            $new_model = $deprecated_models[$current_model];
            $this->_update_option('ai_model', $new_model);
            $current_model = $new_model;
            set_alert('info', "Auto-updated deprecated model to '{$new_model}'. The previous model is no longer supported by OpenAI.");
        }
        
        $data['current_ai_model'] = $current_model;
        $data['current_ai_temperature'] = get_option('ai_temperature') ?: '0.3';
        $data['current_ai_max_tokens'] = get_option('ai_max_tokens') ?: '1000';
        $data['current_ai_prompt'] = $current_prompt;
        
        // Set default values if they don't exist
        if (!get_option('ai_model')) {
            $this->_update_option('ai_model', 'gpt-4o');
        }
        if (!get_option('ai_temperature')) {
            $this->_update_option('ai_temperature', '0.3');
        }
        if (!get_option('ai_max_tokens')) {
            $this->_update_option('ai_max_tokens', '1000');
        }
        
        $this->load->view('admin/ai_setup/index', $data);
    }

    public function fix_database()
    {
        if (!is_admin()) {
            access_denied('AI Setup');
        }

        $data['title'] = 'AI Analysis Database Setup';
        
        if ($this->input->get('action') === 'fix') {
            $this->_fix_database_structure();
        }
        
        $data['table_status'] = $this->_check_table_status();
        $this->load->view('admin/ai_setup/database_fix', $data);
    }

    public function setup_openai()
    {
        if (!is_admin()) {
            access_denied('AI Setup');
        }

        if ($this->input->post()) {
            $api_key = $this->input->post('openai_api_key');
            
            // Update or insert the setting
            $existing = $this->db->where('name', 'openai_api_key')->get(db_prefix() . 'options')->row();
            
            if ($existing) {
                $this->db->where('name', 'openai_api_key')->update(db_prefix() . 'options', ['value' => $api_key]);
            } else {
                $this->db->insert(db_prefix() . 'options', [
                    'name' => 'openai_api_key',
                    'value' => $api_key,
                    'autoload' => 1
                ]);
            }
            
            set_alert('success', 'OpenAI API key updated successfully');
            redirect(admin_url('ai_setup/setup_openai'));
        }

        $data['title'] = 'OpenAI Configuration';
        $data['current_key'] = get_option('openai_api_key');
        $this->load->view('admin/ai_setup/openai_config', $data);
    }

    public function api_logs()
    {
        if (!is_admin()) {
            access_denied('AI Setup');
        }

        $data['title'] = 'AI Analysis API Logs';
        
        // Handle log clearing
        if ($this->input->post('clear_logs')) {
            $this->_clear_api_logs();
            set_alert('success', 'API logs cleared successfully');
            redirect(admin_url('ai_setup/api_logs'));
        }
        
        // Get logs with pagination
        $this->load->library('pagination');
        
        $page = $this->input->get('page') ?: 1;
        $per_page = 50;
        $offset = ($page - 1) * $per_page;
        
        $logs = $this->_get_api_logs($per_page, $offset);
        $total_logs = $this->_count_api_logs();
        
        // Pagination config
        $config['base_url'] = admin_url('ai_setup/api_logs');
        $config['total_rows'] = $total_logs;
        $config['per_page'] = $per_page;
        $config['page_query_string'] = TRUE;
        $config['query_string_segment'] = 'page';
        
        $this->pagination->initialize($config);
        
        $data['logs'] = $logs;
        $data['pagination'] = $this->pagination->create_links();
        $data['total_logs'] = $total_logs;
        
        $this->load->view('admin/ai_setup/api_logs', $data);
    }

    public function get_log_details($log_id)
    {
        // Set JSON header
        header('Content-Type: application/json');
        
        if (!is_admin()) {
            echo json_encode(['success' => false, 'message' => 'Access denied']);
            return;
        }

        if (empty($log_id) || !is_numeric($log_id)) {
            echo json_encode(['success' => false, 'message' => 'Invalid log ID']);
            return;
        }

        $table = db_prefix() . 'ai_api_logs';
        
        // Create table if it doesn't exist
        if (!$this->db->table_exists($table)) {
            $this->_create_api_logs_table();
        }
        
        try {
            $log = $this->db->where('id', $log_id)->get($table)->row();
            
            if ($log) {
                echo json_encode([
                    'success' => true,
                    'data' => [
                        'id' => $log->id,
                        'lead_id' => $log->lead_id,
                        'endpoint' => $log->endpoint,
                        'method' => $log->method,
                        'request_data' => $log->request_data,
                        'response_data' => $log->response_data,
                        'status_code' => $log->status_code,
                        'response_time' => $log->response_time,
                        'success' => $log->success,
                        'error_message' => $log->error_message,
                        'created_at' => $log->created_at
                    ]
                ]);
            } else {
                echo json_encode(['success' => false, 'message' => 'Log not found with ID: ' . $log_id]);
            }
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'message' => 'Database error: ' . $e->getMessage()]);
        }
    }

    public function save_model_config()
    {
        if (!is_admin()) {
            access_denied('AI Setup');
        }

        if ($this->input->post()) {
            $ai_model = $this->input->post('ai_model');
            $ai_temperature = $this->input->post('ai_temperature');
            $ai_max_tokens = $this->input->post('ai_max_tokens');
            $ai_analysis_prompt = $this->input->post('ai_analysis_prompt');
            
            // Validate inputs
            if (empty($ai_model) || empty($ai_analysis_prompt)) {
                set_alert('danger', 'AI model and prompt are required');
                redirect(admin_url('ai_setup'));
            }
            
            // Save options
            $this->_update_option('ai_model', $ai_model);
            $this->_update_option('ai_temperature', $ai_temperature);
            $this->_update_option('ai_max_tokens', $ai_max_tokens);
            $this->_update_option('ai_analysis_prompt', $ai_analysis_prompt);
            
            set_alert('success', 'AI model configuration saved successfully');
        }
        
        redirect(admin_url('ai_setup'));
    }
    
    public function test_prompt()
    {
        if (!is_admin()) {
            ajax_access_denied();
        }

        $prompt = $this->input->post('prompt');
        $model = $this->input->post('model');
        $temperature = floatval($this->input->post('temperature'));
        $max_tokens = intval($this->input->post('max_tokens'));
        
        if (empty($prompt)) {
            echo json_encode(['success' => false, 'message' => 'Prompt is required']);
            return;
        }
        
        // Check if OpenAI API key is configured
        $api_key = get_option('openai_api_key');
        if (empty($api_key)) {
            echo json_encode(['success' => false, 'message' => 'OpenAI API key not configured']);
            return;
        }
        
        try {
            // Test the prompt with sample data
            $test_prompt = str_replace(
                ['{lead_name}', '{lead_email}', '{lead_phone}', '{lead_value}', '{lead_description}'],
                ['John Doe', 'john@example.com', '+1234567890', '$500,000', 'Interested in a 2-bedroom apartment in downtown area'],
                $prompt
            );
            
            $response = $this->_test_openai_call($test_prompt, $model, $temperature, $max_tokens);
            
            if ($response['success']) {
                echo json_encode([
                    'success' => true,
                    'preview' => substr($response['response'], 0, 200) . (strlen($response['response']) > 200 ? '...' : ''),
                    'full_response' => $response['response']
                ]);
            } else {
                echo json_encode(['success' => false, 'message' => $response['error']]);
            }
            
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'message' => $e->getMessage()]);
        }
    }

    public function create_test_log()
    {
        if (!is_admin()) {
            access_denied('AI Setup');
        }

        $this->load->model('leads_ai_model');
        
        $test_log = [
            'lead_id' => 1,
            'endpoint' => 'https://api.openai.com/v1/chat/completions',
            'method' => 'POST',
            'request_data' => json_encode([
                'model' => 'gpt-4o',
                'messages' => [
                    ['role' => 'user', 'content' => 'Test prompt for lead analysis']
                ],
                'temperature' => 0.3,
                'max_tokens' => 1000
            ]),
            'response_data' => json_encode([
                'choices' => [
                    ['message' => ['content' => 'This is a test AI response for demonstration purposes using GPT-4o model.']]
                ]
            ]),
            'status_code' => 200,
            'response_time' => 2.5,
            'success' => 1,
            'error_message' => null
        ];
        
        if ($this->leads_ai_model->log_api_call($test_log)) {
            set_alert('success', 'Test log entry created successfully');
        } else {
            set_alert('danger', 'Failed to create test log entry');
        }
        
        redirect(admin_url('ai_setup/api_logs'));
    }

    /**
     * Debug prompt configuration
     */
    public function debug_prompt()
    {
        if (!is_admin()) {
            access_denied('AI Setup');
        }

        echo "<h3>AI Prompt Configuration Debug</h3>";
        echo "<strong>Current ai_analysis_prompt option:</strong><br>";
        echo "<pre>" . htmlspecialchars(get_option('ai_analysis_prompt') ?: 'NOT SET') . "</pre>";
        
        echo "<strong>Default prompt from method:</strong><br>";
        echo "<pre>" . htmlspecialchars($this->_get_default_prompt()) . "</pre>";
        
        echo "<strong>All AI-related options in database:</strong><br>";
        $ai_options = $this->db->like('name', 'ai_', 'after')->get(db_prefix() . 'options')->result();
        echo "<pre>";
        foreach ($ai_options as $option) {
            echo $option->name . " = " . substr($option->value, 0, 100) . (strlen($option->value) > 100 ? '...' : '') . "\n";
        }
        echo "</pre>";
        
        echo "<strong>Available options table structure:</strong><br>";
        echo "<pre>";
        $query = $this->db->query("DESCRIBE " . db_prefix() . "options");
        foreach ($query->result() as $column) {
            echo $column->Field . " - " . $column->Type . "\n";
        }
        echo "</pre>";
    }

    private function _fix_database_structure()
    {
        $results = [];
        
        // Fix AI reports table
        $table_name = db_prefix() . 'leads_ai_reports';
        
        if (!$this->db->table_exists($table_name)) {
            $sql = "CREATE TABLE IF NOT EXISTS `{$table_name}` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `lead_id` int(11) NOT NULL,
                `score` int(3) NOT NULL DEFAULT 0,
                `verdict` varchar(50) NOT NULL DEFAULT '',
                `rationale` text NOT NULL,
                `recommendations` text,
                `uploaded_files` text,
                `analysis_type` varchar(50) DEFAULT 'document_analysis',
                `staff_id` int(11) NOT NULL,
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                KEY `lead_id` (`lead_id`),
                KEY `staff_id` (`staff_id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;";
            
            if ($this->db->query($sql)) {
                $results[] = ['success', "Table {$table_name} created successfully"];
            } else {
                $results[] = ['danger', "Failed to create table {$table_name}"];
            }
        } else {
            // Check and add missing columns
            $query = $this->db->query("DESCRIBE `{$table_name}`");
            $existing_columns = [];
            
            if ($query) {
                foreach ($query->result() as $column) {
                    $existing_columns[] = $column->Field;
                }
            }
            
            $required_columns = [
                'score' => "ADD COLUMN `score` int(3) NOT NULL DEFAULT 0",
                'verdict' => "ADD COLUMN `verdict` varchar(50) NOT NULL DEFAULT ''",
                'rationale' => "ADD COLUMN `rationale` text NOT NULL",
                'recommendations' => "ADD COLUMN `recommendations` text",
                'uploaded_files' => "ADD COLUMN `uploaded_files` text",
                'analysis_type' => "ADD COLUMN `analysis_type` varchar(50) DEFAULT 'document_analysis'",
                'staff_id' => "ADD COLUMN `staff_id` int(11) NOT NULL DEFAULT 1",
                'created_at' => "ADD COLUMN `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP"
            ];
            
            $columns_added = 0;
            foreach ($required_columns as $column_name => $add_sql) {
                if (!in_array($column_name, $existing_columns)) {
                    $full_sql = "ALTER TABLE `{$table_name}` {$add_sql}";
                    if ($this->db->query($full_sql)) {
                        $columns_added++;
                    }
                }
            }
            
            if ($columns_added > 0) {
                $results[] = ['success', "Added {$columns_added} missing columns to {$table_name}"];
            } else {
                $results[] = ['info', "Table {$table_name} structure is already correct"];
            }
        }
        
        // Fix AI queue table
        $queue_table = db_prefix() . 'leads_ai_analysis_queue';
        
        if (!$this->db->table_exists($queue_table)) {
            $sql = "CREATE TABLE IF NOT EXISTS `{$queue_table}` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `lead_id` int(11) NOT NULL,
                `status` enum('pending','processing','completed','failed') NOT NULL DEFAULT 'pending',
                `priority` int(2) NOT NULL DEFAULT 5,
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                `processed_at` timestamp NULL DEFAULT NULL,
                PRIMARY KEY (`id`),
                KEY `lead_id` (`lead_id`),
                KEY `status` (`status`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;";
            
            if ($this->db->query($sql)) {
                $results[] = ['success', "Table {$queue_table} created successfully"];
            } else {
                $results[] = ['danger', "Failed to create table {$queue_table}"];
            }
        } else {
            $results[] = ['info', "Table {$queue_table} already exists"];
        }
        
        // Store results for display
        $this->session->set_flashdata('ai_setup_results', $results);
        redirect(admin_url('ai_setup/fix_database'));
    }

    private function _check_table_status()
    {
        $status = [];
        
        $ai_reports_table = db_prefix() . 'leads_ai_reports';
        $ai_queue_table = db_prefix() . 'leads_ai_analysis_queue';
        
        // Use simple keys that match what the view expects
        $status['ai_reports'] = $this->db->table_exists($ai_reports_table);
        $status['ai_queue'] = $this->db->table_exists($ai_queue_table);
        
        // Also provide detailed status for full table info
        $status['tables'] = [];
        
        $tables = [
            $ai_reports_table => 'AI Reports Table',
            $ai_queue_table => 'AI Analysis Queue Table'
        ];
        
        foreach ($tables as $table => $name) {
            $status['tables'][$table] = [
                'name' => $name,
                'exists' => $this->db->table_exists($table),
                'columns' => []
            ];
            
            if ($status['tables'][$table]['exists']) {
                $query = $this->db->query("DESCRIBE `{$table}`");
                if ($query) {
                    foreach ($query->result() as $column) {
                        $status['tables'][$table]['columns'][] = $column->Field;
                    }
                }
            }
        }
        
        return $status;
    }

    private function _get_api_logs($limit = 50, $offset = 0)
    {
        $table = db_prefix() . 'ai_api_logs';
        
        // Create table if it doesn't exist
        if (!$this->db->table_exists($table)) {
            $this->_create_api_logs_table();
        }
        
        return $this->db->order_by('created_at', 'DESC')
                       ->limit($limit, $offset)
                       ->get($table)
                       ->result();
    }
    
    private function _count_api_logs()
    {
        $table = db_prefix() . 'ai_api_logs';
        
        if (!$this->db->table_exists($table)) {
            return 0;
        }
        
        return $this->db->count_all($table);
    }
    
    private function _clear_api_logs()
    {
        $table = db_prefix() . 'ai_api_logs';
        
        if ($this->db->table_exists($table)) {
            $this->db->truncate($table);
        }
    }
    
    private function _create_api_logs_table()
    {
        $table = db_prefix() . 'ai_api_logs';
        
        $sql = "CREATE TABLE IF NOT EXISTS `{$table}` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `lead_id` int(11) DEFAULT NULL,
            `endpoint` varchar(255) NOT NULL,
            `method` varchar(10) NOT NULL DEFAULT 'POST',
            `request_data` longtext,
            `response_data` longtext,
            `status_code` int(3) DEFAULT NULL,
            `response_time` float DEFAULT NULL,
            `success` tinyint(1) DEFAULT 0,
            `error_message` text,
            `staff_id` int(11) DEFAULT NULL,
            `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            KEY `lead_id` (`lead_id`),
            KEY `status_code` (`status_code`),
            KEY `success` (`success`),
            KEY `created_at` (`created_at`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8;";
        
        $this->db->query($sql);
    }
    
    private function _update_option($name, $value)
    {
        $existing = $this->db->where('name', $name)->get(db_prefix() . 'options')->row();
        
        if ($existing) {
            $this->db->where('name', $name)->update(db_prefix() . 'options', ['value' => $value]);
        } else {
            $this->db->insert(db_prefix() . 'options', [
                'name' => $name,
                'value' => $value,
                'autoload' => 1
            ]);
        }
    }
    
    private function _get_default_prompt()
    {
        return "You are an expert lead evaluation specialist. Analyze the provided lead information, follow-up notes, conversations, and any attached files to assess purchase likelihood.

=== LEAD INFORMATION ===
- Name: {lead_name}
- Email: {lead_email}
- Phone: {lead_phone}
- Company: {lead_company}
- Source: {lead_source}
- Value: {lead_value}
- Description: {lead_description}
- Date Added: {lead_date_added}
- Last Contact: {lead_last_contact}

=== FOLLOW-UP NOTES & CONVERSATIONS ===
{follow_up_notes}

=== ACTIVITY LOG ===
{activity_log}

=== ATTACHED FILES CONTENT ===
{file_content}

=== ANALYSIS INSTRUCTIONS ===
Analyze the lead information, follow-up notes, conversations, activity log, and any provided images/documents. Consider:
- Lead qualification and interest level based on conversations
- Communication patterns, response times, and engagement quality
- Follow-up types and frequency indicating interest level
- Staff notes about client interactions and feedback
- Budget indicators and financial capability from conversations
- Timeline and urgency signals from communications
- Objections, concerns, or positive signals mentioned in notes
- Social media presence and lifestyle (from images)
- Document content relevance
- Overall purchase intent signals from all touchpoints

IMPORTANT: Respond with valid JSON in this exact format:
{
  \"confidence_score\": 85,
  \"verdict\": \"Hot\",
  \"key_insights\": \"Analysis of behavior and communication patterns\",
  \"recommended_actions\": \"Specific next steps based on conversation history\",
  \"communication_analysis\": \"Assessment of engagement quality\",
  \"urgency_level\": \"High\",
  \"risk_factors\": [\"potential concerns from conversations\"],
  \"positive_signals\": [\"encouraging signs from interactions\"]
}

Ensure your response is valid JSON only, no additional text. Base your analysis heavily on the conversation history and follow-up notes.";
    }
    
    private function _test_openai_call($prompt, $model, $temperature, $max_tokens)
    {
        $api_key = get_option('openai_api_key');
        
        $data = [
            'model' => $model,
            'messages' => [
                [
                    'role' => 'user',
                    'content' => $prompt
                ]
            ],
            'temperature' => $temperature,
            'max_tokens' => $max_tokens
        ];
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, 'https://api.openai.com/v1/chat/completions');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Authorization: Bearer ' . $api_key
        ]);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        
        $start_time = microtime(true);
        $response = curl_exec($ch);
        $response_time = microtime(true) - $start_time;
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        
        curl_close($ch);
        
        if ($response === false) {
            return ['success' => false, 'error' => 'Failed to connect to OpenAI API'];
        }
        
        $response_data = json_decode($response, true);
        
        if ($http_code !== 200) {
            $error = isset($response_data['error']['message']) ? $response_data['error']['message'] : 'Unknown API error';
            return ['success' => false, 'error' => $error];
        }
        
        if (!isset($response_data['choices'][0]['message']['content'])) {
            return ['success' => false, 'error' => 'Invalid response from OpenAI API'];
        }
        
        return [
            'success' => true,
            'response' => $response_data['choices'][0]['message']['content'],
            'response_time' => $response_time
        ];
    }

    /**
     * Force set default prompt and AI configuration
     */
    public function set_defaults()
    {
        if (!is_admin()) {
            access_denied('AI Setup');
        }

        // Set default AI configuration
        $this->_update_option('ai_model', 'gpt-4o');
        $this->_update_option('ai_temperature', '0.3');
        $this->_update_option('ai_max_tokens', '1000');
        $this->_update_option('ai_analysis_prompt', $this->_get_default_prompt());
        
        set_alert('success', 'Default AI configuration has been set successfully!');
        redirect(admin_url('ai_setup'));
    }

    /**
     * Update deprecated models automatically
     */
    public function update_deprecated_models()
    {
        if (!is_admin()) {
            access_denied('AI Setup');
        }

        $current_model = get_option('ai_model');
        $updated = false;
        
        // Check for deprecated models and update them
        $deprecated_models = [
            'gpt-4-vision-preview' => 'gpt-4o',
            'gpt-4-32k' => 'gpt-4-turbo',
            'gpt-3.5-turbo-16k' => 'gpt-3.5-turbo'
        ];
        
        if (isset($deprecated_models[$current_model])) {
            $new_model = $deprecated_models[$current_model];
            $this->_update_option('ai_model', $new_model);
            $updated = true;
            set_alert('success', "Updated deprecated model '{$current_model}' to '{$new_model}'");
        }
        
        if (!$updated) {
            set_alert('info', 'No deprecated models found. Current model is up to date.');
        }
        
        redirect(admin_url('ai_setup'));
    }

    /**
     * Fine-tuning management page
     */
    public function fine_tuning()
    {
        if (!is_admin()) {
            access_denied('AI Setup');
        }

        $data['title'] = 'Fine-Tuning Management';
        
        // Get saved fine-tuned models
        $fine_tuned_models = get_option('ai_fine_tuned_models');
        $data['fine_tuned_models'] = $fine_tuned_models ? json_decode($fine_tuned_models, true) : [];
        
        // Get OpenAI fine-tuning jobs if API key is available
        $data['openai_jobs'] = [];
        $openai_key = get_option('openai_api_key');
        if (!empty($openai_key)) {
            $data['openai_jobs'] = $this->_get_openai_fine_tuning_jobs();
        }
        
        $this->load->view('admin/ai_setup/fine_tuning', $data);
    }

    /**
     * Save fine-tuned model
     */
    public function save_fine_tuned_model()
    {
        if (!is_admin()) {
            access_denied('AI Setup');
        }

        if (!$this->input->post()) {
            redirect(admin_url('ai_setup/fine_tuning'));
        }

        $model_id = $this->input->post('model_id');
        $model_name = $this->input->post('model_name');
        $description = $this->input->post('description');

        if (empty($model_id) || empty($model_name)) {
            set_alert('danger', 'Model ID and name are required');
            redirect(admin_url('ai_setup/fine_tuning'));
        }

        // Get existing models
        $fine_tuned_models = get_option('ai_fine_tuned_models');
        $models = $fine_tuned_models ? json_decode($fine_tuned_models, true) : [];

        // Add new model
        $models[] = [
            'id' => $model_id,
            'name' => $model_name,
            'description' => $description,
            'created_at' => date('Y-m-d H:i:s'),
            'created_by' => get_staff_user_id()
        ];

        // Save updated models
        $this->_update_option('ai_fine_tuned_models', json_encode($models));

        set_alert('success', 'Fine-tuned model added successfully');
        redirect(admin_url('ai_setup/fine_tuning'));
    }

    /**
     * Delete fine-tuned model
     */
    public function delete_fine_tuned_model($index)
    {
        if (!is_admin()) {
            access_denied('AI Setup');
        }

        // Get existing models
        $fine_tuned_models = get_option('ai_fine_tuned_models');
        $models = $fine_tuned_models ? json_decode($fine_tuned_models, true) : [];

        if (isset($models[$index])) {
            unset($models[$index]);
            $models = array_values($models); // Re-index array

            // Save updated models
            $this->_update_option('ai_fine_tuned_models', json_encode($models));

            set_alert('success', 'Fine-tuned model deleted successfully');
        } else {
            set_alert('danger', 'Model not found');
        }

        redirect(admin_url('ai_setup/fine_tuning'));
    }

    /**
     * Create fine-tuning job
     */
    public function create_fine_tuning_job()
    {
        if (!is_admin()) {
            access_denied('AI Setup');
        }

        if (!$this->input->post()) {
            redirect(admin_url('ai_setup/fine_tuning'));
        }

        $training_file = $_FILES['training_file'] ?? null;
        $base_model = $this->input->post('base_model') ?: 'gpt-3.5-turbo';
        $suffix = $this->input->post('suffix') ?: 'lead-analysis';

        if (!$training_file || $training_file['error'] !== UPLOAD_ERR_OK) {
            set_alert('danger', 'Please upload a valid training file');
            redirect(admin_url('ai_setup/fine_tuning'));
        }

        $result = $this->_upload_training_file_and_create_job($training_file, $base_model, $suffix);

        if ($result['success']) {
            set_alert('success', 'Fine-tuning job created successfully. Job ID: ' . $result['job_id']);
        } else {
            set_alert('danger', 'Failed to create fine-tuning job: ' . $result['error']);
        }

        redirect(admin_url('ai_setup/fine_tuning'));
    }

    /**
     * Get OpenAI fine-tuning jobs
     */
    private function _get_openai_fine_tuning_jobs()
    {
        $api_key = get_option('openai_api_key');
        if (empty($api_key)) {
            return [];
        }

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, 'https://api.openai.com/v1/fine_tuning/jobs');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Authorization: Bearer ' . $api_key
        ]);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);

        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($response === false || $http_code !== 200) {
            return [];
        }

        $data = json_decode($response, true);
        return isset($data['data']) ? $data['data'] : [];
    }

    /**
     * Upload training file and create fine-tuning job
     */
    private function _upload_training_file_and_create_job($training_file, $base_model, $suffix)
    {
        $api_key = get_option('openai_api_key');
        if (empty($api_key)) {
            return ['success' => false, 'error' => 'OpenAI API key not configured'];
        }

        // Step 1: Upload training file
        $file_upload_result = $this->_upload_file_to_openai($training_file, $api_key);
        if (!$file_upload_result['success']) {
            return $file_upload_result;
        }

        $file_id = $file_upload_result['file_id'];

        // Step 2: Create fine-tuning job
        $job_data = [
            'training_file' => $file_id,
            'model' => $base_model,
            'suffix' => $suffix
        ];

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, 'https://api.openai.com/v1/fine_tuning/jobs');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($job_data));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Authorization: Bearer ' . $api_key
        ]);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);

        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($response === false) {
            return ['success' => false, 'error' => 'Failed to connect to OpenAI API'];
        }

        $response_data = json_decode($response, true);

        if ($http_code !== 200) {
            $error = isset($response_data['error']['message']) ? $response_data['error']['message'] : 'Unknown API error';
            return ['success' => false, 'error' => $error];
        }

        return [
            'success' => true,
            'job_id' => $response_data['id'],
            'file_id' => $file_id
        ];
    }

    /**
     * Upload file to OpenAI
     */
    private function _upload_file_to_openai($file, $api_key)
    {
        $tmp_path = $file['tmp_name'];
        $filename = $file['name'];

        if (!file_exists($tmp_path)) {
            return ['success' => false, 'error' => 'Training file not found'];
        }

        $post_data = [
            'purpose' => 'fine-tune',
            'file' => new CURLFile($tmp_path, $file['type'], $filename)
        ];

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, 'https://api.openai.com/v1/files');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $post_data);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Authorization: Bearer ' . $api_key
        ]);
        curl_setopt($ch, CURLOPT_TIMEOUT, 60);

        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($response === false) {
            return ['success' => false, 'error' => 'Failed to upload file to OpenAI'];
        }

        $response_data = json_decode($response, true);

        if ($http_code !== 200) {
            $error = isset($response_data['error']['message']) ? $response_data['error']['message'] : 'File upload failed';
            return ['success' => false, 'error' => $error];
        }

        return [
            'success' => true,
            'file_id' => $response_data['id']
        ];
    }
}
?>
