<?php
defined('BASEPATH') or exit('No direct script access allowed');

class Ai_reports extends AdminController
{
    public function __construct()
    {
        parent::__construct();
        $this->load->database();
        $this->load->model('leads_ai_model');
        $this->load->model('leads_model');
        $this->load->model('staff_model');
    }

    /**
     * Main AI Reports listing page
     */
    public function index()
    {
        if (!is_admin() && !has_permission('leads', '', 'view')) {
            access_denied('AI Reports');
        }

        $data['title'] = 'AI Analysis Reports';
        
        // Get filter parameters
        $filters = [
            'lead_id' => $this->input->get('lead_id'),
            'verdict' => $this->input->get('verdict'),
            'date_from' => $this->input->get('date_from'),
            'date_to' => $this->input->get('date_to'),
            'staff_id' => $this->input->get('staff_id'),
            'analysis_type' => $this->input->get('analysis_type')
        ];

        // No longer handle AJAX requests - use direct database approach
        // Get summary statistics
        $data['stats'] = $this->_get_reports_statistics($filters);
        
        // Get filter options
        $data['verdicts'] = $this->_get_available_verdicts();
        $data['analysis_types'] = $this->_get_available_analysis_types();
        $data['staff_members'] = $this->staff_model->get('', ['active' => 1]);
        
        $data['filters'] = $filters;
        $this->load->view('admin/ai_reports/index', $data);
    }

    /**
     * View individual AI report details
     */
    public function view($report_id)
    {
        if (!is_admin() && !has_permission('leads', '', 'view')) {
            access_denied('AI Reports');
        }

        $report = $this->_get_report_with_lead($report_id);
        
        if (!$report) {
            show_404();
        }

        // Check if user can access the lead
        if (!is_admin() && !$this->leads_model->staff_can_access_lead($report->lead_id)) {
            access_denied('AI Report');
        }

        $data['title'] = 'AI Analysis Report #' . $report_id;
        $data['report'] = $report;
        
        // Parse additional data
        $data['uploaded_files'] = $report->uploaded_files ? json_decode($report->uploaded_files, true) : [];
        $data['ai_response_data'] = $report->ai_response ? json_decode($report->ai_response, true) : [];
        
        $this->load->view('admin/ai_reports/view', $data);
    }

    /**
     * Delete AI report
     */
    public function delete($report_id)
    {
        if (!is_admin()) {
            access_denied('AI Reports');
        }

        $report = $this->_get_report_with_lead($report_id);
        
        if (!$report) {
            show_404();
        }

        if ($this->leads_ai_model->delete_report($report_id)) {
            set_alert('success', 'AI report deleted successfully');
            
            // Log activity
            log_activity('AI Report Deleted [Report ID: ' . $report_id . ', Lead: ' . $report->lead_name . ']');
        } else {
            set_alert('danger', 'Failed to delete AI report');
        }

        redirect(admin_url('ai_reports'));
    }

    /**
     * Bulk delete AI reports
     */
    public function bulk_delete()
    {
        // Set JSON content type header
        header('Content-Type: application/json');
        
        try {
            // Use the same authentication pattern as the single delete method
            if (!is_admin()) {
                echo json_encode(['success' => false, 'message' => 'Access denied - Admin privileges required']);
                return;
            }

            // Get report IDs from POST data or FormData
            $report_ids_json = '';
            
            // First try regular POST data
            if (isset($_POST['report_ids']) && !empty($_POST['report_ids'])) {
                $report_ids_json = $_POST['report_ids'];
                log_message('debug', 'Got report_ids from regular POST: ' . $report_ids_json);
            } else {
                // Try to get from raw input (FormData)
                $raw_input = file_get_contents('php://input');
                log_message('debug', 'Raw input data: ' . $raw_input);
                
                // Parse the FormData
                if (!empty($raw_input)) {
                    // For FormData, we need to parse it differently
                    $boundary = null;
                    if (isset($_SERVER['CONTENT_TYPE']) && strpos($_SERVER['CONTENT_TYPE'], 'multipart/form-data') !== false) {
                        // Handle multipart/form-data
                        preg_match('/boundary=(.*)$/', $_SERVER['CONTENT_TYPE'], $matches);
                        $boundary = $matches[1] ?? null;
                        
                        if ($boundary) {
                            $parts = explode('--' . $boundary, $raw_input);
                            foreach ($parts as $part) {
                                if (strpos($part, 'name="report_ids"') !== false) {
                                    $lines = explode("\r\n", $part);
                                    $report_ids_json = end($lines);
                                    break;
                                }
                            }
                        }
                    } else {
                        // Try URL-encoded format
                        parse_str($raw_input, $parsed_data);
                        if (isset($parsed_data['report_ids'])) {
                            $report_ids_json = $parsed_data['report_ids'];
                        }
                    }
                }
            }
            
            // Debug logging
            log_message('debug', 'Bulk delete request - Content-Type: ' . ($_SERVER['CONTENT_TYPE'] ?? 'not set'));
            log_message('debug', 'Bulk delete request - Raw POST data: ' . print_r($_POST, true));
            log_message('debug', 'Bulk delete request - report_ids_json: ' . $report_ids_json);
            
            if (empty($report_ids_json)) {
                echo json_encode(['success' => false, 'message' => 'No report IDs received in request data']);
                return;
            }
            
            $report_ids = json_decode($report_ids_json, true);
            
            // If JSON decode fails, try treating it as a simple array
            if (json_last_error() !== JSON_ERROR_NONE) {
                log_message('error', 'JSON decode error: ' . json_last_error_msg());
                echo json_encode(['success' => false, 'message' => 'Invalid JSON format: ' . json_last_error_msg()]);
                return;
            }
            
            if (empty($report_ids) || !is_array($report_ids)) {
                echo json_encode(['success' => false, 'message' => 'No reports selected or invalid format']);
                return;
            }

            // Check if leads_ai_model is loaded
            if (!isset($this->leads_ai_model)) {
                $this->load->model('leads_ai_model');
            }

            $deleted_count = 0;
            $failed_deletes = [];
            
            foreach ($report_ids as $report_id) {
                if (!is_numeric($report_id)) {
                    log_message('warning', 'Invalid report ID: ' . $report_id);
                    $failed_deletes[] = $report_id . ' (invalid ID)';
                    continue; // Skip invalid IDs
                }
                
                try {
                    if ($this->leads_ai_model->delete_report($report_id)) {
                        $deleted_count++;
                        log_message('info', 'Successfully deleted AI report: ' . $report_id);
                    } else {
                        $failed_deletes[] = $report_id . ' (delete failed)';
                        log_message('warning', 'Failed to delete AI report: ' . $report_id);
                    }
                } catch (Exception $e) {
                    $failed_deletes[] = $report_id . ' (exception: ' . $e->getMessage() . ')';
                    log_message('error', 'Exception deleting AI report ' . $report_id . ': ' . $e->getMessage());
                }
            }

            if ($deleted_count > 0) {
                $message = $deleted_count . ' report' . ($deleted_count > 1 ? 's' : '') . ' deleted successfully';
                
                if (!empty($failed_deletes)) {
                    $message .= '. ' . count($failed_deletes) . ' report' . (count($failed_deletes) > 1 ? 's' : '') . ' could not be deleted.';
                }
                
                echo json_encode([
                    'success' => true,
                    'message' => $message,
                    'deleted_count' => $deleted_count,
                    'failed_count' => count($failed_deletes),
                    'failed_details' => $failed_deletes
                ]);
                
                // Log activity
                log_activity('Bulk AI Reports Deleted [Count: ' . $deleted_count . ', Failed: ' . count($failed_deletes) . ']');
            } else {
                echo json_encode([
                    'success' => false, 
                    'message' => 'Failed to delete any reports. Details: ' . implode(', ', $failed_deletes)
                ]);
            }
        } catch (Exception $e) {
            log_message('error', 'Bulk delete exception: ' . $e->getMessage() . ' - ' . $e->getTraceAsString());
            echo json_encode([
                'success' => false, 
                'message' => 'Server error: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Export AI reports
     */
    public function export()
    {
        if (!is_admin() && !has_permission('leads', '', 'view')) {
            access_denied('AI Reports Export');
        }

        $format = $this->input->get('format') ?: 'excel';
        
        // Get filter parameters
        $filters = [
            'lead_id' => $this->input->get('lead_id'),
            'verdict' => $this->input->get('verdict'),
            'date_from' => $this->input->get('date_from'),
            'date_to' => $this->input->get('date_to'),
            'staff_id' => $this->input->get('staff_id'),
            'analysis_type' => $this->input->get('analysis_type')
        ];

        $reports = $this->_get_all_reports($filters);

        if (empty($reports)) {
            set_alert('warning', 'No reports found to export');
            redirect(admin_url('ai_reports'));
            return;
        }

        switch ($format) {
            case 'excel':
                $this->_export_excel($reports);
                break;
            case 'csv':
                $this->_export_csv($reports);
                break;
            case 'pdf':
                $this->_export_pdf($reports);
                break;
            default:
                $this->_export_excel($reports);
        }
    }

    /**
     * Compare multiple AI reports
     */
    public function compare()
    {
        if (!is_admin() && !has_permission('leads', '', 'view')) {
            access_denied('AI Reports');
        }

        $report_ids = $this->input->get('reports');
        
        if (empty($report_ids)) {
            set_alert('warning', 'No reports selected for comparison');
            redirect(admin_url('ai_reports'));
            return;
        }

        $report_ids = is_array($report_ids) ? $report_ids : explode(',', $report_ids);
        $reports = [];

        foreach ($report_ids as $report_id) {
            $report = $this->_get_report_with_lead($report_id);
            if ($report && (is_admin() || $this->leads_model->staff_can_access_lead($report->lead_id))) {
                $report->ai_response_data = $report->ai_response ? json_decode($report->ai_response, true) : [];
                $reports[] = $report;
            }
        }

        if (empty($reports)) {
            set_alert('warning', 'No valid reports found for comparison');
            redirect(admin_url('ai_reports'));
            return;
        }

        $data['title'] = 'Compare AI Reports';
        $data['reports'] = $reports;
        $this->load->view('admin/ai_reports/compare', $data);
    }

    /**
     * Generate comprehensive analytics
     */
    public function analytics()
    {
        if (!is_admin()) {
            access_denied('AI Reports Analytics');
        }

        $data['title'] = 'AI Analysis Analytics';
        
        // Get date range
        $date_from = $this->input->get('date_from') ?: date('Y-m-01', strtotime('-3 months'));
        $date_to = $this->input->get('date_to') ?: date('Y-m-d');
        
        $data['date_from'] = $date_from;
        $data['date_to'] = $date_to;
        
        // Get analytics data
        $data['analytics'] = $this->_get_analytics_data($date_from, $date_to);
        
        $this->load->view('admin/ai_reports/analytics', $data);
    }

    /**
     * AJAX: Get chart data for analytics
     */
    public function get_chart_data()
    {
        if (!is_admin()) {
            ajax_access_denied();
        }

        $type = $this->input->get('type');
        $date_from = $this->input->get('date_from') ?: date('Y-m-01', strtotime('-3 months'));
        $date_to = $this->input->get('date_to') ?: date('Y-m-d');

        switch ($type) {
            case 'confidence_distribution':
                $data = $this->_get_confidence_distribution($date_from, $date_to);
                break;
            case 'verdict_trends':
                $data = $this->_get_verdict_trends($date_from, $date_to);
                break;
            case 'analysis_volume':
                $data = $this->_get_analysis_volume($date_from, $date_to);
                break;
            case 'staff_performance':
                $data = $this->_get_staff_performance($date_from, $date_to);
                break;
            default:
                $data = [];
        }

        echo json_encode($data);
    }

    /**
     * Test AJAX endpoint
     */
    public function test_ajax()
    {
        header('Content-Type: application/json');
        
        $response = [
            'success' => true,
            'message' => 'AJAX endpoint is working',
            'timestamp' => date('Y-m-d H:i:s'),
            'is_admin' => is_admin(),
            'is_ajax' => $this->input->is_ajax_request(),
            'post_data' => $_POST,
            'get_data' => $_GET
        ];
        
        echo json_encode($response);
    }

    /**
     * Simple test view without AJAX
     */
    public function simple_test()
    {
        if (!is_admin() && !has_permission('leads', '', 'view')) {
            access_denied('AI Reports');
        }

        $data['title'] = 'AI Reports - Simple Test';
        $this->load->view('admin/ai_reports/simple_test', $data);
    }

    /**
     * Get table data for DataTables
     */
    private function _get_table_data($filters = [])
    {
        // Debug logging
        log_message('debug', 'AI Reports _get_table_data called with filters: ' . json_encode($filters));
        log_message('debug', 'POST data: ' . json_encode($_POST));
        
        $columns = [
            'r.id',
            'l.name',
            'r.confidence_score',
            'r.purchase_likelihood',
            'r.analysis_type',
            'r.created_at',
            's.firstname'
        ];

        $limit = $this->input->post('length');
        $start = $this->input->post('start');
        $order_column = $this->input->post('order')[0]['column'];
        $order_dir = $this->input->post('order')[0]['dir'];
        
        // Adjust column index for admin checkbox
        if (is_admin()) {
            $order_column = $order_column > 0 ? $order_column - 1 : 0;
        }
        
        $order = isset($columns[$order_column]) ? $columns[$order_column] : 'r.id';

        // Build query
        $this->db->select('r.*, l.name as lead_name, l.company as lead_company, l.email as lead_email, 
                          CONCAT(s.firstname, " ", s.lastname) as created_by_name')
                 ->from(db_prefix() . 'leads_ai_reports r')
                 ->join(db_prefix() . 'leads l', 'l.id = r.lead_id', 'left')
                 ->join(db_prefix() . 'staff s', 's.staffid = r.created_by', 'left');

        // Apply filters
        if (!empty($filters['lead_id'])) {
            $this->db->where('r.lead_id', $filters['lead_id']);
        }

        if (!empty($filters['verdict'])) {
            $this->db->where('r.verdict', $filters['verdict']);
        }

        if (!empty($filters['date_from'])) {
            $this->db->where('DATE(r.created_at) >=', $filters['date_from']);
        }

        if (!empty($filters['date_to'])) {
            $this->db->where('DATE(r.created_at) <=', $filters['date_to']);
        }

        if (!empty($filters['staff_id'])) {
            $this->db->where('r.created_by', $filters['staff_id']);
        }

        if (!empty($filters['analysis_type'])) {
            $this->db->where('r.analysis_type', $filters['analysis_type']);
        }

        // Search
        if ($this->input->post('search')['value']) {
            $search = $this->input->post('search')['value'];
            $this->db->group_start()
                     ->like('l.name', $search)
                     ->or_like('l.company', $search)
                     ->or_like('l.email', $search)
                     ->or_like('r.key_insights', $search)
                     ->or_like('r.recommendations', $search)
                     ->group_end();
        }

        // If not admin, only show reports for accessible leads
        if (!is_admin()) {
            $staff_id = get_staff_user_id();
            $this->db->where('(l.assigned = ' . $staff_id . ' OR r.created_by = ' . $staff_id . ')');
        }

        $reports = $this->db->order_by($order, $order_dir)
                           ->limit($limit, $start)
                           ->get()
                           ->result();

        // Count total records
        $this->db->select('COUNT(*) as count')
                 ->from(db_prefix() . 'leads_ai_reports r')
                 ->join(db_prefix() . 'leads l', 'l.id = r.lead_id', 'left');

        // Apply same filters for count
        if (!empty($filters['lead_id'])) {
            $this->db->where('r.lead_id', $filters['lead_id']);
        }
        if (!empty($filters['verdict'])) {
            $this->db->where('r.verdict', $filters['verdict']);
        }
        if (!empty($filters['date_from'])) {
            $this->db->where('DATE(r.created_at) >=', $filters['date_from']);
        }
        if (!empty($filters['date_to'])) {
            $this->db->where('DATE(r.created_at) <=', $filters['date_to']);
        }
        if (!empty($filters['staff_id'])) {
            $this->db->where('r.created_by', $filters['staff_id']);
        }
        if (!empty($filters['analysis_type'])) {
            $this->db->where('r.analysis_type', $filters['analysis_type']);
        }

        if (!is_admin()) {
            $staff_id = get_staff_user_id();
            $this->db->where('(l.assigned = ' . $staff_id . ' OR r.created_by = ' . $staff_id . ')');
        }

        $total_count = $this->db->get()->row()->count;

        $output = [
            'draw' => intval($this->input->post('draw')),
            'recordsTotal' => $total_count,
            'recordsFiltered' => $total_count,
            'data' => [],
            'csrf_token' => $this->security->get_csrf_hash()
        ];

        foreach ($reports as $report) {
            $row = [];
            
            // Report ID with link
            $row[] = '<a href="' . admin_url('ai_reports/view/' . $report->id) . '">#' . $report->id . '</a>';
            
            // Lead info
            $lead_link = '<a href="' . admin_url('leads/index/' . $report->lead_id) . '">' . $report->lead_name . '</a>';
            if ($report->lead_company) {
                $lead_link .= '<br><small class="text-muted">' . $report->lead_company . '</small>';
            }
            $row[] = $lead_link;
            
            // Confidence score with progress bar
            $score = $report->confidence_score ?: 0;
            $score_class = $score >= 80 ? 'success' : ($score >= 60 ? 'warning' : 'danger');
            $row[] = '<div class="progress" style="margin-bottom: 0;">
                        <div class="progress-bar progress-bar-' . $score_class . '" style="width: ' . $score . '%">' . $score . '%</div>
                      </div>';
            
            // Verdict with badge
            $verdict = $report->purchase_likelihood ?: 'Unknown';
            $badge_class = $this->_get_verdict_badge_class($verdict);
            $row[] = '<span class="label label-' . $badge_class . '">' . $verdict . '</span>';
            
            // Analysis type
            $row[] = ucfirst(str_replace('_', ' ', $report->analysis_type));
            
            // Created date
            $row[] = _dt($report->created_at);
            
            // Created by
            $row[] = $report->created_by_name ?: 'Unknown';
            
            // Actions
            $actions = '<div class="btn-group">';
            $actions .= '<a href="' . admin_url('ai_reports/view/' . $report->id) . '" class="btn btn-default btn-xs" title="View"><i class="fa fa-eye"></i></a>';
            if (is_admin()) {
                $actions .= '<a href="' . admin_url('ai_reports/delete/' . $report->id) . '" class="btn btn-danger btn-xs" onclick="return confirm(\'Are you sure?\')" title="Delete"><i class="fa fa-trash"></i></a>';
            }
            $actions .= '</div>';
            $row[] = $actions;
            
            $output['data'][] = $row;
        }

        echo json_encode($output);
    }

    /**
     * Get report with lead information
     */
    private function _get_report_with_lead($report_id)
    {
        return $this->db->select('r.*, l.name as lead_name, l.company as lead_company, l.email as lead_email, 
                                 l.phonenumber as lead_phone, CONCAT(s.firstname, " ", s.lastname) as created_by_name')
                        ->from(db_prefix() . 'leads_ai_reports r')
                        ->join(db_prefix() . 'leads l', 'l.id = r.lead_id', 'left')
                        ->join(db_prefix() . 'staff s', 's.staffid = r.created_by', 'left')
                        ->where('r.id', $report_id)
                        ->get()
                        ->row();
    }

    /**
     * Get all reports for export
     */
    private function _get_all_reports($filters = [])
    {
        $this->db->select('r.*, l.name as lead_name, l.company as lead_company, l.email as lead_email, 
                          l.phonenumber as lead_phone, CONCAT(s.firstname, " ", s.lastname) as created_by_name,
                          COALESCE(r.score, r.confidence_score) as score_value,
                          COALESCE(r.verdict, r.purchase_likelihood) as verdict_value')
                 ->from(db_prefix() . 'leads_ai_reports r')
                 ->join(db_prefix() . 'leads l', 'l.id = r.lead_id', 'left')
                 ->join(db_prefix() . 'staff s', 's.staffid = r.created_by', 'left');

        // Apply filters (same as table data)
        if (!empty($filters['lead_id'])) {
            $this->db->where('r.lead_id', $filters['lead_id']);
        }
        if (!empty($filters['verdict'])) {
            $this->db->group_start();
            $this->db->where('r.verdict', $filters['verdict']);
            $this->db->or_where('r.purchase_likelihood', $filters['verdict']);
            $this->db->group_end();
        }
        if (!empty($filters['date_from'])) {
            $this->db->where('DATE(r.created_at) >=', $filters['date_from']);
        }
        if (!empty($filters['date_to'])) {
            $this->db->where('DATE(r.created_at) <=', $filters['date_to']);
        }
        if (!empty($filters['staff_id'])) {
            $this->db->where('r.created_by', $filters['staff_id']);
        }
        if (!empty($filters['analysis_type'])) {
            $this->db->where('r.analysis_type', $filters['analysis_type']);
        }

        if (!is_admin()) {
            $staff_id = get_staff_user_id();
            $this->db->where('(l.assigned = ' . $staff_id . ' OR r.created_by = ' . $staff_id . ')');
        }

        return $this->db->order_by('r.created_at', 'DESC')->get()->result();
    }

    /**
     * Get reports statistics
     */
    private function _get_reports_statistics($filters = [])
    {
        // Build WHERE clause for filters
        $where_conditions = [];
        $where_params = [];
        
        if (!empty($filters['lead_id'])) {
            $where_conditions[] = 'r.lead_id = ?';
            $where_params[] = $filters['lead_id'];
        }
        if (!empty($filters['verdict'])) {
            $where_conditions[] = 'r.verdict = ?';
            $where_params[] = $filters['verdict'];
        }
        if (!empty($filters['date_from'])) {
            $where_conditions[] = 'DATE(r.created_at) >= ?';
            $where_params[] = $filters['date_from'];
        }
        if (!empty($filters['date_to'])) {
            $where_conditions[] = 'DATE(r.created_at) <= ?';
            $where_params[] = $filters['date_to'];
        }
        if (!empty($filters['staff_id'])) {
            $where_conditions[] = 'r.created_by = ?';
            $where_params[] = $filters['staff_id'];
        }
        if (!empty($filters['analysis_type'])) {
            $where_conditions[] = 'r.analysis_type = ?';
            $where_params[] = $filters['analysis_type'];
        }

        if (!is_admin()) {
            $staff_id = get_staff_user_id();
            $where_conditions[] = '(l.assigned = ? OR r.created_by = ?)';
            $where_params[] = $staff_id;
            $where_params[] = $staff_id;
        }
        
        $where_clause = '';
        if (!empty($where_conditions)) {
            $where_clause = 'WHERE ' . implode(' AND ', $where_conditions);
        }
        
        $reports_table = db_prefix() . 'leads_ai_reports';
        $leads_table = db_prefix() . 'leads';
        
        // Query 1: Get total count
        $total_sql = "SELECT COUNT(*) as total 
                      FROM {$reports_table} r 
                      LEFT JOIN {$leads_table} l ON l.id = r.lead_id 
                      {$where_clause}";
        
        $total_query = $this->db->query($total_sql, $where_params);
        $total = $total_query->row()->total ?? 0;
        
        // Query 2: Get verdict distribution (handle both column names)
        $verdict_sql = "SELECT COALESCE(r.verdict, r.purchase_likelihood) as verdict, COUNT(*) as count 
                        FROM {$reports_table} r 
                        LEFT JOIN {$leads_table} l ON l.id = r.lead_id 
                        {$where_clause} 
                        GROUP BY COALESCE(r.verdict, r.purchase_likelihood)";
        
        $verdict_query = $this->db->query($verdict_sql, $where_params);
        $verdict_stats = $verdict_query->result();
        
        // Query 3: Get average confidence score (handle both column names)
        $avg_sql = "SELECT AVG(COALESCE(r.score, r.confidence_score)) as avg_score 
                    FROM {$reports_table} r 
                    LEFT JOIN {$leads_table} l ON l.id = r.lead_id 
                    {$where_clause}";
        
        $avg_query = $this->db->query($avg_sql, $where_params);
        $avg_score = $avg_query->row()->avg_score ?? 0;

        return [
            'total' => $total,
            'verdict_distribution' => $verdict_stats,
            'average_confidence' => round($avg_score, 1)
        ];
    }

    /**
     * Get available verdicts
     */
    private function _get_available_verdicts()
    {
        // Use raw query to handle both possible column names
        $sql = "SELECT DISTINCT COALESCE(verdict, purchase_likelihood) as verdict 
                FROM " . db_prefix() . "leads_ai_reports 
                WHERE COALESCE(verdict, purchase_likelihood) IS NOT NULL 
                AND COALESCE(verdict, purchase_likelihood) != ''";
        
        return $this->db->query($sql)->result_array();
    }

    /**
     * Get available analysis types
     */
    private function _get_available_analysis_types()
    {
        return $this->db->select('analysis_type')
                        ->distinct()
                        ->where('analysis_type IS NOT NULL')
                        ->where('analysis_type !=', '')
                        ->get(db_prefix() . 'leads_ai_reports')
                        ->result_array();
    }

    /**
     * Get verdict badge class
     */
    private function _get_verdict_badge_class($verdict)
    {
        switch (strtolower($verdict)) {
            case 'hot':
                return 'danger';
            case 'warm':
                return 'warning';
            case 'cold':
                return 'info';
            case 'qualified':
                return 'success';
            default:
                return 'default';
        }
    }

    /**
     * Export to Excel
     */
    private function _export_excel($reports)
    {
        // This would use a library like PhpSpreadsheet
        // For now, implementing as CSV with Excel-compatible format
        $this->_export_csv($reports, 'excel');
    }

    /**
     * Export to CSV
     */
    private function _export_csv($reports, $format = 'csv')
    {
        $filename = 'ai_analysis_reports_' . date('Y-m-d_H-i-s') . '.' . $format;
        
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        
        $output = fopen('php://output', 'w');
        
        // Add BOM for UTF-8 compatibility
        fprintf($output, chr(0xEF).chr(0xBB).chr(0xBF));
        
        // Add header info
        fputcsv($output, ['AI Analysis Reports Export']);
        fputcsv($output, ['Generated on: ' . date('F j, Y \a\t g:i A')]);
        fputcsv($output, ['Total Reports: ' . count($reports)]);
        fputcsv($output, []); // Empty row
        
        // Column headers
        fputcsv($output, [
            'Report ID',
            'Lead ID',
            'Lead Name',
            'Company',
            'Email',
            'Phone',
            'Confidence Score (%)',
            'Verdict',
            'Analysis Type',
            'Key Insights',
            'Recommendations',
            'Created By',
            'Created Date',
            'Status'
        ]);
        
        // Data rows
        foreach ($reports as $report) {
            fputcsv($output, [
                $report->id,
                $report->lead_id,
                $report->lead_name ?: 'Unknown',
                $report->lead_company ?: '',
                $report->lead_email ?: '',
                $report->lead_phone ?: '',
                $report->score,
                $report->verdict ?: 'Unknown',
                ucfirst(str_replace('_', ' ', $report->analysis_type)),
                strip_tags($report->key_insights ?: ''),
                strip_tags($report->recommendations ?: ''),
                $report->created_by_name ?: 'Unknown',
                date('F j, Y \a\t g:i A', strtotime($report->created_at)),
                $report->status ?: 'completed'
            ]);
        }
        
        fclose($output);
        exit;
    }

    /**
     * Export to PDF
     */
    private function _export_pdf($reports)
    {
        $data['reports'] = $reports;
        $data['title'] = 'AI Analysis Reports Export';
        
        // Set headers for HTML display that can be printed as PDF
        header('Content-Type: text/html; charset=utf-8');
        header('Cache-Control: no-cache, must-revalidate');
        header('Expires: 0');
        
        // Load the PDF export view
        echo $this->load->view('admin/ai_reports/pdf_export', $data, true);
        exit;
    }

    /**
     * Get analytics data
     */
    private function _get_analytics_data($date_from, $date_to)
    {
        // This would implement various analytics queries
        // Placeholder for now
        return [
            'total_reports' => 0,
            'confidence_trends' => [],
            'verdict_distribution' => [],
            'top_performing_staff' => []
        ];
    }

    /**
     * Get confidence distribution for charts
     */
    private function _get_confidence_distribution($date_from, $date_to)
    {
        // Placeholder - would return chart data
        return [];
    }

    /**
     * Get verdict trends for charts
     */
    private function _get_verdict_trends($date_from, $date_to)
    {
        // Placeholder - would return chart data
        return [];
    }

    /**
     * Get analysis volume for charts
     */
    private function _get_analysis_volume($date_from, $date_to)
    {
        // Placeholder - would return chart data
        return [];
    }

    /**
     * Get staff performance for charts
     */
    private function _get_staff_performance($date_from, $date_to)
    {
        // Placeholder - would return chart data
        return [];
    }

    /**
     * Static test page for debugging
     */
    public function static_test()
    {
        if (!is_admin() && !has_permission('leads', '', 'view')) {
            access_denied('AI Reports');
        }

        $data['title'] = 'AI Reports - Static Test';
        $this->load->view('admin/ai_reports/static_test', $data);
    }

    /**
     * Generate complete HTML table for reports (safe from JavaScript interference)
     */
    private function _generate_reports_table_html($reports)
    {
        $html = '';
        
        if (empty($reports)) {
            $html = '<tr><td colspan="8" class="text-center">No AI reports found</td></tr>';
        } else {
            foreach ($reports as $report) {
                $score = $report->score ?: 0;
                $score_class = $score >= 80 ? 'success' : ($score >= 60 ? 'warning' : 'danger');
                
                $verdict = $report->verdict ?: 'Unknown';
                $badge_class = '';
                switch (strtolower($verdict)) {
                    case 'hot':
                    case 'high':
                        $badge_class = 'success';
                        break;
                    case 'warm':
                    case 'medium':
                        $badge_class = 'warning';
                        break;
                    case 'cold':
                    case 'low':
                        $badge_class = 'danger';
                        break;
                    default:
                        $badge_class = 'default';
                        break;
                }
                
                $html .= '<tr>';
                $html .= '<td><a href="' . admin_url('ai_reports/view/' . $report->id) . '">#' . $report->id . '</a></td>';
                
                // Lead column
                $html .= '<td>';
                $html .= '<a href="' . admin_url('leads/index/' . $report->lead_id) . '">' . e($report->lead_name) . '</a>';
                if (!empty($report->lead_company)) {
                    $html .= '<br><small class="text-muted">' . e($report->lead_company) . '</small>';
                }
                $html .= '</td>';
                
                // Confidence score column
                $html .= '<td data-score="' . $score . '">';
                $html .= '<div class="progress" style="margin-bottom: 0;">';
                $html .= '<div class="progress-bar progress-bar-' . $score_class . '" style="width: ' . $score . '%">' . $score . '%</div>';
                $html .= '</div>';
                $html .= '</td>';
                
                // Verdict column
                $html .= '<td><span class="label label-' . $badge_class . '">' . $verdict . '</span></td>';
                
                // Analysis type
                $html .= '<td>' . ucfirst(str_replace('_', ' ', $report->analysis_type)) . '</td>';
                
                // Created date
                $html .= '<td>' . date('M j, Y g:i A', strtotime($report->created_at)) . '</td>';
                
                // Created by
                $html .= '<td>' . ($report->created_by_name ?: 'Unknown') . '</td>';
                
                // Actions
                $html .= '<td>';
                $html .= '<div class="btn-group">';
                $html .= '<a href="' . admin_url('ai_reports/view/' . $report->id) . '" class="btn btn-default btn-xs" title="View"><i class="fa fa-eye"></i></a>';
                if (is_admin()) {
                    $html .= '<a href="' . admin_url('ai_reports/delete/' . $report->id) . '" class="btn btn-danger btn-xs" onclick="return confirm(\'Are you sure?\')" title="Delete"><i class="fa fa-trash"></i></a>';
                }
                $html .= '</div>';
                $html .= '</td>';
                
                $html .= '</tr>';
            }
        }
        
        return $html;
    }
}
?>
