<?php

defined('BASEPATH') or exit('No direct script access allowed');

/**
 * Facebook Helper Functions
 * Utility functions for Facebook API integration
 */

/**
 * Make a request to Facebook Graph API
 * @param string $endpoint API endpoint (e.g., '/me' or '/123456789/leads')
 * @param array $params Query parameters
 * @param string $access_token Access token
 * @param string $method HTTP method (GET, POST, DELETE)
 * @return array Response data
 */
function facebook_api_request($endpoint, $params = [], $access_token = '', $method = 'GET')
{
    $base_url = 'https://graph.facebook.com/v20.0';
    
    // Add access token to params
    if (!empty($access_token)) {
        $params['access_token'] = $access_token;
    }

    // Build URL
    $url = $base_url . $endpoint;

    // Initialize cURL
    $ch = curl_init();

    if ($method === 'GET') {
        $url .= '?' . http_build_query($params);
        curl_setopt($ch, CURLOPT_URL, $url);
    } elseif ($method === 'POST') {
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($params));
    } elseif ($method === 'DELETE') {
        curl_setopt($ch, CURLOPT_URL, $url . '?' . http_build_query($params));
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'DELETE');
    }

    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Accept: application/json',
        'Content-Type: application/x-www-form-urlencoded'
    ]);

    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curl_error = curl_error($ch);
    
    curl_close($ch);

    // Handle cURL errors
    if ($response === false) {
        throw new Exception('cURL Error: ' . $curl_error);
    }

    // Parse response
    $data = json_decode($response, true);

    // Log API request for debugging
    log_facebook_info('api_request', [
        'endpoint' => $endpoint,
        'method' => $method,
        'http_code' => $http_code,
        'response' => $data
    ]);

    if ($http_code >= 400) {
        log_facebook_error('api_error', "HTTP {$http_code}: " . ($data['error']['message'] ?? 'Unknown error'));
    }

    return $data;
}

/**
 * Get long-lived page access token
 * @param string $app_id App ID
 * @param string $app_secret App Secret
 * @param string $short_lived_token Short-lived user access token
 * @return array|null
 */
function facebook_exchange_token($app_id, $app_secret, $short_lived_token)
{
    try {
        $response = facebook_api_request('/oauth/access_token', [
            'grant_type' => 'fb_exchange_token',
            'client_id' => $app_id,
            'client_secret' => $app_secret,
            'fb_exchange_token' => $short_lived_token
        ]);

        return $response;
    } catch (Exception $e) {
        log_facebook_error('token_exchange', $e->getMessage());
        return null;
    }
}

/**
 * Get page access token from user access token
 * @param string $page_id Page ID
 * @param string $user_access_token User access token
 * @return string|null
 */
function facebook_get_page_access_token($page_id, $user_access_token)
{
    try {
        $response = facebook_api_request('/' . $page_id, [
            'fields' => 'access_token'
        ], $user_access_token);

        return $response['access_token'] ?? null;
    } catch (Exception $e) {
        log_facebook_error('get_page_token', $e->getMessage());
        return null;
    }
}

/**
 * Get all lead forms for a Facebook page
 * @param string $page_id Page ID
 * @param string $page_access_token Page access token
 * @return array
 */
function facebook_get_lead_forms($page_id, $page_access_token)
{
    try {
        $response = facebook_api_request("/{$page_id}/leadgen_forms", [
            'fields' => 'id,name,status,leads_count'
        ], $page_access_token);

        return $response['data'] ?? [];
    } catch (Exception $e) {
        log_facebook_error('get_lead_forms', $e->getMessage());
        return [];
    }
}

/**
 * Test Facebook API connection
 * @param string $page_id Page ID
 * @param string $page_access_token Page access token
 * @return array ['success' => bool, 'message' => string, 'data' => array]
 */
function facebook_test_connection($page_id, $page_access_token)
{
    try {
        $response = facebook_api_request("/{$page_id}", [
            'fields' => 'id,name,access_token'
        ], $page_access_token);

        if (isset($response['error'])) {
            return [
                'success' => false,
                'message' => $response['error']['message'],
                'data' => null
            ];
        }

        return [
            'success' => true,
            'message' => 'Connection successful',
            'data' => $response
        ];
    } catch (Exception $e) {
        return [
            'success' => false,
            'message' => $e->getMessage(),
            'data' => null
        ];
    }
}

/**
 * Log informational message to file
 * @param string $context Context identifier
 * @param mixed $message Message or data to log
 */
function log_facebook_info($context, $message)
{
    $log_file = FCPATH . 'modules/facebook_leads_integration/logs/info_' . date('Y-m-d') . '.log';
    $timestamp = date('Y-m-d H:i:s');
    
    if (is_array($message) || is_object($message)) {
        $message = json_encode($message, JSON_PRETTY_PRINT);
    }
    
    $log_entry = "[{$timestamp}] [{$context}] {$message}\n";
    
    @file_put_contents($log_file, $log_entry, FILE_APPEND);
}

/**
 * Log error message to file
 * @param string $context Context identifier
 * @param string $message Error message
 */
function log_facebook_error($context, $message)
{
    $log_file = FCPATH . 'modules/facebook_leads_integration/logs/error_' . date('Y-m-d') . '.log';
    $timestamp = date('Y-m-d H:i:s');
    
    if (is_array($message) || is_object($message)) {
        $message = json_encode($message, JSON_PRETTY_PRINT);
    }
    
    $log_entry = "[{$timestamp}] [ERROR] [{$context}] {$message}\n";
    
    @file_put_contents($log_file, $log_entry, FILE_APPEND);
}

/**
 * Clean old log files (older than 30 days)
 */
function facebook_clean_old_logs()
{
    $log_dir = FCPATH . 'modules/facebook_leads_integration/logs/';
    
    if (!is_dir($log_dir)) {
        return;
    }
    
    $files = glob($log_dir . '*.log');
    $now = time();
    
    foreach ($files as $file) {
        if (is_file($file)) {
            // Delete files older than 30 days
            if ($now - filemtime($file) >= 30 * 24 * 60 * 60) {
                @unlink($file);
            }
        }
    }
}

/**
 * Validate Facebook webhook signature (for enhanced security)
 * @param string $payload Raw POST data
 * @param string $signature Signature from X-Hub-Signature-256 header
 * @param string $app_secret App Secret
 * @return bool
 */
function facebook_validate_webhook_signature($payload, $signature, $app_secret)
{
    if (empty($signature)) {
        return false;
    }
    
    list($algorithm, $provided_hash) = explode('=', $signature, 2);
    
    if ($algorithm !== 'sha256') {
        return false;
    }
    
    $expected_hash = hash_hmac('sha256', $payload, $app_secret);
    
    return hash_equals($expected_hash, $provided_hash);
}

/**
 * Format phone number to international format
 * @param string $phone Phone number
 * @param string $country_code Default country code (e.g., 'US')
 * @return string
 */
function facebook_format_phone($phone, $country_code = 'US')
{
    // Remove all non-numeric characters
    $phone = preg_replace('/[^0-9]/', '', $phone);
    
    // Add + if not present and number doesn't start with country code
    if (!empty($phone) && substr($phone, 0, 1) !== '+') {
        // You can add more sophisticated country code detection here
        $phone = '+' . $phone;
    }
    
    return $phone;
}

