<?php

defined('BASEPATH') or exit('No direct script access allowed');

/**
 * Facebook Leads Integration Controller
 * Handles all admin panel interactions and API operations
 */
class Facebook_leads_integration extends AdminController
{
    public function __construct()
    {
        parent::__construct();
        $this->load->model('facebook_leads_integration_model');
        $this->load->helper('facebook_helper');
    }

    /**
     * Settings page
     */
    public function settings()
    {
        if (!has_permission('facebook_leads_integration', '', 'view')) {
            access_denied('facebook_leads_integration');
        }

        // Handle form submission
        if ($this->input->post()) {
            $this->handle_settings_update();
        }

        // Load current settings
        $data['settings'] = $this->facebook_leads_integration_model->get_all_settings();
        
        // Get staff list for assignment dropdown
        $this->load->model('staff_model');
        $data['staff_members'] = $this->staff_model->get();
        
        // Get lead statuses
        $this->load->model('leads_model');
        $data['lead_statuses'] = $this->leads_model->get_status();
        
        // Get lead sources
        $data['lead_sources'] = $this->db->get(db_prefix() . 'leads_sources')->result();
        
        // Statistics
        $data['stats'] = $this->facebook_leads_integration_model->get_statistics();
        
        $data['title'] = _l('facebook_leads_settings');
        $this->load->view('settings', $data);
    }

    /**
     * Handle settings form submission
     */
    private function handle_settings_update()
    {
        if (!has_permission('facebook_leads_integration', '', 'edit')) {
            access_denied('facebook_leads_integration');
        }

        $settings = [
            'app_id'                  => $this->input->post('app_id'),
            'app_secret'              => $this->input->post('app_secret'),
            'page_access_token'       => $this->input->post('page_access_token'),
            'page_id'                 => $this->input->post('page_id'),
            'form_ids'                => $this->input->post('form_ids'),
            'default_assigned_staff'  => $this->input->post('default_assigned_staff'),
            'lead_status'             => $this->input->post('lead_status'),
            'lead_source'             => $this->input->post('lead_source'),
            'cron_interval'           => $this->input->post('cron_interval'),
            'enabled'                 => $this->input->post('enabled') ? '1' : '0',
        ];

        foreach ($settings as $name => $value) {
            $this->facebook_leads_integration_model->update_setting($name, $value);
        }

        set_alert('success', _l('fbl_settings_saved'));
        redirect(admin_url('facebook_leads_integration/settings'));
    }

    /**
     * Test Facebook API connection (AJAX)
     */
    public function test_connection()
    {
        if (!has_permission('facebook_leads_integration', '', 'view')) {
            echo json_encode(['success' => false, 'message' => 'Access denied']);
            return;
        }

        $page_access_token = $this->input->post('page_access_token');
        $page_id = $this->input->post('page_id');

        if (empty($page_access_token) || empty($page_id)) {
            echo json_encode([
                'success' => false,
                'message' => 'Please provide Page Access Token and Page ID'
            ]);
            return;
        }

        try {
            $result = facebook_api_request("/{$page_id}", ['fields' => 'name,id'], $page_access_token);
            
            if (isset($result['error'])) {
                echo json_encode([
                    'success' => false,
                    'message' => $result['error']['message'] ?? 'Unknown error'
                ]);
            } else {
                echo json_encode([
                    'success' => true,
                    'message' => sprintf(_l('fbl_connection_success'), $result['name']),
                    'page_name' => $result['name']
                ]);
            }
        } catch (Exception $e) {
            echo json_encode([
                'success' => false,
                'message' => $e->getMessage()
            ]);
        }
    }

    /**
     * Manual fetch leads (AJAX or direct)
     */
    public function fetch_leads()
    {
        if (!has_permission('facebook_leads_integration', '', 'edit')) {
            if ($this->input->is_ajax_request()) {
                echo json_encode(['success' => false, 'message' => 'Access denied']);
                return;
            }
            access_denied('facebook_leads_integration');
        }

        $settings = $this->facebook_leads_integration_model->get_all_settings();

        if ($settings['enabled'] != '1') {
            $message = 'Integration is disabled';
            if ($this->input->is_ajax_request()) {
                echo json_encode(['success' => false, 'message' => $message]);
            } else {
                set_alert('warning', $message);
                redirect(admin_url('facebook_leads_integration/settings'));
            }
            return;
        }

        if (empty($settings['page_access_token']) || empty($settings['form_ids'])) {
            $message = 'Please configure all required settings first';
            if ($this->input->is_ajax_request()) {
                echo json_encode(['success' => false, 'message' => $message]);
            } else {
                set_alert('warning', $message);
                redirect(admin_url('facebook_leads_integration/settings'));
            }
            return;
        }

        try {
            $result = $this->facebook_leads_integration_model->fetch_and_import_leads();
            
            $this->facebook_leads_integration_model->update_setting('last_fetch_time', date('Y-m-d H:i:s'));
            
            if ($this->input->is_ajax_request()) {
                echo json_encode([
                    'success' => true,
                    'message' => sprintf(_l('fbl_leads_fetched'), $result['imported']),
                    'imported' => $result['imported'],
                    'duplicates' => $result['duplicates'],
                    'errors' => $result['errors']
                ]);
            } else {
                set_alert('success', sprintf(_l('fbl_leads_fetched'), $result['imported']));
                redirect(admin_url('facebook_leads_integration/settings'));
            }
        } catch (Exception $e) {
            log_facebook_error('fetch_leads', $e->getMessage());
            
            if ($this->input->is_ajax_request()) {
                echo json_encode([
                    'success' => false,
                    'message' => sprintf(_l('fbl_fetch_error'), $e->getMessage())
                ]);
            } else {
                set_alert('danger', sprintf(_l('fbl_fetch_error'), $e->getMessage()));
                redirect(admin_url('facebook_leads_integration/settings'));
            }
        }
    }

    /**
     * View logs
     */
    public function logs()
    {
        if (!has_permission('facebook_leads_integration', '', 'view')) {
            access_denied('facebook_leads_integration');
        }

        $data['logs'] = $this->facebook_leads_integration_model->get_logs(100);
        $data['title'] = _l('facebook_leads_logs');
        
        $this->load->view('logs', $data);
    }

    /**
     * View single log data (AJAX)
     */
    public function view_log_data($id)
    {
        if (!has_permission('facebook_leads_integration', '', 'view')) {
            echo json_encode(['success' => false, 'message' => 'Access denied']);
            return;
        }

        $log = $this->facebook_leads_integration_model->get_log($id);
        
        if ($log) {
            echo json_encode([
                'success' => true,
                'data' => json_decode($log->data, true)
            ]);
        } else {
            echo json_encode([
                'success' => false,
                'message' => 'Log not found'
            ]);
        }
    }

    /**
     * Facebook Webhook endpoint
     * Receives real-time lead notifications
     */
    public function webhook()
    {
        $method = $_SERVER['REQUEST_METHOD'];

        if ($method === 'GET') {
            // Webhook verification
            $this->verify_webhook();
        } elseif ($method === 'POST') {
            // Receive lead data
            $this->receive_webhook();
        }
    }

    /**
     * Verify Facebook webhook (GET request)
     */
    private function verify_webhook()
    {
        $verify_token = $this->facebook_leads_integration_model->get_setting('webhook_verify_token');
        
        $hub_verify_token = $this->input->get('hub_verify_token');
        $hub_challenge = $this->input->get('hub_challenge');
        $hub_mode = $this->input->get('hub_mode');

        if ($hub_mode === 'subscribe' && $hub_verify_token === $verify_token) {
            echo $hub_challenge;
            http_response_code(200);
        } else {
            http_response_code(403);
            echo 'Forbidden';
        }
        exit;
    }

    /**
     * Receive webhook data (POST request)
     */
    private function receive_webhook()
    {
        $input = file_get_contents('php://input');
        $data = json_decode($input, true);

        // Log raw webhook data
        log_facebook_info('webhook_received', $input);

        if (isset($data['entry'])) {
            foreach ($data['entry'] as $entry) {
                if (isset($entry['changes'])) {
                    foreach ($entry['changes'] as $change) {
                        if ($change['field'] === 'leadgen' && isset($change['value']['leadgen_id'])) {
                            // Process the lead
                            $leadgen_id = $change['value']['leadgen_id'];
                            $form_id = $change['value']['form_id'] ?? '';
                            
                            try {
                                $this->facebook_leads_integration_model->fetch_and_import_single_lead($leadgen_id, $form_id);
                            } catch (Exception $e) {
                                log_facebook_error('webhook_process', $e->getMessage());
                            }
                        }
                    }
                }
            }
        }

        http_response_code(200);
        echo 'OK';
        exit;
    }
}

