<?php

defined('BASEPATH') or exit('No direct script access allowed');

class Reporting_dashboard_model extends App_Model
{
    public function __construct()
    {
        parent::__construct();
    }

    public function get_kpis(array $filters)
    {
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        
        $today = date('Y-m-d');
        $weekStart = date('Y-m-d', strtotime('monday this week'));
        $monthStart = date('Y-m-01');
        
        return [
            'today' => $this->count_lead_activities(['from' => $today, 'to' => $today, 'staff' => $filters['staff'] ?? []]),
            'week' => $this->count_lead_activities(['from' => $weekStart, 'to' => $today, 'staff' => $filters['staff'] ?? []]),
            'month' => $this->count_lead_activities(['from' => $monthStart, 'to' => $today, 'staff' => $filters['staff'] ?? []]),
            'logins_today' => $this->count_logins_today($filters['staff'] ?? []),
            'top_staff' => $this->get_top_staff_simple($filters)
        ];
    }

    public function get_staff_activity_chart(array $filters)
    {
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        
        $this->db->select('s.staffid, CONCAT(s.firstname, " ", s.lastname) as staff_name, COUNT(a.id) as total');
        $this->db->from(db_prefix() . 'lead_activity_log a');
        $this->db->join(db_prefix() . 'staff s', 's.staffid = a.staffid', 'left');
        
        if ($from) $this->db->where('DATE(a.date) >=', $from);
        if ($to) $this->db->where('DATE(a.date) <=', $to);
        if (!empty($filters['staff'])) {
            $this->db->where_in('a.staffid', $filters['staff']);
        }
        
        $this->db->group_by('s.staffid');
        $this->db->order_by('total', 'DESC');
        $this->db->limit(10);
        
        $result = $this->db->get()->result_array();
        
        return [
            'labels' => array_column($result, 'staff_name'),
            'data' => array_map('intval', array_column($result, 'total'))
        ];
    }

    public function get_activity_over_time_chart(array $filters)
    {
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        
        $this->db->select('DATE(a.date) as date, COUNT(a.id) as total');
        $this->db->from(db_prefix() . 'lead_activity_log a');
        
        if ($from) $this->db->where('DATE(a.date) >=', $from);
        if ($to) $this->db->where('DATE(a.date) <=', $to);
        if (!empty($filters['staff'])) {
            $this->db->where_in('a.staffid', $filters['staff']);
        }
        
        $this->db->group_by('DATE(a.date)');
        $this->db->order_by('DATE(a.date)', 'ASC');
        
        $result = $this->db->get()->result_array();
        
        return [
            'labels' => array_column($result, 'date'),
            'data' => array_map('intval', array_column($result, 'total'))
        ];
    }

    public function get_activity_type_pie(array $filters)
    {
        try {
            // Guard: table may not exist yet
            $table = db_prefix() . 'lead_activity_log';
            if(!$this->db->table_exists(str_replace(db_prefix(), '', $table))){
                return [ 'labels'=>[], 'data'=>[] ];
            }
            $from = $filters['from'] ?? date('Y-m-01');
            $to = $filters['to'] ?? date('Y-m-d');

            // Classification aligned with PDF report logic (exact notification keywords)
            // so that PDF distribution matches dashboard chart.
            $this->db->select("CASE\n                WHEN LOWER(a.description) LIKE '%not_activity_new_reminder_created%' THEN 'Reminder Created'\n                WHEN LOWER(a.description) LIKE '%not_lead_activity_added_followup%' THEN 'Followup Added'\n                WHEN LOWER(a.description) LIKE '%not_lead_activity_lead_created%' THEN 'Lead Created'\n                WHEN LOWER(a.description) LIKE '%not_lead_activity_assigned%' THEN 'Lead Assigned'\n                ELSE 'Other'\n            END as activity_type, COUNT(a.id) as total", false);
            $this->db->from($table . ' a');
            if ($from) $this->db->where('DATE(a.date) >=', $from);
            if ($to) $this->db->where('DATE(a.date) <=', $to);
            if (!empty($filters['staff'])) {
                $this->db->where_in('a.staffid', $filters['staff']);
            }
            $this->db->group_by('activity_type');
            $this->db->order_by('total', 'DESC');
            $result = $this->db->get()->result_array();

            return [
                'labels' => array_column($result, 'activity_type'),
                'data' => array_map('intval', array_column($result, 'total'))
            ];
        } catch (Exception $e) {
            log_message('error', 'Reporting Dashboard get_activity_type_pie error: ' . $e->getMessage());
            return [ 'labels'=>[], 'data'=>[] ];
        }
    }

    /**
     * Build stacked follow-up types by staff.
     * Tries to infer follow-up type from activity description (heuristic) unless a dedicated field/table exists.
     * Returns structure with labels (staff) and datasets (array of series) matching createFollowupTypesChart expectations.
     */
    public function get_followup_types_chart(array $filters)
    {
        // Prefer structured follow-up types from lead_notes (more accurate than heuristic)
        $notesTable = db_prefix() . 'lead_notes';
        if(!$this->db->table_exists(str_replace(db_prefix(), '', $notesTable))){
            return [ 'labels'=>[], 'datasets'=>[] ];
        }
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        $this->db->select('ln.followup_type, ln.description, ln.dateadded, s.staffid, s.firstname, s.lastname');
        $this->db->from($notesTable . ' ln');
        $this->db->join(db_prefix().'staff s','s.staffid = ln.addedfrom','left');
        if($from) $this->db->where('DATE(ln.dateadded) >=',$from);
        if($to) $this->db->where('DATE(ln.dateadded) <=',$to);
        if (!empty($filters['staff'])) $this->db->where_in('ln.addedfrom', $filters['staff']);
        $rows = $this->db->get()->result_array();
        $staffNames=[]; $types=[]; $matrix=[]; $palette=[];
        $baseColors=['#4e73df','#1cc88a','#36b9cc','#f6c23e','#e74a3b','#858796','#2c9faf','#6f42c1','#17a673','#fd7e14'];
        $colorIndex=0;
        foreach($rows as $r){
            $sid=(int)($r['staffid'] ?? 0); $name=trim(($r['firstname']??'').' '.($r['lastname']??'')) ?: 'Staff #'.$sid; $staffNames[$sid]=$name;
            $type=$r['followup_type'];
            $desc=$r['description']??'';
            if(!$type && $desc){
                if(preg_match('/<strong>Type:\s*([^<]+)<\/strong>/i',$desc,$m)) $type=trim($m[1]);
                elseif(preg_match('/Type:\s*([^\r\n<]+)/i',$desc,$m)) $type=trim($m[1]);
            }
            if(!$type) $type='Other';
            if(!in_array($type,$types)) { $types[]=$type; if(!isset($palette[$type])) $palette[$type]=$baseColors[$colorIndex++ % count($baseColors)]; }
            if(!isset($matrix[$sid])) $matrix[$sid]=[];
            if(!isset($matrix[$sid][$type])) $matrix[$sid][$type]=0;
            $matrix[$sid][$type]++;
        }
        asort($staffNames); $labels=array_values($staffNames); $staffOrder=array_keys($staffNames);
        // Ensure consistent type order
        sort($types);
        $datasets=[];
        foreach($types as $t){
            $datasets[]=[
                'label'=>$t,
                'backgroundColor'=>$palette[$t] ?? '#666',
                'data'=>array_map(function($sid) use ($matrix,$t){ return isset($matrix[$sid][$t])?(int)$matrix[$sid][$t]:0; }, $staffOrder)
            ];
        }
        return ['labels'=>$labels,'datasets'=>$datasets];
    }

    /**
     * Lightweight activity log (object rows) for reuse (PDF, API) without DataTables decoration.
     */
    public function get_activity_log(array $filters, $limit = 50)
    {
        $table = db_prefix() . 'lead_activity_log';
        if(!$this->db->table_exists(str_replace(db_prefix(), '', $table))){
            return [];
        }
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        $this->db->select('a.*, s.firstname, s.lastname, l.name as lead_name, l.company as lead_company, ln.description as note_description, ln.followup_type');
        $this->db->from($table . ' a');
        $this->db->join(db_prefix() . 'staff s', 's.staffid = a.staffid', 'left');
        $this->db->join(db_prefix() . 'leads l', 'l.id = a.leadid', 'left');
        $this->db->join(db_prefix() . 'lead_notes ln', 'ln.leadid = a.leadid AND DATE(ln.dateadded) = DATE(a.date) AND ln.addedfrom = a.staffid', 'left');
        if ($from) $this->db->where('DATE(a.date) >=', $from);
        if ($to) $this->db->where('DATE(a.date) <=', $to);
        if (!empty($filters['staff'])) { $this->db->where_in('a.staffid', $filters['staff']); }
        $this->db->order_by('a.date', 'DESC');
        if($limit) $this->db->limit((int)$limit);
        $rows = $this->db->get()->result_array();
        $out = [];
        foreach($rows as $r){
            $staffName = trim(($r['firstname'] ?? '') . ' ' . ($r['lastname'] ?? '')) ?: 'System';
            
            // Build proper lead display name
            $leadId = (int)($r['leadid'] ?? 0);
            $leadName = trim($r['lead_name'] ?? '');
            $leadCompany = trim($r['lead_company'] ?? '');
            
            if ($leadName && $leadCompany) {
                $related = $leadName . ' (' . $leadCompany . ')';
            } elseif ($leadName) {
                $related = $leadName;
            } elseif ($leadCompany) {
                $related = $leadCompany;
            } elseif ($leadId > 0) {
                $related = 'Lead #' . $leadId;
            } else {
                $related = 'Unknown Lead';
            }
            
            $descLower = strtolower($r['description'] ?? '');
            $activityType = 'Activity';
            if(strpos($descLower,'not_activity_new_reminder_created') !== false){ $activityType = 'Reminder Created'; }
            elseif(strpos($descLower,'not_lead_activity_added_followup') !== false){ $activityType = 'Followup Added'; }
            elseif(strpos($descLower,'not_lead_activity_lead_created') !== false){ $activityType = 'Lead Created'; }
            elseif(strpos($descLower,'not_lead_activity_assigned') !== false){ $activityType = 'Lead Assigned'; }
            
            // Get follow-up information
            $followupType = trim($r['followup_type'] ?? '');
            $noteDescription = trim($r['note_description'] ?? '');
            
            // Build description based on activity type
            $description = 'Activity performed on lead';
            if ($activityType === 'Followup Added' && $noteDescription) {
                $description = $noteDescription;
            } elseif ($activityType === 'Reminder Created') {
                $description = 'Reminder created for follow-up';
            }
            
            $out[] = [
                'date' => $r['date'],
                'staff_name' => $staffName,
                'activity_type' => $activityType,
                'related' => $related,
                'description' => $description,
                'followup_type' => $followupType
            ];
        }
        return $out;
    }

    /**
     * Follow-up types by staff replication (used by PDF) – extracts type from lead_notes.followup_type or description patterns.
     */
    public function get_followup_types_by_staff(array $filters)
    {
        $notesTable = db_prefix() . 'lead_notes';
        if(!$this->db->table_exists(str_replace(db_prefix(), '', $notesTable))){
            return ['by_staff'=>[], 'all_types'=>[]];
        }
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        $this->db->select('ln.followup_type, ln.description, ln.dateadded, s.firstname, s.lastname');
        $this->db->from($notesTable . ' ln');
        $this->db->join(db_prefix() . 'staff s', 's.staffid = ln.addedfrom', 'left');
        if ($from) $this->db->where('DATE(ln.dateadded) >=', $from);
        if ($to) $this->db->where('DATE(ln.dateadded) <=', $to);
        $this->db->order_by('ln.dateadded','DESC');
        $rows = $this->db->get()->result_array();
        $byStaff = [];
        $allTypes = [];
        foreach($rows as $r){
            $staffName = trim(($r['firstname']??'').' '.($r['lastname']??''));
            if(!$staffName) $staffName = 'Unknown';
            $type = $r['followup_type'];
            $desc = $r['description'] ?? '';
            if(!$type && $desc){
                if(preg_match('/<strong>Type:\s*([^<]+)<\/strong>/i',$desc,$m)) $type = trim($m[1]);
                elseif(preg_match('/Type:\s*([^\r\n<]+)/i',$desc,$m)) $type = trim($m[1]);
            }
            if(!$type) continue;
            if(!isset($byStaff[$staffName])) $byStaff[$staffName] = [];
            if(!isset($byStaff[$staffName][$type])) $byStaff[$staffName][$type] = 0;
            $byStaff[$staffName][$type]++;
            if(!in_array($type,$allTypes)) $allTypes[] = $type;
        }
        // Fallback include general followup types from activity log if none found
        if(empty($byStaff)){
            $activityTable = db_prefix().'lead_activity_log';
            if($this->db->table_exists(str_replace(db_prefix(), '', $activityTable))){
                $this->db->select('la.staffid, COUNT(*) as total, s.firstname, s.lastname');
                $this->db->from($activityTable.' la');
                $this->db->join(db_prefix().'staff s','s.staffid=la.staffid','left');
                if ($from) $this->db->where('DATE(la.date) >=', $from);
                if ($to) $this->db->where('DATE(la.date) <=', $to);
                $this->db->like('la.description','not_lead_activity_added_followup');
                $this->db->group_by('la.staffid');
                $res = $this->db->get()->result_array();
                foreach($res as $r){
                    $staffName = trim(($r['firstname']??'').' '.($r['lastname']??''));
                    if(!$staffName) $staffName = 'Unknown';
                    $byStaff[$staffName] = ['General Followup'=>(int)$r['total']];
                }
                if(!empty($byStaff)) $allTypes[] = 'General Followup';
            }
        }
        return ['by_staff'=>$byStaff,'all_types'=>$allTypes];
    }

    public function get_activity_table(array $filters)
    {
        // If no lead activity log table, return empty structure
        $table = db_prefix() . 'lead_activity_log';
        if(!$this->db->table_exists(str_replace(db_prefix(), '', $table))){
            return ['data'=>[]];
        }

        // Determine if DataTables style request (draw param present)
        $draw = (int)$this->input->post('draw');
        $isDataTables = $draw > 0;
        $start = (int)$this->input->post('start');
        $length = (int)$this->input->post('length');
        if ($length <= 0 || $length > 1000) { $length = 25; }

        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');

        $this->db->select('a.*, s.firstname, s.lastname, l.name as lead_name, l.company as lead_company, ln.description as note_description, ln.followup_type');
        $this->db->from($table . ' a');
        $this->db->join(db_prefix() . 'staff s', 's.staffid = a.staffid', 'left');
        $this->db->join(db_prefix() . 'leads l', 'l.id = a.leadid', 'left');
        $this->db->join(db_prefix() . 'lead_notes ln', 'ln.leadid = a.leadid AND DATE(ln.dateadded) = DATE(a.date) AND ln.addedfrom = a.staffid', 'left');
        // Use a more generous date range for now
        if ($from) $this->db->where('DATE(a.date) >=', $from);
        if ($to) $this->db->where('DATE(a.date) <=', $to);
        if (!empty($filters['staff'])) { 
            error_log('Staff filter applied in get_activity_table: ' . print_r($filters['staff'], true));
            $this->db->where_in('a.staffid', $filters['staff']); 
        } else {
            error_log('No staff filter applied in get_activity_table - showing all staff');
        }

        // Optional search (DataTables only)
        if ($isDataTables) {
            $search = $this->input->post('search');
            if (!empty($search['value'])) {
                $val = $search['value'];
                $this->db->group_start();
                $this->db->like('a.description', $val);
                $this->db->or_like('s.firstname', $val);
                $this->db->or_like('s.lastname', $val);
                $this->db->or_like('l.name', $val);
                $this->db->or_like('l.company', $val);
                $this->db->or_like('ln.description', $val);
                $this->db->or_like('ln.followup_type', $val);
                $this->db->group_end();
            }
        }

        // Count total before pagination (clone not needed if we re-run minimal count)
        $countQuery = clone $this->db;
        $total = $countQuery->count_all_results();

        $this->db->order_by('a.date', 'DESC');
        $this->db->limit($length, $start);
        $rows = $this->db->get()->result_array();

        $data = [];
        foreach($rows as $r){
            $staffName = trim(($r['firstname'] ?? '') . ' ' . ($r['lastname'] ?? '')) ?: 'System';
            
            // Build proper lead display name and link
            $leadId = (int)($r['leadid'] ?? 0);
            $leadName = trim($r['lead_name'] ?? '');
            $leadCompany = trim($r['lead_company'] ?? '');
            
            if ($leadName && $leadCompany) {
                $related = $leadName . ' (' . $leadCompany . ')';
            } elseif ($leadName) {
                $related = $leadName;
            } elseif ($leadCompany) {
                $related = $leadCompany;
            } elseif ($leadId > 0) {
                $related = 'Lead #' . $leadId;
            } else {
                $related = 'Unknown Lead';
            }
            
            $descLower = strtolower($r['description'] ?? '');
            $activityType = 'Lead Activity';
            if(strpos($descLower,'not_activity_new_reminder_created') !== false){ $activityType = 'Reminder Created'; }
            elseif(strpos($descLower,'not_lead_activity_added_followup') !== false){ $activityType = 'Followup Added'; }
            elseif(strpos($descLower,'not_lead_activity_lead_created') !== false){ $activityType = 'Lead Created'; }
            elseif(strpos($descLower,'not_lead_activity_assigned') !== false){ $activityType = 'Lead Assigned'; }

            // Get follow-up information
            $followupType = trim($r['followup_type'] ?? '');
            $noteDescription = trim($r['note_description'] ?? '');
            
            // Build description based on activity type
            $description = 'Activity performed on lead';
            if ($activityType === 'Followup Added' && $noteDescription) {
                $description = $noteDescription;
            } elseif ($activityType === 'Reminder Created') {
                $description = 'Reminder created for follow-up';
            }

            // Build proper view link
            $viewLink = $leadId > 0 ? 'leads/index/' . $leadId : '#';

            // Raw row object (API friendly)
            $rowObj = [
                'id' => (int)$r['id'],
                'date' => e(_dt($r['date'])),
                'staff_name' => e($staffName),
                'activity_type' => $activityType,
                'related' => e($related),
                'description' => e($description),
                'followup_type' => e($followupType),
                'lead_id' => $leadId,
                'view_link' => $viewLink
            ];

            if($isDataTables){
                // Keep legacy array format for DataTables
                $staffImage = staff_profile_image($r['staffid'], ['staff-profile-image-small']);
                $staffColumn = '<div class="d-flex align-items-center">' . $staffImage . '<span class="ml-2">' . e($staffName) . '</span></div>';
                $data[] = [
                    $rowObj['date'],
                    $staffColumn,
                    $activityType,
                    $rowObj['related'],
                    process_text_content_for_display($r['description']),
                    '-',
                    '<button class="btn btn-sm btn-default view-details" data-id="' . $r['id'] . '"><i class="fa fa-eye"></i></button>'
                ];
            } else {
                $data[] = $rowObj;
            }
        }

        if($isDataTables){
            return [
                'draw' => $draw,
                'recordsTotal' => $total,
                'recordsFiltered' => $total,
                'data' => $data
            ];
        }
        return [ 'data' => $data ];
    }

    // Helper methods
    private function count_lead_activities($filters)
    {
        $this->db->from(db_prefix() . 'lead_activity_log');
        if (!empty($filters['from'])) {
            $this->db->where('DATE(date) >=', $filters['from']);
        }
        if (!empty($filters['to'])) {
            $this->db->where('DATE(date) <=', $filters['to']);
        }
        if (!empty($filters['staff'])) {
            $this->db->where_in('staffid', $filters['staff']);
        }
        return $this->db->count_all_results();
    }

    private function count_logins_today($staffFilter = [])
    {
        // Check if general activity log exists for login tracking
        if ($this->db->table_exists(str_replace(db_prefix(), '', db_prefix() . 'activitylog'))) {
            $this->db->from(db_prefix() . 'activitylog');
            $this->db->like('description', 'logged in');
            $this->db->where('DATE(date)', date('Y-m-d'));
            if (!empty($staffFilter)) {
                $this->db->where_in('staffid', $staffFilter);
            }
            return $this->db->count_all_results();
        }
        return 0;
    }

    private function get_top_staff_simple($filters = [])
    {
        $this->db->select('s.staffid, CONCAT(s.firstname, " ", s.lastname) as staff_name, COUNT(a.id) as total');
        $this->db->from(db_prefix() . 'lead_activity_log a');
        $this->db->join(db_prefix() . 'staff s', 's.staffid = a.staffid', 'left');
        $this->db->where('DATE(a.date) >=', date('Y-m-01')); // This month
        if (!empty($filters['staff'])) {
            $this->db->where_in('a.staffid', $filters['staff']);
        }
        $this->db->group_by('s.staffid');
        $this->db->order_by('total', 'DESC');
        $this->db->limit(5);
        
        return $this->db->get()->result_array();
    }

    // Method for email summary
    public function get_email_summary_data()
    {
        return [
            'kpis' => $this->get_kpis([])
        ];
    }

    /**
     * Return reminders grouped for dashboard (basic placeholder implementation)
     * Structure: ['todays'=>[], 'due'=>[], 'upcoming'=>[]]
     */
    public function get_reminders(array $filters)
    {
        $out = [ 'todays'=>[], 'due'=>[], 'upcoming'=>[] ];

        // Attempt to query reminders table if exists (Perfex usually tblreminders)
        $table = db_prefix() . 'reminders';
        if(!$this->db->table_exists(str_replace(db_prefix(), '', $table))){
            return $out; // table not present, return empty structure
        }

        $today = date('Y-m-d');
        $this->db->select('r.id, r.description, r.date, r.rel_type, r.rel_id, r.isnotified');
        $this->db->from($table . ' r');
        
        // Left join with leads table to check junk/lost status
        $this->db->join(db_prefix() . 'leads l', 'l.id = r.rel_id AND r.rel_type = "lead"', 'left');
        
        // Exclude reminders for leads that are marked as junk or lost
        $this->db->where('(r.rel_type != "lead" OR (l.junk = 0 AND l.lost = 0))');
        
        // Optional staff filter if provided
        if(!empty($filters['staff'])){
            // assuming reminders table has staff field (adjust if different)
            $this->db->where_in('r.staff', (array)$filters['staff']);
        }
        $rows = $this->db->get()->result_array();

        foreach($rows as $r){
            $d = substr($r['date'],0,10);
            $isNotified = (int)($r['isnotified'] ?? 0);
            
            if($d === $today){
                // Only include if not marked as read/notified
                if($isNotified == 0){
                    $out['todays'][] = $r;
                }
            } elseif($d < $today){
                // For overdue: only include if not marked as read/notified and date is in the past
                if($isNotified == 0){
                    $out['due'][] = $r;
                }
            } else {
                // For upcoming: include all future reminders regardless of notification status
                $out['upcoming'][] = $r;
            }
        }
        return $out;
    }


}