<?php

defined('BASEPATH') or exit('No direct script access allowed');

/**
 * AI Lead Analysis Hooks
 * 
 * Hooks for automatic AI re-analysis when lead attachments are updated
 * 
 * @package    Application
 * @subpackage Hooks
 * @category   Lead Management
 * @author     Ibraya Group
 * @since      Version 1.0.0
 */

/**
 * Hook: Add AI Setup menu item to admin sidebar menu
 */
hooks()->add_action('admin_init', 'ai_analysis_setup_menu');

/**
 * Hook: After lead attachment is added
 * Automatically trigger AI re-analysis if the lead already has an AI report
 */
hooks()->add_action('lead_attachment_added', 'ai_analysis_check_reanalysis');

/**
 * Hook: After lead is updated
 * Check if AI re-analysis is needed based on significant changes
 */
hooks()->add_action('lead_updated', 'ai_analysis_check_lead_changes');

/**
 * Hook: Daily cleanup
 * Clean up old AI analysis files and reports
 */
hooks()->add_action('daily_cronjob', 'ai_analysis_daily_cleanup');

/**
 * Check if AI re-analysis should be triggered after attachment upload
 * 
 * @param array $data Attachment data
 */
function ai_analysis_check_reanalysis($data)
{
    $CI = &get_instance();
    
    if (!isset($data['lead_id'])) {
        return;
    }
    
    $lead_id = $data['lead_id'];
    
    // Load required models
    $CI->load->model('leads_ai_model');
    $CI->load->model('leads_model');
    
    // Check if auto re-analysis is enabled
    $auto_reanalysis = get_option('ai_analysis_auto_reanalysis');
    if (!$auto_reanalysis) {
        return;
    }
    
    // Check if lead has existing AI reports
    if (!$CI->leads_ai_model->has_reports($lead_id)) {
        return;
    }
    
    // Get the latest report
    $latest_report = $CI->leads_ai_model->get_latest_report($lead_id);
    if (!$latest_report) {
        return;
    }
    
    // Check if enough time has passed since last analysis (prevent spam)
    $min_interval = get_option('ai_analysis_min_reanalysis_interval') ?: 24; // hours
    $last_analysis_time = strtotime($latest_report->created_at);
    $min_time = time() - ($min_interval * 3600);
    
    if ($last_analysis_time > $min_time) {
        log_message('info', "AI re-analysis skipped for lead {$lead_id} - too soon since last analysis");
        return;
    }
    
    // Check if the new attachment contains screenshots (image files)
    if (isset($data['file_name'])) {
        $file_ext = strtolower(pathinfo($data['file_name'], PATHINFO_EXTENSION));
        $image_extensions = ['jpg', 'jpeg', 'png'];
        
        if (in_array($file_ext, $image_extensions)) {
            // Schedule AI re-analysis
            ai_analysis_schedule_reanalysis($lead_id, 'attachment_added');
        }
    }
}

/**
 * Check if significant lead changes require AI re-analysis
 * 
 * @param array $data Lead update data
 */
function ai_analysis_check_lead_changes($data)
{
    if (!isset($data['lead_id'])) {
        return;
    }
    
    $lead_id = $data['lead_id'];
    
    // Check for significant changes that might affect AI analysis
    $significant_fields = [
        'lead_value', // Price change
        'title',      // Property/apartment change
        'description' // Description change
    ];
    
    $has_significant_change = false;
    if (isset($data['changes'])) {
        foreach ($significant_fields as $field) {
            if (isset($data['changes'][$field])) {
                $has_significant_change = true;
                break;
            }
        }
    }
    
    if ($has_significant_change) {
        ai_analysis_schedule_reanalysis($lead_id, 'lead_updated');
    }
}

/**
 * Schedule AI re-analysis for a lead
 * 
 * @param int    $lead_id Lead ID
 * @param string $reason  Reason for re-analysis
 */
function ai_analysis_schedule_reanalysis($lead_id, $reason = 'manual')
{
    $CI = &get_instance();
    
    // Create a scheduled task or trigger immediate analysis based on settings
    $immediate_reanalysis = get_option('ai_analysis_immediate_reanalysis');
    
    if ($immediate_reanalysis) {
        // Trigger immediate background analysis
        ai_analysis_background_process($lead_id, $reason);
    } else {
        // Add to queue for later processing
        ai_analysis_queue_analysis($lead_id, $reason);
    }
    
    log_message('info', "AI re-analysis scheduled for lead {$lead_id}, reason: {$reason}");
}

/**
 * Process AI analysis in background
 * 
 * @param int    $lead_id Lead ID
 * @param string $reason  Reason for analysis
 */
function ai_analysis_background_process($lead_id, $reason)
{
    $CI = &get_instance();
    
    // This would typically be handled by a queue system or background job
    // For now, we'll create a simple flag that can be processed by cron
    
    $queue_data = [
        'lead_id' => $lead_id,
        'reason' => $reason,
        'scheduled_at' => date('Y-m-d H:i:s'),
        'status' => 'pending'
    ];
    
    // Store in a simple queue table or file
    ai_analysis_add_to_queue($queue_data);
}

/**
 * Add analysis job to queue
 * 
 * @param array $queue_data Queue data
 */
function ai_analysis_add_to_queue($queue_data)
{
    $CI = &get_instance();
    
    // Create queue table if it doesn't exist
    $table_name = db_prefix() . 'ai_analysis_queue';
    
    if (!$CI->db->table_exists($table_name)) {
        $CI->db->query("
            CREATE TABLE {$table_name} (
                id int(11) NOT NULL AUTO_INCREMENT,
                lead_id int(11) NOT NULL,
                reason varchar(100) NOT NULL,
                scheduled_at datetime NOT NULL,
                processed_at datetime NULL,
                status enum('pending','processing','completed','failed') DEFAULT 'pending',
                error_message text NULL,
                created_at timestamp DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (id),
                KEY lead_id (lead_id),
                KEY status (status)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
        ");
    }
    
    $CI->db->insert($table_name, $queue_data);
}

/**
 * Queue analysis for later processing
 * 
 * @param int    $lead_id Lead ID
 * @param string $reason  Reason for analysis
 */
function ai_analysis_queue_analysis($lead_id, $reason)
{
    ai_analysis_add_to_queue([
        'lead_id' => $lead_id,
        'reason' => $reason,
        'scheduled_at' => date('Y-m-d H:i:s'),
        'status' => 'pending'
    ]);
}

/**
 * Process queued AI analysis jobs
 * Called by cron job
 */
function ai_analysis_process_queue()
{
    $CI = &get_instance();
    
    $queue_table = db_prefix() . 'ai_analysis_queue';
    
    if (!$CI->db->table_exists($queue_table)) {
        return;
    }
    
    // Get pending jobs
    $pending_jobs = $CI->db->where('status', 'pending')
                          ->where('scheduled_at <=', date('Y-m-d H:i:s'))
                          ->limit(5) // Process 5 at a time to avoid overload
                          ->get($queue_table)
                          ->result();
    
    foreach ($pending_jobs as $job) {
        // Mark as processing
        $CI->db->where('id', $job->id)
               ->update($queue_table, [
                   'status' => 'processing',
                   'processed_at' => date('Y-m-d H:i:s')
               ]);
        
        try {
            // Process the analysis
            ai_analysis_process_job($job);
            
            // Mark as completed
            $CI->db->where('id', $job->id)
                   ->update($queue_table, ['status' => 'completed']);
                   
        } catch (Exception $e) {
            // Mark as failed
            $CI->db->where('id', $job->id)
                   ->update($queue_table, [
                       'status' => 'failed',
                       'error_message' => $e->getMessage()
                   ]);
                   
            log_message('error', "AI analysis queue job failed: " . $e->getMessage());
        }
    }
}

/**
 * Process individual analysis job
 * 
 * @param object $job Job data
 */
function ai_analysis_process_job($job)
{
    $CI = &get_instance();
    
    // Load required models
    $CI->load->model('leads_model');
    $CI->load->model('leads_ai_model');
    
    // Get lead data
    $lead = $CI->leads_model->get($job->lead_id);
    if (!$lead) {
        throw new Exception("Lead {$job->lead_id} not found");
    }
    
    // Get lead attachments that are images
    $attachments = $CI->leads_model->get_lead_attachments($job->lead_id);
    $image_attachments = [];
    
    foreach ($attachments as $attachment) {
        $file_ext = strtolower(pathinfo($attachment['file_name'], PATHINFO_EXTENSION));
        if (in_array($file_ext, ['jpg', 'jpeg', 'png'])) {
            $image_attachments[] = $attachment['file_name'];
        }
    }
    
    if (empty($image_attachments)) {
        throw new Exception("No image attachments found for lead {$job->lead_id}");
    }
    
    // Call AI analysis - this would reuse the existing AI analysis logic
    // For now, we'll just log that the job would be processed
    log_message('info', "Processing AI analysis job for lead {$job->lead_id}, found " . count($image_attachments) . " image attachments");
    
    // In a real implementation, you would call the OpenAI API here
    // using the same logic as in the Leads controller
}

/**
 * Daily cleanup of AI analysis data
 */
function ai_analysis_daily_cleanup()
{
    $CI = &get_instance();
    
    $CI->load->model('leads_ai_model');
    
    // Clean up old files
    $cleanup_days = get_option('ai_analysis_cleanup_days') ?: 30;
    $cleaned_files = $CI->leads_ai_model->cleanup_old_files($cleanup_days);
    
    if ($cleaned_files > 0) {
        log_message('info', "AI analysis daily cleanup: removed {$cleaned_files} old files");
    }
    
    // Clean up old queue entries
    $queue_table = db_prefix() . 'ai_analysis_queue';
    if ($CI->db->table_exists($queue_table)) {
        $cutoff_date = date('Y-m-d H:i:s', strtotime("-7 days"));
        $CI->db->where('created_at <', $cutoff_date)
               ->where_in('status', ['completed', 'failed'])
               ->delete($queue_table);
               
        $deleted_rows = $CI->db->affected_rows();
        if ($deleted_rows > 0) {
            log_message('info', "AI analysis daily cleanup: removed {$deleted_rows} old queue entries");
        }
    }
}

/**
 * Cron job to process AI analysis queue
 * Add this to your system cron or call from existing cron jobs
 */
function ai_analysis_cron_process()
{
    ai_analysis_process_queue();
}

// Register cron job hook
hooks()->add_action('app_init', function() {
    // Process queue every 15 minutes if running via cron
    if (defined('CRON_JOB') && constant('CRON_JOB')) {
        ai_analysis_process_queue();
    }
});

/**
 * Add AI Setup menu item to the sidebar menu
 */
function ai_analysis_setup_menu()
{
    $CI = &get_instance();
    
    // Only show to admins
    if (!is_admin()) {
        return;
    }
    
    // Add AI Analysis main menu item to sidebar
    $CI->app_menu->add_sidebar_menu_item('ai_analysis', [
        'name'     => 'AI Analysis',
        'href'     => admin_url('ai_setup'),
        'icon'     => 'fa fa-brain',
        'position' => 25,
    ]);
    
    // Add sub-menu items to sidebar
    $CI->app_menu->add_sidebar_children_item('ai_analysis', [
        'slug' => 'ai_setup_dashboard',
        'name' => 'Dashboard',
        'href' => admin_url('ai_setup'),
        'icon' => 'fa fa-tachometer-alt'
    ]);
    
    $CI->app_menu->add_sidebar_children_item('ai_analysis', [
        'slug' => 'ai_setup_database',
        'name' => 'Fix Database',
        'href' => admin_url('ai_setup/fix_database'),
        'icon' => 'fa fa-database'
    ]);
    
    $CI->app_menu->add_sidebar_children_item('ai_analysis', [
        'slug' => 'ai_setup_openai', 
        'name' => 'Configure OpenAI',
        'href' => admin_url('ai_setup/setup_openai'),
        'icon' => 'fa fa-cog'
    ]);
    
    $CI->app_menu->add_sidebar_children_item('ai_analysis', [
        'slug' => 'ai_reports',
        'name' => 'AI Reports',
        'href' => admin_url('ai_reports'),
        'icon' => 'fa fa-file-text'
    ]);

    $CI->app_menu->add_sidebar_children_item('ai_analysis', [
        'slug' => 'ai_api_logs',
        'name' => 'API Logs', 
        'href' => admin_url('ai_setup/api_logs'),
        'icon' => 'fa fa-list-alt'
    ]);
    
    $CI->app_menu->add_sidebar_children_item('ai_analysis', [
        'slug' => 'ai_fine_tuning',
        'name' => 'Fine-Tuning', 
        'href' => admin_url('ai_setup/fine_tuning'),
        'icon' => 'fa fa-cogs'
    ]);
    
    $CI->app_menu->add_sidebar_children_item('ai_analysis', [
        'slug' => 'openai_settings',
        'name' => 'Settings', 
        'href' => admin_url('openai_settings'),
        'icon' => 'fa fa-key'
    ]);
}
