<?php

defined('BASEPATH') or exit('No direct script access allowed');

/**
 * AI Analysis Controller
 * 
 * Controller for AI analysis operations across the CRM
 * Demonstrates usage of the AI Analysis module
 * 
 * @package    Application
 * @subpackage Controllers
 * @category   AI Analysis
 * @author     Ibraya Group
 * @since      Version 1.0.0
 */
class AI_Analysis_Controller extends AdminController
{
    /**
     * Constructor
     */
    public function __construct()
    {
        parent::__construct();
        
        // Load required libraries and services
        $this->load->library('AI_Analysis');
        $this->load->library('AI_Analysis_Service', null, 'ai_analysis_service');
        
        // Check if user has permission
        if (!has_permission('ai_analysis', '', 'view')) {
            access_denied('AI Analysis');
        }
    }
    
    /**
     * AI Analysis dashboard
     */
    public function index()
    {
        // Get statistics for dashboard
        $data['statistics'] = $this->ai_analysis_service->get_statistics();
        $data['recent_logs'] = $this->ai_analysis_service->get_recent_logs(10);
        
        $data['title'] = 'AI Analysis Dashboard';
        $this->load->view('admin/ai_analysis/dashboard', $data);
    }
    
    /**
     * Analyze a specific lead
     */
    public function analyze_lead($lead_id = null)
    {
        if (!$lead_id) {
            set_alert('danger', 'Lead ID is required');
            redirect(admin_url('leads'));
        }
        
        if ($this->input->post()) {
            // Handle form submission for custom analysis
            $custom_prompt = $this->input->post('custom_prompt');
            $options = [
                'model' => $this->input->post('model') ?: 'gpt-4o',
                'temperature' => (float)($this->input->post('temperature') ?: 0.3),
                'max_tokens' => (int)($this->input->post('max_tokens') ?: 1000)
            ];
            
            if (!empty($custom_prompt)) {
                // Custom analysis
                $this->load->model('leads_model');
                $lead = $this->leads_model->get($lead_id);
                $notes = $this->leads_model->get_notes($lead_id);
                $activity_log = $this->leads_model->get_lead_activity_log($lead_id);
                
                $lead_data = [
                    'name' => $lead->name ?? '',
                    'email' => $lead->email ?? '',
                    'phonenumber' => $lead->phonenumber ?? '',
                    'company' => $lead->company ?? '',
                    'description' => $lead->description ?? '',
                    'notes' => $notes,
                    'activity_log' => $activity_log
                ];
                
                $result = $this->ai_analysis_service->custom_analysis($custom_prompt, $lead_data, [], $options);
            } else {
                // Standard lead analysis
                $result = $this->ai_analysis_service->quick_lead_analysis($lead_id, $options);
            }
            
            if ($result['success']) {
                set_alert('success', 'Lead analysis completed successfully');
            } else {
                set_alert('danger', 'Analysis failed: ' . $result['error']);
            }
            
            // Return result as JSON for AJAX requests
            if ($this->input->is_ajax_request()) {
                echo json_encode($result);
                return;
            }
        }
        
        // Load lead data for display
        $this->load->model(['leads_model', 'leads_ai_model']);
        $data['lead'] = $this->leads_model->get($lead_id);
        $data['recent_reports'] = $this->leads_ai_model->get_reports($lead_id);
        
        $data['title'] = 'Analyze Lead: ' . $data['lead']->name;
        $this->load->view('admin/ai_analysis/analyze_lead', $data);
    }
    
    /**
     * Analyze a customer
     */
    public function analyze_customer($customer_id = null)
    {
        if (!$customer_id) {
            set_alert('danger', 'Customer ID is required');
            redirect(admin_url('clients'));
        }
        
        if ($this->input->post()) {
            $options = [
                'model' => $this->input->post('model') ?: 'gpt-4o',
                'temperature' => (float)($this->input->post('temperature') ?: 0.3),
                'max_tokens' => (int)($this->input->post('max_tokens') ?: 1000)
            ];
            
            $result = $this->ai_analysis_service->quick_customer_analysis($customer_id, $options);
            
            if ($this->input->is_ajax_request()) {
                echo json_encode($result);
                return;
            }
            
            if ($result['success']) {
                set_alert('success', 'Customer analysis completed successfully');
            } else {
                set_alert('danger', 'Analysis failed: ' . $result['error']);
            }
        }
        
        // Load customer data
        $this->load->model('clients_model');
        $data['customer'] = $this->clients_model->get($customer_id);
        
        $data['title'] = 'Analyze Customer: ' . $data['customer']->company;
        $this->load->view('admin/ai_analysis/analyze_customer', $data);
    }
    
    /**
     * Analyze a project
     */
    public function analyze_project($project_id = null)
    {
        if (!$project_id) {
            set_alert('danger', 'Project ID is required');
            redirect(admin_url('projects'));
        }
        
        if ($this->input->post()) {
            $options = [
                'model' => $this->input->post('model') ?: 'gpt-4o',
                'temperature' => (float)($this->input->post('temperature') ?: 0.3),
                'max_tokens' => (int)($this->input->post('max_tokens') ?: 1000)
            ];
            
            $result = $this->ai_analysis_service->quick_project_analysis($project_id, $options);
            
            if ($this->input->is_ajax_request()) {
                echo json_encode($result);
                return;
            }
            
            if ($result['success']) {
                set_alert('success', 'Project analysis completed successfully');
            } else {
                set_alert('danger', 'Analysis failed: ' . $result['error']);
            }
        }
        
        // Load project data
        $this->load->model('projects_model');
        $data['project'] = $this->projects_model->get($project_id);
        
        $data['title'] = 'Analyze Project: ' . $data['project']->name;
        $this->load->view('admin/ai_analysis/analyze_project', $data);
    }
    
    /**
     * Custom analysis interface
     */
    public function custom()
    {
        if ($this->input->post()) {
            $prompt = $this->input->post('prompt');
            $data_input = $this->input->post('data');
            $options = [
                'model' => $this->input->post('model') ?: 'gpt-4o',
                'temperature' => (float)($this->input->post('temperature') ?: 0.3),
                'max_tokens' => (int)($this->input->post('max_tokens') ?: 1000)
            ];
            
            // Parse data input as JSON if provided
            $data = [];
            if (!empty($data_input)) {
                $data = json_decode($data_input, true);
                if (json_last_error() !== JSON_ERROR_NONE) {
                    $data = ['content' => $data_input]; // Fallback to text
                }
            }
            
            $result = $this->ai_analysis_service->custom_analysis($prompt, $data, [], $options);
            
            if ($this->input->is_ajax_request()) {
                echo json_encode($result);
                return;
            }
            
            $data['result'] = $result;
        }
        
        $data['title'] = 'Custom AI Analysis';
        $this->load->view('admin/ai_analysis/custom', $data);
    }
    
    /**
     * Batch analysis interface
     */
    public function batch()
    {
        if ($this->input->post()) {
            $items_input = $this->input->post('items');
            $prompt = $this->input->post('prompt');
            $options = [
                'model' => $this->input->post('model') ?: 'gpt-4o',
                'temperature' => (float)($this->input->post('temperature') ?: 0.3),
                'max_tokens' => (int)($this->input->post('max_tokens') ?: 1000)
            ];
            
            // Parse items as JSON
            $items = json_decode($items_input, true);
            if (json_last_error() !== JSON_ERROR_NONE) {
                set_alert('danger', 'Invalid JSON format for items');
                redirect(admin_url('ai_analysis/batch'));
            }
            
            $result = $this->ai_analysis_service->batch_analysis($items, $prompt, $options);
            
            if ($this->input->is_ajax_request()) {
                echo json_encode($result);
                return;
            }
            
            $data['result'] = $result;
        }
        
        $data['title'] = 'Batch AI Analysis';
        $this->load->view('admin/ai_analysis/batch', $data);
    }
    
    /**
     * API logs viewer
     */
    public function logs()
    {
        $limit = $this->input->get('limit') ?: 50;
        $filters = [
            'success_only' => $this->input->get('success_only'),
            'errors_only' => $this->input->get('errors_only')
        ];
        
        $data['logs'] = $this->ai_analysis_service->get_recent_logs($limit, $filters);
        $data['statistics'] = $this->ai_analysis_service->get_statistics();
        
        $data['title'] = 'AI Analysis Logs';
        $this->load->view('admin/ai_analysis/logs', $data);
    }
    
    /**
     * Test API connection
     */
    public function test_connection()
    {
        $result = $this->ai_analysis_service->test_api_connection();
        
        if ($this->input->is_ajax_request()) {
            echo json_encode($result);
            return;
        }
        
        if ($result['success']) {
            set_alert('success', 'API connection test successful');
        } else {
            set_alert('danger', 'API connection test failed: ' . $result['message']);
        }
        
        redirect(admin_url('ai_analysis'));
    }
    
    /**
     * Settings page
     */
    public function settings()
    {
        if ($this->input->post()) {
            $settings = [
                'openai_api_key' => $this->input->post('openai_api_key'),
                'ai_model' => $this->input->post('ai_model'),
                'ai_temperature' => $this->input->post('ai_temperature'),
                'ai_max_tokens' => $this->input->post('ai_max_tokens'),
                'ai_lead_analysis_prompt' => $this->input->post('ai_lead_analysis_prompt'),
                'ai_customer_analysis_prompt' => $this->input->post('ai_customer_analysis_prompt'),
                'ai_project_analysis_prompt' => $this->input->post('ai_project_analysis_prompt')
            ];
            
            foreach ($settings as $key => $value) {
                update_option($key, $value);
            }
            
            set_alert('success', 'AI Analysis settings updated successfully');
            redirect(admin_url('ai_analysis/settings'));
        }
        
        $data['title'] = 'AI Analysis Settings';
        $this->load->view('admin/ai_analysis/settings', $data);
    }
    
    /**
     * Clean up old logs
     */
    public function cleanup_logs()
    {
        $days = $this->input->post('days') ?: 30;
        $count = $this->ai_analysis_service->cleanup_logs($days);
        
        if ($this->input->is_ajax_request()) {
            echo json_encode(['success' => true, 'cleaned' => $count]);
            return;
        }
        
        set_alert('success', "Cleaned up {$count} old log entries");
        redirect(admin_url('ai_analysis/logs'));
    }
    
    /**
     * AJAX endpoint for file validation
     */
    public function validate_files()
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }
        
        $files = $this->input->post('files') ?: [];
        $result = $this->ai_analysis_service->validate_files($files);
        
        echo json_encode($result);
    }
}
