<?php
defined('BASEPATH') or exit('No direct script access allowed');

class Darkglass_theme extends AdminController
{
    public function __construct()
    {
        parent::__construct();
        
        // Check permissions
        if (!has_permission('settings', '', 'view')) {
            access_denied('settings');
        }
    }

    /**
     * Settings page
     */
    public function settings()
    {
        if ($this->input->post()) {
            // Handle form submission
            $this->save_settings();
        }

        $data['title'] = 'Dark Glass Theme Settings';
        $data['settings'] = $this->get_settings();
        
        $this->load->view('darkglass_theme/settings', $data);
    }

    /**
     * Save theme settings
     */
    private function save_settings()
    {
        // Theme activation
        update_option('darkglass_theme_activated', $this->input->post('theme_activated') ? 1 : 0);
        
        // Colors
        update_option('darkglass_theme_primary_color', $this->input->post('primary_color'));
        update_option('darkglass_theme_secondary_color', $this->input->post('secondary_color'));
        update_option('darkglass_theme_accent_color', $this->input->post('accent_color'));
        
        // Effects
        update_option('darkglass_theme_blur_intensity', $this->input->post('blur_intensity'));
        update_option('darkglass_theme_animation_speed', $this->input->post('animation_speed'));
        update_option('darkglass_theme_orbs_enabled', $this->input->post('orbs_enabled') ? 1 : 0);
        update_option('darkglass_theme_particles_enabled', $this->input->post('particles_enabled') ? 1 : 0);
        
        // Glass effects
        update_option('darkglass_theme_glass_opacity', $this->input->post('glass_opacity'));
        update_option('darkglass_theme_border_opacity', $this->input->post('border_opacity'));
        
        set_alert('success', 'Theme settings updated successfully');
        redirect(admin_url('darkglass_theme/settings'));
    }

    /**
     * Get current settings
     */
    private function get_settings()
    {
        return [
            'theme_activated' => get_option('darkglass_theme_activated'),
            'primary_color' => get_option('darkglass_theme_primary_color') ?: '#06b6d4',
            'secondary_color' => get_option('darkglass_theme_secondary_color') ?: '#14b8a6',
            'accent_color' => get_option('darkglass_theme_accent_color') ?: '#fb923c',
            'blur_intensity' => get_option('darkglass_theme_blur_intensity') ?: '20',
            'animation_speed' => get_option('darkglass_theme_animation_speed') ?: 'normal',
            'orbs_enabled' => get_option('darkglass_theme_orbs_enabled'),
            'particles_enabled' => get_option('darkglass_theme_particles_enabled'),
            'glass_opacity' => get_option('darkglass_theme_glass_opacity') ?: '0.04',
            'border_opacity' => get_option('darkglass_theme_border_opacity') ?: '0.08',
        ];
    }

    /**
     * Export theme settings
     */
    public function export_settings()
    {
        $settings = $this->get_settings();
        
        header('Content-Type: application/json');
        header('Content-Disposition: attachment; filename="darkglass-theme-settings.json"');
        
        echo json_encode($settings, JSON_PRETTY_PRINT);
    }

    /**
     * Import theme settings
     */
    public function import_settings()
    {
        if (!empty($_FILES['settings_file']['tmp_name'])) {
            $json = file_get_contents($_FILES['settings_file']['tmp_name']);
            $settings = json_decode($json, true);
            
            if ($settings) {
                foreach ($settings as $key => $value) {
                    update_option('darkglass_theme_' . $key, $value);
                }
                
                set_alert('success', 'Settings imported successfully');
            } else {
                set_alert('danger', 'Invalid settings file');
            }
        }
        
        redirect(admin_url('darkglass_theme/settings'));
    }

    /**
     * Reset to default settings
     */
    public function reset_settings()
    {
        // Reset to defaults
        update_option('darkglass_theme_primary_color', '#06b6d4');
        update_option('darkglass_theme_secondary_color', '#14b8a6');
        update_option('darkglass_theme_accent_color', '#fb923c');
        update_option('darkglass_theme_blur_intensity', '20');
        update_option('darkglass_theme_animation_speed', 'normal');
        update_option('darkglass_theme_orbs_enabled', 1);
        update_option('darkglass_theme_particles_enabled', 0);
        update_option('darkglass_theme_glass_opacity', '0.04');
        update_option('darkglass_theme_border_opacity', '0.08');
        
        set_alert('success', 'Settings reset to defaults');
        redirect(admin_url('darkglass_theme/settings'));
    }
}
