<?php

defined('BASEPATH') or exit('No direct script access allowed');

class Reporting_dashboard_model extends App_Model
{
    public function __construct()
    {
        parent::__construct();
    }

    public function get_kpis(array $filters)
    {
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        
        $today = date('Y-m-d');
        $weekStart = date('Y-m-d', strtotime('monday this week'));
        $monthStart = date('Y-m-01');
        
        return [
            'today' => $this->count_lead_activities(['from' => $today, 'to' => $today, 'staff' => $filters['staff'] ?? []]),
            'week' => $this->count_lead_activities(['from' => $weekStart, 'to' => $today, 'staff' => $filters['staff'] ?? []]),
            'month' => $this->count_lead_activities(['from' => $monthStart, 'to' => $today, 'staff' => $filters['staff'] ?? []]),
            'logins_today' => $this->count_logins_today($filters['staff'] ?? []),
            'top_staff' => $this->get_top_staff_simple($filters)
        ];
    }

    public function get_staff_activity_chart(array $filters)
    {
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        
        $this->db->select('s.staffid, CONCAT(s.firstname, " ", s.lastname) as staff_name, COUNT(a.id) as total');
        $this->db->from(db_prefix() . 'lead_activity_log a');
        $this->db->join(db_prefix() . 'staff s', 's.staffid = a.staffid', 'left');
        
        if ($from) $this->db->where('DATE(a.date) >=', $from);
        if ($to) $this->db->where('DATE(a.date) <=', $to);
        if (!empty($filters['staff'])) {
            $this->db->where_in('a.staffid', $filters['staff']);
        }
        
        $this->db->group_by('s.staffid');
        $this->db->order_by('total', 'DESC');
        $this->db->limit(10);
        
        $result = $this->db->get()->result_array();
        
        return [
            'labels' => array_column($result, 'staff_name'),
            'data' => array_map('intval', array_column($result, 'total'))
        ];
    }

    public function get_activity_over_time_chart(array $filters)
    {
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        
        $this->db->select('DATE(a.date) as date, COUNT(a.id) as total');
        $this->db->from(db_prefix() . 'lead_activity_log a');
        
        if ($from) $this->db->where('DATE(a.date) >=', $from);
        if ($to) $this->db->where('DATE(a.date) <=', $to);
        if (!empty($filters['staff'])) {
            $this->db->where_in('a.staffid', $filters['staff']);
        }
        
        $this->db->group_by('DATE(a.date)');
        $this->db->order_by('DATE(a.date)', 'ASC');
        
        $result = $this->db->get()->result_array();
        
        return [
            'labels' => array_column($result, 'date'),
            'data' => array_map('intval', array_column($result, 'total'))
        ];
    }

    public function get_activity_type_pie(array $filters)
    {
        try {
            // Guard: table may not exist yet
            $table = db_prefix() . 'lead_activity_log';
            if(!$this->db->table_exists(str_replace(db_prefix(), '', $table))){
                return [ 'labels'=>[], 'data'=>[] ];
            }
            $from = $filters['from'] ?? date('Y-m-01');
            $to = $filters['to'] ?? date('Y-m-d');

            // Classification aligned with PDF report logic (exact notification keywords)
            // so that PDF distribution matches dashboard chart.
            $this->db->select("CASE\n                WHEN LOWER(a.description) LIKE '%not_activity_new_reminder_created%' THEN 'Reminder Created'\n                WHEN LOWER(a.description) LIKE '%not_lead_activity_added_followup%' THEN 'Followup Added'\n                WHEN LOWER(a.description) LIKE '%not_lead_activity_lead_created%' THEN 'Lead Created'\n                WHEN LOWER(a.description) LIKE '%not_lead_activity_assigned%' THEN 'Lead Assigned'\n                ELSE 'Other'\n            END as activity_type, COUNT(a.id) as total", false);
            $this->db->from($table . ' a');
            if ($from) $this->db->where('DATE(a.date) >=', $from);
            if ($to) $this->db->where('DATE(a.date) <=', $to);
            if (!empty($filters['staff'])) {
                $this->db->where_in('a.staffid', $filters['staff']);
            }
            $this->db->group_by('activity_type');
            $this->db->order_by('total', 'DESC');
            $result = $this->db->get()->result_array();

            return [
                'labels' => array_column($result, 'activity_type'),
                'data' => array_map('intval', array_column($result, 'total'))
            ];
        } catch (Exception $e) {
            log_message('error', 'Reporting Dashboard get_activity_type_pie error: ' . $e->getMessage());
            return [ 'labels'=>[], 'data'=>[] ];
        }
    }

    /**
     * Build stacked follow-up types by staff.
     * Tries to infer follow-up type from activity description (heuristic) unless a dedicated field/table exists.
     * Returns structure with labels (staff) and datasets (array of series) matching createFollowupTypesChart expectations.
     */
    public function get_followup_types_chart(array $filters)
    {
        // Prefer structured follow-up types from lead_notes (more accurate than heuristic)
        $notesTable = db_prefix() . 'lead_notes';
        if(!$this->db->table_exists(str_replace(db_prefix(), '', $notesTable))){
            return [ 'labels'=>[], 'datasets'=>[] ];
        }
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        $this->db->select('ln.followup_type, ln.description, ln.dateadded, s.staffid, s.firstname, s.lastname');
        $this->db->from($notesTable . ' ln');
        $this->db->join(db_prefix().'staff s','s.staffid = ln.addedfrom','left');
        if($from) $this->db->where('DATE(ln.dateadded) >=',$from);
        if($to) $this->db->where('DATE(ln.dateadded) <=',$to);
        if (!empty($filters['staff'])) $this->db->where_in('ln.addedfrom', $filters['staff']);
        $rows = $this->db->get()->result_array();
        $staffNames=[]; $types=[]; $matrix=[]; $palette=[];
        $baseColors=['#4e73df','#1cc88a','#36b9cc','#f6c23e','#e74a3b','#858796','#2c9faf','#6f42c1','#17a673','#fd7e14'];
        $colorIndex=0;
        foreach($rows as $r){
            $sid=(int)($r['staffid'] ?? 0); $name=trim(($r['firstname']??'').' '.($r['lastname']??'')) ?: 'Staff #'.$sid; $staffNames[$sid]=$name;
            $type=$r['followup_type'];
            $desc=$r['description']??'';
            if(!$type && $desc){
                if(preg_match('/<strong>Type:\s*([^<]+)<\/strong>/i',$desc,$m)) $type=trim($m[1]);
                elseif(preg_match('/Type:\s*([^\r\n<]+)/i',$desc,$m)) $type=trim($m[1]);
            }
            if(!$type) $type='Other';
            if(!in_array($type,$types)) { $types[]=$type; if(!isset($palette[$type])) $palette[$type]=$baseColors[$colorIndex++ % count($baseColors)]; }
            if(!isset($matrix[$sid])) $matrix[$sid]=[];
            if(!isset($matrix[$sid][$type])) $matrix[$sid][$type]=0;
            $matrix[$sid][$type]++;
        }
        asort($staffNames); $labels=array_values($staffNames); $staffOrder=array_keys($staffNames);
        // Ensure consistent type order
        sort($types);
        $datasets=[];
        foreach($types as $t){
            $datasets[]=[
                'label'=>$t,
                'backgroundColor'=>$palette[$t] ?? '#666',
                'data'=>array_map(function($sid) use ($matrix,$t){ return isset($matrix[$sid][$t])?(int)$matrix[$sid][$t]:0; }, $staffOrder)
            ];
        }
        return ['labels'=>$labels,'datasets'=>$datasets];
    }

    /**
     * Lightweight activity log (object rows) for reuse (PDF, API) without DataTables decoration.
     */
    public function get_activity_log(array $filters, $limit = 50)
    {
        $table = db_prefix() . 'lead_activity_log';
        if(!$this->db->table_exists(str_replace(db_prefix(), '', $table))){
            return [];
        }
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        $this->db->select('a.*, s.firstname, s.lastname, l.name as lead_name, l.company as lead_company, ln.description as note_description, ln.followup_type');
        $this->db->from($table . ' a');
        $this->db->join(db_prefix() . 'staff s', 's.staffid = a.staffid', 'left');
        $this->db->join(db_prefix() . 'leads l', 'l.id = a.leadid', 'left');
        $this->db->join(db_prefix() . 'lead_notes ln', 'ln.leadid = a.leadid AND DATE(ln.dateadded) = DATE(a.date) AND ln.addedfrom = a.staffid', 'left');
        if ($from) $this->db->where('DATE(a.date) >=', $from);
        if ($to) $this->db->where('DATE(a.date) <=', $to);
        if (!empty($filters['staff'])) { $this->db->where_in('a.staffid', $filters['staff']); }
        $this->db->order_by('a.date', 'DESC');
        if($limit) $this->db->limit((int)$limit);
        $rows = $this->db->get()->result_array();
        $out = [];
        foreach($rows as $r){
            $staffName = trim(($r['firstname'] ?? '') . ' ' . ($r['lastname'] ?? '')) ?: 'System';
            
            // Build proper lead display name
            $leadId = (int)($r['leadid'] ?? 0);
            $leadName = trim($r['lead_name'] ?? '');
            $leadCompany = trim($r['lead_company'] ?? '');
            
            if ($leadName && $leadCompany) {
                $related = $leadName . ' (' . $leadCompany . ')';
            } elseif ($leadName) {
                $related = $leadName;
            } elseif ($leadCompany) {
                $related = $leadCompany;
            } elseif ($leadId > 0) {
                $related = 'Lead #' . $leadId;
            } else {
                $related = 'Unknown Lead';
            }
            
            $descLower = strtolower($r['description'] ?? '');
            $activityType = 'Activity';
            if(strpos($descLower,'not_activity_new_reminder_created') !== false){ $activityType = 'Reminder Created'; }
            elseif(strpos($descLower,'not_lead_activity_added_followup') !== false){ $activityType = 'Followup Added'; }
            elseif(strpos($descLower,'not_lead_activity_lead_created') !== false){ $activityType = 'Lead Created'; }
            elseif(strpos($descLower,'not_lead_activity_assigned') !== false){ $activityType = 'Lead Assigned'; }
            
            // Get follow-up information
            $followupType = trim($r['followup_type'] ?? '');
            $noteDescription = trim($r['note_description'] ?? '');
            
            // Build description based on activity type
            $description = 'Activity performed on lead';
            if ($activityType === 'Followup Added' && $noteDescription) {
                $description = $noteDescription;
            } elseif ($activityType === 'Reminder Created') {
                $description = 'Reminder created for follow-up';
            }
            
            $out[] = [
                'date' => $r['date'],
                'staff_name' => $staffName,
                'activity_type' => $activityType,
                'related' => $related,
                'description' => $description,
                'followup_type' => $followupType
            ];
        }
        return $out;
    }

    /**
     * Follow-up types by staff replication (used by PDF) – extracts type from lead_notes.followup_type or description patterns.
     */
    public function get_followup_types_by_staff(array $filters)
    {
        $notesTable = db_prefix() . 'lead_notes';
        if(!$this->db->table_exists(str_replace(db_prefix(), '', $notesTable))){
            return ['by_staff'=>[], 'all_types'=>[]];
        }
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        $this->db->select('ln.followup_type, ln.description, ln.dateadded, s.firstname, s.lastname');
        $this->db->from($notesTable . ' ln');
        $this->db->join(db_prefix() . 'staff s', 's.staffid = ln.addedfrom', 'left');
        if ($from) $this->db->where('DATE(ln.dateadded) >=', $from);
        if ($to) $this->db->where('DATE(ln.dateadded) <=', $to);
        $this->db->order_by('ln.dateadded','DESC');
        $rows = $this->db->get()->result_array();
        $byStaff = [];
        $allTypes = [];
        foreach($rows as $r){
            $staffName = trim(($r['firstname']??'').' '.($r['lastname']??''));
            if(!$staffName) $staffName = 'Unknown';
            $type = $r['followup_type'];
            $desc = $r['description'] ?? '';
            if(!$type && $desc){
                if(preg_match('/<strong>Type:\s*([^<]+)<\/strong>/i',$desc,$m)) $type = trim($m[1]);
                elseif(preg_match('/Type:\s*([^\r\n<]+)/i',$desc,$m)) $type = trim($m[1]);
            }
            if(!$type) continue;
            if(!isset($byStaff[$staffName])) $byStaff[$staffName] = [];
            if(!isset($byStaff[$staffName][$type])) $byStaff[$staffName][$type] = 0;
            $byStaff[$staffName][$type]++;
            if(!in_array($type,$allTypes)) $allTypes[] = $type;
        }
        // Fallback include general followup types from activity log if none found
        if(empty($byStaff)){
            $activityTable = db_prefix().'lead_activity_log';
            if($this->db->table_exists(str_replace(db_prefix(), '', $activityTable))){
                $this->db->select('la.staffid, COUNT(*) as total, s.firstname, s.lastname');
                $this->db->from($activityTable.' la');
                $this->db->join(db_prefix().'staff s','s.staffid=la.staffid','left');
                if ($from) $this->db->where('DATE(la.date) >=', $from);
                if ($to) $this->db->where('DATE(la.date) <=', $to);
                $this->db->like('la.description','not_lead_activity_added_followup');
                $this->db->group_by('la.staffid');
                $res = $this->db->get()->result_array();
                foreach($res as $r){
                    $staffName = trim(($r['firstname']??'').' '.($r['lastname']??''));
                    if(!$staffName) $staffName = 'Unknown';
                    $byStaff[$staffName] = ['General Followup'=>(int)$r['total']];
                }
                if(!empty($byStaff)) $allTypes[] = 'General Followup';
            }
        }
        return ['by_staff'=>$byStaff,'all_types'=>$allTypes];
    }

    public function get_activity_table(array $filters)
    {
        // Enhanced activity table with modern features and better data extraction
        $table = db_prefix() . 'lead_activity_log';
        $table_name_only = 'lead_activity_log';
        
        // Check if table exists, fallback to alternative table names
        $table_variations = ['lead_activity_log', 'activity_log', 'activities'];
        $actual_table = null;
        
        foreach ($table_variations as $variation) {
            if ($this->db->table_exists($variation)) {
                $actual_table = db_prefix() . $variation;
                $table_name_only = $variation;
                break;
            }
        }
        
        if (!$actual_table) {
            error_log('No activity table found. Checked: ' . implode(', ', $table_variations));
            return ['data' => [], 'recordsTotal' => 0, 'recordsFiltered' => 0, 'draw' => (int)$this->input->post('draw')];
        }

        error_log('Using activity table: ' . $actual_table);

        // DataTables parameters
        $draw = (int)$this->input->post('draw');
        $start = (int)$this->input->post('start');
        $length = (int)$this->input->post('length');
        if ($length <= 0 || $length > 100) { $length = 25; }

        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');

        error_log('Enhanced Activity Query - From: ' . $from . ', To: ' . $to . ', Table: ' . $actual_table);

        // Enhanced followup type extraction with multiple strategies
        $followup_type_case = $this->build_advanced_followup_type_extraction();
        
        // Build comprehensive select statement
        $select_fields = [
            'a.*',
            'CONCAT(s.firstname, " ", s.lastname) as staff_name',
            's.firstname', 
            's.lastname',
            'l.name as lead_name', 
            'l.company as lead_company',
            'l.phonenumber as lead_phone',
            'l.email as lead_email',
            $followup_type_case . ' as followup_type',
            'DATE_FORMAT(a.date, "%Y-%m-%d %H:%i") as formatted_date',
            'DATE_FORMAT(a.date, "%W, %M %d, %Y at %H:%i") as readable_date',
            'TIMESTAMPDIFF(HOUR, a.date, NOW()) as hours_ago'
        ];
        
        $this->db->select(implode(', ', $select_fields));
        $this->db->from($actual_table . ' a');
        $this->db->join(db_prefix() . 'staff s', 's.staffid = a.staffid', 'left');
        $this->db->join(db_prefix() . 'leads l', 'l.id = a.leadid', 'left');
        
        // Enhanced date filtering
        if ($from) $this->db->where('DATE(a.date) >=', $from);
        if ($to) $this->db->where('DATE(a.date) <=', $to);
        
        // Advanced activity type filtering
        if (!empty($filters['activity_type']) && is_array($filters['activity_type'])) {
            $this->apply_activity_type_filter($filters['activity_type']);
        }
        
        // Enhanced staff filtering
        if (!empty($filters['staff']) && is_array($filters['staff'])) {
            $this->db->where_in('a.staffid', $filters['staff']);
            error_log('Staff filter applied: ' . implode(', ', $filters['staff']));
        }

        // Advanced search functionality
        $search = $this->input->post('search');
        if (!empty($search['value'])) {
            $search_term = $search['value'];
            $this->db->group_start();
            $this->db->like('a.description', $search_term);
            $this->db->or_like('CONCAT(s.firstname, " ", s.lastname)', $search_term);
            $this->db->or_like('l.name', $search_term);
            $this->db->or_like('l.company', $search_term);
            $this->db->or_like('l.phonenumber', $search_term);
            $this->db->or_like('l.email', $search_term);
            $this->db->group_end();
        }

        // Get total count for pagination
        $total_query = clone $this->db;
        $total = $total_query->count_all_results();
        
        // Apply ordering and pagination
        $order_column = $this->input->post('order')[0]['column'] ?? 0;
        $order_dir = $this->input->post('order')[0]['dir'] ?? 'desc';
        
        $order_columns = ['a.date', 'staff_name', 'lead_name', 'followup_type', 'a.description'];
        if (isset($order_columns[$order_column])) {
            $this->db->order_by($order_columns[$order_column], $order_dir);
        } else {
            $this->db->order_by('a.date', 'DESC');
        }
        
        $this->db->limit($length, $start);
        
        try {
            $query_result = $this->db->get();
            $rows = $query_result->result_array();
            error_log('Enhanced Activity table query successful: ' . count($rows) . ' rows returned');
        } catch (Exception $e) {
            error_log('Enhanced Activity table query error: ' . $e->getMessage());
            error_log('Last query: ' . $this->db->last_query());
            return ['data' => [], 'recordsTotal' => 0, 'recordsFiltered' => 0, 'draw' => $draw];
        }

        $data = [];
        foreach ($rows as $r) {
            // Enhanced staff name handling
            $staffName = trim(($r['firstname'] ?? '') . ' ' . ($r['lastname'] ?? '')) ?: 'System';
            
            // Enhanced lead information
            $leadId = (int)($r['leadid'] ?? 0);
            $leadName = trim($r['lead_name'] ?? '');
            $leadCompany = trim($r['lead_company'] ?? '');
            $leadPhone = trim($r['lead_phone'] ?? '');
            $leadEmail = trim($r['lead_email'] ?? '');
            
            // Build comprehensive lead display
            $leadDisplay = $this->build_lead_display($leadName, $leadCompany, $leadPhone, $leadEmail);
            
            // Enhanced followup type with fallback
            $followupType = $this->extract_enhanced_followup_type($r);
            
            // Enhanced description processing
            $description = $this->process_activity_description($r['description'] ?? '');
            
            // Time formatting
            $hoursAgo = (int)($r['hours_ago'] ?? 0);
            $timeAgo = $this->format_time_ago($hoursAgo);
            
            // Build the enhanced data row
            $data[] = [
                'id' => $r['id'],
                'date' => $r['formatted_date'] ?? '',
                'readable_date' => $r['readable_date'] ?? '',
                'time_ago' => $timeAgo,
                'staff' => $staffName,
                'staff_id' => $r['staffid'] ?? 0,
                'lead' => $leadDisplay,
                'lead_id' => $leadId,
                'lead_name' => $leadName,
                'lead_company' => $leadCompany,
                'lead_phone' => $leadPhone,
                'lead_email' => $leadEmail,
                'followup_type' => $followupType,
                'description' => $description,
                'raw_description' => $r['description'] ?? '',
                'additional_data' => $r['additional_data'] ?? '',
                'priority' => $this->determine_activity_priority($r)
            ];
        }

        return [
            'draw' => $draw,
            'recordsTotal' => $total,
            'recordsFiltered' => $total,
            'data' => $data
        ];
    }

    /**
     * Build advanced followup type extraction with multiple strategies
     */
    private function build_advanced_followup_type_extraction()
    {
        return "(
            CASE 
                WHEN a.description REGEXP '(whatsapp|whats app|wa)' THEN 'WhatsApp'
                WHEN a.description REGEXP '(call|phone|tel|dialed|rang)' THEN 'Phone Call'
                WHEN a.description REGEXP '(email|mail|sent message)' THEN 'Email'
                WHEN a.description REGEXP '(sms|text message|texted)' THEN 'SMS'
                WHEN a.description REGEXP '(meeting|appointment|scheduled)' THEN 'Meeting'
                WHEN a.description REGEXP '(reminder|follow.?up|callback)' THEN 'Reminder'
                WHEN a.description REGEXP '(status|updated|changed)' THEN 'Status Update'
                WHEN a.description REGEXP '(proposal|quote|estimate)' THEN 'Proposal'
                WHEN a.description REGEXP '(contract|agreement|signed)' THEN 'Contract'
                WHEN a.description REGEXP '(invoice|bill|payment)' THEN 'Invoice'
                WHEN a.description REGEXP '(note|comment|remark)' THEN 'Note'
                WHEN a.description REGEXP '(assigned|delegated)' THEN 'Assignment'
                WHEN a.description REGEXP '(qualified|qualification)' THEN 'Qualification'
                WHEN a.description REGEXP '(demo|demonstration|presentation)' THEN 'Demo'
                WHEN a.description REGEXP '(visit|visited|on.?site)' THEN 'Site Visit'
                ELSE 'General'
            END
        )";
    }

    /**
     * Apply advanced activity type filtering
     */
    private function apply_activity_type_filter($activity_types)
    {
        if (empty($activity_types)) return;
        
        $this->db->group_start();
        foreach ($activity_types as $type) {
            switch (strtolower($type)) {
                case 'whatsapp':
                    $this->db->or_like('a.description', 'whatsapp');
                    $this->db->or_like('a.description', 'whats app');
                    $this->db->or_like('a.description', 'wa');
                    break;
                case 'call':
                case 'phone':
                    $this->db->or_like('a.description', 'call');
                    $this->db->or_like('a.description', 'phone');
                    $this->db->or_like('a.description', 'dialed');
                    break;
                case 'email':
                    $this->db->or_like('a.description', 'email');
                    $this->db->or_like('a.description', 'mail');
                    break;
                case 'sms':
                    $this->db->or_like('a.description', 'sms');
                    $this->db->or_like('a.description', 'text');
                    break;
                case 'meeting':
                    $this->db->or_like('a.description', 'meeting');
                    $this->db->or_like('a.description', 'appointment');
                    break;
                default:
                    $this->db->or_like('a.description', $type);
            }
        }
        $this->db->group_end();
    }

    /**
     * Build enhanced lead display with contact info
     */
    private function build_lead_display($name, $company, $phone, $email)
    {
        $display = '';
        
        if ($name && $company) {
            $display = $name . ' (' . $company . ')';
        } elseif ($name) {
            $display = $name;
        } elseif ($company) {
            $display = $company;
        } else {
            $display = 'Unknown Lead';
        }
        
        $contact_info = [];
        if ($phone) $contact_info[] = $phone;
        if ($email) $contact_info[] = $email;
        
        if (!empty($contact_info)) {
            $display .= ' • ' . implode(' • ', $contact_info);
        }
        
        return $display;
    }

    /**
     * Extract enhanced followup type with multiple strategies
     */
    private function extract_enhanced_followup_type($row)
    {
        // Use the SQL-generated followup type first
        if (!empty($row['followup_type']) && $row['followup_type'] !== 'General') {
            return $row['followup_type'];
        }
        
        $description = strtolower($row['description'] ?? '');
        
        // Enhanced pattern matching
        $patterns = [
            'WhatsApp' => ['whatsapp', 'whats app', 'wa message', 'wa sent'],
            'Phone Call' => ['call', 'phone', 'dialed', 'rang', 'spoke with'],
            'Email' => ['email', 'mail sent', 'emailed', 'sent message'],
            'SMS' => ['sms', 'text message', 'texted', 'sent text'],
            'Meeting' => ['meeting', 'appointment', 'scheduled', 'met with'],
            'Reminder' => ['reminder', 'follow up', 'callback', 'follow-up'],
            'Status Update' => ['status', 'updated', 'changed to', 'moved to'],
            'Proposal' => ['proposal', 'quote', 'estimate', 'quotation'],
            'Contract' => ['contract', 'agreement', 'signed', 'signature'],
            'Invoice' => ['invoice', 'bill', 'payment', 'paid'],
            'Note' => ['note added', 'comment', 'remark', 'observation'],
            'Assignment' => ['assigned', 'delegated', 'transferred'],
            'Demo' => ['demo', 'demonstration', 'presentation', 'showed']
        ];
        
        foreach ($patterns as $type => $keywords) {
            foreach ($keywords as $keyword) {
                if (strpos($description, $keyword) !== false) {
                    return $type;
                }
            }
        }
        
        return 'General';
    }

    /**
     * Process and enhance activity descriptions
     */
    private function process_activity_description($description)
    {
        // Clean up and enhance the description
        $description = trim($description);
        
        // Add emoji indicators for common activities
        $emoji_map = [
            'whatsapp' => '💬',
            'call' => '📞',
            'email' => '📧',
            'sms' => '💬',
            'meeting' => '🤝',
            'reminder' => '⏰',
            'proposal' => '📄',
            'contract' => '📋',
            'invoice' => '💰'
        ];
        
        $lower_desc = strtolower($description);
        foreach ($emoji_map as $keyword => $emoji) {
            if (strpos($lower_desc, $keyword) !== false) {
                $description = $emoji . ' ' . $description;
                break;
            }
        }
        
        return $description;
    }

    /**
     * Format time ago in a human-readable way
     */
    private function format_time_ago($hours)
    {
        if ($hours < 1) {
            return 'Just now';
        } elseif ($hours < 24) {
            return $hours . ' hour' . ($hours > 1 ? 's' : '') . ' ago';
        } elseif ($hours < 48) {
            return 'Yesterday';
        } else {
            $days = floor($hours / 24);
            return $days . ' day' . ($days > 1 ? 's' : '') . ' ago';
        }
    }

    /**
     * Determine activity priority based on type and timing
     */
    private function determine_activity_priority($row)
    {
        $hours_ago = (int)($row['hours_ago'] ?? 999);
        $description = strtolower($row['description'] ?? '');
        
        // High priority: Recent important activities
        if ($hours_ago <= 2 && (
            strpos($description, 'call') !== false ||
            strpos($description, 'meeting') !== false ||
            strpos($description, 'urgent') !== false
        )) {
            return 'high';
        }
        
        // Medium priority: Recent general activities
        if ($hours_ago <= 24) {
            return 'medium';
        }
        
        return 'normal';
    }

    /**
     * Helper functions for KPI calculation
     */
    private function count_lead_activities($filters) 
    {
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        $staff = $filters['staff'] ?? [];
        
        // Try multiple table names
        $tables = ['lead_activity_log', 'activity_log', 'activities'];
        $table_found = null;
        
        foreach ($tables as $table) {
            if ($this->db->table_exists($table)) {
                $table_found = db_prefix() . $table;
                break;
            }
        }
        
        if (!$table_found) {
            return 0;
        }
        
        $this->db->from($table_found);
        $this->db->where('DATE(date) >=', $from);
        $this->db->where('DATE(date) <=', $to);
        
        if (!empty($staff)) {
            $this->db->where_in('staffid', $staff);
        }
        
        return $this->db->count_all_results();
    }
    
    private function count_logins_today($staff) 
    {
        // Placeholder - implement based on your login tracking system
        return 0;
    }
    
    private function get_top_staff_simple($filters) 
    {
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        
        // Try to find the correct activity table
        $tables = ['lead_activity_log', 'activity_log', 'activities'];
        $table_found = null;
        
        foreach ($tables as $table) {
            if ($this->db->table_exists($table)) {
                $table_found = db_prefix() . $table;
                break;
            }
        }
        
        if (!$table_found) {
            return [];
        }
        
        $this->db->select('s.staffid, CONCAT(s.firstname, " ", s.lastname) as staff_name, COUNT(a.id) as total');
        $this->db->from($table_found . ' a');
        $this->db->join(db_prefix() . 'staff s', 's.staffid = a.staffid', 'left');
        $this->db->where('DATE(a.date) >=', $from);
        $this->db->where('DATE(a.date) <=', $to);
        $this->db->group_by('s.staffid');
        $this->db->order_by('total', 'DESC');
        $this->db->limit(5);
        
        return $this->db->get()->result_array();
    }

    // Method for email summary
                $related = 'Unknown Lead';
            }
            
            $descLower = strtolower($r['description'] ?? '');
            $activityType = 'Lead Activity';
            if(strpos($descLower,'not_activity_new_reminder_created') !== false){ $activityType = 'Reminder Created'; }
            elseif(strpos($descLower,'not_lead_activity_added_followup') !== false){ $activityType = 'Followup Added'; }
            elseif(strpos($descLower,'not_lead_activity_lead_created') !== false){ $activityType = 'Lead Created'; }
            elseif(strpos($descLower,'not_lead_activity_assigned') !== false){ $activityType = 'Lead Assigned'; }

            // Get follow-up information from the query result  
            $followupType = trim($r['followup_type'] ?? '');
            $description = $r['description'] ?? 'Activity performed on lead';
            
            // Debug: Log the raw data for debugging
            error_log("Activity ID {$r['id']}: raw followup_type = '" . ($r['followup_type'] ?? 'NULL') . "'");
            error_log("Activity ID {$r['id']}: description = '" . substr($description, 0, 100) . "'");
            
            // If no followup type from subquery, try multiple extraction methods
            if (!$followupType) {
                // Method 1: Extract from "Type: X" pattern in description
                if (preg_match('/<strong>Type:\s*([^<]+)<\/strong>/i', $description, $matches)) {
                    $followupType = trim($matches[1]);
                    error_log("Activity ID {$r['id']}: extracted from <strong>Type: pattern = '$followupType'");
                } elseif (preg_match('/Type:\s*([^<\s\r\n]+)/i', $description, $matches)) {
                    $followupType = trim($matches[1]);
                    error_log("Activity ID {$r['id']}: extracted from Type: pattern = '$followupType'");
                }
                
                // Method 2: Look for common followup keywords in description
                if (!$followupType) {
                    $desc_lower = strtolower($description);
                    if (strpos($desc_lower, 'phone') !== false || strpos($desc_lower, 'call') !== false) {
                        $followupType = 'Phone Call';
                    } elseif (strpos($desc_lower, 'email') !== false) {
                        $followupType = 'Email';
                    } elseif (strpos($desc_lower, 'whatsapp') !== false) {
                        $followupType = 'WhatsApp Call';
                    } elseif (strpos($desc_lower, 'visit') !== false) {
                        $followupType = 'Visit';
                    } elseif (strpos($desc_lower, 'message') !== false || strpos($desc_lower, 'sms') !== false) {
                        $followupType = 'Message';
                    } else {
                        // Try to extract any word after "Type:" or similar patterns
                        if (preg_match('/(?:type|followup|follow.up):\s*(\w+)/i', $description, $matches)) {
                            $followupType = ucfirst(trim($matches[1]));
                        } else {
                            $followupType = 'General'; // Default for unmatched
                        }
                    }
                    error_log("Activity ID {$r['id']}: extracted from keywords = '$followupType'");
                }
            }
            
            // If still no type, set default based on activity type
            if (!$followupType) {
                if ($activityType === 'Reminder Created') {
                    $followupType = 'Reminder';
                } else {
                    $followupType = 'Message'; // Default fallback
                }
                error_log("Activity ID {$r['id']}: using default fallback = '$followupType'");
            }
            
            // Build description based on activity type
            if ($activityType === 'Reminder Created') {
                $description = 'Reminder created for follow-up';
            } elseif ($activityType === 'Followup Added') {
                // Keep the original description or clean it up
                $description = strip_tags($description);
            }

            // Build proper view link
            $viewLink = $leadId > 0 ? 'leads/index/' . $leadId : '#';

            // Raw row object (API friendly)
            $rowObj = [
                'id' => (int)$r['id'],
                'date' => e(_dt($r['date'])),
                'staff_name' => e($staffName),
                'activity_type' => $activityType,
                'related' => e($related),
                'description' => e($description),
                'followup_type' => e($followupType),
                'lead_id' => $leadId,
                'view_link' => $viewLink
            ];

            if($isDataTables){
                // Keep legacy array format for DataTables
                $staffImage = staff_profile_image($r['staffid'], ['staff-profile-image-small']);
                $staffColumn = '<div class="d-flex align-items-center">' . $staffImage . '<span class="ml-2">' . e($staffName) . '</span></div>';
                $data[] = [
                    $rowObj['date'],
                    $staffColumn,
                    $activityType,
                    $rowObj['related'],
                    process_text_content_for_display($r['description']),
                    '-',
                    '<button class="btn btn-sm btn-default view-details" data-id="' . $r['id'] . '"><i class="fa fa-eye"></i></button>'
                ];
            } else {
                $data[] = $rowObj;
            }
        }

        if($isDataTables){
            return [
                'draw' => $draw,
                'recordsTotal' => $total,
                'recordsFiltered' => $total,
                'data' => $data
            ];
        }
        return [ 'data' => $data ];
    }

    // Helper methods
    private function count_lead_activities($filters)
    {
        $this->db->from(db_prefix() . 'lead_activity_log');
        if (!empty($filters['from'])) {
            $this->db->where('DATE(date) >=', $filters['from']);
        }
        if (!empty($filters['to'])) {
            $this->db->where('DATE(date) <=', $filters['to']);
        }
        if (!empty($filters['staff'])) {
            $this->db->where_in('staffid', $filters['staff']);
        }
        return $this->db->count_all_results();
    }

    private function count_logins_today($staffFilter = [])
    {
        // Check if general activity log exists for login tracking
        if ($this->db->table_exists(str_replace(db_prefix(), '', db_prefix() . 'activitylog'))) {
            $this->db->from(db_prefix() . 'activitylog');
            $this->db->like('description', 'logged in');
            $this->db->where('DATE(date)', date('Y-m-d'));
            if (!empty($staffFilter)) {
                $this->db->where_in('staffid', $staffFilter);
            }
            return $this->db->count_all_results();
        }
        return 0;
    }

    private function get_top_staff_simple($filters = [])
    {
        $this->db->select('s.staffid, CONCAT(s.firstname, " ", s.lastname) as staff_name, COUNT(a.id) as total');
        $this->db->from(db_prefix() . 'lead_activity_log a');
        $this->db->join(db_prefix() . 'staff s', 's.staffid = a.staffid', 'left');
        $this->db->where('DATE(a.date) >=', date('Y-m-01')); // This month
        if (!empty($filters['staff'])) {
            $this->db->where_in('a.staffid', $filters['staff']);
        }
        $this->db->group_by('s.staffid');
        $this->db->order_by('total', 'DESC');
        $this->db->limit(5);
        
        return $this->db->get()->result_array();
    }

    // Method for email summary
    public function get_email_summary_data()
    {
        return [
            'kpis' => $this->get_kpis([])
        ];
    }

    /**
     * Return reminders grouped for dashboard (basic placeholder implementation)
     * Structure: ['todays'=>[], 'due'=>[], 'upcoming'=>[]]
     */
    public function get_reminders(array $filters)
    {
        $out = [ 'todays'=>[], 'due'=>[], 'upcoming'=>[] ];

        // Attempt to query reminders table if exists (Perfex usually tblreminders)
        $table = db_prefix() . 'reminders';
        if(!$this->db->table_exists(str_replace(db_prefix(), '', $table))){
            return $out; // table not present, return empty structure
        }

        $today = date('Y-m-d');
        
        // Enhanced query with staff names and related entity names
        $this->db->select('
            r.id, 
            r.description, 
            r.date, 
            r.rel_type, 
            r.rel_id, 
            r.isnotified,
            r.staff,
            CONCAT(s.firstname, " ", s.lastname) as staff_name,
            CASE 
                WHEN r.rel_type = "lead" THEN l.name
                WHEN r.rel_type = "customer" THEN c.company
                WHEN r.rel_type = "project" THEN p.name
                WHEN r.rel_type = "proposal" THEN pr.subject
                WHEN r.rel_type = "estimate" THEN CONCAT("Estimate #", e.number)
                WHEN r.rel_type = "invoice" THEN CONCAT("Invoice #", i.number)
                WHEN r.rel_type = "contract" THEN co.subject
                ELSE CONCAT(r.rel_type, " #", r.rel_id)
            END as related_name,
            CASE 
                WHEN r.rel_type = "lead" THEN l.email
                WHEN r.rel_type = "customer" THEN c.email
                ELSE NULL
            END as related_email
        ');
        $this->db->from($table . ' r');
        
        // Join with staff table for staff names
        $this->db->join(db_prefix() . 'staff s', 's.staffid = r.staff', 'left');
        
        // Join with leads table
        $this->db->join(db_prefix() . 'leads l', 'l.id = r.rel_id AND r.rel_type = "lead"', 'left');
        
        // Join with customers table
        $this->db->join(db_prefix() . 'clients c', 'c.userid = r.rel_id AND r.rel_type = "customer"', 'left');
        
        // Join with projects table
        $this->db->join(db_prefix() . 'projects p', 'p.id = r.rel_id AND r.rel_type = "project"', 'left');
        
        // Join with proposals table
        $this->db->join(db_prefix() . 'proposals pr', 'pr.id = r.rel_id AND r.rel_type = "proposal"', 'left');
        
        // Join with estimates table
        $this->db->join(db_prefix() . 'estimates e', 'e.id = r.rel_id AND r.rel_type = "estimate"', 'left');
        
        // Join with invoices table
        $this->db->join(db_prefix() . 'invoices i', 'i.id = r.rel_id AND r.rel_type = "invoice"', 'left');
        
        // Join with contracts table
        $this->db->join(db_prefix() . 'contracts co', 'co.id = r.rel_id AND r.rel_type = "contract"', 'left');
        
        // Exclude reminders for leads that are marked as junk or lost
        $this->db->where('(r.rel_type != "lead" OR (l.junk = 0 AND l.lost = 0))');
        
        // Optional staff filter if provided
        if(!empty($filters['staff'])){
            $this->db->where_in('r.staff', (array)$filters['staff']);
        }
        
        $this->db->order_by('r.date', 'ASC');
        $rows = $this->db->get()->result_array();

        foreach($rows as $r){
            $d = substr($r['date'],0,10);
            $isNotified = (int)($r['isnotified'] ?? 0);
            
            if($d === $today){
                // Only include if not marked as read/notified
                if($isNotified == 0){
                    $out['todays'][] = $r;
                }
            } elseif($d < $today){
                // For overdue: only include if not marked as read/notified and date is in the past
                if($isNotified == 0){
                    $out['due'][] = $r;
                }
            } else {
                // For upcoming: include all future reminders regardless of notification status
                $out['upcoming'][] = $r;
            }
        }
        return $out;
    }


}