<?php

defined('BASEPATH') or exit('No direct script access allowed');

class Reporting_dashboard_model extends App_Model
{
    public function __construct()
    {
        parent::__construct();
    }

    public function get_kpis(array $filters)
    {
        $today = date('Y-m-d');
        $weekStart = date('Y-m-d', strtotime('monday this week'));
        $monthStart = date('Y-m-01');
        
        return [
            'today' => $this->count_lead_activities(['from' => $today, 'to' => $today]),
            'week' => $this->count_lead_activities(['from' => $weekStart, 'to' => $today]),
            'month' => $this->count_lead_activities(['from' => $monthStart, 'to' => $today]),
            'logins_today' => $this->count_logins_today(),
            'top_staff' => $this->get_top_staff_simple()
        ];
    }

    public function get_staff_activity_chart(array $filters)
    {
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        
        $this->db->select('s.staffid, CONCAT(s.firstname, " ", s.lastname) as staff_name, COUNT(a.id) as total');
        $this->db->from(db_prefix() . 'lead_activity_log a');
        $this->db->join(db_prefix() . 'staff s', 's.staffid = a.staffid', 'left');
        
        if ($from) $this->db->where('DATE(a.date) >=', $from);
        if ($to) $this->db->where('DATE(a.date) <=', $to);
        if (!empty($filters['staff'])) {
            $this->db->where_in('a.staffid', $filters['staff']);
        }
        
        $this->db->group_by('s.staffid');
        $this->db->order_by('total', 'DESC');
        $this->db->limit(10);
        
        $result = $this->db->get()->result_array();
        
        return [
            'labels' => array_column($result, 'staff_name'),
            'data' => array_map('intval', array_column($result, 'total'))
        ];
    }

    public function get_activity_over_time_chart(array $filters)
    {
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        
        $this->db->select('DATE(a.date) as date, COUNT(a.id) as total');
        $this->db->from(db_prefix() . 'lead_activity_log a');
        
        if ($from) $this->db->where('DATE(a.date) >=', $from);
        if ($to) $this->db->where('DATE(a.date) <=', $to);
        if (!empty($filters['staff'])) {
            $this->db->where_in('a.staffid', $filters['staff']);
        }
        
        $this->db->group_by('DATE(a.date)');
        $this->db->order_by('DATE(a.date)', 'ASC');
        
        $result = $this->db->get()->result_array();
        
        return [
            'labels' => array_column($result, 'date'),
            'data' => array_map('intval', array_column($result, 'total'))
        ];
    }

    public function get_activity_type_pie(array $filters)
    {
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        
        $this->db->select('
            CASE
                WHEN a.description LIKE "%created%" THEN "Lead Created"
                WHEN a.description LIKE "%assigned%" THEN "Lead Assigned"
                WHEN a.description LIKE "%status%" THEN "Status Changed"
                WHEN a.description LIKE "%note%" THEN "Note Added"
                WHEN a.description LIKE "%updated%" THEN "Lead Updated"
                ELSE "Other"
            END as activity_type,
            COUNT(a.id) as total
        ');
        $this->db->from(db_prefix() . 'lead_activity_log a');
        
        if ($from) $this->db->where('DATE(a.date) >=', $from);
        if ($to) $this->db->where('DATE(a.date) <=', $to);
        if (!empty($filters['staff'])) {
            $this->db->where_in('a.staffid', $filters['staff']);
        }
        
        $this->db->group_by('activity_type');
        $this->db->order_by('total', 'DESC');
        
        $result = $this->db->get()->result_array();
        
        return [
            'labels' => array_column($result, 'activity_type'),
            'data' => array_map('intval', array_column($result, 'total'))
        ];
    }

    public function get_activity_table(array $filters)
    {
        // DataTables server-side processing
        $draw = (int)$this->input->post('draw');
        $start = (int)$this->input->post('start');
        $length = (int)$this->input->post('length');
        
        if ($length <= 0 || $length > 1000) {
            $length = 25;
        }
        
        $from = $filters['from'] ?? date('Y-m-01');
        $to = $filters['to'] ?? date('Y-m-d');
        
        // Base query
        $this->db->select('a.*, s.firstname, s.lastname, l.name as lead_name, l.company as lead_company');
        $this->db->from(db_prefix() . 'lead_activity_log a');
        $this->db->join(db_prefix() . 'staff s', 's.staffid = a.staffid', 'left');
        $this->db->join(db_prefix() . 'leads l', 'l.id = a.leadid', 'left');
        
        // Apply filters
        if ($from) $this->db->where('DATE(a.date) >=', $from);
        if ($to) $this->db->where('DATE(a.date) <=', $to);
        if (!empty($filters['staff'])) {
            $this->db->where_in('a.staffid', $filters['staff']);
        }
        
        // Search
        $search = $this->input->post('search');
        if (!empty($search['value'])) {
            $searchValue = $search['value'];
            $this->db->group_start();
            $this->db->like('a.description', $searchValue);
            $this->db->or_like('s.firstname', $searchValue);
            $this->db->or_like('s.lastname', $searchValue);
            $this->db->or_like('l.name', $searchValue);
            $this->db->or_like('l.company', $searchValue);
            $this->db->group_end();
        }
        
        // Get total count
        $totalQuery = clone $this->db;
        $total = $totalQuery->count_all_results();
        
        // Add pagination and ordering
        $this->db->order_by('a.date', 'DESC');
        $this->db->limit($length, $start);
        
        $activities = $this->db->get()->result_array();
        
        // Format data for DataTables
        $data = [];
        foreach ($activities as $activity) {
            $staffName = trim(($activity['firstname'] ?? '') . ' ' . ($activity['lastname'] ?? ''));
            if (empty($staffName)) $staffName = 'System';
            
            $staffImage = staff_profile_image($activity['staffid'], ['staff-profile-image-small']);
            $staffColumn = '<div class="d-flex align-items-center">' . $staffImage . '<span class="ml-2">' . e($staffName) . '</span></div>';
            
            $leadInfo = '';
            if (!empty($activity['lead_name'])) {
                $leadInfo = e($activity['lead_name']);
            } elseif (!empty($activity['lead_company'])) {
                $leadInfo = e($activity['lead_company']);
            } else {
                $leadInfo = 'Lead #' . $activity['leadid'];
            }
            
            $data[] = [
                e(_dt($activity['date'])),
                $staffColumn,
                'Lead Activity',
                $leadInfo,
                process_text_content_for_display($activity['description']),
                '-',
                '<button class="btn btn-sm btn-default view-details" data-id="' . $activity['id'] . '"><i class="fa fa-eye"></i></button>'
            ];
        }
        
        return [
            'draw' => $draw,
            'recordsTotal' => $total,
            'recordsFiltered' => $total,
            'data' => $data
        ];
    }

    // Helper methods
    private function count_lead_activities($dateRange)
    {
        $this->db->from(db_prefix() . 'lead_activity_log');
        if (!empty($dateRange['from'])) {
            $this->db->where('DATE(date) >=', $dateRange['from']);
        }
        if (!empty($dateRange['to'])) {
            $this->db->where('DATE(date) <=', $dateRange['to']);
        }
        return $this->db->count_all_results();
    }

    private function count_logins_today()
    {
        // Check if general activity log exists for login tracking
        if ($this->db->table_exists(str_replace(db_prefix(), '', db_prefix() . 'activitylog'))) {
            $this->db->from(db_prefix() . 'activitylog');
            $this->db->like('description', 'logged in');
            $this->db->where('DATE(date)', date('Y-m-d'));
            return $this->db->count_all_results();
        }
        return 0;
    }

    private function get_top_staff_simple()
    {
        $this->db->select('s.staffid, CONCAT(s.firstname, " ", s.lastname) as staff_name, COUNT(a.id) as total');
        $this->db->from(db_prefix() . 'lead_activity_log a');
        $this->db->join(db_prefix() . 'staff s', 's.staffid = a.staffid', 'left');
        $this->db->where('DATE(a.date) >=', date('Y-m-01')); // This month
        $this->db->group_by('s.staffid');
        $this->db->order_by('total', 'DESC');
        $this->db->limit(5);
        
        return $this->db->get()->result_array();
    }

    // Method for email summary
    public function get_email_summary_data()
    {
        return [
            'kpis' => $this->get_kpis([])
        ];
    }

    // Create sample data if none exists
    public function create_sample_data()
    {
        // Check if we have any lead activities
        $count = $this->db->count_all(db_prefix() . 'lead_activity_log');
        
        if ($count > 0) {
            return ['success' => false, 'message' => 'Data already exists'];
        }
        
        // Get first lead
        $this->db->limit(1);
        $lead = $this->db->get(db_prefix() . 'leads')->row();
        
        if (!$lead) {
            return ['success' => false, 'message' => 'No leads found. Create a lead first.'];
        }
        
        // Sample activities
        $activities = [
            'Lead created by staff member',
            'Lead status updated to Customer',
            'Note added to lead record',
            'Lead assigned to sales team',
            'Follow-up scheduled for lead',
            'Lead source information updated',
            'Contact information verified',
            'Lead marked as hot prospect'
        ];
        
        $inserted = 0;
        foreach ($activities as $i => $description) {
            $data = [
                'date' => date('Y-m-d H:i:s', strtotime('-' . (count($activities) - $i) . ' days')),
                'description' => $description,
                'leadid' => $lead->id,
                'staffid' => 1, // Admin user
                'full_name' => 'Admin User'
            ];
            
            if ($this->db->insert(db_prefix() . 'lead_activity_log', $data)) {
                $inserted++;
            }
        }
        
        return ['success' => true, 'message' => "Created $inserted sample activities"];
    }
}