(function($) {
    'use strict';
    
    var activityTable = null;
    var charts = {
        staff: null,
        timeline: null,
        types: null
    };

    // Wait for both jQuery and debug panel
    function initReportingDashboard() {
        console.log('Reporting Dashboard JS initializing...');
        
        // Debug integration
        if (window.reportingDashboardDebug) {
            window.reportingDashboardDebug.log('JavaScript file loaded and initializing');
            window.reportingDashboardDebug.updateStatus('button-status', 'Initializing...');
        }
        
        initializeDatePickers();
        initializeSelectPickers();
        initializeDataTable();
        initializeCharts();
        
        // Apply filters button
        $('#apply-filters').on('click', function(e) {
            e.preventDefault();
            console.log('Apply filters button clicked!');
            
            if (window.reportingDashboardDebug) {
                window.reportingDashboardDebug.log('Apply button clicked');
                window.reportingDashboardDebug.updateStatus('last-click', new Date().toLocaleTimeString());
                window.reportingDashboardDebug.updateStatus('filters', JSON.stringify(getFilters()));
            }
            
            applyFilters();
        });
        
        // Debug: Check if button exists
        var buttonExists = $('#apply-filters').length > 0;
        console.log('Apply button found:', buttonExists);
        
        if (window.reportingDashboardDebug) {
            window.reportingDashboardDebug.updateStatus('button-status', buttonExists ? 'Found and Attached' : 'NOT FOUND');
            
            // Show current URLs
            var baseUrl = window.location.origin + window.location.pathname.split('/admin/')[0] + '/admin/';
            window.reportingDashboardDebug.updateStatus('urls', baseUrl + 'reporting_dashboard/get_kpis');
        }
        
        // Preset date buttons
        $('.preset-btn').on('click', function(e) {
            e.preventDefault();
            var preset = $(this).data('preset');
            setDatePreset(preset);
        });
        
        // Export buttons
        $('#export-csv, #table-export-csv').on('click', exportCSV);
        $('#export-excel, #table-export-excel').on('click', exportExcel);
        $('#export-print').on('click', printTable);
        
        // Load initial data with this month preset
        setDatePreset('month');
        setTimeout(function() {
            applyFilters();
        }, 500);
    }

    $(document).ready(function() {
        // Wait a bit for debug panel to be ready
        setTimeout(initReportingDashboard, 100);
    });

    function initializeDatePickers() {
        if (typeof $.fn.datepicker !== 'undefined') {
            $('.datepicker').datepicker({
                format: 'yyyy-mm-dd',
                autoclose: true
            });
        }
    }

    function initializeSelectPickers() {
        if (typeof $.fn.selectpicker !== 'undefined') {
            $('.selectpicker').selectpicker();
        }
    }

    function initializeDataTable() {
        if (typeof $.fn.DataTable !== 'undefined') {
            var baseUrl = window.location.origin + window.location.pathname.split('/admin/')[0] + '/admin/';
            
            activityTable = $('#activity-log-table').DataTable({
                processing: true,
                serverSide: true,
                ajax: {
                    url: baseUrl + 'reporting_dashboard/get_activity_table',
                    type: 'POST',
                    data: function(d) {
                        var filters = getFilters();
                        console.log('DataTable sending data:', $.extend({}, d, filters));
                        return $.extend({}, d, filters);
                    },
                    error: function(xhr, error, code) {
                        console.error('DataTable AJAX error:', error, code, xhr.responseText);
                    }
                },
                columns: [
                    { title: 'Date & Time', width: '15%' },
                    { title: 'Staff Member', width: '20%' },
                    { title: 'Activity Type', width: '15%' },
                    { title: 'Related Entity', width: '15%' },
                    { title: 'Description', width: '25%' },
                    { title: 'IP Address', width: '10%' },
                    { title: 'Actions', width: '10%', orderable: false }
                ],
                order: [[0, 'desc']],
                pageLength: 25,
                responsive: true,
                language: {
                    processing: "Loading activities...",
                    emptyTable: "No activity data found",
                    zeroRecords: "No matching activities found"
                }
            });
        }
    }

    function initializeCharts() {
        if (typeof Chart !== 'undefined') {
            // Staff Activity Chart
            var staffCtx = document.getElementById('staffActivityChart');
            if (staffCtx) {
                charts.staff = new Chart(staffCtx, {
                    type: 'bar',
                    data: {
                        labels: [],
                        datasets: [{
                            label: 'Activities',
                            data: [],
                            backgroundColor: 'rgba(54, 162, 235, 0.6)',
                            borderColor: 'rgba(54, 162, 235, 1)',
                            borderWidth: 1
                        }]
                    },
                    options: {
                        responsive: true,
                        scales: {
                            y: {
                                beginAtZero: true
                            }
                        }
                    }
                });
            }

            // Timeline Chart
            var timelineCtx = document.getElementById('timelineChart');
            if (timelineCtx) {
                charts.timeline = new Chart(timelineCtx, {
                    type: 'line',
                    data: {
                        labels: [],
                        datasets: [{
                            label: 'Daily Activities',
                            data: [],
                            borderColor: 'rgba(75, 192, 192, 1)',
                            backgroundColor: 'rgba(75, 192, 192, 0.2)',
                            tension: 0.1
                        }]
                    },
                    options: {
                        responsive: true,
                        scales: {
                            y: {
                                beginAtZero: true
                            }
                        }
                    }
                });
            }

            // Activity Types Pie Chart
            var typesCtx = document.getElementById('activityTypesChart');
            if (typesCtx) {
                charts.types = new Chart(typesCtx, {
                    type: 'pie',
                    data: {
                        labels: [],
                        datasets: [{
                            data: [],
                            backgroundColor: [
                                '#FF6384',
                                '#36A2EB',
                                '#FFCE56',
                                '#4BC0C0',
                                '#9966FF',
                                '#FF9F40',
                                '#FF6384',
                                '#C9CBCF'
                            ]
                        }]
                    },
                    options: {
                        responsive: true
                    }
                });
            }
        }
    }

    function setDatePreset(preset) {
        var today = new Date();
        var from, to;
        
        switch(preset) {
            case 'today':
                from = to = formatDate(today);
                break;
            case 'week':
                var monday = new Date(today);
                monday.setDate(today.getDate() - today.getDay() + 1);
                from = formatDate(monday);
                to = formatDate(today);
                break;
            case 'month':
                from = formatDate(new Date(today.getFullYear(), today.getMonth(), 1));
                to = formatDate(today);
                break;
        }
        
        $('#date-from').val(from);
        $('#date-to').val(to);
        
        // Update preset button states
        $('.preset-btn').removeClass('active');
        $('.preset-btn[data-preset="' + preset + '"]').addClass('active');
    }
    
    function formatDate(date) {
        var year = date.getFullYear();
        var month = String(date.getMonth() + 1).padStart(2, '0');
        var day = String(date.getDate()).padStart(2, '0');
        return year + '-' + month + '-' + day;
    }

    function getFilters() {
        return {
            from: $('#date-from').val(),
            to: $('#date-to').val(),
            staff: $('#staff-filter').val() || [],
            activity_type: $('#activity-type-filter').val() || []
        };
    }

    function applyFilters() {
        console.log('Applying filters:', getFilters());
        
        loadKPIs();
        loadCharts();
        
        if (activityTable) {
            activityTable.ajax.reload();
        }
    }

    function loadKPIs() {
        var filters = getFilters();
        console.log('Loading KPIs with filters:', filters);
        
        // Build URL manually to ensure it's correct
        var baseUrl = window.location.origin + window.location.pathname.split('/admin/')[0] + '/admin/';
        var ajaxUrl = baseUrl + 'reporting_dashboard/get_kpis';
        
        console.log('AJAX URL:', ajaxUrl);
        
        $.ajax({
            url: ajaxUrl,
            type: 'POST',
            data: filters,
            dataType: 'json',
            beforeSend: function() {
                console.log('Sending KPI request...');
            },
            success: function(data) {
                console.log('KPIs loaded successfully:', data);
                
                if (window.reportingDashboardDebug) {
                    window.reportingDashboardDebug.log('KPIs loaded successfully', data);
                    window.reportingDashboardDebug.updateStatus('error', 'Last load: SUCCESS');
                }
                
                $('#kpi-today').text(data.today || 0);
                $('#kpi-week').text(data.week || 0);
                $('#kpi-month').text(data.month || 0);
                $('#kpi-logins').text(data.logins_today || 0);
                
                // Top staff
                var topStaffHtml = '';
                if (data.top_staff && data.top_staff.length > 0) {
                    data.top_staff.forEach(function(staff) {
                        topStaffHtml += '<div class="staff-item">';
                        topStaffHtml += '<span class="staff-name">' + (staff.staff_name || 'Unknown') + '</span>';
                        topStaffHtml += '<span class="staff-count">' + (staff.total || 0) + '</span>';
                        topStaffHtml += '</div>';
                    });
                } else {
                    topStaffHtml = '<div class="text-muted text-center">No data available</div>';
                }
                $('#top-staff-list').html(topStaffHtml);
            },
            error: function(xhr, status, error) {
                console.error('Error loading KPIs:', error, xhr.responseText);
                console.error('Status:', status);
                console.error('XHR:', xhr);
                
                if (window.reportingDashboardDebug) {
                    window.reportingDashboardDebug.log('KPI AJAX Error', {
                        status: status,
                        error: error,
                        response: xhr.responseText
                    });
                    window.reportingDashboardDebug.updateStatus('error', 'KPI Load Failed: ' + error);
                }
            }
        });
    }

    function loadCharts() {
        var filters = getFilters();
        var baseUrl = window.location.origin + window.location.pathname.split('/admin/')[0] + '/admin/';
        
        // Staff Activity Chart
        $.ajax({
            url: baseUrl + 'reporting_dashboard/get_staff_activity_chart',
            type: 'POST',
            data: filters,
            dataType: 'json',
            success: function(data) {
                console.log('Staff chart data:', data);
                if (charts.staff) {
                    charts.staff.data.labels = data.labels || [];
                    charts.staff.data.datasets[0].data = data.data || [];
                    charts.staff.update();
                }
            },
            error: function(xhr, status, error) {
                console.error('Error loading staff chart:', error, xhr.responseText);
            }
        });

        // Timeline Chart
        $.ajax({
            url: baseUrl + 'reporting_dashboard/get_activity_over_time_chart',
            type: 'POST',
            data: filters,
            dataType: 'json',
            success: function(data) {
                console.log('Timeline chart data:', data);
                if (charts.timeline) {
                    charts.timeline.data.labels = data.labels || [];
                    charts.timeline.data.datasets[0].data = data.data || [];
                    charts.timeline.update();
                }
            },
            error: function(xhr, status, error) {
                console.error('Error loading timeline chart:', error, xhr.responseText);
            }
        });

        // Activity Types Pie Chart
        $.ajax({
            url: baseUrl + 'reporting_dashboard/get_activity_type_pie',
            type: 'POST',
            data: filters,
            dataType: 'json',
            success: function(data) {
                console.log('Pie chart data:', data);
                if (charts.types) {
                    charts.types.data.labels = data.labels || [];
                    charts.types.data.datasets[0].data = data.data || [];
                    charts.types.update();
                }
            },
            error: function(xhr, status, error) {
                console.error('Error loading pie chart:', error, xhr.responseText);
            }
        });
    }

    function exportCSV() {
        if (activityTable) {
            var data = activityTable.rows().data().toArray();
            var csv = 'Date & Time,Staff Member,Activity Type,Related Entity,Description,IP Address\n';
            
            data.forEach(function(row) {
                csv += '"' + row.join('","') + '"\n';
            });
            
            downloadFile(csv, 'activity_log.csv', 'text/csv');
        }
    }

    function exportExcel() {
        if (activityTable) {
            var data = activityTable.rows().data().toArray();
            var headers = ['Date & Time', 'Staff Member', 'Activity Type', 'Related Entity', 'Description', 'IP Address'];
            
            var worksheet = [headers].concat(data);
            var csv = worksheet.map(function(row) {
                return row.join('\t');
            }).join('\n');
            
            downloadFile(csv, 'activity_log.xls', 'application/vnd.ms-excel');
        }
    }

    function printTable() {
        if (activityTable) {
            var printWindow = window.open('', '_blank');
            var tableHtml = $('#activity-log-table').parent().html();
            
            printWindow.document.write(`
                <html>
                <head>
                    <title>Activity Log Report</title>
                    <style>
                        body { font-family: Arial, sans-serif; }
                        table { border-collapse: collapse; width: 100%; }
                        th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
                        th { background-color: #f2f2f2; }
                    </style>
                </head>
                <body>
                    <h1>Activity Log Report</h1>
                    <p>Generated on: ${new Date().toLocaleString()}</p>
                    ${tableHtml}
                </body>
                </html>
            `);
            
            printWindow.document.close();
            printWindow.print();
        }
    }

    function downloadFile(content, filename, mimeType) {
        var blob = new Blob([content], { type: mimeType });
        var url = window.URL.createObjectURL(blob);
        var a = document.createElement('a');
        a.href = url;
        a.download = filename;
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
        window.URL.revokeObjectURL(url);
    }

})(jQuery);