<?php

defined('BASEPATH') or exit('No direct script access allowed');

/**
 * Staff Report Service
 *
 * Handles AI analysis and WhatsApp delivery for staff performance reports.
 */
class Staff_report_service
{
    /**
     * @var CI_Controller
     */
    protected $CI;

    /**
     * Cached Autograph model availability.
     *
     * @var bool|null
     */
    protected $autographModelLoaded = null;

    public function __construct()
    {
        $this->CI = &get_instance();

        if (!class_exists('Reporting_dashboard_model', false)) {
            $this->CI->load->model('reporting_dashboard/reporting_dashboard_model');
        }
    }

    /**
     * Analyze summary data with OpenAI, returning structured suggestions.
     *
     * @param array $summary
     * @param array $options
     * @return array
     */
    public function analyze_summary(array $summary, array $options = [])
    {
        $credentials = $this->get_openai_credentials();

        if (empty($credentials['api_key'])) {
            return [
                'success' => false,
                'error' => 'OpenAI API key is not configured. Add it in Autograph Feedback Insights settings or Reporting Dashboard options.',
                'analysis' => null
            ];
        }

        $prompt = $this->build_ai_prompt($summary, $options);

        return $this->call_openai_api($prompt, $credentials['model'], $credentials['api_key'], $options);
    }

    /**
     * Format WhatsApp message for the staff summary.
     *
     * @param array $summary
     * @param array|null $analysis
     * @return string
     */
    public function format_whatsapp_message(array $summary, $analysis = null)
    {
        $periodLabel = isset($summary['period']['label']) ? $summary['period']['label'] : '';
        $totals = $summary['totals'] ?? [];

        $message = "📞 *Staff Activity Performance*\n";
        if ($periodLabel !== '') {
            $message .= "📅 Period: {$periodLabel}\n";
        } else {
            $message .= "📅 Period: " . date('M j, Y', strtotime($summary['period']['from'] ?? 'now')) . ' - ' . date('M j, Y', strtotime($summary['period']['to'] ?? 'now')) . "\n";
        }

        $message .= "━━━━━━━━━━━━━━━━━━━━\n";
        $message .= "• Total activities: " . (int)($totals['total_activities'] ?? 0) . "\n";
        $message .= "• Phone/WA calls: " . (int)($totals['total_calls'] ?? 0) . " (" . number_format((float)($totals['call_ratio_overall'] ?? 0), 1) . "%)\n";
        $message .= "• Avg activities/staff: " . number_format((float)($totals['average_per_staff'] ?? 0), 2) . "\n";
        if (!empty($summary['followup_totals'])) {
            $message .= "• Activity Mix: " . implode(', ', $this->format_followup_mix($summary['followup_totals'], 4)) . "\n";
        }
        
        // Add reminders status
        if (!empty($summary['reminders_followups']['totals'])) {
            $remTotals = $summary['reminders_followups']['totals'];
            $overdueCount = (int)($remTotals['total_overdue'] ?? 0);
            $upcomingCount = (int)($remTotals['total_upcoming'] ?? 0);
            if ($overdueCount > 0 || $upcomingCount > 0) {
                $message .= "• Reminders: ";
                if ($overdueCount > 0) {
                    $message .= "⚠️ {$overdueCount} overdue";
                }
                if ($upcomingCount > 0) {
                    if ($overdueCount > 0) $message .= ", ";
                    $message .= "📅 {$upcomingCount} upcoming (7d)";
                }
                $message .= "\n";
            }
        }
        
        $message .= "━━━━━━━━━━━━━━━━━━━━\n";

        if (!empty($analysis) && !empty($analysis['overview'])) {
            $message .= "🧠 *AI Overview*\n" . $this->truncate_line($analysis['overview'], 380) . "\n";
        }

        if (!empty($analysis['pros'])) {
            $message .= "✅ *Strengths*\n";
            foreach (array_slice($analysis['pros'], 0, 3) as $item) {
                $message .= "• " . $this->truncate_line($item, 300) . "\n";
            }
        }

        if (!empty($analysis['cons'])) {
            $message .= "⚠️ *Risks*\n";
            foreach (array_slice($analysis['cons'], 0, 3) as $item) {
                $message .= "• " . $this->truncate_line($item, 300) . "\n";
            }
        }

        if (!empty($analysis['recommendations'])) {
            $message .= "💡 *Recommended Actions*\n";
            foreach (array_slice($analysis['recommendations'], 0, 3) as $item) {
                $message .= "• " . $this->truncate_line($item, 300) . "\n";
            }
        }

        $message .= "━━━━━━━━━━━━━━━━━━━━\n";

        $staffRows = $summary['staff'] ?? [];
        if (!empty($analysis['highlights'])) {
            $highlightsMap = [];
            foreach ($analysis['highlights'] as $highlight) {
                if (!empty($highlight['staff'])) {
                    $highlightsMap[$highlight['staff']] = $highlight;
                }
            }
        } else {
            $highlightsMap = [];
        }

        $maxStaffLines = 6;
        $count = 0;

        // Create a map of staff reminders for quick lookup
        $staffRemindersMap = [];
        if (!empty($summary['reminders_followups']['staff_summary'])) {
            foreach ($summary['reminders_followups']['staff_summary'] as $remStaff) {
                $staffRemindersMap[$remStaff['staff_id']] = $remStaff;
            }
        }

        foreach ($staffRows as $row) {
            if ($count >= $maxStaffLines) {
                $message .= "• ...\n";
                break;
            }

            $staffId = $row['staff_id'];
            $remInfo = $staffRemindersMap[$staffId] ?? null;
            
            $line = sprintf(
                "%s — %d activities (%.2f/day) • %d calls (%.1f%%)",
                $row['staff_name'],
                (int)$row['total_activities'],
                (float)$row['activities_per_day'],
                (int)$row['total_calls'],
                (float)$row['call_ratio']
            );

            $message .= "• " . $line . "\n";

            if (!empty($row['followup_counts'])) {
                $mixLines = $this->format_followup_mix($row['followup_counts'], 3);
                if (!empty($mixLines)) {
                    $message .= "    ↳ Types: " . implode(', ', $mixLines) . "\n";
                }
            }
            
            // Add reminder status if exists
            if ($remInfo) {
                $overdueCount = (int)$remInfo['overdue_count'];
                $upcomingCount = (int)$remInfo['upcoming_count'];
                if ($overdueCount > 0 || $upcomingCount > 0) {
                    $remLine = "    ↳ Reminders: ";
                    if ($overdueCount > 0) {
                        $remLine .= "⚠️ {$overdueCount} overdue";
                    }
                    if ($upcomingCount > 0) {
                        if ($overdueCount > 0) $remLine .= ", ";
                        $remLine .= "📅 {$upcomingCount} due soon";
                    }
                    $message .= $remLine . "\n";
                }
            }

            if (!empty($highlightsMap[$row['staff_name']]['insight'])) {
                $message .= "    ↳ " . $this->truncate_line($highlightsMap[$row['staff_name']]['insight'], 300) . "\n";
            } elseif (!empty($row['last_activity'])) {
                $message .= "    ↳ Last activity: " . date('M j, g:i a', strtotime($row['last_activity'])) . "\n";
            }

            $count++;
        }

        $message .= "━━━━━━━━━━━━━━━━━━━━\n";
        $message .= "_Automated via Reporting Dashboard_\n";

        return $this->enforce_whatsapp_limit($message);
    }

    /**
     * Send a text message to multiple WhatsApp recipients via Meta API.
     *
     * @param string $message
     * @param array  $numbers
     * @return array
     */
    public function send_whatsapp_message($message, array $numbers)
    {
        $results = [
            'success' => false,
            'sent' => 0,
            'failed' => 0,
            'results' => [],
            'error' => null,
        ];

        $numbers = $this->normalize_numbers($numbers);

        if (empty($numbers)) {
            $results['error'] = 'No valid WhatsApp numbers provided.';
            return $results;
        }

        $phoneNumberId = get_option('phone_number_id');
        $accessToken = get_option('whatsapp_access_token');

        if (empty($phoneNumberId) || empty($accessToken)) {
            $results['error'] = 'WhatsApp API credentials are missing. Configure them in Autograph Feedback Insights settings.';
            return $results;
        }

        $endpoint = "https://graph.facebook.com/v18.0/{$phoneNumberId}/messages";

        foreach ($numbers as $number) {
            $payload = [
                'messaging_product' => 'whatsapp',
                'recipient_type' => 'individual',
                'to' => $number,
                'type' => 'text',
                'text' => [
                    'preview_url' => false,
                    'body' => $message,
                ],
            ];

            $ch = curl_init($endpoint);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Content-Type: application/json',
                'Authorization: Bearer ' . $accessToken,
            ]);

            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $curlError = curl_error($ch);
            curl_close($ch);

            if ($httpCode >= 200 && $httpCode < 300 && !$curlError) {
                $results['sent']++;
                $results['results'][] = [
                    'number' => $number,
                    'status' => 'sent',
                    'http_code' => $httpCode,
                ];
            } else {
                $results['failed']++;
                $errorMessage = $curlError ?: ($this->extract_whatsapp_error($response) ?: 'Unknown error');
                $results['results'][] = [
                    'number' => $number,
                    'status' => 'failed',
                    'http_code' => $httpCode,
                    'error' => $errorMessage,
                    'response' => $response,
                ];

                if ($results['error'] === null) {
                    $results['error'] = $errorMessage;
                }
            }
        }

        $results['success'] = $results['sent'] > 0 && $results['failed'] === 0;

        return $results;
    }

    /**
     * Build AI prompt from staff summary.
     *
     * @param array $summary
     * @param array $options
     * @return string
     */
    protected function build_ai_prompt(array $summary, array $options = [])
    {
        $lines = [];
        $periodLabel = isset($summary['period']['label']) ? $summary['period']['label'] : '';

        $lines[] = 'Analyze the following staff activity metrics from a sales CRM.';
        $lines[] = 'Provide insights on productivity, call effectiveness, and coaching opportunities.';
        $lines[] = 'Respond strictly in JSON format with keys: overview (string), pros (array of strings), cons (array of strings), recommendations (array of strings), highlights (array of objects with staff, insight, metric_summary).';
        $lines[] = '';

        if ($periodLabel !== '') {
            $lines[] = 'Period: ' . $periodLabel;
        } else {
            $lines[] = 'Period: ' . ($summary['period']['from'] ?? '') . ' to ' . ($summary['period']['to'] ?? '');
        }

        $totals = $summary['totals'] ?? [];
        $lines[] = 'Overall Totals:';
        $lines[] = sprintf(
            '- Total activities: %d; Total calls: %d; Avg activities per staff: %.2f; Call ratio: %.1f%%; Staff count: %d',
            (int)($totals['total_activities'] ?? 0),
            (int)($totals['total_calls'] ?? 0),
            (float)($totals['average_per_staff'] ?? 0),
            (float)($totals['call_ratio_overall'] ?? 0),
            (int)($totals['staff_count'] ?? 0)
        );
        if (!empty($summary['followup_totals'])) {
            $lines[] = 'Follow-up mix (overall): ' . implode(', ', $this->format_followup_mix($summary['followup_totals'], 5));
        }
        
        // Add reminders/followups accountability data
        if (!empty($summary['reminders_followups'])) {
            $remData = $summary['reminders_followups'];
            $remTotals = $remData['totals'] ?? [];
            $lines[] = '';
            $lines[] = 'Reminders & Follow-ups Status:';
            $lines[] = sprintf(
                '- Overdue reminders: %d; Upcoming reminders (7 days): %d; Staff with overdue: %d',
                (int)($remTotals['total_overdue'] ?? 0),
                (int)($remTotals['total_upcoming'] ?? 0),
                (int)($remTotals['staff_with_overdue'] ?? 0)
            );
        }
        
        $lines[] = '';
        $lines[] = 'Staff Breakdown:';

        // Create a map of staff reminders for quick lookup
        $staffRemindersMap = [];
        if (!empty($summary['reminders_followups']['staff_summary'])) {
            foreach ($summary['reminders_followups']['staff_summary'] as $remStaff) {
                $staffRemindersMap[$remStaff['staff_id']] = $remStaff;
            }
        }

        foreach ($summary['staff'] as $index => $row) {
            $staffId = $row['staff_id'];
            $remInfo = $staffRemindersMap[$staffId] ?? null;
            
            $overdueCount = $remInfo ? (int)$remInfo['overdue_count'] : 0;
            $upcomingCount = $remInfo ? (int)$remInfo['upcoming_count'] : 0;
            
            $lines[] = sprintf(
                '%d. %s — activities: %d, calls: %d, call ratio: %.1f%%, activities/day: %.2f, unique leads: %d, last activity: %s, overdue reminders: %d, upcoming: %d',
                $index + 1,
                $row['staff_name'],
                (int)$row['total_activities'],
                (int)$row['total_calls'],
                (float)$row['call_ratio'],
                (float)$row['activities_per_day'],
                (int)$row['unique_leads'],
                $row['last_activity'] ?? 'N/A',
                $overdueCount,
                $upcomingCount
            );
            if (!empty($row['followup_counts'])) {
                $lines[] = '   Follow-up mix: ' . implode(', ', $this->format_followup_mix($row['followup_counts'], 4));
            }
        }

        if (!empty($options['notes'])) {
            $lines[] = '';
            $lines[] = 'Additional Context: ' . $options['notes'];
        }

        return implode("\n", $lines);
    }

    /**
     * Perform OpenAI API call and parse JSON response safely.
     *
     * @param string $prompt
     * @param string $model
     * @param string $apiKey
     * @param array  $options
     * @return array
     */
    protected function call_openai_api($prompt, $model, $apiKey, array $options = [])
    {
        $model = $model ?: 'gpt-4o-mini';
        $temperature = isset($options['temperature']) ? (float)$options['temperature'] : 0.4;
        $maxTokens = isset($options['max_tokens']) ? (int)$options['max_tokens'] : 900;

        $payload = [
            'model' => $model,
            'messages' => [
                [
                    'role' => 'system',
                    'content' => 'You are a revenue operations analyst. Always respond with valid JSON and provide measurable, objective insights.'
                ],
                [
                    'role' => 'user',
                    'content' => $prompt
                ],
            ],
            'temperature' => $temperature,
            'max_tokens' => $maxTokens,
        ];

        $ch = curl_init('https://api.openai.com/v1/chat/completions');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Authorization: ' . 'Bearer ' . $apiKey,
        ]);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);

        if ($error) {
            return [
                'success' => false,
                'error' => 'OpenAI request failed: ' . $error,
                'analysis' => null,
            ];
        }

        if ($httpCode !== 200) {
            $errorPayload = json_decode($response, true);
            $message = $errorPayload['error']['message'] ?? 'Unexpected OpenAI response.';
            return [
                'success' => false,
                'error' => 'OpenAI error (' . $httpCode . '): ' . $message,
                'analysis' => null,
                'raw' => $response,
            ];
        }

        $data = json_decode($response, true);
        $content = $data['choices'][0]['message']['content'] ?? '';

        if (preg_match('/```(?:json)?\s*(\{.*?\})\s*```/s', $content, $matches)) {
            $content = $matches[1];
        }

        $analysis = json_decode($content, true);
        if (json_last_error() !== JSON_ERROR_NONE || !is_array($analysis)) {
            return [
                'success' => false,
                'error' => 'Failed to parse AI output as JSON: ' . json_last_error_msg(),
                'analysis' => null,
                'raw' => $content,
            ];
        }

        // Ensure expected keys exist
        $analysis += [
            'overview' => '',
            'pros' => [],
            'cons' => [],
            'recommendations' => [],
            'highlights' => [],
        ];

        return [
            'success' => true,
            'analysis' => $analysis,
            'model' => $model,
        ];
    }

    /**
     * Obtain OpenAI credentials with fallback.
     *
     * @return array
     */
    protected function get_openai_credentials()
    {
        $apiKey = null;
        $model = null;

        if ($this->autographModelLoaded !== false) {
            try {
                if ($this->autographModelLoaded === null) {
                    $this->CI->load->model('autograph_feedback_insights/autograph_feedback_insights_model');
                    $this->autographModelLoaded = true;
                }

                if (isset($this->CI->autograph_feedback_insights_model)) {
                    $apiKey = $this->CI->autograph_feedback_insights_model->get_setting('openai_api_key');
                    $model = $this->CI->autograph_feedback_insights_model->get_setting('openai_model');
                }
            } catch (Exception $e) {
                log_message('error', 'Staff_report_service: failed loading Autograph Feedback Insights model - ' . $e->getMessage());
                $this->autographModelLoaded = false;
            } catch (Error $e) {
                log_message('error', 'Staff_report_service: Autograph Feedback Insights model missing - ' . $e->getMessage());
                $this->autographModelLoaded = false;
            }
        }

        if (empty($apiKey)) {
            $apiKey = get_option('reporting_dashboard_openai_api_key');
        }

        if (empty($model)) {
            $model = get_option('reporting_dashboard_openai_model') ?: 'gpt-4o-mini';
        }

        return [
            'api_key' => $apiKey,
            'model' => $model,
        ];
    }

    /**
     * Extract error message from WhatsApp API response.
     *
     * @param string $response
     * @return string|null
     */
    protected function extract_whatsapp_error($response)
    {
        if (empty($response)) {
            return null;
        }

        $decoded = json_decode($response, true);
        if (!is_array($decoded)) {
            return null;
        }

        return $decoded['error']['message'] ?? null;
    }

    /**
     * Keep message below WhatsApp limit (4096 characters).
     *
     * @param string $message
     * @return string
     */
    protected function enforce_whatsapp_limit($message)
    {
        $limit = 4000;
        if (strlen($message) <= $limit) {
            return $message;
        }

        return substr($message, 0, $limit - 20) . "\n... (truncated)";
    }

    /**
     * Shorten a single line to a safe length.
     *
     * @param string $text
     * @param int    $limit
     * @return string
     */
    protected function truncate_line($text, $limit = 320)
    {
        $text = trim((string)$text);
        if (strlen($text) <= $limit) {
            return $text;
        }

        return substr($text, 0, $limit - 3) . '...';
    }

    /**
     * Format follow-up distribution lines for messaging/AI.
     *
     * @param array $counts
     * @param int   $limit
     * @return array
     */
    protected function format_followup_mix(array $counts, $limit = 3)
    {
        if (empty($counts)) {
            return [];
        }

        arsort($counts);
        $mix = [];
        $total = array_sum(array_map('intval', $counts));

        if ($total <= 0) {
            return [];
        }

        $shown = 0;
        $usedTotal = 0;
        foreach ($counts as $type => $count) {
            $count = (int)$count;
            if ($count <= 0) {
                continue;
            }

            $percent = $total > 0 ? round(($count / $total) * 100, 1) : 0;
            $mix[] = $this->truncate_line(sprintf('%s %d (%.1f%%)', $type, $count, $percent), 60);
            $usedTotal += $count;
            $shown++;

            if ($shown >= $limit) {
                break;
            }
        }

        if ($usedTotal < $total) {
            $remaining = $total - $usedTotal;
            if ($remaining > 0) {
                $mix[] = $this->truncate_line(sprintf('Other %d', $remaining), 60);
            }
        }

        return $mix;
    }

    /**
     * Normalize WhatsApp numbers for API delivery.
     *
     * @param array|string $numbers
     * @return array
     */
    protected function normalize_numbers($numbers)
    {
        if (empty($numbers)) {
            return [];
        }

        if (is_string($numbers)) {
            $numbers = preg_split('/[\s,;]+/', $numbers);
        }

        if (!is_array($numbers)) {
            return [];
        }

        $normalized = [];

        foreach ($numbers as $number) {
            if ($number === null) {
                continue;
            }

            $clean = preg_replace('/[^0-9+]/', '', (string)$number);
            if ($clean === '') {
                continue;
            }

            if ($clean[0] !== '+' && strlen($clean) >= 10) {
                $clean = '+' . $clean;
            }

            $normalized[] = $clean;
        }

        return array_values(array_unique($normalized));
    }
}


