<?php
// Process test upload from the debug script
error_reporting(E_ALL);
ini_set('display_errors', 1);

echo '<style>
    body { font-family: Arial, sans-serif; margin: 20px; }
    h1, h2, h3 { margin-top: 20px; }
    .success { color: green; }
    .warning { color: orange; }
    .error { color: red; }
    .info { background: #f4f4f4; padding: 10px; margin: 10px 0; }
    pre { background: #f8f8f8; padding: 10px; overflow-x: auto; }
</style>';

echo '<h1>Lead Note Upload Test</h1>';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    die('<p class="error">This script should be accessed via the form submission.</p>');
}

// Check if we have the required post data
if (!isset($_POST['lead_id']) || !isset($_POST['note_content']) || !isset($_FILES['note_attachment'])) {
    die('<p class="error">Missing required form data.</p>');
}

$lead_id = (int)$_POST['lead_id'];
$note_content = $_POST['note_content'];

echo '<h2>Upload Parameters</h2>';
echo '<ul>';
echo '<li>Lead ID: ' . htmlspecialchars($lead_id) . '</li>';
echo '<li>Note Content: ' . htmlspecialchars($note_content) . '</li>';
echo '<li>File: ' . htmlspecialchars($_FILES['note_attachment']['name']) . ' (' . $_FILES['note_attachment']['size'] . ' bytes)</li>';
echo '</ul>';

// Define helper function to check upload paths
function check_upload_path($type) {
    // Simple implementation to check path by type
    $path = '';
    
    // Check for our constants first
    if ($type === 'lead_note' && defined('LEAD_NOTES_FOLDER')) {
        return LEAD_NOTES_FOLDER;
    }
    
    // Fallback check based on common structures
    switch ($type) {
        case 'lead_note':
            $path = __DIR__ . '/uploads/leads/notes/';
            break;
        case 'note':
            $path = __DIR__ . '/uploads/notes/';
            break;
    }
    
    return $path;
}

// Let's see what paths we would try
echo '<h2>Checking Possible Upload Paths</h2>';

// Dummy implementation of the upload process
echo '<div class="info">';
echo '<strong>Step 1:</strong> We would normally call get_upload_path_by_type('lead_note')...<br>';

$path_lead_note = check_upload_path('lead_note');
$path_note = check_upload_path('note');

echo '<br>Path for "lead_note": ' . ($path_lead_note ?: '<span class="error">EMPTY</span>') . '<br>';
echo 'Path for "note": ' . ($path_note ?: '<span class="error">EMPTY</span>') . '<br>';

// Check if the directories exist and are writable
$lead_note_dir = $path_lead_note . $lead_id;
$note_dir = $path_note . $lead_id;

echo '<br><strong>Step 2:</strong> We would check/create the directory...<br>';

echo '<br>Directory for lead_note: ' . $lead_note_dir;
if (file_exists($lead_note_dir)) {
    echo ' - <span class="success">EXISTS</span>';
    echo ' - ' . (is_writable($lead_note_dir) ? '<span class="success">Writable</span>' : '<span class="error">Not Writable</span>');
} else {
    echo ' - <span class="warning">DOES NOT EXIST</span>';
    
    // Try to create it
    echo '<br>Attempting to create directory: ' . $lead_note_dir . '<br>';
    $create_result = @mkdir($lead_note_dir, 0755, true);
    echo $create_result ? '<span class="success">Directory created!</span>' : '<span class="error">Failed to create directory!</span>';
}

echo '<br><br>Directory for note: ' . $note_dir;
if (file_exists($note_dir)) {
    echo ' - <span class="success">EXISTS</span>';
    echo ' - ' . (is_writable($note_dir) ? '<span class="success">Writable</span>' : '<span class="error">Not Writable</span>');
} else {
    echo ' - <span class="warning">DOES NOT EXIST</span>';
}

echo '<br><br><strong>Step 3:</strong> We would upload the file...<br>';

// Try to save the file to each directory
$lead_note_file_path = $lead_note_dir . '/' . $_FILES['note_attachment']['name'];
$note_file_path = $note_dir . '/' . $_FILES['note_attachment']['name'];

echo '<br>Attempting to save to lead_note path: ' . $lead_note_file_path . '<br>';
$lead_note_upload_result = move_uploaded_file($_FILES['note_attachment']['tmp_name'], $lead_note_file_path);

if ($lead_note_upload_result) {
    echo '<span class="success">File saved successfully!</span><br>';
    echo '<strong>File URL would be:</strong> http://' . $_SERVER['HTTP_HOST'] . '/' . basename(__DIR__) . '/uploads/leads/notes/' . $lead_id . '/' . htmlspecialchars($_FILES['note_attachment']['name']) . '<br>';
} else {
    echo '<span class="error">Failed to save file!</span><br>';
    echo 'Error code: ' . $_FILES['note_attachment']['error'] . '<br>';
}

echo '<br><strong>Step 4:</strong> We would update the database...<br>';

// Connect to the database
$config_file = 'application/config/database.php';

if (!file_exists($config_file)) {
    die('<p class="error">Database config file not found</p>');
}

// Parse config file manually to avoid CodeIgniter dependency
$config_content = file_get_contents($config_file);
preg_match("/\['hostname'\]\s*=\s*'([^']+)'/", $config_content, $host_match);
preg_match("/\['username'\]\s*=\s*'([^']+)'/", $config_content, $user_match);
preg_match("/\['password'\]\s*=\s*'([^']+)'/", $config_content, $pass_match);
preg_match("/\['database'\]\s*=\s*'([^']+)'/", $config_content, $db_match);
preg_match("/\['dbprefix'\]\s*=\s*'([^']+)'/", $config_content, $prefix_match);

$host = $host_match[1] ?? 'localhost';
$username = $user_match[1] ?? 'root';
$password = $pass_match[1] ?? '';
$database = $db_match[1] ?? 'crm';
$db_prefix = $prefix_match[1] ?? '';

try {
    $pdo = new PDO("mysql:host=$host;dbname=$database;charset=utf8", $username, $password);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    echo '<p class="success">Database connection successful!</p>';
    
    // Start transaction
    $pdo->beginTransaction();
    
    // 1. Create the note first
    $stmt = $pdo->prepare("INSERT INTO {$db_prefix}notes (rel_id, rel_type, description, date_contacted, addedfrom, dateadded, has_attachment) 
                          VALUES (?, ?, ?, NOW(), 1, NOW(), 1)");
    $stmt->execute([$lead_id, 'lead', $note_content]);
    
    $note_id = $pdo->lastInsertId();
    echo '<p>Note created with ID: ' . $note_id . '</p>';
    
    // 2. Add the file record
    if ($lead_note_upload_result) {
        $stmt = $pdo->prepare("INSERT INTO {$db_prefix}files (rel_id, rel_type, file_name, filetype, dateadded) 
                              VALUES (?, ?, ?, ?, NOW())");
        
        // Here is the critical part - we need to use the same rel_type as the Leads model
        // But we don't know for sure if it uses 'lead_note' or 'note'
        $rel_type = 'lead_note'; // This is our best guess
        
        $stmt->execute([$note_id, $rel_type, $_FILES['note_attachment']['name'], $_FILES['note_attachment']['type']]);
        
        echo '<p class="success">File record added to database with rel_type "' . $rel_type . '"</p>';
    }
    
    // Commit transaction
    $pdo->commit();
    
    echo '<h2>Test Completed Successfully</h2>';
    echo '<p>The note with attachment has been added. You can now check if it appears correctly in the interface.</p>';
    
    // Add a link back to the debug script
    echo '<p><a href="debug_lead_note_attachments.php?lead_id=' . $lead_id . '">Back to Debug Script</a></p>';
    
} catch (Exception $e) {
    // Roll back the transaction
    if (isset($pdo)) $pdo->rollBack();
    
    echo '<p class="error">Database error: ' . $e->getMessage() . '</p>';
}

echo '</div>';
?>