<?php

defined('BASEPATH') || exit('No direct script access allowed');

class Whatsapp_api_model extends App_Model
{

    private $whatsapp_api_log_table;
    private $scheduled_whatsapp_messages;

    public function __construct()
    {
        parent::__construct();
        $this->whatsapp_api_log_table = db_prefix() . 'whatsapp_api_debug_log';
        $this->scheduled_whatsapp_messages = db_prefix() . 'scheduled_whatsapp_messages';
    }

    public function get_template_data($id)
    {
        return $this->db->get_where(db_prefix() . 'whatsapp_templates', ['id' => $id])->row();
    }

    public function save_template_map_info($map_info)
    {
        modules\whatsapp_api\core\Apiinit::ease_of_mind('whatsapp_api');
        $result = $this->db->insert(db_prefix() . 'whatsapp_templates_mapping', $map_info);
        if($result){
            return $this->db->insert_id();
        }
        return false;
    }

    public function update_template_map_info($map_info, $where)
    {
        return $this->db->update(db_prefix() . 'whatsapp_templates_mapping', $map_info, $where);
    }

    public function delete_whatsapp_templates_mapping($id)
    {
        return $this->db->delete(db_prefix() . 'whatsapp_templates_mapping', $id);
    }

    public function get_mapping_data($where)
    {
        modules\whatsapp_api\core\Apiinit::ease_of_mind('whatsapp_api');
        return $this->db
            ->select(db_prefix() . 'whatsapp_templates_mapping.*')
            ->select('wt.template_name, wt.language, wt.header_data_format, wt.header_data_text, wt.body_data, wt.footer_data, wt.buttons_data, wt.header_params_count, wt.body_params_count, wt.footer_params_count, whatsapp_templates_mapping.send_to')
            ->join(db_prefix() . 'whatsapp_templates wt', db_prefix() . 'whatsapp_templates_mapping.template_id = wt.id')
            ->get_where(db_prefix() . 'whatsapp_templates_mapping', $where)->result();
    }    /**
     * Get upcoming reminders that need WhatsApp notification
     * @return array
     */    public function get_upcoming_reminders()
    {
        $current_time = date('Y-m-d H:i:s');
        $past_time = date('Y-m-d H:i:s', strtotime('-2 hours')); // Look back 2 hours for recently due reminders
        
        $this->db->select('r.*, s.firstname, s.lastname, s.phonenumber');
        $this->db->from(db_prefix() . 'reminders r');
        $this->db->join(db_prefix() . 'staff s', 's.staffid = r.staff');
        $this->db->where('r.date >=', $past_time); // Include recently due reminders
        $this->db->where('r.date <=', $current_time); // Up to current time (due now)
        $this->db->where('(r.whatsapp_notified IS NULL OR r.whatsapp_notified = 0)');
        $this->db->where('s.phonenumber !=', '');
        $this->db->where('s.phonenumber IS NOT NULL');
        $this->db->order_by('r.date', 'ASC');
        
        return $this->db->get()->result_array();
    }/**
     * Mark reminder as WhatsApp notified with atomic check
     * @param int $reminder_id
     * @return bool
     */
    public function mark_reminder_whatsapp_notified($reminder_id)
    {
        // Use atomic update to prevent race conditions
        $this->db->where('id', $reminder_id);
        $this->db->where('(whatsapp_notified IS NULL OR whatsapp_notified = 0)'); // Only update if not already notified
        return $this->db->update(
            db_prefix() . 'reminders',
            ['whatsapp_notified' => 1, 'whatsapp_notified_at' => date('Y-m-d H:i:s')]
        );
    }

    /**
     * Check if reminder has already been WhatsApp notified
     * @param int $reminder_id
     * @return bool
     */
    public function is_reminder_whatsapp_notified($reminder_id)
    {
        $this->db->select('whatsapp_notified');
        $this->db->where('id', $reminder_id);
        $result = $this->db->get(db_prefix() . 'reminders')->row();
        
        return $result && $result->whatsapp_notified == 1;
    }

    /**
     * Add WhatsApp notification column to reminders table if it doesn't exist
     */
    public function add_whatsapp_notification_columns()
    {
        if (!$this->db->field_exists('whatsapp_notified', db_prefix() . 'reminders')) {
            $fields = [
                'whatsapp_notified' => [
                    'type' => 'TINYINT',
                    'constraint' => 1,
                    'default' => 0,
                    'null' => false
                ],
                'whatsapp_notified_at' => [
                    'type' => 'DATETIME',
                    'null' => true
                ]
            ];
            
            $this->dbforge->add_column(db_prefix() . 'reminders', $fields);
        }
    }

    public function clear_webhook_log()
    {
        if ($this->db->truncate($this->whatsapp_api_log_table)) {
            return true;
        }

        return false;
    }

    public function get_whatsapp_api_log_info($id)
    {
        return $this->db->get_where(db_prefix() . 'whatsapp_api_debug_log', ['id' => $id])->row();
    }

    public function change_whatsapp_template_status($data, $where)
    {
        if ($this->db->update(db_prefix() . 'whatsapp_templates_mapping', $data, $where)) {
            return true;
        }

        return false;
    }

    public function add_request_log($data)
    {
        return $this->db->insert(db_prefix() . 'whatsapp_api_debug_log', $data);
    }

    public function updateWhatsappAuthStatus($data)
    {
        $status = $data['whatsapp_auth_enabled'];

        if ($status == '1') {
            $data['two_factor_auth_enabled'] = '0';
        }

        return $this->db->update(db_prefix() . 'staff', $data, ['staffid' => get_staff_user_id()]);
    }

    public function isWhatsappAuthCodeValid($code, $email)
    {
        $this->db->select('whatsapp_auth_code_requested');
        $this->db->where('whatsapp_auth_code', $code);
        $this->db->where('email', $email);
        $user = $this->db->get(db_prefix() . 'staff')->row();

        if (!$user) {
            return false;
        }

        $timestamp_minus_1_hour = time() - (60 * 60);
        $new_code_key_requested = strtotime($user->whatsapp_auth_code_requested);
        
        if ($timestamp_minus_1_hour > $new_code_key_requested) {
            return false;
        }

        return true;
    }

    public function getUserByWhatsappAuthCode($code)
    {
        $this->db->where('whatsapp_auth_code', $code);

        return $this->db->get(db_prefix() . 'staff')->row();
    }

    public function clearWhatsappAuthCode($id)
    {
        $this->db->where('staffid', $id);
        $this->db->update(db_prefix() . 'staff', [
            'whatsapp_auth_code' => null,
        ]);

        return true;
    }

    public function whatsappAuthLogin($user)
    {
        $this->session->set_userdata([
            'staff_user_id'   => $user->staffid,
            'staff_logged_in' => true,
        ]);

        $remember = null;
        if ($this->session->has_userdata('tfa_remember')) {
            $remember = true;
            $this->session->unset_userdata('tfa_remember');
        }

        if ($remember) {
            $this->create_autologin($user->staffid, true);
        }

        $this->update_login_info($user->staffid, true);

        return true;
    }

    private function create_autologin($user_id, $staff)
    {
        $this->load->helper('cookie');
        $key = substr(md5(uniqid(rand() . get_cookie($this->config->item('sess_cookie_name')))), 0, 16);
        $this->user_autologin->delete($user_id, $key, $staff);
        if ($this->user_autologin->set($user_id, md5($key), $staff)) {
            set_cookie([
                'name'  => 'autologin',
                'value' => serialize([
                    'user_id' => $user_id,
                    'key'     => $key,
                ]),
                'expire' => 60 * 60 * 24 * 31 * 2, // 2 months
            ]);

            return true;
        }

        return false;
    }

    private function update_login_info($user_id, $staff)
    {
        $table = db_prefix() . 'contacts';
        $_id   = 'id';
        if ($staff == true) {
            $table = db_prefix() . 'staff';
            $_id   = 'staffid';
        }
        $this->db->set('last_ip', $this->input->ip_address());
        $this->db->set('last_login', date('Y-m-d H:i:s'));
        $this->db->where($_id, $user_id);
        $this->db->update($table);

        log_activity('User Successfully Logged In [User Id: ' . $user_id . ', Is Staff Member: ' . ($staff == true ? 'Yes' : 'No') . ', IP: ' . $this->input->ip_address() . ']');
    }

    public function delete_scheduled_call($where)
    {
        return $this->db->delete($this->scheduled_whatsapp_messages, $where);
    }

    public function copy_mapped_template($data) {
        $_data = reset($data);
        $copy_data = [
            'template_id'   => $_data->template_id,
            'category'      => $_data->category,
            'send_to'       => $_data->send_to,
            'header_params' => $_data->header_params,
            'body_params'   => $_data->body_params,
            'footer_params' => $_data->footer_params,
            'active'        => $_data->active,
            'debug_mode'    => $_data->debug_mode,
        ];
        return $this->save_template_map_info($copy_data);
    }

    /**
     * Get recent WhatsApp reminder activity from logs
     * @param int $limit
     * @return array
     */
    public function get_recent_activity($limit = 20)
    {
        $this->db->select('l.*, s.firstname, s.lastname');
        $this->db->from($this->whatsapp_api_log_table . ' l');
        $this->db->join(db_prefix() . 'staff s', 's.staffid = l.staff_id', 'left');
        $this->db->where('l.message_category', 'staff_reminder');
        $this->db->order_by('l.created_at', 'DESC');
        $this->db->limit($limit);
        
        return $this->db->get()->result_array();
    }

    /**
     * Process reminders safely with limits to prevent spam
     * @param array $reminders
     * @param string $source
     * @return array
     */
    public function process_reminders_safely($reminders, $source = 'unknown')
    {
        $processed = 0;
        $skipped = 0;
        $max_per_run = 3; // Limit to 3 notifications per execution
        $processed_today = $this->get_whatsapp_notifications_sent_today();
        $daily_limit = 50; // Daily limit to prevent spam
        
        if ($processed_today >= $daily_limit) {
            log_activity("WhatsApp reminder daily limit reached ($daily_limit). Skipping processing.");
            return ['processed' => 0, 'skipped' => count($reminders), 'reason' => 'daily_limit'];
        }
        
        foreach ($reminders as $reminder) {
            if ($processed >= $max_per_run) {
                $skipped += count($reminders) - ($processed + $skipped);
                break;
            }            // Double-check that this reminder hasn't been WhatsApp notified
            if (!$this->is_reminder_whatsapp_notified($reminder['id'])) {
                try {
                    $CI = &get_instance();
                    // Send WhatsApp notification using template
                    $CI->whatsapp_api_lib->send_mapped_template('staff_reminder', false, 'MANUAL', $reminder['id']);
                    
                    // Mark as WhatsApp notified (atomic operation)
                    $marked = $this->mark_reminder_whatsapp_notified($reminder['id']);
                    
                    if ($marked) {
                        $processed++;
                        log_activity("WhatsApp reminder notification sent ($source) to: " . $reminder['firstname'] . ' ' . $reminder['lastname']);
                    } else {
                        $skipped++;
                    }
                } catch (Exception $e) {
                    log_activity("Failed to send WhatsApp reminder ($source): " . $e->getMessage());
                    $skipped++;
                }
            } else {
                $skipped++;
            }
        }
        
        return ['processed' => $processed, 'skipped' => $skipped];
    }

    /**
     * Get count of WhatsApp notifications sent today
     * @return int
     */
    public function get_whatsapp_notifications_sent_today()
    {
        $this->db->where('whatsapp_notified', 1);
        $this->db->where('DATE(whatsapp_notified_at)', date('Y-m-d'));
        return $this->db->count_all_results(db_prefix() . 'reminders');
    }
}
