<?php
/**
 * UPLOAD INSTRUCTIONS FOR CPANEL
 * ================================
 * 
 * 1. Upload this file to: application/helpers/facebook_helper.php
 * 
 * 2. Set file permissions to 644:
 *    - Right-click file in cPanel File Manager
 *    - Click "Change Permissions"
 *    - Set to: 644 (rw-r--r--)
 * 
 * 3. Verify encoding:
 *    - File must be UTF-8 WITHOUT BOM
 *    - If using Notepad++: Encoding → Convert to UTF-8 without BOM
 *    - If using VS Code: Save with encoding "UTF-8"
 * 
 * 4. Upload method:
 *    - Use cPanel File Manager (RECOMMENDED)
 *    - OR use FTP in TEXT/ASCII mode (NOT binary)
 * 
 * 5. After upload:
 *    - Clear cache: Delete all files in application/cache/
 *    - Test: Access your CRM admin panel
 * 
 * TROUBLESHOOTING:
 * If still getting "Unable to load" error:
 * - Check file is exactly at: /public_html/crm/application/helpers/facebook_helper.php
 * - Verify permissions are 644
 * - Check there's no .txt extension (should be .php only)
 * - Verify first line is exactly: <?php
 * - No spaces or characters before <?php
 */

defined('BASEPATH') or exit('No direct script access allowed');

/**
 * Facebook Helper Functions - cPanel Compatible Version
 * Version: 2.1 (cPanel Optimized)
 * Last Updated: October 10, 2025
 */

if (!function_exists('facebook_api_request')) {
    /**
     * Make a request to Facebook Graph API
     * @param string $endpoint API endpoint
     * @param array $params Query parameters
     * @param string $access_token Access token
     * @param string $method HTTP method
     * @return array Response data
     */
    function facebook_api_request($endpoint, $params = [], $access_token = '', $method = 'GET')
    {
        $base_url = 'https://graph.facebook.com/v21.0';
        
        if (!empty($access_token)) {
            $params['access_token'] = $access_token;
        }

        $url = $base_url . $endpoint;
        $ch = curl_init();

        if ($method === 'GET') {
            $url .= '?' . http_build_query($params);
            curl_setopt($ch, CURLOPT_URL, $url);
        } elseif ($method === 'POST') {
            curl_setopt($ch, CURLOPT_URL, $url);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($params));
        } elseif ($method === 'DELETE') {
            curl_setopt($ch, CURLOPT_URL, $url . '?' . http_build_query($params));
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'DELETE');
        }

        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Accept: application/json',
            'Content-Type: application/x-www-form-urlencoded'
        ]);

        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curl_error = curl_error($ch);
        curl_close($ch);

        if ($response === false) {
            throw new Exception('cURL Error: ' . $curl_error);
        }

        $data = json_decode($response, true);

        if (function_exists('log_facebook_info')) {
            log_facebook_info('api_request', [
                'endpoint' => $endpoint,
                'method' => $method,
                'http_code' => $http_code,
                'response' => $data
            ]);
        }

        if ($http_code >= 400 && function_exists('log_facebook_error')) {
            log_facebook_error('api_error', "HTTP {$http_code}: " . ($data['error']['message'] ?? 'Unknown error'));
        }

        return $data;
    }
}

if (!function_exists('facebook_exchange_token')) {
    function facebook_exchange_token($app_id, $app_secret, $short_lived_token)
    {
        try {
            $response = facebook_api_request('/oauth/access_token', [
                'grant_type' => 'fb_exchange_token',
                'client_id' => $app_id,
                'client_secret' => $app_secret,
                'fb_exchange_token' => $short_lived_token
            ]);
            return $response;
        } catch (Exception $e) {
            if (function_exists('log_facebook_error')) {
                log_facebook_error('token_exchange', $e->getMessage());
            }
            return null;
        }
    }
}

if (!function_exists('facebook_get_page_access_token')) {
    function facebook_get_page_access_token($page_id, $user_access_token)
    {
        try {
            $response = facebook_api_request('/' . $page_id, [
                'fields' => 'access_token'
            ], $user_access_token);
            return $response['access_token'] ?? null;
        } catch (Exception $e) {
            if (function_exists('log_facebook_error')) {
                log_facebook_error('get_page_token', $e->getMessage());
            }
            return null;
        }
    }
}

if (!function_exists('facebook_get_lead_forms')) {
    function facebook_get_lead_forms($page_id, $page_access_token)
    {
        try {
            $response = facebook_api_request("/{$page_id}/leadgen_forms", [
                'fields' => 'id,name,status,leads_count'
            ], $page_access_token);
            return $response['data'] ?? [];
        } catch (Exception $e) {
            if (function_exists('log_facebook_error')) {
                log_facebook_error('get_lead_forms', $e->getMessage());
            }
            return [];
        }
    }
}

if (!function_exists('facebook_test_connection')) {
    function facebook_test_connection($page_id, $page_access_token)
    {
        try {
            $response = facebook_api_request("/{$page_id}", [
                'fields' => 'id,name,access_token'
            ], $page_access_token);

            if (isset($response['error'])) {
                return [
                    'success' => false,
                    'message' => $response['error']['message'],
                    'data' => null
                ];
            }

            return [
                'success' => true,
                'message' => 'Connection successful',
                'data' => $response
            ];
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => $e->getMessage(),
                'data' => null
            ];
        }
    }
}

if (!function_exists('log_facebook_info')) {
    function log_facebook_info($context, $message)
    {
        // Multiple fallback methods for path detection
        $log_dir = '';
        
        if (defined('FCPATH')) {
            $log_dir = FCPATH . 'modules/facebook_leads_integration/logs/';
        } elseif (defined('APPPATH')) {
            $log_dir = dirname(APPPATH) . '/modules/facebook_leads_integration/logs/';
        } else {
            $log_dir = dirname(dirname(dirname(__FILE__))) . '/modules/facebook_leads_integration/logs/';
        }
        
        if (!is_dir($log_dir)) {
            @mkdir($log_dir, 0755, true);
        }
        
        $log_file = $log_dir . 'info_' . date('Y-m-d') . '.log';
        $timestamp = date('Y-m-d H:i:s');
        
        if (is_array($message) || is_object($message)) {
            $message = json_encode($message, JSON_PRETTY_PRINT);
        }
        
        $log_entry = "[{$timestamp}] [{$context}] {$message}\n";
        @file_put_contents($log_file, $log_entry, FILE_APPEND);
    }
}

if (!function_exists('log_facebook_error')) {
    function log_facebook_error($context, $message)
    {
        // Multiple fallback methods for path detection
        $log_dir = '';
        
        if (defined('FCPATH')) {
            $log_dir = FCPATH . 'modules/facebook_leads_integration/logs/';
        } elseif (defined('APPPATH')) {
            $log_dir = dirname(APPPATH) . '/modules/facebook_leads_integration/logs/';
        } else {
            $log_dir = dirname(dirname(dirname(__FILE__))) . '/modules/facebook_leads_integration/logs/';
        }
        
        if (!is_dir($log_dir)) {
            @mkdir($log_dir, 0755, true);
        }
        
        $log_file = $log_dir . 'error_' . date('Y-m-d') . '.log';
        $timestamp = date('Y-m-d H:i:s');
        
        if (is_array($message) || is_object($message)) {
            $message = json_encode($message, JSON_PRETTY_PRINT);
        }
        
        $log_entry = "[{$timestamp}] [ERROR] [{$context}] {$message}\n";
        @file_put_contents($log_file, $log_entry, FILE_APPEND);
    }
}

if (!function_exists('facebook_clean_old_logs')) {
    function facebook_clean_old_logs()
    {
        $log_dir = '';
        
        if (defined('FCPATH')) {
            $log_dir = FCPATH . 'modules/facebook_leads_integration/logs/';
        } elseif (defined('APPPATH')) {
            $log_dir = dirname(APPPATH) . '/modules/facebook_leads_integration/logs/';
        } else {
            $log_dir = dirname(dirname(dirname(__FILE__))) . '/modules/facebook_leads_integration/logs/';
        }
        
        if (!is_dir($log_dir)) {
            return;
        }
        
        $files = glob($log_dir . '*.log');
        $now = time();
        
        foreach ($files as $file) {
            if (is_file($file)) {
                if ($now - filemtime($file) >= 30 * 24 * 60 * 60) {
                    @unlink($file);
                }
            }
        }
    }
}

if (!function_exists('facebook_validate_webhook_signature')) {
    function facebook_validate_webhook_signature($payload, $signature, $app_secret)
    {
        if (empty($signature)) {
            return false;
        }
        
        list($algorithm, $provided_hash) = explode('=', $signature, 2);
        
        if ($algorithm !== 'sha256') {
            return false;
        }
        
        $expected_hash = hash_hmac('sha256', $payload, $app_secret);
        return hash_equals($expected_hash, $provided_hash);
    }
}

if (!function_exists('facebook_format_phone')) {
    function facebook_format_phone($phone, $country_code = 'US')
    {
        $phone = preg_replace('/[^0-9]/', '', $phone);
        
        if (!empty($phone) && substr($phone, 0, 1) !== '+') {
            $phone = '+' . $phone;
        }
        
        return $phone;
    }
}

// File loaded successfully
return true;
