<?php
defined('BASEPATH') or define('BASEPATH', true);

// Include the app configuration
require_once(__DIR__ . '/application/config/app-config.php');

// Database configuration
$db_config = array(
    'hostname' => APP_DB_HOSTNAME,
    'username' => APP_DB_USERNAME,
    'password' => APP_DB_PASSWORD,
    'database' => 'crm'  // Your database name
);

// Initialize database connection
$mysqli = new mysqli(
    $db_config['hostname'],
    $db_config['username'],
    $db_config['password'],
    $db_config['database']
);

if ($mysqli->connect_error) {
    die('Connect Error (' . $mysqli->connect_errno . ') ' . $mysqli->connect_error);
}

// Function to get status name
function get_status_name($status_id, $mysqli) {
    $sql = "SELECT name FROM tblleads_status WHERE id = ?";
    $stmt = $mysqli->prepare($sql);
    $stmt->bind_param("i", $status_id);
    $stmt->execute();
    $result = $stmt->get_result();
    if ($row = $result->fetch_assoc()) {
        return $row['name'];
    }
    return 'Unknown';
}

// Function to get source name
function get_source_name($source_id, $mysqli) {
    $sql = "SELECT name FROM tblleads_sources WHERE id = ?";
    $stmt = $mysqli->prepare($sql);
    $stmt->bind_param("i", $source_id);
    $stmt->execute();
    $result = $stmt->get_result();
    if ($row = $result->fetch_assoc()) {
        return $row['name'];
    }
    return 'Unknown';
}

// Function to get country name
function get_country_name($country_id, $mysqli) {
    $sql = "SELECT short_name FROM tblcountries WHERE country_id = ?";
    $stmt = $mysqli->prepare($sql);
    $stmt->bind_param("i", $country_id);
    $stmt->execute();
    $result = $stmt->get_result();
    if ($row = $result->fetch_assoc()) {
        return $row['short_name'];
    }
    return 'Unknown';
}

// Function to get staff name
function get_staff_name($staff_id, $mysqli) {
    $sql = "SELECT CONCAT(firstname, ' ', lastname) as full_name FROM tblstaff WHERE staffid = ?";
    $stmt = $mysqli->prepare($sql);
    $stmt->bind_param("i", $staff_id);
    $stmt->execute();
    $result = $stmt->get_result();
    if ($row = $result->fetch_assoc()) {
        return $row['full_name'];
    }
    return 'Unassigned';
}

// Get lead statistics
function get_lead_statistics($mysqli) {
    $stats = [];
    
    // Total leads
    $sql = "SELECT COUNT(*) as total FROM tblleads";
    $result = $mysqli->query($sql);
    $stats['total'] = $result->fetch_assoc()['total'];

    // Leads by status
    $sql = "SELECT s.name, COUNT(l.id) as count 
            FROM tblleads_status s 
            LEFT JOIN tblleads l ON s.id = l.status 
            GROUP BY s.id";
    $result = $mysqli->query($sql);
    $stats['by_status'] = [];
    while ($row = $result->fetch_assoc()) {
        $stats['by_status'][$row['name']] = $row['count'];
    }

    // Leads with/without phone
    $sql = "SELECT 
                SUM(CASE WHEN phonenumber != '' OR phone_numbers IS NOT NULL THEN 1 ELSE 0 END) as with_phone,
                SUM(CASE WHEN phonenumber = '' AND phone_numbers IS NULL THEN 1 ELSE 0 END) as without_phone
            FROM tblleads";
    $result = $mysqli->query($sql);
    $phone_stats = $result->fetch_assoc();
    $stats['phone_stats'] = $phone_stats;

    // Leads with/without address
    $sql = "SELECT 
                SUM(CASE WHEN address != '' OR city != '' OR state != '' OR country != 0 OR zip != '' THEN 1 ELSE 0 END) as with_address,
                SUM(CASE WHEN address = '' AND city = '' AND state = '' AND (country = 0 OR country IS NULL) AND zip = '' THEN 1 ELSE 0 END) as without_address
            FROM tblleads";
    $result = $mysqli->query($sql);
    $address_stats = $result->fetch_assoc();
    $stats['address_stats'] = $address_stats;

    return $stats;
}

// Get all leads
$sql = "SELECT * FROM tblleads ORDER BY dateadded DESC";
$result = $mysqli->query($sql);

// Get statistics
$stats = get_lead_statistics($mysqli);

// HTML output
?>
<!DOCTYPE html>
<html>
<head>
    <title>Leads Data Check</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; }
        table { border-collapse: collapse; width: 100%; margin-top: 20px; }
        th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
        th { background-color: #f4f4f4; }
        tr:nth-child(even) { background-color: #f9f9f9; }
        .lead-info { margin-bottom: 30px; border-bottom: 2px solid #eee; padding-bottom: 20px; }
        .section-title { background-color: #007bff; color: white; padding: 10px; margin-top: 20px; }
        .error { color: red; }
        .success { color: green; }
        .null-value { color: #999; font-style: italic; }
        .search-box { padding: 10px; margin-bottom: 20px; }
        .search-box input { padding: 5px; width: 300px; }
        .filter-box { margin-bottom: 20px; }
        .filter-box select { padding: 5px; margin-right: 10px; }
        .stats-container { 
            background: #f8f9fa; 
            padding: 20px; 
            margin-bottom: 30px; 
            border-radius: 5px;
        }
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin-top: 20px;
        }
        .stat-card {
            background: white;
            padding: 15px;
            border-radius: 5px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        .stat-card h4 { margin: 0 0 10px 0; color: #007bff; }
    </style>
</head>
<body>
    <h1>Leads Data Check</h1>

    <!-- Statistics Section -->
    <div class="stats-container">
        <h2>Lead Statistics</h2>
        <div class="stats-grid">
            <div class="stat-card">
                <h4>Total Leads</h4>
                <p><?php echo $stats['total']; ?></p>
            </div>
            <div class="stat-card">
                <h4>Phone Numbers</h4>
                <p>With Phone: <?php echo $stats['phone_stats']['with_phone']; ?></p>
                <p>Without Phone: <?php echo $stats['phone_stats']['without_phone']; ?></p>
            </div>
            <div class="stat-card">
                <h4>Address Information</h4>
                <p>With Address: <?php echo $stats['address_stats']['with_address']; ?></p>
                <p>Without Address: <?php echo $stats['address_stats']['without_address']; ?></p>
            </div>
            <div class="stat-card">
                <h4>Status Distribution</h4>
                <?php foreach ($stats['by_status'] as $status => $count): ?>
                    <p><?php echo htmlspecialchars($status) . ': ' . $count; ?></p>
                <?php endforeach; ?>
            </div>
        </div>
    </div>
    
    <!-- Search and Filter Section -->
    <div class="search-box">
        <input type="text" id="searchInput" placeholder="Search leads...">
        <select id="fieldFilter">
            <option value="all">All Fields</option>
            <option value="name">Name</option>
            <option value="email">Email</option>
            <option value="phone">Phone</option>
            <option value="company">Company</option>
            <option value="city">City</option>
        </select>
    </div>

    <?php
    if ($result->num_rows > 0) {
        while ($lead = $result->fetch_assoc()) {
            echo '<div class="lead-info">';
            echo '<h2 class="section-title">Lead ID: ' . htmlspecialchars($lead['id']) . '</h2>';
            
            // Basic Information
            echo '<h3>Basic Information</h3>';
            echo '<table>';
            echo '<tr><td>Name:</td><td>' . htmlspecialchars($lead['name']) . '</td></tr>';
            echo '<tr><td>Title:</td><td>' . ($lead['title'] ? htmlspecialchars($lead['title']) : '<span class="null-value">Not set</span>') . '</td></tr>';
            echo '<tr><td>Company:</td><td>' . ($lead['company'] ? htmlspecialchars($lead['company']) : '<span class="null-value">Not set</span>') . '</td></tr>';
            echo '<tr><td>Description:</td><td>' . ($lead['description'] ? nl2br(htmlspecialchars($lead['description'])) : '<span class="null-value">Not set</span>') . '</td></tr>';
            echo '<tr><td>Email:</td><td>' . ($lead['email'] ? htmlspecialchars($lead['email']) : '<span class="null-value">Not set</span>') . '</td></tr>';
            echo '<tr><td>Website:</td><td>' . ($lead['website'] ? htmlspecialchars($lead['website']) : '<span class="null-value">Not set</span>') . '</td></tr>';
            echo '<tr><td>Phone:</td><td>' . ($lead['phonenumber'] ? htmlspecialchars($lead['phonenumber']) : '<span class="null-value">Not set</span>') . '</td></tr>';
            echo '</table>';

            // Additional Phone Numbers
            if (!empty($lead['phone_numbers'])) {
                echo '<h3>Additional Phone Numbers</h3>';
                echo '<table>';
                $additional_phones = json_decode($lead['phone_numbers'], true);
                if (is_array($additional_phones)) {
                    foreach ($additional_phones as $phone) {
                        echo '<tr><td>Additional Phone:</td><td>' . htmlspecialchars($phone) . '</td></tr>';
                    }
                }
                echo '</table>';
            }

            // Address Information
            echo '<h3>Address Information</h3>';
            echo '<table>';
            echo '<tr><td>Address:</td><td>' . ($lead['address'] ? nl2br(htmlspecialchars($lead['address'])) : '<span class="null-value">Not set</span>') . '</td></tr>';
            echo '<tr><td>City:</td><td>' . ($lead['city'] ? htmlspecialchars($lead['city']) : '<span class="null-value">Not set</span>') . '</td></tr>';
            echo '<tr><td>State:</td><td>' . ($lead['state'] ? htmlspecialchars($lead['state']) : '<span class="null-value">Not set</span>') . '</td></tr>';
            echo '<tr><td>Country:</td><td>' . ($lead['country'] ? get_country_name($lead['country'], $mysqli) : '<span class="null-value">Not set</span>') . '</td></tr>';
            echo '<tr><td>Zip:</td><td>' . ($lead['zip'] ? htmlspecialchars($lead['zip']) : '<span class="null-value">Not set</span>') . '</td></tr>';
            echo '</table>';

            // Status and Source Information
            echo '<h3>Status and Source Information</h3>';
            echo '<table>';
            echo '<tr><td>Status:</td><td>' . get_status_name($lead['status'], $mysqli) . '</td></tr>';
            echo '<tr><td>Source:</td><td>' . get_source_name($lead['source'], $mysqli) . '</td></tr>';
            echo '<tr><td>Assigned To:</td><td>' . get_staff_name($lead['assigned'], $mysqli) . '</td></tr>';
            echo '<tr><td>Date Added:</td><td>' . date('Y-m-d H:i:s', strtotime($lead['dateadded'])) . '</td></tr>';
            echo '<tr><td>Last Contact:</td><td>' . ($lead['lastcontact'] ? date('Y-m-d H:i:s', strtotime($lead['lastcontact'])) : '<span class="null-value">Never</span>') . '</td></tr>';
            echo '<tr><td>Lead Value:</td><td>' . ($lead['lead_value'] ? htmlspecialchars($lead['lead_value']) : '<span class="null-value">Not set</span>') . '</td></tr>';
            echo '</table>';

            // Status Flags
            echo '<h3>Status Flags</h3>';
            echo '<table>';
            echo '<tr><td>Is Public:</td><td>' . ($lead['is_public'] ? '<span class="success">Yes</span>' : '<span class="error">No</span>') . '</td></tr>';
            echo '<tr><td>Is Junk:</td><td>' . ($lead['junk'] ? '<span class="error">Yes</span>' : '<span class="success">No</span>') . '</td></tr>';
            echo '<tr><td>Is Lost:</td><td>' . ($lead['lost'] ? '<span class="error">Yes</span>' : '<span class="success">No</span>') . '</td></tr>';
            echo '</table>';

            echo '</div>';
        }
    } else {
        echo '<p>No leads found in the database.</p>';
    }
    ?>

    <script>
    // Search functionality
    document.getElementById('searchInput').addEventListener('keyup', function() {
        var searchText = this.value.toLowerCase();
        var searchField = document.getElementById('fieldFilter').value;
        var leads = document.getElementsByClassName('lead-info');

        Array.from(leads).forEach(function(lead) {
            var text = lead.textContent.toLowerCase();
            var show = false;

            if (searchField === 'all') {
                show = text.includes(searchText);
            } else {
                var table = lead.querySelector('table');
                var rows = table.getElementsByTagName('tr');
                for (var row of rows) {
                    var label = row.cells[0].textContent.toLowerCase();
                    if (label.includes(searchField.toLowerCase())) {
                        var value = row.cells[1].textContent.toLowerCase();
                        if (value.includes(searchText)) {
                            show = true;
                            break;
                        }
                    }
                }
            }

            lead.style.display = show ? '' : 'none';
        });
    });

    // Field filter functionality
    document.getElementById('fieldFilter').addEventListener('change', function() {
        document.getElementById('searchInput').dispatchEvent(new Event('keyup'));
    });
    </script>
</body>
</html>
<?php
$mysqli->close();
?>
