/**
 * Reminder Sound Generator - AUTOMATIC PLAYBACK VERSION
 * Creates audio notifications for reminder alerts
 * Prioritizes automatic playback without user interaction requirements
 * Uses HTML5 audio and optimized techniques for browser compatibility
 */

// Global flag to track if user has interacted with the page (for enhanced features)
let userHasInteracted = false;

// Audio context for enhanced sounds (only created after user interaction)
let audioContext = null;

/**
 * Initialize user interaction detection for enhanced features (optional)
 */
function initUserInteractionDetection() {
    if (userHasInteracted) return;
    
    const events = ['click', 'touchstart', 'keydown', 'mousedown'];
    const handleFirstInteraction = () => {
        userHasInteracted = true;
        console.log('User interaction detected - enhanced audio features available');
        
        // Remove event listeners
        events.forEach(event => {
            document.removeEventListener(event, handleFirstInteraction, true);
        });
        
        // Initialize AudioContext for enhanced sounds (optional)
        initAudioContextSafely();
    };
    
    events.forEach(event => {
        document.addEventListener(event, handleFirstInteraction, true);
    });
}

/**
 * Initialize audio context safely (only after user interaction, for enhanced features)
 */
function initAudioContextSafely() {
    if (!audioContext && userHasInteracted) {
        try {
            audioContext = new (window.AudioContext || window.webkitAudioContext)();
            console.log('AudioContext initialized for enhanced audio features');
        } catch (e) {
            console.log('AudioContext not supported, using HTML5 audio only');
            audioContext = null;
        }
    }
    return audioContext && audioContext.state === 'running';
}

/**
 * Create simple beep using HTML5 audio with data URI (works automatically)
 */
function playSimpleBeep(volume = 0.3) {
    try {
        // Use a simple beep data URI that works without user interaction in most browsers
        const beepDataUri = 'data:audio/wav;base64,UklGRnoGAABXQVZFZm10IBAAAAABAAEAQB8AAEAfAAABAAgAZGF0YQoGAACBhYqFbF1fdJivrJBhNjVgodDbq2EcBj+a2/LDciUFLIHO8tiJNwgZaLvt559NEAxQp+PwtmMcBjiR1/LMeSwFJHfH8N2QQAoUXrTp66hVFApGn+DyvmEaAi2Fzfa/cCIBHnfF89OZSA0PVqzn77BdGAhAl+L0txQELHfL9cN1KgQkf8X113pQDQlZrOf0uHUhBUCS3vK9eCMFJHzJ9N2QQAoUXrTp66hVFApGn+DyvmEaAi2Fzfa/cCIBHnfF89OZSA0PVqzn77BdGAhAl+L0txQELHfL9cN1KgQkf8X113pQDQlZrOf0uHUhBUCS3vK9eCMFJHzJ9N2QQAoUXrTp66hVFApGn+DyvmEaAi2Fzfa/cCIBHnfF89OZSA0PVqzn77BdGAhAl+L0txQELHfL9cN1KgQkf8X113pQDQlZrOf0uHUhBUCS3vK9eCMFJHzJ9N2QQAoUXrTp66hVFApGn+DyvmEaAi2Fzfa/cCIBHnfF89OZSA0PVqzn77BdGAhAl+L0txQELHfL9cN1KgQkf8X113pQDQlZrOf0uHUhBUCS3vK9eCMFJHzJ9N2QQAoUXrTp66hVFApGn+DyvmEaAi2Fzfa/cCIBHnfF89OZSA0PVqzn77BdGAhAl+L0';
        
        const audio = new Audio(beepDataUri);
        audio.volume = Math.min(volume, 0.5);
        
        // Attempt to play immediately
        const playPromise = audio.play();
        if (playPromise !== undefined) {
            playPromise.then(() => {
                console.log('Simple beep played successfully');
            }).catch(error => {
                // Fallback to visual notification if audio fails
                console.log('Audio playback requires user interaction first');
                showVisualNotification();
            });
        }
        return true;
    } catch (error) {
        console.log('Simple beep failed:', error.message);
        showVisualNotification();
        return false;
    }
}

/**
 * Create and play dynamic tone using Web Audio API (when user has interacted)
 */
function playWebAudioTone(frequency = 800, duration = 200, volume = 0.3) {
    if (!userHasInteracted || !initAudioContextSafely()) {
        return false;
    }
    
    try {
        // Create oscillator for the beep sound
        const oscillator = audioContext.createOscillator();
        const gainNode = audioContext.createGain();
        
        // Connect nodes
        oscillator.connect(gainNode);
        gainNode.connect(audioContext.destination);
        
        // Configure sound
        oscillator.frequency.value = frequency;
        oscillator.type = 'sine';
        
        // Configure volume envelope
        gainNode.gain.setValueAtTime(0, audioContext.currentTime);
        gainNode.gain.linearRampToValueAtTime(volume, audioContext.currentTime + 0.01);
        gainNode.gain.exponentialRampToValueAtTime(0.001, audioContext.currentTime + duration / 1000);
        
        // Play sound
        oscillator.start(audioContext.currentTime);
        oscillator.stop(audioContext.currentTime + duration / 1000);
        
        console.log('Web Audio tone played successfully');
        return true;
    } catch (e) {
        console.log('Web Audio tone failed:', e.message);
        return false;
    }
}

/**
 * Show visual notification as fallback when audio can't play
 */
function showVisualNotification() {
    // Create a brief visual flash as audio fallback
    const flash = document.createElement('div');
    flash.style.cssText = `
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background-color: rgba(255, 165, 0, 0.3);
        z-index: 9999;
        pointer-events: none;
        animation: reminderFlash 0.3s ease-out;
    `;
    
    // Add flash animation
    if (!document.querySelector('#reminderFlashStyle')) {
        const style = document.createElement('style');
        style.id = 'reminderFlashStyle';
        style.textContent = `
            @keyframes reminderFlash {
                0% { opacity: 0; }
                50% { opacity: 1; }
                100% { opacity: 0; }
            }
        `;
        document.head.appendChild(style);
    }
    
    document.body.appendChild(flash);
    setTimeout(() => {
        if (flash.parentNode) {
            flash.parentNode.removeChild(flash);
        }
    }, 300);
    
    console.log('Visual reminder notification shown');
}

/**
 * Play reminder notification sound - AUTOMATIC VERSION
 * @param {number} frequency - Sound frequency (default: 800)
 * @param {number} duration - Sound duration in milliseconds (default: 200)
 * @param {number} volume - Sound volume 0-1 (default: 0.3)
 */
function playReminderSound(frequency = 800, duration = 200, volume = 0.3) {
    // Try Web Audio API first if user has interacted (best quality)
    if (playWebAudioTone(frequency, duration, volume)) {
        return;
    }
    
    // Fallback to simple beep (works automatically in most cases)
    if (playSimpleBeep(volume)) {
        return;
    }
    
    // Final fallback: visual notification
    showVisualNotification();
    console.log('Reminder sound attempted (frequency: ' + frequency + 'Hz)');
}

/**
 * Play multiple beeps for urgent reminders
 * @param {number} count - Number of beeps
 * @param {number} interval - Interval between beeps in milliseconds
 */
function playUrgentReminderSound(count = 3, interval = 300) {
    for (let i = 0; i < count; i++) {
        setTimeout(() => {
            playReminderSound(900, 150, 0.4);
        }, i * interval);
    }
}

/**
 * Play a gentle notification sound
 */
function playGentleReminderSound() {
    playReminderSound(600, 200, 0.2);
    setTimeout(() => {
        playReminderSound(800, 200, 0.2);
    }, 250);
}

/**
 * Play a bell notification sound
 */
function playBellReminderSound() {
    playReminderSound(523.25, 300, 0.25); // C5
    setTimeout(() => {
        playReminderSound(659.25, 400, 0.2); // E5
    }, 100);
    setTimeout(() => {
        playReminderSound(783.99, 500, 0.15); // G5
    }, 200);
}

/**
 * Play a ding notification sound
 */
function playDingReminderSound() {
    playReminderSound(1000, 100, 0.3);
    setTimeout(() => {
        playReminderSound(800, 150, 0.2);
    }, 120);
}

/**
 * Play standard reminder sound (alias for playReminderSound)
 */
function playStandardReminderSound() {
    playReminderSound();
}

/**
 * Play custom uploaded reminder sound - AUTOMATIC VERSION
 */
function playCustomReminderSound() {
    // Get custom sound file from global settings
    var customSoundFile = window.custom_reminder_sound_file || null;
    
    if (!customSoundFile) {
        console.warn('No custom sound file configured, falling back to standard sound');
        playReminderSound();
        return;
    }

    try {
        // Create audio element and play custom sound
        var audio = new Audio();
        audio.volume = 0.3;
        audio.src = (window.site_url || '') + 'uploads/custom_sounds/' + customSoundFile;
        
        // Error handling
        audio.onerror = function(e) {
            var errorMessage = 'Failed to load custom sound file';
            if (e.target && e.target.error) {
                var error = e.target.error;
                switch(error.code) {
                    case error.MEDIA_ERR_ABORTED:
                        errorMessage = 'Custom sound playback was aborted';
                        break;
                    case error.MEDIA_ERR_NETWORK:
                        errorMessage = 'Custom sound network error';
                        break;
                    case error.MEDIA_ERR_DECODE:
                        errorMessage = 'Custom sound format/decode error';
                        break;
                    case error.MEDIA_ERR_SRC_NOT_SUPPORTED:
                        errorMessage = 'Custom sound format not supported';
                        break;
                    default:
                        errorMessage = 'Custom sound error (code: ' + error.code + ')';
                }
            }
            console.warn(errorMessage + ', falling back to standard sound');
            playReminderSound();
        };
        
        // Attempt to play immediately (automatic)
        const playPromise = audio.play();
        if (playPromise !== undefined) {
            playPromise.then(() => {
                console.log('Custom sound played successfully');
            }).catch(function(error) {
                console.log('Custom sound requires user interaction, falling back');
                playReminderSound();
            });
        }
        
    } catch (error) {
        console.warn('Error with custom sound playback: ' + error.message + ', falling back to standard sound');
        playReminderSound();
    }
}

/**
 * Play reminder sound based on type - AUTOMATIC VERSION
 * @param {string} soundType - Type of sound to play (gentle, standard, urgent, bell, ding, none, custom)
 */
function playReminderSoundByType(soundType = 'standard') {
    console.log('Playing reminder sound type:', soundType);
    
    switch(soundType) {
        case 'gentle':
            playGentleReminderSound();
            break;
        case 'urgent':
            playUrgentReminderSound();
            break;
        case 'bell':
            playBellReminderSound();
            break;
        case 'ding':
            playDingReminderSound();
            break;
        case 'custom':
            playCustomReminderSound();
            break;
        case 'none':
            console.log('Sound disabled');
            break;
        case 'standard':
        default:
            playReminderSound();
            break;
    }
}

// Initialize user interaction detection when script loads (for enhanced features)
initUserInteractionDetection();

// Make functions globally available (for direct script includes)
window.playReminderSound = playReminderSound;
window.playStandardReminderSound = playStandardReminderSound;
window.playUrgentReminderSound = playUrgentReminderSound;
window.playGentleReminderSound = playGentleReminderSound;
window.playBellReminderSound = playBellReminderSound;
window.playDingReminderSound = playDingReminderSound;
window.playCustomReminderSound = playCustomReminderSound;
window.playReminderSoundByType = playReminderSoundByType;
window.initUserInteractionDetection = initUserInteractionDetection;

// Export functions if using modules
if (typeof module !== 'undefined' && module.exports) {
    module.exports = {
        playReminderSound,
        playStandardReminderSound,
        playUrgentReminderSound,
        playGentleReminderSound,
        playBellReminderSound,
        playDingReminderSound,
        playCustomReminderSound,
        playReminderSoundByType,
        initUserInteractionDetection
    };
}
