<?php
defined('BASEPATH') or exit('No direct script access allowed');

class Test_reminder extends AdminController
{
    public function __construct()
    {
        parent::__construct();
        $this->load->model('leads_model');
        $this->load->model('misc_model');
    }

    public function index()
    {
        if (!is_staff_member()) {
            echo "Access denied - not logged in as staff";
            return;
        }

        echo "<h2>Test Follow-up Reminder Functionality</h2>";

        // Test creating a reminder manually
        $test_lead_id = 44; // Use an existing lead
        $future_date = date('Y-m-d H:i:s', strtotime('+1 hour')); // 1 hour from now

        echo "<h3>Test 1: Check if lead exists</h3>";
        $lead = $this->leads_model->get($test_lead_id);
        if ($lead) {
            echo "<p style='color: green;'>✓ Lead found: " . htmlspecialchars($lead->name) . "</p>";
        } else {
            echo "<p style='color: red;'>✗ Lead not found</p>";
            return;
        }

        echo "<h3>Test 2: Create reminder manually</h3>";
        $reminder_data = [
            'description' => 'Test follow-up reminder for lead: ' . $lead->name,
            'date' => $future_date,
            'staff' => get_staff_user_id(),
            'rel_id' => $test_lead_id,
            'rel_type' => 'lead',
            'notify_by_email' => 1,
        ];

        echo "<p><strong>Reminder data:</strong></p>";
        echo "<pre>" . print_r($reminder_data, true) . "</pre>";

        $result = $this->misc_model->add_reminder($reminder_data, $test_lead_id);
        if ($result) {
            echo "<p style='color: green;'>✓ Reminder created successfully!</p>";
        } else {
            echo "<p style='color: red;'>✗ Failed to create reminder</p>";
        }

        echo "<h3>Test 3: Check existing reminders for this lead</h3>";
        $this->db->select('id, description, date, staff, isnotified');
        $this->db->where('rel_id', $test_lead_id);
        $this->db->where('rel_type', 'lead');
        $this->db->order_by('date', 'DESC');
        $this->db->limit(5);
        $reminders = $this->db->get(db_prefix() . 'reminders')->result();

        if ($reminders) {
            echo "<table border='1' cellpadding='5'>";
            echo "<tr><th>ID</th><th>Description</th><th>Date</th><th>Staff</th><th>Notified</th></tr>";
            foreach ($reminders as $reminder) {
                echo "<tr>";
                echo "<td>" . $reminder->id . "</td>";
                echo "<td>" . htmlspecialchars(strip_tags($reminder->description)) . "</td>";
                echo "<td>" . $reminder->date . "</td>";
                echo "<td>" . $reminder->staff . "</td>";
                echo "<td>" . ($reminder->isnotified ? 'Yes' : 'No') . "</td>";
                echo "</tr>";
            }
            echo "</table>";
        } else {
            echo "<p>No reminders found for this lead</p>";
        }

        echo "<h3>Test 4: Test the automatic reminder function</h3>";
        $test_description = 'Test note for automatic reminder creation';
        $test_date = date('Y-m-d H:i:s', strtotime('+2 hours'));
        
        // Simulate the exact process that happens in add_note
        echo "<h4>4a. Test data preparation</h4>";
        $test_note_data = [
            'description' => $test_description,
            'next_followup_date' => $test_date,
            'create_reminder' => '1'
        ];
        echo "<pre>Test note data: " . print_r($test_note_data, true) . "</pre>";
        
        // Test the _create_automatic_followup_reminder logic manually
        echo "<h4>4b. Manual reminder creation test</h4>";
        try {
            // Get lead information for better reminder description
            $reminder_description = 'Follow-up reminder for lead: ' . ($lead ? $lead->name : 'Unknown Lead');
            if (!empty($test_description)) {
                $clean_description = strip_tags($test_description);
                if (strlen($clean_description) > 100) {
                    $clean_description = substr($clean_description, 0, 100) . '...';
                }
                $reminder_description .= "\n\nNote: " . $clean_description;
            }
            
            // Prepare reminder data exactly as in the controller
            $reminder_data_exact = [
                'description' => $reminder_description,
                'date' => $test_date,
                'staff' => get_staff_user_id(),
                'rel_id' => $test_lead_id,
                'rel_type' => 'lead',
                'notify_by_email' => 1,
            ];
            
            echo "<pre>Exact reminder data: " . print_r($reminder_data_exact, true) . "</pre>";
            
            $auto_result = $this->misc_model->add_reminder($reminder_data_exact, $test_lead_id);
            if ($auto_result) {
                echo "<p style='color: green;'>✓ Manual automatic reminder creation successful!</p>";
            } else {
                echo "<p style='color: red;'>✗ Manual automatic reminder creation failed</p>";
                echo "<p>Database error: " . print_r($this->db->error(), true) . "</p>";
            }
        } catch (Exception $e) {
            echo "<p style='color: red;'>✗ Exception: " . $e->getMessage() . "</p>";
        }
        
        echo "<h4>4c. Test actual add_note functionality</h4>";
        echo "<p><strong>Important:</strong> This tests if the controller method conditions work correctly.</p>";
        
        // Check if the conditions in add_note would trigger reminder creation
        $would_create_reminder = false;
        if (isset($test_note_data['next_followup_date']) && !empty($test_note_data['next_followup_date']) && 
            isset($test_note_data['create_reminder']) && $test_note_data['create_reminder'] == '1') {
            $would_create_reminder = true;
        }
        
        if ($would_create_reminder) {
            echo "<p style='color: green;'>✓ Controller conditions would trigger reminder creation</p>";
        } else {
            echo "<p style='color: red;'>✗ Controller conditions would NOT trigger reminder creation</p>";
            echo "<p>Check: next_followup_date = '" . ($test_note_data['next_followup_date'] ?? 'not set') . "'</p>";
            echo "<p>Check: create_reminder = '" . ($test_note_data['create_reminder'] ?? 'not set') . "'</p>";
        }

        echo "<h3>Instructions</h3>";
        echo "<ol>";
        echo "<li>Go to <a href='" . admin_url('leads/lead/' . $test_lead_id) . "'>Lead " . $test_lead_id . "</a></li>";
        echo "<li>Add a note with a future follow-up date</li>";
        echo "<li>Check the 'Create automatic reminder' checkbox</li>";
        echo "<li>Submit the note</li>";
        echo "<li>Go to the Reminders tab to see the automatic reminder</li>";
        echo "<li>Or check <a href='" . admin_url('misc/reminders') . "'>All Reminders</a></li>";
        echo "</ol>";
        
        echo "<h3>Debug Information</h3>";
        echo "<h4>Controller File Check</h4>";
        $controller_file = APPPATH . 'controllers/admin/Leads.php';
        if (file_exists($controller_file)) {
            echo "<p style='color: green;'>✓ Leads controller file exists</p>";
            $controller_content = file_get_contents($controller_file);
            if (strpos($controller_content, '_create_automatic_followup_reminder') !== false) {
                echo "<p style='color: green;'>✓ _create_automatic_followup_reminder method found in controller</p>";
            } else {
                echo "<p style='color: red;'>✗ _create_automatic_followup_reminder method NOT found in controller</p>";
            }
            if (strpos($controller_content, 'create_reminder') !== false) {
                echo "<p style='color: green;'>✓ create_reminder checkbox handling found</p>";
            } else {
                echo "<p style='color: red;'>✗ create_reminder checkbox handling NOT found</p>";
            }
        } else {
            echo "<p style='color: red;'>✗ Leads controller file not found</p>";
        }
        
        echo "<h4>Database Schema Check</h4>";
        $schema_check = $this->db->query("SHOW COLUMNS FROM " . db_prefix() . "lead_notes LIKE 'next_followup_date'");
        if ($schema_check->num_rows() > 0) {
            echo "<p style='color: green;'>✓ next_followup_date column exists in lead_notes table</p>";
        } else {
            echo "<p style='color: red;'>✗ next_followup_date column missing from lead_notes table</p>";
        }
        
        echo "<h4>Recent Error Logs</h4>";
        $log_file = APPPATH . '../application/logs/log-' . date('Y-m-d') . '.php';
        if (file_exists($log_file)) {
            $log_content = file_get_contents($log_file);
            $lines = explode("\n", $log_content);
            $recent_errors = array_slice($lines, -20); // Last 20 lines
            echo "<pre style='max-height: 200px; overflow-y: scroll; background: #f0f0f0; padding: 10px;'>";
            echo htmlspecialchars(implode("\n", $recent_errors));
            echo "</pre>";
        } else {
            echo "<p>No log file found for today</p>";
        }
    }
}
?>
