<?php
defined('BASEPATH') or exit('No direct script access allowed');

class Test_add_note extends AdminController
{
    public function __construct()
    {
        parent::__construct();
        $this->load->model('leads_model');
        $this->load->model('misc_model');
    }

    public function index()
    {
        if (!is_staff_member()) {
            echo "Access denied - not logged in as staff";
            return;
        }

        echo "<h2>Test Add Note with Automatic Reminder</h2>";

        $test_lead_id = 108;
        $lead = $this->leads_model->get($test_lead_id);
        
        if (!$lead) {
            echo "<p style='color: red;'>Lead $test_lead_id not found</p>";
            return;
        }

        echo "<p><strong>Testing with Lead:</strong> " . htmlspecialchars($lead->name) . " (ID: $test_lead_id)</p>";

        // Simulate the exact POST data that would be sent from the form
        $future_date = date('Y-m-d H:i:s', strtotime('+24 hours'));
        
        echo "<h3>Step 1: Simulate Form Data</h3>";
        $_POST = [
            'description' => 'Test note with automatic reminder creation',
            'next_followup_date' => $future_date,
            'create_reminder' => '1',
            'contacted_indicator' => 'yes',
            'custom_contact_date' => date('Y-m-d H:i:s')
        ];
        
        echo "<pre>Simulated POST data: " . print_r($_POST, true) . "</pre>";

        echo "<h3>Step 2: Process Data (simulating controller logic)</h3>";
        
        // Simulate the data processing from add_note method
        $data = $_POST;
        
        // Handle contact date information
        if (isset($data['contacted_indicator'])) {
            if ($data['contacted_indicator'] == 'yes') {
                $contacted_date = $data['custom_contact_date'];
                if (empty($contacted_date)) {
                    $contacted_date = date('Y-m-d H:i:s');
                }
                $data['date_contacted'] = $contacted_date;
            }
            unset($data['contacted_indicator']);
            unset($data['custom_contact_date']);
        }

        // Handle next follow-up date (simplified version)
        if (isset($data['next_followup_date']) && !empty($data['next_followup_date'])) {
            $next_followup = $data['next_followup_date'];
            // Keep as is since it's already in correct format for test
            $data['next_followup_date'] = $next_followup;
        } else {
            unset($data['next_followup_date']);
        }

        echo "<pre>Processed data: " . print_r($data, true) . "</pre>";

        echo "<h3>Step 3: Insert Note</h3>";
        
        $insert_data = [
            'description' => nl2br($data['description']),
            'leadid' => $test_lead_id,
            'addedfrom' => get_staff_user_id(),
            'dateadded' => date('Y-m-d H:i:s')
        ];
        
        // Add date_contacted if provided
        if (isset($data['date_contacted'])) {
            $insert_data['date_contacted'] = $data['date_contacted'];
        }
        
        // Add next_followup_date if provided
        if (isset($data['next_followup_date'])) {
            $insert_data['next_followup_date'] = $data['next_followup_date'];
        }
        
        echo "<pre>Insert data: " . print_r($insert_data, true) . "</pre>";

        // Insert into lead_notes table
        $this->db->insert(db_prefix() . 'lead_notes', $insert_data);
        $note_id = $this->db->insert_id();
        
        if ($note_id) {
            echo "<p style='color: green;'>✓ Note inserted successfully with ID: $note_id</p>";
            
            // Update lastcontact if date_contacted is set
            if (isset($insert_data['date_contacted'])) {
                $this->db->where('id', $test_lead_id);
                $this->db->update(db_prefix() . 'leads', [
                    'lastcontact' => $insert_data['date_contacted']
                ]);
                echo "<p style='color: green;'>✓ Last contact updated</p>";
            }
            
        } else {
            echo "<p style='color: red;'>✗ Failed to insert note</p>";
            echo "<p>Error: " . print_r($this->db->error(), true) . "</p>";
            return;
        }

        echo "<h3>Step 4: Test Automatic Reminder Creation</h3>";
        
        // Check if reminder should be created
        $should_create_reminder = isset($data['next_followup_date']) && !empty($data['next_followup_date']) && 
                                 isset($data['create_reminder']) && $data['create_reminder'] == '1';
        
        echo "<p><strong>Should create reminder:</strong> " . ($should_create_reminder ? 'YES' : 'NO') . "</p>";
        
        if ($should_create_reminder) {
            // Manual reminder creation using the exact logic from the controller
            try {
                $reminder_description = 'Follow-up reminder for lead: ' . $lead->name;
                if (!empty($data['description'])) {
                    $clean_description = strip_tags($data['description']);
                    if (strlen($clean_description) > 100) {
                        $clean_description = substr($clean_description, 0, 100) . '...';
                    }
                    $reminder_description .= "\n\nNote: " . $clean_description;
                }
                
                $reminder_data = [
                    'description' => $reminder_description,
                    'date' => $data['next_followup_date'],
                    'staff' => get_staff_user_id(),
                    'rel_id' => $test_lead_id,
                    'rel_type' => 'lead',
                    'notify_by_email' => 1,
                ];
                
                echo "<pre>Reminder data: " . print_r($reminder_data, true) . "</pre>";
                
                $result = $this->misc_model->add_reminder($reminder_data, $test_lead_id);
                
                if ($result) {
                    echo "<p style='color: green;'>✓ Automatic reminder created successfully!</p>";
                } else {
                    echo "<p style='color: red;'>✗ Failed to create automatic reminder</p>";
                    echo "<p>Error: " . print_r($this->db->error(), true) . "</p>";
                }
                
            } catch (Exception $e) {
                echo "<p style='color: red;'>✗ Exception creating reminder: " . $e->getMessage() . "</p>";
            }
        }

        echo "<h3>Step 5: Verify Results</h3>";
        
        // Check the inserted note
        $this->db->select('*');
        $this->db->where('id', $note_id);
        $inserted_note = $this->db->get(db_prefix() . 'lead_notes')->row();
        
        if ($inserted_note) {
            echo "<h4>Inserted Note:</h4>";
            echo "<table border='1' cellpadding='5'>";
            echo "<tr><th>Field</th><th>Value</th></tr>";
            foreach ($inserted_note as $field => $value) {
                echo "<tr><td>$field</td><td>" . htmlspecialchars($value ?: 'NULL') . "</td></tr>";
            }
            echo "</table>";
        }
        
        // Check created reminders
        $this->db->select('*');
        $this->db->where('rel_id', $test_lead_id);
        $this->db->where('rel_type', 'lead');
        $this->db->order_by('id', 'DESC');
        $this->db->limit(3);
        $recent_reminders = $this->db->get(db_prefix() . 'reminders')->result();
        
        if ($recent_reminders) {
            echo "<h4>Recent Reminders for this Lead:</h4>";
            echo "<table border='1' cellpadding='5'>";
            echo "<tr><th>ID</th><th>Description</th><th>Date</th><th>Staff</th><th>Created</th></tr>";
            foreach ($recent_reminders as $reminder) {
                echo "<tr>";
                echo "<td>" . $reminder->id . "</td>";
                echo "<td>" . htmlspecialchars(strip_tags($reminder->description)) . "</td>";
                echo "<td>" . $reminder->date . "</td>";
                echo "<td>" . $reminder->staff . "</td>";
                echo "<td>" . ($reminder->creator ?? 'N/A') . "</td>";
                echo "</tr>";
            }
            echo "</table>";
        } else {
            echo "<p>No reminders found for this lead</p>";
        }

        echo "<h3>Conclusion</h3>";
        echo "<p>This test simulates exactly what happens when you submit the lead note form.</p>";
        echo "<p>If the reminder was created here but not through the actual form, there might be:</p>";
        echo "<ul>";
        echo "<li>JavaScript issues preventing form submission</li>";
        echo "<li>CSRF token issues</li>";
        echo "<li>Missing form fields</li>";
        echo "<li>Controller file not updated on server</li>";
        echo "</ul>";
        
        // Clean up POST data
        $_POST = [];
    }
}
?>
