<?php
defined('BASEPATH') or exit('No direct script access allowed');

class Reporting_dashboard_permissions extends AdminController
{
    public function __construct()
    {
        parent::__construct();
        
        // Enable error reporting and logging
        error_reporting(E_ALL);
        ini_set('display_errors', 1);
        ini_set('log_errors', 1);
        
        $this->load->model('reporting_dashboard/reporting_dashboard_permissions_model');
        $this->load->model('staff_model');
        
        // Only admin can access permission management
        if (!is_admin()) {
            access_denied('Reporting Dashboard Permissions');
        }
    }

    /**
     * Permission management page
     */
    public function index()
    {
        $data['title'] = 'Reporting Dashboard - Permissions Management';
        $data['staff_permissions'] = $this->reporting_dashboard_permissions_model->get_all_staff_permissions();
        $data['available_staff'] = $this->reporting_dashboard_permissions_model->get_available_staff_for_permission();
        $this->load->view('permissions_management', $data);
    }

    /**
     * Get available staff for specific permission assignment via AJAX
     */
    public function get_available_staff()
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $exclude_staff_id = $this->input->get('exclude_staff_id');
        $available_staff = $this->reporting_dashboard_permissions_model->get_available_staff_for_permission($exclude_staff_id);
        
        echo json_encode([
            'success' => true,
            'staff' => $available_staff
        ]);
    }

    /**
     * Update staff permission via AJAX
     */
    public function update_permission()
    {
        // Start output buffering
        ob_start();
        
        // Set headers for JSON response
        header('Content-Type: application/json');
        
        if (!$this->input->is_ajax_request()) {
            ob_end_clean();
            show_404();
            return;
        }

        try {
            $staff_id = $this->input->post('staff_id');
            $permission_type = $this->input->post('permission_type');
            $allowed_staff_ids = $this->input->post('allowed_staff_ids');
            $can_generate_reports = (bool)$this->input->post('can_generate_reports');
            $current_user_id = get_staff_user_id();

            // Log the attempt
            log_activity('Permission update attempt for staff ' . $staff_id . ' by user ' . $current_user_id);

            // Validate inputs
            if (empty($staff_id) || empty($permission_type)) {
                log_activity('Permission update failed: Invalid parameters');
                
                ob_end_clean();
                echo json_encode([
                    'success' => false,
                    'message' => 'Invalid parameters - Staff ID and permission type are required'
                ]);
                return;
            }

            // Validate staff_id is numeric
            if (!is_numeric($staff_id) || $staff_id <= 0) {
                ob_end_clean();
                echo json_encode([
                    'success' => false,
                    'message' => 'Invalid staff ID'
                ]);
                return;
            }

            // Validate permission type
            if (!in_array($permission_type, ['view_all_staff', 'view_own_only', 'view_specific_staff'])) {
                log_activity('Permission update failed: Invalid permission type - ' . $permission_type);
                
                ob_end_clean();
                echo json_encode([
                    'success' => false,
                    'message' => 'Invalid permission type: ' . $permission_type
                ]);
                return;
            }

            // For view_specific_staff, validate allowed_staff_ids
            $allowed_staff_array = [];
            if ($permission_type === 'view_specific_staff') {
                if (empty($allowed_staff_ids)) {
                    log_activity('Permission update failed: No staff selected for specific staff permission');
                    
                    ob_end_clean();
                    echo json_encode([
                        'success' => false,
                        'message' => 'Please select at least one staff member for specific staff permission'
                    ]);
                    return;
                }
                
                $allowed_staff_array = is_array($allowed_staff_ids) ? $allowed_staff_ids : explode(',', $allowed_staff_ids);
                $allowed_staff_array = array_filter(array_map('intval', $allowed_staff_array));
                
                if (empty($allowed_staff_array)) {
                    log_activity('Permission update failed: Invalid staff selection');
                    
                    ob_end_clean();
                    echo json_encode([
                        'success' => false,
                        'message' => 'Invalid staff selection'
                    ]);
                    return;
                }
            }

            // Update permission
            $success = $this->reporting_dashboard_permissions_model->set_staff_permission(
                $staff_id,
                $permission_type,
                $current_user_id,
                $allowed_staff_array,
                $can_generate_reports
            );

            ob_end_clean();
            
            if ($success) {
                log_activity('Permission updated successfully for staff ' . $staff_id . ' to ' . $permission_type);
                echo json_encode([
                    'success' => true,
                    'message' => 'Permission updated successfully'
                ]);
            } else {
                log_activity('Permission update failed for staff ' . $staff_id . ' - Database error');
                echo json_encode([
                    'success' => false,
                    'message' => 'Failed to update permission - Database error'
                ]);
            }
        } catch (Exception $e) {
            log_activity('Permission update exception: ' . $e->getMessage());
            
            ob_end_clean();
            echo json_encode([
                'success' => false,
                'message' => 'An error occurred: ' . $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
        }
    }

    /**
     * Remove staff permission (revert to default)
     */
    public function remove_permission()
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $staff_id = $this->input->post('staff_id');

        if (empty($staff_id)) {
            echo json_encode([
                'success' => false,
                'message' => 'Invalid staff ID'
            ]);
            return;
        }

        $success = $this->reporting_dashboard_permissions_model->remove_staff_permission($staff_id);

        if ($success) {
            echo json_encode([
                'success' => true,
                'message' => 'Permission reset to default successfully'
            ]);
        } else {
            echo json_encode([
                'success' => false,
                'message' => 'Failed to reset permission'
            ]);
        }
    }

    /**
     * Bulk update permissions
     */
    public function bulk_update()
    {
        // Start output buffering to catch any errors
        ob_start();
        
        // Set headers for JSON response
        header('Content-Type: application/json');
        
        // Allow both AJAX and direct POST for debugging
        if (!$this->input->is_ajax_request() && !$this->input->post()) {
            ob_end_clean();
            echo json_encode([
                'success' => false,
                'message' => 'Invalid request method'
            ]);
            return;
        }

        try {
            $permissions = $this->input->post('permissions');
            $current_user_id = get_staff_user_id();

            // Log the received data for debugging
            log_activity('Permission bulk update attempt by user: ' . $current_user_id);
            
            if (empty($permissions) || !is_array($permissions)) {
                log_activity('Permission bulk update failed: No permissions provided or not an array. Type: ' . gettype($permissions));
                
                // Clear any output
                ob_end_clean();
                
                echo json_encode([
                    'success' => false,
                    'message' => 'No permissions provided or invalid format'
                ]);
                return;
            }

            $updated_count = 0;
            $errors = [];

            foreach ($permissions as $staff_id => $permission_data) {
                try {
                    $permission_type = isset($permission_data['type']) ? $permission_data['type'] : '';
                    $allowed_staff_ids = isset($permission_data['allowed_staff_ids']) ? $permission_data['allowed_staff_ids'] : [];
                    $can_generate_reports = isset($permission_data['can_generate_reports']) ? (bool)$permission_data['can_generate_reports'] : false;
                    
                    // Validate staff_id
                    if (!is_numeric($staff_id) || $staff_id <= 0) {
                        $errors[] = "Invalid staff ID: $staff_id";
                        continue;
                    }
                    
                    // Validate permission type
                    if (!in_array($permission_type, ['view_all_staff', 'view_own_only', 'view_specific_staff'])) {
                        $errors[] = "Invalid permission type for staff ID $staff_id: $permission_type";
                        continue;
                    }
                    
                    // Validate allowed_staff_ids for specific staff permission
                    if ($permission_type === 'view_specific_staff') {
                        if (empty($allowed_staff_ids)) {
                            $errors[] = "Staff ID $staff_id: Specific staff permission requires at least one staff member to be selected";
                            continue;
                        }
                        // Ensure it's an array
                        if (!is_array($allowed_staff_ids)) {
                            $allowed_staff_ids = explode(',', $allowed_staff_ids);
                        }
                        // Clean and validate IDs
                        $allowed_staff_ids = array_filter(array_map('intval', $allowed_staff_ids));
                        if (empty($allowed_staff_ids)) {
                            $errors[] = "Staff ID $staff_id: Invalid staff selection";
                            continue;
                        }
                    }
                    
                    $success = $this->reporting_dashboard_permissions_model->set_staff_permission(
                        $staff_id,
                        $permission_type,
                        $current_user_id,
                        $allowed_staff_ids,
                        $can_generate_reports
                    );
                    
                    if ($success) {
                        $updated_count++;
                        log_activity('Permission updated for staff ID ' . $staff_id . ' to ' . $permission_type);
                    } else {
                        $errors[] = "Failed to update permission for staff ID: $staff_id (database error)";
                        log_activity('Permission update failed for staff ID ' . $staff_id);
                    }
                } catch (Exception $e) {
                    $errors[] = "Exception updating staff ID $staff_id: " . $e->getMessage();
                    log_activity('Permission update exception for staff ID ' . $staff_id . ': ' . $e->getMessage());
                }
            }

            $response = [
                'success' => empty($errors),
                'message' => empty($errors) ? 
                    "Successfully updated $updated_count permissions" :
                    "Updated $updated_count permissions with " . count($errors) . " errors",
                'errors' => $errors,
                'updated_count' => $updated_count
            ];
            
            log_activity('Permission bulk update completed: ' . $updated_count . ' updated, ' . count($errors) . ' errors');
            
            // Clear any output and send JSON
            ob_end_clean();
            echo json_encode($response);
            
        } catch (Exception $e) {
            log_activity('Permission bulk update fatal error: ' . $e->getMessage());
            
            // Clear any output
            ob_end_clean();
            
            echo json_encode([
                'success' => false,
                'message' => 'An error occurred while updating permissions: ' . $e->getMessage(),
                'errors' => [$e->getMessage()],
                'trace' => $e->getTraceAsString()
            ]);
        }
    }
}
?>
