/**
 * Dark Glass Theme JavaScript
 * Handles interactive effects and animations
 */

(function() {
    'use strict';

    // Initialize theme when DOM is ready
    document.addEventListener('DOMContentLoaded', function() {
        initDarkGlassTheme();
    });

    function initDarkGlassTheme() {
        // Add entrance animations to elements
        addEntranceAnimations();
        
        // Initialize interactive glass effects
        initInteractiveGlass();
        
        // Initialize glass morphism hover effects
        initGlassMorphismEffects();
        
        // Initialize animated background orbs
        initBackgroundOrbs();
        
        // Initialize theme toggle if needed
        initThemeToggle();
        
        // Apply glass effect to existing Perfex elements
        applyGlassToElements();
        
        // Initialize neon glow effects
        initNeonGlow();
        
        // Initialize smooth scroll
        initSmoothScroll();
    }

    /**
     * Add entrance animations to glass elements
     */
    function addEntranceAnimations() {
        const elements = document.querySelectorAll('.panel, .card, .box, .glass-card');
        
        elements.forEach((element, index) => {
            element.style.opacity = '0';
            element.style.transform = 'translateY(30px)';
            
            setTimeout(() => {
                element.style.transition = 'all 0.8s cubic-bezier(0.4, 0, 0.2, 1)';
                element.style.opacity = '1';
                element.style.transform = 'translateY(0)';
            }, index * 60);
        });
    }

    /**
     * Initialize interactive glass effects on mouse move
     */
    function initInteractiveGlass() {
        const glassElements = document.querySelectorAll('.glass-card, .panel, .card');
        
        glassElements.forEach(element => {
            element.addEventListener('mousemove', function(e) {
                const rect = this.getBoundingClientRect();
                const x = ((e.clientX - rect.left) / rect.width) * 100;
                const y = ((e.clientY - rect.top) / rect.height) * 100;
                
                // Create gradient effect following mouse
                this.style.background = `
                    radial-gradient(
                        circle at ${x}% ${y}%,
                        rgba(6, 182, 212, 0.08) 0%,
                        rgba(255, 255, 255, 0.04) 100%
                    )
                `;
            });
            
            element.addEventListener('mouseleave', function() {
                this.style.background = '';
            });
        });
    }

    /**
     * Initialize glass morphism hover effects
     */
    function initGlassMorphismEffects() {
        // Add ripple effect on click
        document.addEventListener('click', function(e) {
            const target = e.target.closest('.btn, .glass-btn');
            if (target) {
                createRipple(e, target);
            }
        });
    }

    /**
     * Create ripple effect
     */
    function createRipple(e, element) {
        const ripple = document.createElement('span');
        ripple.className = 'ripple';
        
        const rect = element.getBoundingClientRect();
        const size = Math.max(rect.width, rect.height);
        const x = e.clientX - rect.left - size / 2;
        const y = e.clientY - rect.top - size / 2;
        
        ripple.style.width = ripple.style.height = size + 'px';
        ripple.style.left = x + 'px';
        ripple.style.top = y + 'px';
        
        // Add ripple styles
        ripple.style.position = 'absolute';
        ripple.style.borderRadius = '50%';
        ripple.style.background = 'rgba(255, 255, 255, 0.3)';
        ripple.style.transform = 'scale(0)';
        ripple.style.animation = 'ripple-animation 0.6s ease-out';
        ripple.style.pointerEvents = 'none';
        
        element.style.position = 'relative';
        element.style.overflow = 'hidden';
        element.appendChild(ripple);
        
        setTimeout(() => {
            ripple.remove();
        }, 600);
    }

    /**
     * Initialize animated background orbs
     */
    function initBackgroundOrbs() {
        // Create additional dynamic orbs
        const orbCount = 3;
        const colors = ['rgba(6, 182, 212, 0.15)', 'rgba(20, 184, 166, 0.15)', 'rgba(251, 146, 60, 0.15)'];
        
        for (let i = 0; i < orbCount; i++) {
            createDynamicOrb(colors[i % colors.length], i);
        }
    }

    /**
     * Create dynamic background orb
     */
    function createDynamicOrb(color, index) {
        const orb = document.createElement('div');
        orb.className = 'dynamic-orb';
        orb.style.position = 'fixed';
        orb.style.width = (300 + Math.random() * 200) + 'px';
        orb.style.height = orb.style.width;
        orb.style.background = `radial-gradient(circle, ${color} 0%, transparent 70%)`;
        orb.style.borderRadius = '50%';
        orb.style.pointerEvents = 'none';
        orb.style.zIndex = '0';
        orb.style.opacity = '0.3';
        
        // Random position
        const positions = [
            { top: '10%', left: '80%' },
            { bottom: '20%', right: '10%' },
            { top: '60%', left: '5%' }
        ];
        
        const pos = positions[index % positions.length];
        Object.assign(orb.style, pos);
        
        // Animate
        orb.style.animation = `float ${15 + index * 3}s ease-in-out infinite ${index % 2 ? 'reverse' : ''}`;
        
        document.body.appendChild(orb);
    }

    /**
     * Initialize theme toggle functionality
     */
    function initThemeToggle() {
        // Check if theme toggle exists or create one
        const existingToggle = document.querySelector('.darkglass-theme-toggle');
        if (!existingToggle && document.querySelector('#header')) {
            createThemeToggle();
        }
    }

    /**
     * Create theme toggle button
     */
    function createThemeToggle() {
        const toggle = document.createElement('div');
        toggle.className = 'darkglass-theme-toggle';
        toggle.innerHTML = `
            <button class="glass-btn theme-toggle-btn" title="Toggle Dark Glass Theme">
                <span class="toggle-icon">🌙</span>
            </button>
        `;
        
        // Style the toggle
        toggle.style.position = 'fixed';
        toggle.style.bottom = '20px';
        toggle.style.right = '20px';
        toggle.style.zIndex = '9999';
        
        const btn = toggle.querySelector('.theme-toggle-btn');
        btn.style.width = '50px';
        btn.style.height = '50px';
        btn.style.borderRadius = '50%';
        btn.style.display = 'flex';
        btn.style.alignItems = 'center';
        btn.style.justifyContent = 'center';
        btn.style.fontSize = '20px';
        btn.style.cursor = 'pointer';
        
        document.body.appendChild(toggle);
        
        // Toggle functionality
        btn.addEventListener('click', function() {
            document.body.classList.toggle('darkglass-theme-active');
            const isActive = document.body.classList.contains('darkglass-theme-active');
            this.querySelector('.toggle-icon').textContent = isActive ? '☀️' : '🌙';
            
            // Save preference
            localStorage.setItem('darkglass-theme', isActive ? 'active' : 'inactive');
        });
    }

    /**
     * Apply glass effect to existing Perfex elements
     */
    function applyGlassToElements() {
        // Add glass classes to specific Perfex elements
        const elements = {
            '.content-wrapper': 'glass-fade-in',
            '.stats-item': 'glass-card glass-scale-in',
            '.widget': 'glass-card',
            '.todo-item': 'glass-interactive',
            '.activity-feed': 'frosted-glass'
        };
        
        Object.keys(elements).forEach(selector => {
            const items = document.querySelectorAll(selector);
            items.forEach(item => {
                const classes = elements[selector].split(' ');
                classes.forEach(className => {
                    item.classList.add(className);
                });
            });
        });
    }

    /**
     * Initialize neon glow effects for important elements
     */
    function initNeonGlow() {
        // Add neon glow to success messages
        const successElements = document.querySelectorAll('.alert-success, .label-success, .badge-success');
        successElements.forEach(el => {
            el.classList.add('glass-neon-success');
        });
        
        // Add neon glow to primary buttons
        const primaryButtons = document.querySelectorAll('.btn-primary, .btn-info');
        primaryButtons.forEach(btn => {
            btn.classList.add('glass-neon');
        });
        
        // Add neon glow to active menu items
        const activeMenuItems = document.querySelectorAll('.nav li.active');
        activeMenuItems.forEach(item => {
            item.classList.add('glass-neon');
        });
    }

    /**
     * Initialize smooth scroll behavior
     */
    function initSmoothScroll() {
        // Add smooth scroll to internal links
        document.querySelectorAll('a[href^="#"]').forEach(anchor => {
            anchor.addEventListener('click', function(e) {
                const targetId = this.getAttribute('href');
                if (targetId && targetId !== '#') {
                    const targetElement = document.querySelector(targetId);
                    if (targetElement) {
                        e.preventDefault();
                        targetElement.scrollIntoView({
                            behavior: 'smooth',
                            block: 'start'
                        });
                    }
                }
            });
        });
    }

    /**
     * Create floating particles effect
     */
    function createParticles() {
        const particleContainer = document.createElement('div');
        particleContainer.className = 'particle-container';
        particleContainer.style.position = 'fixed';
        particleContainer.style.top = '0';
        particleContainer.style.left = '0';
        particleContainer.style.width = '100%';
        particleContainer.style.height = '100%';
        particleContainer.style.pointerEvents = 'none';
        particleContainer.style.zIndex = '1';
        
        for (let i = 0; i < 20; i++) {
            const particle = document.createElement('div');
            particle.className = 'particle';
            particle.style.position = 'absolute';
            particle.style.width = '2px';
            particle.style.height = '2px';
            particle.style.background = 'rgba(6, 182, 212, 0.5)';
            particle.style.borderRadius = '50%';
            particle.style.left = Math.random() * 100 + '%';
            particle.style.top = Math.random() * 100 + '%';
            particle.style.animation = `particle-float ${10 + Math.random() * 20}s linear infinite`;
            particle.style.animationDelay = Math.random() * 5 + 's';
            
            particleContainer.appendChild(particle);
        }
        
        document.body.appendChild(particleContainer);
    }

    /**
     * Check and apply saved theme preference
     */
    function applySavedTheme() {
        const savedTheme = localStorage.getItem('darkglass-theme');
        if (savedTheme === 'inactive') {
            document.body.classList.remove('darkglass-theme-active');
        }
    }

    // Apply saved theme on load
    applySavedTheme();

    // Add CSS for ripple animation
    const style = document.createElement('style');
    style.textContent = `
        @keyframes ripple-animation {
            to {
                transform: scale(4);
                opacity: 0;
            }
        }
        
        @keyframes particle-float {
            from {
                transform: translateY(100vh) translateX(0);
            }
            to {
                transform: translateY(-10vh) translateX(100px);
            }
        }
    `;
    document.head.appendChild(style);

    // Listen for ajax content loads (Perfex uses ajax)
    $(document).ajaxComplete(function() {
        setTimeout(() => {
            addEntranceAnimations();
            applyGlassToElements();
            initNeonGlow();
        }, 100);
    });

    // Optimize performance by throttling mouse events
    function throttle(func, delay) {
        let timeoutId;
        let lastExecTime = 0;
        return function() {
            const context = this;
            const args = arguments;
            const currentTime = Date.now();
            if (currentTime - lastExecTime > delay) {
                func.apply(context, args);
                lastExecTime = currentTime;
            }
        };
    }

    // Apply throttling to mousemove events
    const throttledMouseMove = throttle(function(e) {
        // Handle global mouse effects if needed
    }, 50);

    document.addEventListener('mousemove', throttledMouseMove);

})();
